/* gui.c
 *
 * Copyright 2002-2004 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <gmodule.h>
#include <gnome.h>
#include <glade/glade.h>
#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "cellrendererspin.h"

#define new_column gtk_tree_view_column_new_with_attributes

gboolean gui_quit = FALSE;

/* Return the length of an integer in string representation */
static int intlen(int i)
{
  if (i == 0)
    return 1;

  if (i < 0)
    return (int)log10((double)-i) + 2;

  return (int)log10((double)i) + 1;
}

/* Connects signals to objects and passes the gui struct on as user_data */
void gui_gladexmlconnectfunc(const gchar * handler_name, GObject * object, const gchar * signal_name, const gchar * signal_data, GObject * connect_object, gboolean after, gpointer user_data)
{
  GCallback func;

  /* Use the gmodule interface to resolve the function name to a pointer */
  if (!g_module_symbol(g_module_open(NULL, 0), handler_name, (gpointer *) & func))
    g_warning("could not find signal handler '%s'.", handler_name);

  /* Connect the signal */
  if (after)
    g_signal_connect_after(object, signal_name, func, user_data);
  else
    g_signal_connect(object, signal_name, func, user_data);

};

void gtk_main_quit()
{
  gui_quit = TRUE;
}

/* Initializes a GUI */
struct gui *gui_open(struct editor *editor)
{
  struct gui *gui = (struct gui *)calloc(1, sizeof(struct gui));
  GtkCellRenderer *renderer;
  GtkTreeView *treeview;
  GtkTreeSelection *select;
  GtkListStore *store;
  GdkPixbuf *pixbuf;

  gui->editor = editor;

  /* Mutex for the refreshing function */
  gui->refresh_mutex = g_mutex_new();

  /* Load the GUI from a GLADE file and connect signals */
  gui->xml = glade_xml_new(PACKAGE_DATA_DIR "/glade/tutka.glade", NULL, NULL);
  if (gui->xml == NULL) {
    gui_close(gui);
    return NULL;
  }
  glade_xml_signal_autoconnect_full(gui->xml, gui_gladexmlconnectfunc, gui);
  gui->window_main = glade_xml_get_widget(gui->xml, "window_main");
  gtk_widget_add_events(gui->window_main, GDK_KEY_RELEASE_MASK);

  /* Create the MVC classes for various list view widgets */
  /* Section list */
  store = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_sectionlist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Section", renderer, "text", GUI_SECTIONLIST_COLUMN_SECTION, NULL));
  if (gui->numplayseqs == NULL)
    renderer = gui_cell_renderer_spin_new(1, 1, 1, 2, 2, 1, 0, &gui->numplayseqs);
  else
    renderer = gui_cell_renderer_spin_new(1, *gui->numplayseqs, 1, 2, 2, 1, 0, &gui->numplayseqs);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_sectionlist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence", renderer, "text", GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence Name", renderer, "text", GUI_SECTIONLIST_COLUMN_PLAYSEQ_NAME, NULL));

  /* Playseq List */
  store = gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseqlist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_playseqlist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_PLAYSEQLIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PLAYSEQLIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_playseqlist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_PLAYSEQLIST_COLUMN_NAME, NULL));

  /* Playing Sequence */
  store = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_playseq_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Position", renderer, "text", GUI_PLAYSEQ_COLUMN_POSITION, NULL));
  if (gui->numblocks == NULL)
    renderer = gui_cell_renderer_spin_new(1, 1, 1, 2, 2, 1, 0, &gui->numblocks);
  else
    renderer = gui_cell_renderer_spin_new(1, *gui->numblocks, 1, 2, 2, 1, 0, &gui->numblocks);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_playseq_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Block Number", renderer, "text", GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Block Name", renderer, "text", GUI_PLAYSEQ_COLUMN_BLOCK_NAME, NULL));

  /* Block List */
  store = gtk_list_store_new(5, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT, G_TYPE_INT, G_TYPE_INT);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_blocklist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_BLOCKLIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_BLOCKLIST_COLUMN_NAME, NULL));
  renderer = gui_cell_renderer_spin_new(1, 256, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_TRACKS));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Tracks", renderer, "text", GUI_BLOCKLIST_COLUMN_TRACKS, NULL));
  renderer = gui_cell_renderer_spin_new(1, 4096, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_LENGTH));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Length", renderer, "text", GUI_BLOCKLIST_COLUMN_LENGTH, NULL));
  renderer = gui_cell_renderer_spin_new(1, 64, 1, 2, 2, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_BLOCKLIST_COLUMN_COMMAND_PAGES));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_blocklist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Command pages", renderer, "text", GUI_BLOCKLIST_COLUMN_COMMAND_PAGES, NULL));

  /* Message List */
  store = gtk_list_store_new(4, G_TYPE_INT, G_TYPE_STRING, G_TYPE_INT, G_TYPE_BOOLEAN);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_messagelist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(store));
  renderer = gtk_cell_renderer_text_new();
  gtk_tree_view_append_column(treeview, new_column("Number", renderer, "text", GUI_MESSAGELIST_COLUMN_NUMBER, NULL));
  renderer = gtk_cell_renderer_text_new();
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_NAME));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_messagelist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Name", renderer, "text", GUI_MESSAGELIST_COLUMN_NAME, NULL));
  renderer = gui_cell_renderer_spin_new(0, 2097152, 1, 512, 512, 1, 0, NULL);
  g_object_set(renderer, "editable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_LENGTH));
  g_signal_connect(renderer, "edited", G_CALLBACK(on_cell_messagelist_edited), gui);
  gtk_tree_view_append_column(treeview, new_column("Length", renderer, "text", GUI_MESSAGELIST_COLUMN_LENGTH, NULL));
  renderer = gtk_cell_renderer_toggle_new();
  g_object_set(renderer, "activatable", TRUE, NULL);
  g_object_set_data(G_OBJECT(renderer), "column", GUINT_TO_POINTER(GUI_MESSAGELIST_COLUMN_AUTOSEND));
  g_signal_connect(renderer, "toggled", G_CALLBACK(on_cell_messagelist_toggled), gui);
  gtk_tree_view_append_column(treeview, new_column("Automatically Send Message After Loading Song", renderer, "active", GUI_MESSAGELIST_COLUMN_AUTOSEND, NULL));

  /* Set the keyboard octave */
  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_keyboard")), 3);
  gui->editor->octave = 3;

  /* Set default path in file dialogs */
  if (gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL) != NULL) {
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_file_open")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_file_saveas")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL));
  }
  if (gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL) != NULL) {
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_message_load")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_message_save")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL));
  }

  /* Add icons */
  pixbuf = gdk_pixbuf_new_from_file(PACKAGE_PIXMAP_DIR "/tutka.png", NULL);
  if (pixbuf != NULL)
    gtk_window_set_default_icon_list(g_list_append(NULL, pixbuf));

  return gui;
}

/* Closes a GUI */
void gui_close(struct gui *gui)
{
  g_mutex_free(gui->refresh_mutex);

  free(gui);
}

/* Refreshes all parts of the GUI */
void gui_refresh_all(struct gui *gui)
{
  /* Reset the tracker */
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  tracker_reset(tracker);
  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_num_channels(tracker, gui->editor->song->blocks[gui->editor->block]->tracks);

  /* Refresh the GUI */
  gui_info_refresh(gui);
  gui_tempo_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_playseqlist_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
  gui_instrument_refresh(gui);
  gui_messagelist_refresh(gui, TRUE);
  gui_trackvolumes_refresh(gui);
  gui_song_refresh(gui);
  gui_window_title_refresh(gui);

  /* Show the main window */
  gtk_widget_show(gui->window_main);
  gtk_widget_show(GTK_WIDGET(tracker));

  /* FIX: Tracker widget kludge */
  tracker->cursor_ch = 0;

  /* The current CellRendererSpin implementation needs a pointer to the upper limit values */
  if (gui->editor->song == NULL) {
    gui->numplayseqs = NULL;
    gui->numblocks = NULL;
  } else {
    gui->numplayseqs = &gui->editor->song->numplayseqs;
    gui->numblocks = &gui->editor->song->numblocks;
  }
}

Tracker *gui_get_tracker(struct gui *gui)
{
  return (Tracker *) glade_xml_get_widget(gui->xml, "tracker");
}

int gui_get_tracker_position(struct gui *gui)
{
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  return tracker->patpos;
}

/* Set the refresh mask */
void gui_set_refresh(struct gui *gui, unsigned int mask)
{
  if (g_thread_self() != gui->editor->thread)
    g_mutex_lock(gui->refresh_mutex);
  gui->refresh_mask |= mask;
  if (g_thread_self() != gui->editor->thread)
    g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh all those parts of the GUI that need to be refreshed */
void gui_refresh(struct gui *gui)
{
  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INSTRUMENT) {
    gui_instrument_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_INSTRUMENT;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TEMPO) {
    gui_tempo_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TEMPO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INFO) {
    gui_info_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_INFO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_WINDOW_TITLE) {
    gui_window_title_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_WINDOW_TITLE;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_STATUSBAR) {
    gui_statusbar_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_STATUSBAR;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TIMER) {
    gui_timer_refresh(gui, gui->editor->time);
    gui->refresh_mask ^= GUI_REFRESH_TIMER;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_BLOCKLIST) {
    gui_blocklist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_BLOCKLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ) {
    gui_playseq_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ_RECREATE) {
    gui_playseq_refresh(gui, TRUE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ_RECREATE;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQLIST) {
    gui_playseqlist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SECTIONLIST) {
    gui_sectionlist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_SECTIONLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_MESSAGELIST) {
    gui_messagelist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_MESSAGELIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKVOLUMES) {
    gui_trackvolumes_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKVOLUMES;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SONG) {
    gui_song_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_SONG;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKER) {
    gui_tracker_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKER;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();
}

/* Refresh the tracker */
void gui_tracker_refresh(struct gui *gui)
{
  Tracker *tracker = gui_get_tracker(gui);

  /* Check that the line is valid */
  if (gui->editor->line >= gui->editor->song->blocks[gui->editor->block]->length)
    gui->editor->line = gui->editor->song->blocks[gui->editor->block]->length - 1;
  /* Check that the commad page is valid */
  if (gui->editor->cmdpage > gui->editor->song->blocks[gui->editor->block]->commandpages)
    gui->editor->cmdpage = gui->editor->song->blocks[gui->editor->block]->commandpages - 1;

  tracker_set_num_channels(tracker, gui->editor->song->blocks[gui->editor->block]->tracks);
  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_patpos(tracker, gui->editor->line);
  tracker_set_cmdpage(tracker, gui->editor->cmdpage);
}

/* Refresh the song properties window widgets */
void gui_song_refresh(struct gui *gui)
{
  GtkEntry *entry = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_songname"));
  GtkToggleButton *togglebutton = GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_sendsync"));

  g_signal_handlers_block_by_func(entry, on_entry_songname_changed, gui);
  g_signal_handlers_block_by_func(togglebutton, on_checkbutton_sendsync_toggled, gui);

  /* Refresh the song name widget */
  if (gui->editor->song->name != NULL)
    gtk_entry_set_text(entry, gui->editor->song->name);
  else
    gtk_entry_set_text(entry, "");

  /* Refresh the send sync widget */
  if (gui->editor->song->sendsync)
    gtk_toggle_button_set_active(togglebutton, TRUE);
  else
    gtk_toggle_button_set_active(togglebutton, FALSE);

  g_signal_handlers_unblock_by_func(entry, on_entry_songname_changed, gui);
  g_signal_handlers_unblock_by_func(togglebutton, on_checkbutton_sendsync_toggled, gui);
}

/* Refresh widgets related to the properties of the current instrument */
void gui_instrument_refresh(struct gui *gui)
{
  char instrument[3];
  snprintf(instrument, 3, "%.2d", gui->editor->instrument + 1);
  GtkSpinButton *spinbutton = GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_instrument"));
  GtkEntry *entry = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrumentproperties_name"));
  GtkRange *range_volume = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_volume"));
  GtkRange *range_transpose = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_transpose"));
  GtkRange *range_hold = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_hold"));
  GtkRange *range_midich = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_midich"));

  /* Refresh the main window widgets */
  g_signal_handlers_block_by_func(spinbutton, on_spinbutton_instrument_changed, gui);
  g_signal_handlers_block_by_func(entry, on_entry_instrumentproperties_name_changed, gui);
  g_signal_handlers_block_by_func(range_volume, on_scale_volume_value_changed, gui);
  g_signal_handlers_block_by_func(range_transpose, on_scale_transpose_value_changed, gui);
  g_signal_handlers_block_by_func(range_hold, on_scale_hold_value_changed, gui);
  g_signal_handlers_block_by_func(range_midich, on_scale_midich_value_changed, gui);

  gtk_spin_button_set_value(spinbutton, gui->editor->instrument + 1);

  if (gui->editor->song->instruments[gui->editor->instrument]->name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), "");

  /* Refresh the instrument window widgets */
  if (gui->editor->song->instruments[gui->editor->instrument]->name != NULL)
    gtk_entry_set_text(entry, gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(entry, "");

  gtk_adjustment_set_value(gtk_range_get_adjustment(range_volume), gui->editor->song->instruments[gui->editor->instrument]->defaultvelocity);
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_transpose), gui->editor->song->instruments[gui->editor->instrument]->transpose);
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_hold), gui->editor->song->instruments[gui->editor->instrument]->hold);
  gtk_adjustment_set_value(gtk_range_get_adjustment(range_midich), gui->editor->song->instruments[gui->editor->instrument]->midichannel + 1);

  g_signal_handlers_unblock_by_func(spinbutton, on_spinbutton_instrument_changed, gui);
  g_signal_handlers_unblock_by_func(entry, on_entry_instrumentproperties_name_changed, gui);
  g_signal_handlers_unblock_by_func(range_volume, on_scale_volume_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_transpose, on_scale_transpose_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_hold, on_scale_hold_value_changed, gui);
  g_signal_handlers_unblock_by_func(range_midich, on_scale_midich_value_changed, gui);
}

/* Refresh tempo related widgets */
void gui_tempo_refresh(struct gui *gui)
{
  GtkRange *scale_tempo = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tempo"));
  GtkRange *scale_tpl = GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tpl"));

  /* Refresh the song properties window widgets */
  g_signal_handlers_block_by_func(scale_tempo, on_scale_tempo_value_changed, gui);
  g_signal_handlers_block_by_func(scale_tpl, on_scale_tpl_value_changed, gui);

  gtk_adjustment_set_value(gtk_range_get_adjustment(scale_tempo), gui->editor->song->tempo);
  gtk_adjustment_set_value(gtk_range_get_adjustment(scale_tpl), gui->editor->song->ticksperline);

  g_signal_handlers_unblock_by_func(scale_tempo, on_scale_tempo_value_changed, gui);
  g_signal_handlers_unblock_by_func(scale_tpl, on_scale_tpl_value_changed, gui);
}

/* Refresh the main window title */
void gui_window_title_refresh(struct gui *gui)
{
  gtk_window_set_title(GTK_WINDOW(gui->window_main), gui->editor->song->name);
}

/* Refresh the statusbar */
void gui_statusbar_refresh(struct gui *gui)
{
  GtkLabel *status = GTK_LABEL(glade_xml_get_widget(gui->xml, "label_status"));

  /* Status bar */
  switch (gui->editor->player->mode) {
  case MODE_PLAY_SONG:
    gtk_label_set_text(status, "Playing song");
    break;
  case MODE_PLAY_BLOCK:
    gtk_label_set_text(status, "Playing block");
    break;
  case MODE_IDLE:
    gtk_label_set_text(status, "Stopped");
    break;
  }
}

/* Refresh the main window information */
void gui_info_refresh(struct gui *gui)
{
  char *section, *playseq, *position, *block, *cmdpage;
  /* Calculate lengths of the info strings and allocate memory for them */
  int sectionlength = 8 + intlen(gui->editor->section + 1) + 1 + intlen(gui->editor->song->numsections) + 1;
  int playseqlength;
  int positionlength = 9 + intlen(gui->editor->position + 1) + 1 + intlen(gui->editor->song->playseqs[gui->editor->playseq]->length) + 1;
  int blocklength;
  int cmdpagelength = 13 + intlen(gui->editor->cmdpage + 1) + 1 + intlen(gui->editor->song->blocks[gui->editor->block]->commandpages) + 1;

  if (gui->editor->song->playseqs[gui->editor->playseq]->name != NULL)
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(gui->editor->song->numplayseqs) + 2 + strlen(gui->editor->song->playseqs[gui->editor->playseq]->name) + 1;
  else
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(gui->editor->song->numplayseqs) + 1;

  if (gui->editor->song->blocks[gui->editor->block]->name != NULL)
    blocklength = 6 + intlen(gui->editor->block + 1) + 1 + intlen(gui->editor->song->numblocks) + 2 + strlen(gui->editor->song->blocks[gui->editor->block]->name) + 1;
  else
    blocklength = 6 + intlen(gui->editor->block + 1) + 1 + intlen(gui->editor->song->numblocks) + 2;

  section = malloc(sectionlength);
  playseq = malloc(playseqlength);
  position = malloc(positionlength);
  block = malloc(blocklength);
  cmdpage = malloc(cmdpagelength);

  /* Print the info strings */
  snprintf(section, sectionlength, "Section %d/%d", gui->editor->section + 1, gui->editor->song->numsections);
  if (gui->editor->song->playseqs[gui->editor->playseq]->name != NULL)
    snprintf(playseq, playseqlength, "Playing Sequence %d/%d: %s", gui->editor->playseq + 1, gui->editor->song->numplayseqs, gui->editor->song->playseqs[gui->editor->playseq]->name);
  else
    snprintf(playseq, playseqlength, "Playing Sequence %d/%d", gui->editor->playseq + 1, gui->editor->song->numplayseqs);
  snprintf(position, playseqlength, "Position %d/%d", gui->editor->position + 1, gui->editor->song->playseqs[gui->editor->playseq]->length);
  if (gui->editor->song->blocks[gui->editor->block]->name != NULL)
    snprintf(block, blocklength, "Block %d/%d: %s", gui->editor->block + 1, gui->editor->song->numblocks, gui->editor->song->blocks[gui->editor->block]->name);
  else
    snprintf(block, blocklength, "Block %d/%d", gui->editor->block + 1, gui->editor->song->numblocks);
  snprintf(cmdpage, cmdpagelength, "Command Page %d/%d", gui->editor->cmdpage + 1, gui->editor->song->blocks[gui->editor->block]->commandpages);
  /* Refresh the info string widgets */
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_section")), section);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_playseq")), playseq);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_position")), position);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_block")), block);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_commandpage")), cmdpage);

  /* Free the info strings */
  free(section);
  free(playseq);
  free(position);
  free(block);
}

/* Refresh the timer */
void gui_timer_refresh(struct gui *gui, unsigned int seconds)
{
  char time[6];

  /* Refresh the main window widgets if the window exists */
  if (gui->window_main != NULL && GTK_IS_WINDOW(gui->window_main)) {
    snprintf(time, 6, "%.2d:%.2d", seconds / 60, seconds % 60);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_timer")), time);
  }
}

/* Recreate and refresh the block list window contents */
void gui_blocklist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char block[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(block, 10, "%d", gui->editor->block);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));

  if (recreate) {
    /* Recreate the block list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_blocklist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < gui->editor->song->numblocks; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_BLOCKLIST_COLUMN_NUMBER, i + 1, GUI_BLOCKLIST_COLUMN_NAME, gui->editor->song->blocks[i]->name, GUI_BLOCKLIST_COLUMN_TRACKS, gui->editor->song->blocks[i]->tracks, GUI_BLOCKLIST_COLUMN_LENGTH, gui->editor->song->blocks[i]->length, GUI_BLOCKLIST_COLUMN_COMMAND_PAGES, gui->editor->song->blocks[i]->commandpages, -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_blocklist_changed, gui);
  }

  /* Move back to the previously selected block */
  treepath = gtk_tree_path_new_from_string(block);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete button sensitive/insensitive */
  if (gui->editor->song->numblocks > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), FALSE);
}

/* Refresh the playing sequence list window widgets */
void gui_playseqlist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char playseq[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(playseq, 10, "%d", gui->editor->playseq);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseqlist"));

  if (recreate) {
    /* Recreate the playing sequence list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_playseqlist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < gui->editor->song->numplayseqs; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_PLAYSEQLIST_COLUMN_NUMBER, i + 1, GUI_PLAYSEQLIST_COLUMN_NAME, gui->editor->song->playseqs[i]->name, -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_playseqlist_changed, gui);
  }

  /* Move to the correct position */
  treepath = gtk_tree_path_new_from_string(playseq);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete playing sequence button sensitive/insensitive */
  if (gui->editor->song->numplayseqs > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseqlist_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseqlist_delete")), FALSE);
}

/* Refresh the playing sequence window widgets */
void gui_playseq_refresh(struct gui *gui, gboolean recreate)
{
  int i, playseqlength;
  char playseqpos[10], *title;
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(playseqpos, 10, "%d", gui->editor->position);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));

  /* Calculate length of the window title and allocate memory for it */
  if (gui->editor->song->playseqs[gui->editor->playseq]->name != NULL)
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(gui->editor->song->numplayseqs) + 2 + strlen(gui->editor->song->playseqs[gui->editor->playseq]->name) + 1;
  else
    playseqlength = 17 + intlen(gui->editor->playseq + 1) + 1 + intlen(gui->editor->song->numplayseqs) + 1;

  title = malloc(playseqlength);

  /* Print the window title */
  if (gui->editor->song->playseqs[gui->editor->playseq]->name != NULL)
    snprintf(title, playseqlength, "Playing Sequence %d/%d: %s", gui->editor->playseq + 1, gui->editor->song->numplayseqs, gui->editor->song->playseqs[gui->editor->playseq]->name);
  else
    snprintf(title, playseqlength, "Playing Sequence %d/%d", gui->editor->playseq + 1, gui->editor->song->numplayseqs);

  if (recreate) {
    /* Recreate the playing sequence */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_playseq_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < gui->editor->song->playseqs[gui->editor->playseq]->length; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_PLAYSEQ_COLUMN_POSITION, i + 1, GUI_PLAYSEQ_COLUMN_BLOCK_NUMBER, gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i] + 1, GUI_PLAYSEQ_COLUMN_BLOCK_NAME, gui->editor->song->blocks[gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i]]->name, -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_playseq_changed, gui);
  }

  /* Move to the correct position */
  treepath = gtk_tree_path_new_from_string(playseqpos);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Set window title */
  gtk_window_set_title(GTK_WINDOW(glade_xml_get_widget(gui->xml, "dialog_playseq")), title);

  /* Make the delete position button sensitive/insensitive */
  if (gui->editor->song->playseqs[gui->editor->playseq]->length > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), FALSE);

  free(title);
}

/* Refresh all section list widgets */
void gui_sectionlist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char section[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;

  snprintf(section, 10, "%d", gui->editor->section);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));

  if (recreate) {
    /* Recreate the section list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_sectionlist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < gui->editor->song->numsections; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_SECTIONLIST_COLUMN_SECTION, i + 1, GUI_SECTIONLIST_COLUMN_PLAYSEQ_NUMBER, gui->editor->song->sections[i] + 1, GUI_SECTIONLIST_COLUMN_PLAYSEQ_NAME, gui->editor->song->playseqs[gui->editor->song->sections[i]]->name, -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_sectionlist_changed, gui);
  }

  /* Move back to the previously selected section */
  treepath = gtk_tree_path_new_from_string(section);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  /* Make the delete button sensitive/insensitive */
  if (gui->editor->song->numsections > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), FALSE);
}

/* Refresh the message list widgets */
void gui_messagelist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char oldmessage[10];
  GtkTreeView *treeview;

  if (gui->editor->song->nummessages == 0)
    gui->messagelist_message = -1;

  snprintf(oldmessage, 10, "%d", gui->messagelist_message);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));

  if (recreate) {
    /* Recreate the Message list */
    GtkListStore *store = GTK_LIST_STORE(gtk_tree_view_get_model(treeview));
    g_signal_handlers_block_by_func(treeview, on_selection_messagelist_changed, gui);
    gtk_list_store_clear(store);
    for (i = 0; i < gui->editor->song->nummessages; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(store, &iter);
      gtk_list_store_set(store, &iter, GUI_MESSAGELIST_COLUMN_NUMBER, i, GUI_MESSAGELIST_COLUMN_NAME, gui->editor->song->messages[i]->name, GUI_MESSAGELIST_COLUMN_LENGTH, gui->editor->song->messages[i]->length, GUI_MESSAGELIST_COLUMN_AUTOSEND, gui->editor->song->messages[i]->autosend, -1);
    }
    g_signal_handlers_unblock_by_func(treeview, on_selection_messagelist_changed, gui);
  }

  /* Move back to the previously selected message and make widgets sensitive */
  if (gui->messagelist_message != -1) {
    gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"))), gtk_tree_path_new_from_string(oldmessage));

    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), TRUE);
  } else {
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), FALSE);
  }
}

void gui_trackvolumes_refresh(struct gui *gui)
{
  /* Refresh the track volumes window widgets */
  int i;
  char track[10];
  GtkWidget *table = glade_xml_get_widget(gui->xml, "table_trackvolumes");

  /* Destroy all children */
  gtk_container_foreach(GTK_CONTAINER(table), (GtkCallback) gtk_widget_destroy, NULL);
  gtk_table_resize(GTK_TABLE(table), 3, gui->editor->song->maxtracks);

  /* Create widgets for each channel */
  for (i = 0; i < gui->editor->song->maxtracks; i++) {
    GtkWidget *w, *x;

    /* Track numbers */
    snprintf(track, 10, "Track %d", i + 1);
    x = gtk_label_new(track);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 2, 4);

    /* Volume sliders */
    w = (GtkWidget *) gtk_adjustment_new((gui->editor->song->trackvolumes[i]) & 127, 0, 127, 1, 8, 0);
    g_signal_connect(G_OBJECT(w), "value_changed", G_CALLBACK(on_adjustment_trackvolumes_changed), (gpointer) & gui->editor->song->trackvolumes[i]);
    x = gtk_vscale_new(GTK_ADJUSTMENT(w));
    gtk_range_set_inverted(GTK_RANGE(x), TRUE);
    gtk_scale_set_digits(GTK_SCALE(x), 0);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 1, 2, GTK_EXPAND, GTK_EXPAND | GTK_FILL, 2, 4);

    /* Mute buttons */
    w = gtk_check_button_new_with_label("Mute");
    if (gui->editor->song->trackvolumes[i] > 127)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
    g_signal_connect(G_OBJECT(w), "toggled", G_CALLBACK(on_togglebutton_trackvolumes_toggled), (gpointer) & gui->editor->song->trackvolumes[i]);
    gtk_table_attach(GTK_TABLE(table), w, i, i + 1, 2, 3, GTK_EXPAND, GTK_FILL, 2, 4);
  }

  gtk_widget_show_all(table);
}

void gui_preferences_refresh(struct gui *gui)
{
  GtkEntry *entry = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_preferences_midi_device"));
  GtkMenuItem *menu_rtc = GTK_MENU_ITEM(glade_xml_get_widget(gui->xml, "menuitem_preferences_rtc"));
  GtkMenuItem *menu_nanosleep = GTK_MENU_ITEM(glade_xml_get_widget(gui->xml, "menuitem_preferences_nanosleep"));

  /* Refresh the preferences window widgets */
  g_signal_handlers_block_by_func(entry, on_entry_preferences_midi_device_changed, gui);
  g_signal_handlers_block_by_func(menu_rtc, on_menuitem_preferences_rtc_activate, gui);
  g_signal_handlers_block_by_func(menu_nanosleep, on_menuitem_preferences_nanosleep_activate, gui);

  if (gconf_client_get_string(gui->editor->gconfclient, "/apps/tutka/mididevice", NULL) != NULL)
    gtk_entry_set_text(entry, gconf_client_get_string(gui->editor->gconfclient, "/apps/tutka/mididevice", NULL));
  else
    gtk_entry_set_text(entry, "");

  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_preferences_scheduling_mode")), gconf_client_get_int(gui->editor->gconfclient, "/apps/tutka/scheduler", NULL));

  g_signal_handlers_unblock_by_func(entry, on_entry_preferences_midi_device_changed, gui);
  g_signal_handlers_unblock_by_func(menu_rtc, on_menuitem_preferences_rtc_activate, gui);
  g_signal_handlers_unblock_by_func(menu_nanosleep, on_menuitem_preferences_nanosleep_activate, gui);
}
