/* editor.c
 *
 * Copyright 2002-2004 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "conversion.h"
#include "editor.h"
#include "player.h"
#include "callbacks.h"

/* Initializes the editor */
struct editor *editor_open()
{
  struct editor *editor = (struct editor *)calloc(1, sizeof(struct editor));

  /* Use GConf */
  editor->gconfclient = gconf_client_get_default();
  gconf_client_add_dir(editor->gconfclient, "/apps/tutka", GCONF_CLIENT_PRELOAD_NONE, NULL);
  
  /* Set defaults */
  editor->space = 1;
  
  /* Get a pointer to the main thread */
  editor->thread = g_thread_self();

  /* Open a GUI file */
  editor->gui = gui_open(editor);
  if(editor->gui == NULL) {
    editor_close(editor);
    return NULL;
  }

  gui_preferences_refresh(editor->gui);
  editor->gconfnotifyid=gconf_client_notify_add(editor->gconfclient, "/apps/tutka", gconf_notify_function, editor->gui, NULL, NULL);
  
  return editor;
}

/* Opens an existing song or creates new */
void editor_song_open(struct editor *editor, char *filename)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_song_open() called with null editor\n");
    return;
  }

  /* Close the old player */
  if (editor->player != NULL)
    player_close(editor->player);

  /* Free the old song */
  if (editor->song != NULL)
    song_free(editor->song);

  if (editor->filename != NULL)
    free(editor->filename);

  /* If a filename was supplied open the file */
  if (filename != NULL) {
    FILE *file=fopen(filename, "r");
    unsigned char data[4]={ 0, 0, 0, 0 };
    
    editor->filename = strdup(filename);

    /* Read first four bytes for checking the file type */
    if(file!=NULL) {
      fread(data, sizeof(unsigned char), 4, file);
      fclose(file);
    }

    /* Is it an MMD2 file? */
    if(data[0]==(ID_MMD2>>24) && data[1]==((ID_MMD2>>16)&0xff) && data[2]==((ID_MMD2>>8)&0xff) && data[3]==(ID_MMD2&0xff)) {
      struct MMD2 *mmd;
      mmd = MMD2_load(filename);
      editor->song = MMD2_convert(mmd);
      MMD2_free(mmd);
    } else
      editor->song = song_load(filename);

    /* Just allocate a new song if the song couldn't be opened */
    if (editor->song == NULL)
      editor->song = song_alloc();
  } else {
    editor->filename = NULL;
    editor->song = song_alloc();
  }

  /* Reset editor location */
  editor->section = 0;
  editor->playseq = 0;
  editor->position = 0;
  editor->block = 0;
  editor->line = 0;
  editor->time = 0;
  editor->cmdpage = 0;

  /* Open the player */
  editor->player = player_open(editor->song, editor);

  /* Refresh the GUI */
  gui_refresh_all(editor->gui);
}

/* Deinitializes the editor */
void editor_close(struct editor *editor)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_close() called with null editor\n");
    return;
  }

  /* Close the player */
  if (editor->player != NULL)
    player_close(editor->player);

  /* Remove the GConf notification */
  if(editor->gconfclient != NULL)
    gconf_client_notify_remove(editor->gconfclient, editor->gconfnotifyid);

  /* Close the GUI */
  if (editor->gui != NULL)
    gui_close(editor->gui);

  /* Free the song */
  if (editor->song != NULL)
    song_free(editor->song);

  /* Free the filename */
  if (editor->filename != NULL)
    free(editor->filename);

  /* Free the copy & paste buffers */
  if (editor->copyblock != NULL)
    block_free(editor->copyblock);

  if (editor->copytrack != NULL)
    block_free(editor->copytrack);

  if (editor->copyarea != NULL)
    block_free(editor->copyarea);

  /* Unref GConf client */
  if(editor->gconfclient != NULL)
    g_object_unref(editor->gconfclient);
  
  free(editor);
}

/* Changes the current section and refreshes the GUI accordingly */
void editor_set_section(struct editor *editor, int section)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_section() called with null editor\n");
    return;
  }

  if (section != editor->section) {
    /* Bounds checking */
    if (section < 0)
      section = 0;
    if (section >= editor->song->numsections)
      section = editor->song->numsections - 1;

    editor->section = section;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_section(editor->player, section);
    gui_set_refresh(editor->gui, GUI_REFRESH_SECTIONLIST | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_playseq(struct editor *editor, int playseq)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_playseq() called with null editor\n");
    return;
  }

  if (playseq != editor->playseq) {
    /* Bounds checking */
    if (playseq < 0)
      playseq = 0;
    if (playseq >= editor->song->numplayseqs)
      playseq = editor->song->numplayseqs - 1;

    editor->playseq = playseq;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_playseq(editor->player, playseq);
    gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQLIST | GUI_REFRESH_PLAYSEQ_RECREATE | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_position(struct editor *editor, int position)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_position() called with null editor\n");
    return;
  }

  if (position != editor->position) {
    /* Bounds checking */
    if (position < 0)
      position = 0;
    if (position >= editor->song->playseqs[editor->playseq]->length)
      position = editor->song->playseqs[editor->playseq]->length - 1;

    editor->position = position;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_position(editor->player, position);
    gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQ | GUI_REFRESH_INFO);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_block(struct editor *editor, int block)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_block() called with null editor\n");
    return;
  }

  if (block != editor->block) {
    /* Bounds checking */
    if (block < 0)
      block = 0;
    if (block >= editor->song->numblocks)
      block = editor->song->numblocks - 1;
    while (editor->cmdpage >= editor->song->blocks[block]->commandpages)
      editor->cmdpage--;

    editor->block = block;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread)
      player_set_block(editor->player, block);
    gui_set_refresh(editor->gui, GUI_REFRESH_BLOCKLIST | GUI_REFRESH_INFO | GUI_REFRESH_TRACKER);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_line(struct editor *editor, int line)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_line() called with null editor\n");
    return;
  }

  if (line != editor->line) {
    /* Bounds checking */
    while (line < 0)
      line += editor->song->blocks[editor->block]->length;
    while (line >= editor->song->blocks[editor->block]->length)
      line -= editor->song->blocks[editor->block]->length;

    editor->line = line;

    /* If called from GUI set player position, always refresh GUI */
    if (g_thread_self() == editor->thread) {
      player_set_line(editor->player, line);
      gui_tracker_refresh(editor->gui);
    } else
      gui_set_refresh(editor->gui, GUI_REFRESH_TRACKER);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_commandpage(struct editor *editor, int cmdpage)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_commandpage() called with null editor\n");
    return;
  }

  if (cmdpage != editor->cmdpage) {
    /* Bounds checking */
    while (cmdpage < 0)
      cmdpage += editor->song->blocks[editor->block]->commandpages;
    while (cmdpage >= editor->song->blocks[editor->block]->commandpages)
      cmdpage -= editor->song->blocks[editor->block]->commandpages;

    editor->cmdpage = cmdpage;

    gui_set_refresh(editor->gui, GUI_REFRESH_INFO | GUI_REFRESH_TRACKER);
  }
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_time(struct editor *editor, int time)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_time() called with null editor\n");
    return;
  }

  editor->time = time;
  gui_set_refresh(editor->gui, GUI_REFRESH_TIMER);
}

/* Changes the position of the editor and refreshes the GUI accordingly */
void editor_set_tempo(struct editor *editor)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_set_tempo() called with null editor\n");
    return;
  }

  gui_set_refresh(editor->gui, GUI_REFRESH_TEMPO);
}

/* Refreshes playseq from section and block from position */
void editor_refresh_playseq_and_block(struct editor *editor)
{
  struct song *song;

  if (editor == NULL) {
    fprintf(stderr, "editor_refresh_playseq_and_block() called with null editor\n");
    return;
  }

  song = editor->song;

  /* Bounds checking */
  if (editor->section >= song->numsections)
    editor->section = 0;

  /* Get playseq number */
  editor->playseq = song->sections[editor->section];

  /* Bounds checking */
  if (editor->position >= song->playseqs[editor->playseq]->length) {
    editor->position = 0;
    editor_refresh_playseq_and_block(editor);
  } else
    editor->block = song->playseqs[editor->playseq]->blocknumbers[editor->position];

  /* If called from GUI set player position, always refresh GUI */
  if (g_thread_self() == editor->thread)
    player_set_playseq(editor->player, editor->playseq);
  gui_set_refresh(editor->gui, GUI_REFRESH_PLAYSEQLIST | GUI_REFRESH_PLAYSEQ | GUI_REFRESH_BLOCKLIST | GUI_REFRESH_INFO);
}

/* Reads MIDI input */
void editor_midi_input(struct editor *editor, unsigned char *data)
{
  Tracker *tracker;
  struct song *song;

  if (editor == NULL) {
    fprintf(stderr, "editor_midi_input() called with null editor\n");
    return;
  }

  tracker = (Tracker *) glade_xml_get_widget(editor->gui->xml, "tracker");
  song = editor->song;

  switch (data[0] & 0xf0) {
  case 0x80:
    /* Note off */
    if (editor->edit) {
      if (editor->chord) {
        /* Do not allow negative chord status: it should never happen anyway though */
        if (editor->chordstatus > 0)
          editor->chordstatus--;
        
        /* Move the cursor to the previous channel */
	tracker_step_cursor_channel(tracker, -1);
        
        /* If all notes of the chord have been released move to the next line */
	if (editor->chordstatus == 0)
	  editor_set_line(editor, editor->line + editor->space);
      }
    }
    break;
  case 0x90:
    /* Note on */
    if (editor->edit) {
      block_set_note(song->blocks[editor->block], tracker->patpos, tracker->cursor_ch, data[1] / 12, data[1] % 12 + 1, editor->instrument + 1);
      block_set_command_full(song->blocks[editor->block], tracker->patpos, tracker->cursor_ch, editor->cmdpage, COMMAND_VELOCITY, data[2]);
      tracker_redraw_row(tracker, tracker->patpos);
      if (editor->chord) {
	editor->chordstatus++;

        /* Move the cursor to the next channel */
	tracker_step_cursor_channel(tracker, 1);
      } else
	editor_set_line(editor, editor->line + editor->space);
    }
    break;
  case 0xb0:
    /* MIDI controller */
    if (editor->edit && editor->recordcontrollers) {
      block_set_command_full(song->blocks[editor->block], tracker->patpos, tracker->cursor_ch, editor->cmdpage, COMMAND_MIDI_CONTROLLERS + data[1], data[2]);
      tracker_redraw_row(tracker, tracker->patpos);
    }
    break;
  case 0xe0:
    /* Pitch wheel */
    if (editor->edit) {
      block_set_command_full(song->blocks[editor->block], tracker->patpos, tracker->cursor_ch, editor->cmdpage, COMMAND_PITCH_WHEEL, data[2]);
      tracker_redraw_row(tracker, tracker->patpos);
    }
    break;
  default:
    break;
  }
}

/* Refreshes the GUI when the player has stopped */
void editor_stop(struct editor *editor)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_stop() called with null editor\n");
    return;
  }

  gui_set_refresh(editor->gui, GUI_REFRESH_STATUSBAR);
}

/* Resets the timer to 00:00 */
void editor_reset_time(struct editor *editor)
{
  if (editor == NULL) {
    fprintf(stderr, "editor_reset_timer() called with null editor\n");
    return;
  }

  /* If called from GUI reset player timer */
  if (g_thread_self() == editor->thread)
    player_reset_time(editor->player, TRUE);
  
  editor->time = 0;
  gui_set_refresh(editor->gui, GUI_REFRESH_TIMER);
}
