/* block.c
 *
 * Copyright 2002-2004 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include "block.h"

/* Allocates a block */
struct block *block_alloc(int tracks, int length, int commandpages)
{
  struct block *block;

  block = (struct block *)calloc(1, sizeof(struct block));
  block->tracks = tracks;
  block->length = length;
  block->notes = (unsigned char *)calloc(2 * tracks * length, sizeof(unsigned char));
  block->commandpages = commandpages;
  block->commands = (unsigned char *)calloc(commandpages * 2 * tracks * length, sizeof(unsigned char));

  return block;
}

/* Frees a block */
void block_free(struct block *block)
{
  if (block == NULL) {
    fprintf(stderr, "block_free() called with null block\n");
    return;
  }

  if (block->name)
    free(block->name);

  if (block->notes)
    free(block->notes);

  if (block->commands)
    free(block->commands);

  free(block);
}

/* Sets a note in a block */
void block_set_note(struct block *block, unsigned int line, unsigned int track, unsigned char octave, unsigned char note, unsigned char instrument)
{
  if (block == NULL) {
    fprintf(stderr, "block_set_note() called with null block\n");
    return;
  }

  if (note) {
    block->notes[2 * (block->tracks * line + track)] = octave * 12 + note;
    block->notes[2 * (block->tracks * line + track) + 1] = instrument;
  } else {
    block->notes[2 * (block->tracks * line + track)] = 0;
    block->notes[2 * (block->tracks * line + track) + 1] = 0;
  }
}

/* Sets an instrument in a block */
void block_set_instrument(struct block *block, unsigned int line, unsigned int track, unsigned char instrument)
{
  if (block == NULL) {
    fprintf(stderr, "block_set_instrument() called with null block\n");
    return;
  }

  block->notes[2 * (block->tracks * line + track) + 1] = instrument;
}

/* Sets a part of a command in a block */
void block_set_command(struct block *block, unsigned int line, unsigned int track, unsigned int commandpage, unsigned char slot, unsigned char data)
{
  if (block == NULL) {
    fprintf(stderr, "block_set_command() called with null block\n");
    return;
  }

  if (slot & 1) {
    block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track) + slot / 2] &= 0xf0;
    block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track) + slot / 2] |= data;
  } else {
    block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track) + slot / 2] &= 0x0f;
    block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track) + slot / 2] |= (data << 4);
  }
}

/* Sets a command in a block */
void block_set_command_full(struct block *block, unsigned int line, unsigned int track, unsigned int commandpage, unsigned char command, unsigned char data)
{
  if (block == NULL) {
    fprintf(stderr, "block_set_command_full() called with null block\n");
    return;
  }

  block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track)] = command;
  block->commands[commandpage * 2 * block->tracks * block->length + 2 * (block->tracks * line + track) + 1] = data;
}

/* Sets the number of tracks in a block */
void block_set_tracks(struct block *block, unsigned int tracks)
{
  unsigned char *notes, *commands;
  unsigned int existingtracks;
  int l, t, ep, nl, nt;

  if (block == NULL) {
    fprintf(stderr, "block_set_tracks() called with null block\n");
    return;
  }

  /* Allocate new arrays */
  notes = (unsigned char *)calloc(2 * tracks * block->length, sizeof(unsigned char));
  commands = (unsigned char *)calloc(block->commandpages * 2 * tracks * block->length, sizeof(unsigned char));
  nl = block->length;
  nt = block->tracks;

  /* How many tracks from the old block to use */
  if (tracks < nt)
    existingtracks = tracks;
  else
    existingtracks = nt;

  /* Copy the notes and the commands of the block to a new data array */
  for (l = 0; l < nl; l++)
    for (t = 0; t < existingtracks; t++) {
      notes[(l * tracks + t) * 2] = block->notes[(l * nt + t) * 2];
      notes[(l * tracks + t) * 2 + 1] = block->notes[(l * nt + t) * 2 + 1];
      for (ep = 0; ep < block->commandpages; ep++) {
	commands[ep * 2 * tracks * nl + (l * tracks + t) * 2] = block->commands[ep * 2 * nt * nl + (l * nt + t) * 2];
	commands[ep * 2 * tracks * nl + (l * tracks + t) * 2 + 1] = block->commands[ep * 2 * nt * nl + (l * nt + t) * 2 + 1];
      }
    }

  /* Free old arrays */
  if (block->notes)
    free(block->notes);
  if (block->commands)
    free(block->commands);

  /* Use new arrays */
  block->notes = notes;
  block->commands = commands;
  block->tracks = tracks;
}

/* Sets the length of a block */
void block_set_length(struct block *block, unsigned int length)
{
  unsigned char *notes, *commands;
  unsigned int existinglength;
  int l, t, ep, nl, nt;

  if (block == NULL) {
    fprintf(stderr, "block_set_length() called with null block\n");
    return;
  }

  /* Allocate new arrays */
  notes = (unsigned char *)calloc(2 * block->tracks * length, sizeof(unsigned char));
  commands = (unsigned char *)calloc(block->commandpages * 2 * block->tracks * length, sizeof(unsigned char));
  nl = block->length;
  nt = block->tracks;

  /* How many lines from the old block to use */
  if (length < nl)
    existinglength = length;
  else
    existinglength = nl;

  /* Copy the notes and the commands of the block to a new data array */
  for (l = 0; l < existinglength; l++)
    for (t = 0; t < nt; t++) {
      notes[(l * nt + t) * 2] = block->notes[(l * nt + t) * 2];
      notes[(l * nt + t) * 2 + 1] = block->notes[(l * nt + t) * 2 + 1];
      for (ep = 0; ep < block->commandpages; ep++) {
	commands[ep * 2 * nt * length + (l * nt + t) * 2] = block->commands[ep * 2 * nt * nl + (l * nt + t) * 2];
	commands[ep * 2 * nt * length + (l * nt + t) * 2 + 1] = block->commands[ep * 2 * nt * nl + (l * nt + t) * 2 + 1];
      }
    }

  /* Free old arrays */
  if (block->notes)
    free(block->notes);
  if (block->commands)
    free(block->commands);

  /* Use new arrays */
  block->notes = notes;
  block->commands = commands;
  block->length = length;
}

/* Sets the number of command pages in a block */
void block_set_commandpages(struct block *block, unsigned int commandpages)
{
  unsigned char *commands;
  unsigned int existingcommandpages;

  if (block == NULL) {
    fprintf(stderr, "block_set_commandpages() called with null block\n");
    return;
  }

  /* Allocate new command page array */
  commands = (unsigned char *)calloc(commandpages * 2 * block->tracks * block->length, sizeof(unsigned char));

  /* How many command pages from the old block to use */
  if (commandpages < block->commandpages)
    existingcommandpages = commandpages;
  else
    existingcommandpages = block->commandpages;

  /* Copy the command pages to a new data array */
  memcpy(commands, block->commands, existingcommandpages * 2 * block->tracks * block->length);

  /* Free old array */
  if (block->commands)
    free(block->commands);

  /* Use new array */
  block->commands = commands;
  block->commandpages = commandpages;
}

/* Copies a part of a block to a new block */
struct block *block_copy(struct block *block, int starttrack, int startline, int endtrack, int endline)
{
  struct block *newblock;
  int t, l, ep, onl, nnl, ont, nnt;

  if (block == NULL) {
    fprintf(stderr, "block_copy() called with null block\n");
    return NULL;
  }

  /* Bounds checking */
  if (starttrack < 0)
    starttrack = 0;
  if (startline < 0)
    startline = 0;
  if (starttrack >= block->tracks)
    starttrack = block->tracks - 1;
  if (startline >= block->length)
    startline = block->length - 1;
  if (endtrack < 0)
    endtrack = 0;
  if (endline < 0)
    endline = 0;
  if (endtrack >= block->tracks)
    endtrack = block->tracks - 1;
  if (endline >= block->length)
    endline = block->length - 1;

  /* Make sure start and end are the correct way around */
  if (starttrack > endtrack) {
    t = starttrack;
    starttrack = endtrack;
    endtrack = t;
  }
  if (startline > endline) {
    l = startline;
    startline = endline;
    endline = l;
  }
    
  /* Allocate new block */
  newblock = block_alloc(endtrack - starttrack + 1, endline - startline + 1, block->commandpages);
  onl = block->length;
  nnl = newblock->length;
  ont = block->tracks;
  nnt = newblock->tracks;

  /* Copy the given part of the block to a new block */
  for (l = startline; l <= endline; l++) {
    for (t = starttrack; t <= endtrack; t++) {
      newblock->notes[((l - startline) * nnt + (t - starttrack)) * 2] = block->notes[(l * ont + t) * 2];
      newblock->notes[((l - startline) * nnt + (t - starttrack)) * 2 + 1] = block->notes[(l * ont + t) * 2 + 1];
      for (ep = 0; ep < block->commandpages; ep++) {
	newblock->commands[ep * 2 * nnt * nnl + ((l - startline) * nnt + (t - starttrack)) * 2] = block->commands[ep * 2 * ont * onl + (l * ont + t) * 2];
	newblock->commands[ep * 2 * nnt * nnl + ((l - startline) * nnt + (t - starttrack)) * 2 + 1] = block->commands[ep * 2 * ont * onl + (l * ont + t) * 2 + 1];
      }
    }
  }

  return newblock;
}

/* Pastes a block to another block in the given position */
void block_paste(struct block *block, struct block *from, int track, int line)
{
  int t, l, ep, nl, nt, fnl, fnt, cnl, cnt;

  if (block == NULL || from == NULL) {
    fprintf(stderr, "block_paste() called with null block\n");
    return;
  }

  nl = block->length;
  nt = block->tracks;
  fnl = from->length;
  fnt = from->tracks;
  cnl = fnl;
  cnt = fnt;

  if (line + fnl > nl)
    cnl = nl - line;
  if (track + fnt > nt)
    cnt = nt - track;

  /* Copy the from block to the destination block; make sure it fits */
  for (l = 0; l < cnl; l++) {
    for (t = 0; t < cnt; t++) {
      block->notes[((line + l) * nt + (track + t)) * 2] = from->notes[(l * fnt + t) * 2];
      block->notes[((line + l) * nt + (track + t)) * 2 + 1] = from->notes[(l * fnt + t) * 2 + 1];
      for (ep = 0; ep < block->commandpages; ep++) {
	block->commands[ep * 2 * nt * nl + ((line + l) * nt + (track + t)) * 2] = from->commands[ep * 2 * fnt * fnl + (l * fnt + t) * 2];
	block->commands[ep * 2 * nt * nl + ((line + l) * nt + (track + t)) * 2 + 1] = from->commands[ep * 2 * fnt * fnl + (l * fnt + t) * 2 + 1];
      }
    }
  }
}

/* Clears a part of a block */
void block_clear(struct block *block, int starttrack, int startline, int endtrack, int endline)
{
  int t, l, ep, nl, nt;

  if (block == NULL) {
    fprintf(stderr, "block_clear() called with null block\n");
    return;
  }

  /* Bounds checking */
  if (starttrack < 0)
    starttrack = 0;
  if (startline < 0)
    startline = 0;
  if (starttrack >= block->tracks)
    starttrack = block->tracks - 1;
  if (startline >= block->length)
    startline = block->length - 1;
  if (endtrack < 0)
    endtrack = 0;
  if (endline < 0)
    endline = 0;
  if (endtrack >= block->tracks)
    endtrack = block->tracks - 1;
  if (endline >= block->length)
    endline = block->length - 1;

  /* Make sure start and end are the correct way around */
  if (starttrack > endtrack) {
    t = starttrack;
    starttrack = endtrack;
    endtrack = t;
  }
  if (startline > endline) {
    l = startline;
    startline = endline;
    endline = l;
  }
    
  nl = block->length;
  nt = block->tracks;

  for (l = startline; l <= endline; l++) {
    for (t = starttrack; t <= endtrack; t++) {
      block->notes[(l * nt + t) * 2] = 0;
      block->notes[(l * nt + t) * 2 + 1] = 0;
      for (ep = 0; ep < block->commandpages; ep++) {
	block->commands[ep * 2 * nt * nl + (l * nt + t) * 2] = 0;
	block->commands[ep * 2 * nt * nl + (l * nt + t) * 2 + 1] = 0;
      }
    }
  }
}

/* Parses a block element in an XML file */
struct block *block_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct block *block = NULL;

  if ((!xmlStrcmp(cur->name, "block")) && (cur->ns == ns)) {
    char *prop;
    int tracks = 4, length = 64, commandpages = 1;

    /* Get block properties */
    prop = xmlGetProp(cur, "tracks");
    if (prop != NULL)
      tracks = atoi(prop);
    prop = xmlGetProp(cur, "length");
    if (prop != NULL)
      length = atoi(prop);
    prop = xmlGetProp(cur, "commandpages");
    if (prop != NULL)
      commandpages = atoi(prop);

    /* Allocate block */
    block = block_alloc(tracks, length, commandpages);
    block->name = xmlGetProp(cur, "name");

    /* Get block contents */
    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
      xmlNodePtr temp;
      if ((!xmlStrcmp(cur->name, "note")) && (cur->ns == ns)) {
	int line = 0, track = 0, note = 0, instrument = 0;

	/* Get note properties */
	prop = xmlGetProp(cur, "line");
	if (prop != NULL)
	  line = atoi(prop);
	prop = xmlGetProp(cur, "track");
	if (prop != NULL)
	  track = atoi(prop);
	prop = xmlGetProp(cur, "instrument");
	if (prop != NULL)
	  instrument = atoi(prop);

	/* Get the note */
	temp = cur->xmlChildrenNode;
	if (temp != NULL)
	  note = atoi(temp->content);

	/* Set the note */
	block_set_note(block, line, track, 0, note, instrument);
      } else if ((!xmlStrcmp(cur->name, "command")) && (cur->ns == ns)) {
	int line = 0, track = 0, commandpage = 0, command = 0, value = 0;

	/* Get command properties */
	prop = xmlGetProp(cur, "line");
	if (prop != NULL)
	  line = atoi(prop);
	prop = xmlGetProp(cur, "track");
	if (prop != NULL)
	  track = atoi(prop);
	prop = xmlGetProp(cur, "commandpage");
	if (prop != NULL)
	  commandpage = atoi(prop);
	prop = xmlGetProp(cur, "value");
	if (prop != NULL)
	  value = atoi(prop);

	/* Get the note */
	temp = cur->xmlChildrenNode;
	if (temp != NULL)
	  command = atoi(temp->content);

	/* Set the command */
	block_set_command_full(block, line, track, commandpage, command, value);
      }
      cur = cur->next;
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected block, got %s\n", cur->name);

  return block;
}

/* Saves a block to an XML document */
void block_save(struct block *block, int number, xmlNodePtr parent)
{
  char c[10];
  int i, j, k;
  xmlNodePtr node, subnode;

  if (block == NULL) {
    fprintf(stderr, "block_save() called with null block\n");
    return;
  }

  /* Set block properties */
  node = xmlNewChild(parent, NULL, "block", NULL);
  snprintf(c, 10, "%d", number);
  xmlSetProp(node, "number", c);
  xmlSetProp(node, "name", block->name);
  snprintf(c, 10, "%d", block->tracks);
  xmlSetProp(node, "tracks", c);
  snprintf(c, 10, "%d", block->length);
  xmlSetProp(node, "length", c);
  snprintf(c, 10, "%d", block->commandpages);
  xmlSetProp(node, "commandpages", c);
  xmlAddChild(node, xmlNewText("\n"));

  /* Notation data */
  for (i = 0; i < block->tracks; i++)
    for (j = 0; j < block->length; j++)
      if (block->notes[2 * (block->tracks * j + i)] != 0 || block->notes[2 * (block->tracks * j + i) + 1] != 0) {
	snprintf(c, 10, "%d", block->notes[2 * (block->tracks * j + i)]);
	subnode = xmlNewChild(node, NULL, "note", c);
	snprintf(c, 10, "%d", j);
	xmlSetProp(subnode, "line", c);
	snprintf(c, 10, "%d", i);
	xmlSetProp(subnode, "track", c);
	snprintf(c, 10, "%d", block->notes[2 * (block->tracks * j + i) + 1]);
	xmlSetProp(subnode, "instrument", c);
	xmlAddChild(node, xmlNewText("\n"));
      }

  /* Command data */
  for (i = 0; i < block->tracks; i++)
    for (j = 0; j < block->length; j++)
      for (k = 0; k < block->commandpages; k++)
	if (block->commands[k * 2 * block->tracks * block->length + 2 * (block->tracks * j + i)] != 0 || block->commands[k * 2 * block->tracks * block->length + 2 * (block->tracks * j + i) + 1] != 0) {
	  snprintf(c, 10, "%d", block->commands[k * 2 * block->tracks * block->length + 2 * (block->tracks * j + i)]);
	  subnode = xmlNewChild(node, NULL, "command", c);
	  snprintf(c, 10, "%d", j);
	  xmlSetProp(subnode, "line", c);
	  snprintf(c, 10, "%d", i);
	  xmlSetProp(subnode, "track", c);
	  snprintf(c, 10, "%d", k);
	  xmlSetProp(subnode, "commandpage", c);
	  snprintf(c, 10, "%d", block->commands[k * 2 * block->tracks * block->length + 2 * (block->tracks * j + i) + 1]);
	  xmlSetProp(subnode, "value", c);
	  xmlAddChild(node, xmlNewText("\n"));
	}

  xmlAddChild(parent, xmlNewText("\n"));
}

/* Transposes a block or a part of it */
void block_transpose(struct block *block, int instrument, int halfnotes, int starttrack, int startline, int endtrack, int endline)
{
  int l, t;

  if (block == NULL) {
    fprintf(stderr, "block_transpose() called with null block\n");
    return;
  }

  /* Bounds checking */
  if (starttrack < 0)
    starttrack = 0;
  if (startline < 0)
    startline = 0;
  if (starttrack >= block->tracks)
    starttrack = block->tracks - 1;
  if (startline >= block->length)
    startline = block->length - 1;
  if (endtrack < 0)
    endtrack = 0;
  if (endline < 0)
    endline = 0;
  if (endtrack >= block->tracks)
    endtrack = block->tracks - 1;
  if (endline >= block->length)
    endline = block->length - 1;

  /* Make sure start and end are the correct way around */
  if (starttrack > endtrack) {
    t = starttrack;
    starttrack = endtrack;
    endtrack = t;
  }
  if (startline > endline) {
    l = startline;
    startline = endline;
    endline = l;
  }
    
  /* Either check for correct instrument or transpose all instruments */
  if (instrument < 0)
    for (l = startline; l <= endline; l++)
      for (t = starttrack; t <= endtrack; t++) {
	if (block->notes[(l * block->tracks + t) * 2] > 0) {
	  if (block->notes[(l * block->tracks + t) * 2] + halfnotes < 1)
	    block->notes[(l * block->tracks + t) * 2] = 1;
	  else if (block->notes[(l * block->tracks + t) * 2] + halfnotes > 127)
	    block->notes[(l * block->tracks + t) * 2] = 127;
	  else
	    block->notes[(l * block->tracks + t) * 2] += halfnotes;
	}
  } else
    for (l = startline; l <= endline; l++)
      for (t = starttrack; t <= endtrack; t++) {
	if (block->notes[(l * block->tracks + t) * 2] > 0 && block->notes[(l * block->tracks + t) * 2 + 1] == instrument + 1) {
	  if (block->notes[(l * block->tracks + t) * 2] + halfnotes < 0)
	    block->notes[(l * block->tracks + t) * 2] = 1;
	  else if(block->notes[(l * block->tracks + t) * 2] + halfnotes > 127)
	    block->notes[(l * block->tracks + t) * 2] = 127;
	  else
	    block->notes[(l * block->tracks + t) * 2] += halfnotes;
	}
      }
}

/* Expands/shrinks a block or a part of it */
void block_expandshrink(struct block *block, int factor, int starttrack, int startline, int endtrack, int endline)
{
  int l, t, p;

  if (block == NULL) {
    fprintf(stderr, "block_expandshrink() called with null block\n");
    return;
  }

  if (factor > -2 && factor < 2)
    return;

  /* Bounds checking */
  if (starttrack < 0)
    starttrack = 0;
  if (startline < 0)
    startline = 0;
  if (starttrack >= block->tracks)
    starttrack = block->tracks - 1;
  if (startline >= block->length)
    startline = block->length - 1;
  if (endtrack < 0)
    endtrack = 0;
  if (endline < 0)
    endline = 0;
  if (endtrack >= block->tracks)
    endtrack = block->tracks - 1;
  if (endline >= block->length)
    endline = block->length - 1;

  /* Make sure start and end are the correct way around */
  if (starttrack > endtrack) {
    t = starttrack;
    starttrack = endtrack;
    endtrack = t;
  }
  if (startline > endline) {
    l = startline;
    startline = endline;
    endline = l;
  }
    
  if (factor < 0) {
    /* Shrink */
    for (l = startline; l <= endline; l++)
      if ((l - startline) < (endline + 1 - startline) / -factor) {
	for (t = starttrack; t <= endtrack; t++) {
	  block->notes[(l * block->tracks + t) * 2] = block->notes[((startline + (l - startline) * -factor) * block->tracks + t) * 2];
	  block->notes[(l * block->tracks + t) * 2 + 1] = block->notes[((startline + (l - startline) * -factor) * block->tracks + t) * 2 + 1];
	  for (p = 0; p < block->commandpages; p++) {
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2] = block->commands[p * block->tracks * block->length * 2 + ((startline + (l - startline) * -factor) * block->tracks + t) * 2];
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2 + 1] = block->commands[p * block->tracks * block->length * 2 + ((startline + (l - startline) * -factor) * block->tracks + t) * 2 + 1];
	  }
	}
      } else {
	for (t = starttrack; t <= endtrack; t++) {
	  block->notes[(l * block->tracks + t) * 2] = 0;
	  block->notes[(l * block->tracks + t) * 2 + 1] = 0;
	  for (p = 0; p < block->commandpages; p++) {
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2] = 0;
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2 + 1] = 0;
	  }
	}
      }
  } else {
    /* Expand */
    for (l = endline; l >= startline; l--)
      if ((l - startline) % factor == 0) {
	for (t = starttrack; t <= endtrack; t++) {
	  block->notes[(l * block->tracks + t) * 2] = block->notes[((startline + (l - startline) / factor) * block->tracks + t) * 2];
	  block->notes[(l * block->tracks + t) * 2 + 1] = block->notes[((startline + (l - startline) / factor) * block->tracks + t) * 2 + 1];
	  for (p = 0; p < block->commandpages; p++) {
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2] = block->commands[p * block->tracks * block->length * 2 + ((startline + (l - startline) / factor) * block->tracks + t) * 2];
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2 + 1] = block->commands[p * block->tracks * block->length * 2 + ((startline + (l - startline) / factor) * block->tracks + t) * 2 + 1];
	  }
	}
      } else {
	for (t = starttrack; t <= endtrack; t++) {
	  block->notes[(l * block->tracks + t) * 2] = 0;
	  block->notes[(l * block->tracks + t) * 2 + 1] = 0;
	  for (p = 0; p < block->commandpages; p++) {
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2] = 0;
	    block->commands[p * block->tracks * block->length * 2 + (l * block->tracks + t) * 2 + 1] = 0;
	  }
	}
      }
  }
}

/* Changes an instrument to another */
void block_changeinstrument(struct block *block, int from, int to, int swap, int starttrack, int startline, int endtrack, int endline)
{
  int l, t;

  if (block == NULL) {
    fprintf(stderr, "block_changeinstrument() called with null block\n");
    return;
  }

  /* Bounds checking */
  if (starttrack < 0)
    starttrack = 0;
  if (startline < 0)
    startline = 0;
  if (starttrack >= block->tracks)
    starttrack = block->tracks - 1;
  if (startline >= block->length)
    startline = block->length - 1;
  if (endtrack < 0)
    endtrack = 0;
  if (endline < 0)
    endline = 0;
  if (endtrack >= block->tracks)
    endtrack = block->tracks - 1;
  if (endline >= block->length)
    endline = block->length - 1;

  /* Make sure start and end are the correct way around */
  if (starttrack > endtrack) {
    t = starttrack;
    starttrack = endtrack;
    endtrack = t;
  }
  if (startline > endline) {
    l = startline;
    startline = endline;
    endline = l;
  }
    
  if (swap)
    for (l = endline; l >= startline; l--)
      for (t = starttrack; t <= endtrack; t++) {
	if (block->notes[(l * block->tracks + t) * 2 + 1] == from)
	  block->notes[(l * block->tracks + t) * 2 + 1] = to;
	else if (block->notes[(l * block->tracks + t) * 2 + 1] == to)
	  block->notes[(l * block->tracks + t) * 2 + 1] = from;
  } else
    for (l = endline; l >= startline; l--)
      for (t = starttrack; t <= endtrack; t++)
	if (block->notes[(l * block->tracks + t) * 2 + 1] == from)
	  block->notes[(l * block->tracks + t) * 2 + 1] = to;
}
