/* gui.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <gmodule.h>
#include <gnome.h>
#include <glade/glade.h>

#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"

#define new_column gtk_tree_view_column_new_with_attributes

gboolean gui_quit = FALSE;

/* Return the length of an integer in string representation */
static int intlen(int i)
{
  if (i == 0)
    return 1;

  if (i < 0)
    return (int)log10((double)-i) + 2;

  return (int)log10((double)i) + 1;
}

/* Connects signals to objects and passes the gui struct on as user_data */
void gui_gladexmlconnectfunc(const gchar * handler_name, GObject * object, const gchar * signal_name, const gchar * signal_data, GObject * connect_object, gboolean after, gpointer user_data)
{
  GCallback func;

  /* Use the gmodule interface to resolve the function name to a pointer */
  if (!g_module_symbol(g_module_open(NULL, 0), handler_name, (gpointer *) & func))
    g_warning("could not find signal handler '%s'.", handler_name);

  /* Connect the signal */
  if (after)
    g_signal_connect_after(object, signal_name, func, user_data);
  else
    g_signal_connect(object, signal_name, func, user_data);

};

void gtk_main_quit()
{
  gui_quit = TRUE;
}

/* Initializes a GUI */
struct gui *gui_open(struct editor *editor, char *filename)
{
  struct gui *gui = (struct gui *)calloc(1, sizeof(struct gui));
  GtkCellRenderer *renderer;
  GtkTreeView *treeview;
  GtkTreeSelection *select;

  gui->editor = editor;

  /* Mutex for the refreshing function */
  gui->refresh_mutex = g_mutex_new();

  /* Load the GUI from a GLADE file and connect signals */
  gui->xml = glade_xml_new(filename, NULL, NULL);
  glade_xml_signal_autoconnect_full(gui->xml, gui_gladexmlconnectfunc, gui);
  gui->window_main = glade_xml_get_widget(gui->xml, "window_main");
  gtk_widget_add_events(gui->window_main, GDK_KEY_RELEASE_MASK);

  /* Create the MVC classes for various list view widgets */
  renderer = gtk_cell_renderer_text_new();

  /* Section list */
  gui->store_sectionlist = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_sectionlist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_sectionlist));
  gtk_tree_view_append_column(treeview, new_column("Section", renderer, "text", 0, NULL));
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence", renderer, "text", 1, NULL));
  gtk_tree_view_append_column(treeview, new_column("Playing Sequence Name", renderer, "text", 2, NULL));

  /* Playing Sequence */
  gui->store_playseq = gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_playseq_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_playseq));
  gtk_tree_view_append_column(treeview, new_column("Position", renderer, "text", 0, NULL));
  gtk_tree_view_append_column(treeview, new_column("Block Number", renderer, "text", 1, NULL));
  gtk_tree_view_append_column(treeview, new_column("Block Name", renderer, "text", 2, NULL));

  /* Block List */
  gui->store_blocklist = gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_blocklist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_blocklist));
  gtk_tree_view_append_column(treeview, new_column("Block Number", renderer, "text", 0, NULL));
  gtk_tree_view_append_column(treeview, new_column("Block Name", renderer, "text", 1, NULL));

  /* Message List */
  gui->store_messagelist = gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"));
  select = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT(select), "changed", G_CALLBACK(on_selection_messagelist_changed), gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_messagelist));
  gtk_tree_view_append_column(treeview, new_column("Message Number", renderer, "text", 0, NULL));
  gtk_tree_view_append_column(treeview, new_column("Message Name", renderer, "text", 1, NULL));

  /* Set the keyboard octave */
  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_keyboard")), 3);
  gui->editor->octave = 3;

  /* Set default path in file dialogs */
  if(gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL) != NULL) {
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_file_open")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_file_saveas")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/songpath", NULL));
  }
  if(gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL) != NULL) {
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_message_load")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL));
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(glade_xml_get_widget(gui->xml, "fileselection_message_save")), gconf_client_get_string(editor->gconfclient, "/apps/tutka/messagepath", NULL));
  }

  return gui;
}

/* Closes a GUI */
void gui_close(struct gui *gui)
{
  g_mutex_free(gui->refresh_mutex);

  free(gui);
}

/* Refreshes all parts of the GUI */
void gui_refresh_all(struct gui *gui)
{
  /* Reset the tracker */
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  tracker_reset(tracker);
  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_num_channels(tracker, gui->editor->song->blocks[gui->editor->block]->tracks);

  /* Refresh the GUI */
  gui_info_refresh(gui);
  gui_tempo_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
  gui_instrument_refresh(gui, TRUE);
  gui_messagelist_refresh(gui, TRUE);
  gui_trackvolumes_refresh(gui);
  gui_song_refresh(gui);
  gui_window_title_refresh(gui);

  /* Show the main window */
  gtk_widget_show(gui->window_main);
  gtk_widget_show(GTK_WIDGET(tracker));

  /* FIX: Tracker widget kludge */
  tracker->cursor_ch = 0;
}

Tracker *gui_get_tracker(struct gui *gui)
{
  return (Tracker *) glade_xml_get_widget(gui->xml, "tracker");
}

int gui_get_tracker_position(struct gui *gui)
{
  Tracker *tracker = (Tracker *) glade_xml_get_widget(gui->xml, "tracker");

  return tracker->patpos;
}

/* Set the refresh mask */
void gui_set_refresh(struct gui *gui, unsigned int mask)
{
  if (g_thread_self() != gui->editor->thread)
    g_mutex_lock(gui->refresh_mutex);
  gui->refresh_mask |= mask;
  if (g_thread_self() != gui->editor->thread)
    g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh all those parts of the GUI that need to be refreshed */
void gui_refresh(struct gui *gui)
{
  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INSTRUMENT) {
    gui_instrument_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_INSTRUMENT;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TEMPO) {
    gui_tempo_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TEMPO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_INFO) {
    gui_info_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_INFO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_WINDOW_TITLE) {
    gui_window_title_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_WINDOW_TITLE;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_STATUSBAR) {
    gui_statusbar_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_STATUSBAR;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TIMER) {
    gui_timer_refresh(gui, gui->editor->time);
    gui->refresh_mask ^= GUI_REFRESH_TIMER;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_BLOCKLIST) {
    gui_blocklist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_BLOCKLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_BLOCKLIST_INFO) {
    gui_blocklist_info_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_BLOCKLIST_INFO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ) {
    gui_playseq_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_PLAYSEQ_INFO) {
    gui_playseq_info_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_PLAYSEQ_INFO;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SECTIONLIST) {
    gui_sectionlist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_SECTIONLIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SECTIONLIST_BUTTONS) {
    gui_sectionlist_buttons_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_SECTIONLIST_BUTTONS;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_MESSAGELIST) {
    gui_messagelist_refresh(gui, FALSE);
    gui->refresh_mask ^= GUI_REFRESH_MESSAGELIST;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKVOLUMES) {
    gui_trackvolumes_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKVOLUMES;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_SONG) {
    gui_song_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_SONG;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();

  g_mutex_lock(gui->refresh_mutex);
  if (gui->refresh_mask & GUI_REFRESH_TRACKER) {
    gui_tracker_refresh(gui);
    gui->refresh_mask ^= GUI_REFRESH_TRACKER;
  }
  g_mutex_unlock(gui->refresh_mutex);
  g_thread_yield();
}

/* Refresh the tracker */
void gui_tracker_refresh(struct gui *gui)
{
  Tracker *tracker = gui_get_tracker(gui);

  /* Check that the line is valid */
  if (gui->editor->line > gui->editor->song->blocks[gui->editor->block]->length)
    gui->editor->line = gui->editor->song->blocks[gui->editor->block]->length - 1;
  /* Check that the commad page is valid */
  if (gui->editor->cmdpage > gui->editor->song->blocks[gui->editor->block]->commandpages)
    gui->editor->cmdpage = gui->editor->song->blocks[gui->editor->block]->commandpages - 1;

  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_patpos(tracker, gui->editor->line);
  tracker_set_cmdpage(tracker, gui->editor->cmdpage);
}

/* Refresh the song properties window widgets */
void gui_song_refresh(struct gui *gui)
{
  if (gui->editor->song->name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_songname")), gui->editor->song->name);

  if (gui->editor->song->sendsync)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_sendsync")), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "checkbutton_sendsync")), FALSE);
}

void gui_instrument_refresh(struct gui *gui, gboolean refresh_spin)
{
  char instrument[3];
  snprintf(instrument, 3, "%.2d", gui->editor->instrument + 1);

  /* Refresh the main window widgets */
  /* Only refresh the spin button if necessary to avoid forever loops */
  if (refresh_spin)
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_instrument")), gui->editor->instrument + 1);
  if (gui->editor->song->instruments[gui->editor->instrument]->name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrument")), "");

  /* Refresh the instrument window widgets */
  if (gui->editor->song->instruments[gui->editor->instrument]->name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrumentproperties_name")), gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_instrumentproperties_name")), "");
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_volume"))), gui->editor->song->instruments[gui->editor->instrument]->defaultvelocity);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_transpose"))), gui->editor->song->instruments[gui->editor->instrument]->transpose);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_hold"))), gui->editor->song->instruments[gui->editor->instrument]->hold);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_instrumentproperties_midich"))), gui->editor->song->instruments[gui->editor->instrument]->midichannel + 1);
}

void gui_tempo_refresh(struct gui *gui)
{
  /* Refresh the song properties window widgets */
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tempo"))), gui->editor->song->tempo);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "scale_tpl"))), gui->editor->song->ticksperline);
}

void gui_window_title_refresh(struct gui *gui)
{
  gtk_window_set_title(GTK_WINDOW(gui->window_main), gui->editor->song->name);
}

void gui_statusbar_refresh(struct gui *gui)
{
  GtkLabel *status = GTK_LABEL(glade_xml_get_widget(gui->xml, "label_status"));
  
  /* Status bar */
  switch (gui->editor->player->mode) {
  case MODE_PLAY_SONG:
    gtk_label_set_text(status, "Playing song");
    break;
  case MODE_PLAY_BLOCK:
    gtk_label_set_text(status, "Playing block");
    break;
  case MODE_IDLE:
    gtk_label_set_text(status, "Stopped");
    break;
  }
}

void gui_info_refresh(struct gui *gui)
{
  char *section, *playseq, *block;
  int sectionlength=8+intlen(gui->editor->section + 1)+1+intlen(gui->editor->song->numsections)+1;
  int playseqlength=9+intlen(gui->editor->position + 1)+1+intlen(gui->editor->song->playseqs[gui->editor->playseq]->length)+1;
  int blocklength;
  
  if (gui->editor->song->blocks[gui->editor->block]->name != NULL)
    blocklength=6+intlen(gui->editor->block + 1)+1+intlen(gui->editor->song->numblocks)+15+intlen(gui->editor->cmdpage + 1)+1+intlen(gui->editor->song->blocks[gui->editor->block]->commandpages)+3+strlen(gui->editor->song->blocks[gui->editor->block]->name)+1;
  else
    blocklength=6+intlen(gui->editor->block + 1)+1+intlen(gui->editor->song->numblocks)+15+intlen(gui->editor->cmdpage + 1)+1+intlen(gui->editor->song->blocks[gui->editor->block]->commandpages)+2;

  section=malloc(sectionlength);
  playseq=malloc(playseqlength);
  block=malloc(blocklength);

  /* Refresh the main window widgets */
  snprintf(section, sectionlength, "Section %d/%d", gui->editor->section + 1, gui->editor->song->numsections);
  snprintf(playseq, playseqlength, "Position %d/%d", gui->editor->position + 1, gui->editor->song->playseqs[gui->editor->playseq]->length);
  if (gui->editor->song->blocks[gui->editor->block]->name != NULL)
    snprintf(block, blocklength, "Block %d/%d (Command Page %d/%d): %s", gui->editor->block + 1, gui->editor->song->numblocks, gui->editor->cmdpage + 1, gui->editor->song->blocks[gui->editor->block]->commandpages, gui->editor->song->blocks[gui->editor->block]->name);
  else
    snprintf(block, blocklength, "Block %d/%d (Command Page %d/%d)", gui->editor->block + 1, gui->editor->song->numblocks, gui->editor->cmdpage + 1, gui->editor->song->blocks[gui->editor->block]->commandpages);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_section")), section);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_position")), playseq);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_block")), block);

  free(section);
  free(playseq);
  free(block);
}

void gui_timer_refresh(struct gui *gui, unsigned int seconds)
{
  char time[6];

  /* Refresh the main window widgets if the window exists */
  if (gui->window_main != NULL && GTK_IS_WINDOW(gui->window_main)) {
    snprintf(time, 6, "%.2d:%.2d", seconds / 60, seconds % 60);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_timer")), time);
  }
}

/* Updates the current block information in the block list window */
void gui_blocklist_info_refresh(struct gui *gui)
{
  GtkEntry *name = GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_blocklist"));
  char label[6 + 8];

  /* Show the number of the selected block */
  snprintf(label, 6 + 8, "Block %d/%d", gui->editor->block + 1, gui->editor->song->numblocks);
  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "frame_blocklist")), label);

  /* Show the name of the selected block */
  if (gui->editor->song->blocks[gui->editor->block]->name != NULL)
    gtk_entry_set_text(name, gui->editor->song->blocks[gui->editor->block]->name);
  else
    gtk_entry_set_text(name, "");

  /* Show the properties of the selected block */
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_blocklist_tracks"))), gui->editor->song->blocks[gui->editor->block]->tracks);
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_blocklist_length"))), gui->editor->song->blocks[gui->editor->block]->length);
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_blocklist_commandpages"))), gui->editor->song->blocks[gui->editor->block]->commandpages);

  /* Make the delete button sensitive/insensitive */
  if (gui->editor->song->numblocks > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_blocklist_delete"), FALSE);

}

/* Recreates and refreshes the block list window contents */
void gui_blocklist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char block[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(block, 10, "%d", gui->editor->block);

  if (recreate) {
    /* Recreate the block list */
    gtk_list_store_clear(gui->store_blocklist);
    for (i = 0; i < gui->editor->song->numblocks; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_blocklist, &iter);
      gtk_list_store_set(gui->store_blocklist, &iter, 0, i + 1, 1, gui->editor->song->blocks[i]->name, -1);
    }
  }

  /* Move back to the previously selected block */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_blocklist"));
  treepath = gtk_tree_path_new_from_string(block);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  gui_blocklist_info_refresh(gui);
}

/* Updates the current playseq information in the playing sequence window */
void gui_playseq_info_refresh(struct gui *gui)
{
  char position[9 + 8], block[6 + 8], title[24];

  /* Show current block */
  snprintf(position, 9 + 8, "Position %d/%d", gui->editor->position + 1, gui->editor->song->playseqs[gui->editor->playseq]->length);
  snprintf(block, 6 + 8, "Block %d/%d", gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position] + 1, gui->editor->song->numblocks);
  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "frame_playseq")), position);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "label_playseq_block")), block);

  /* Show playing sequence name */
  if (gui->editor->song->playseqs[gui->editor->playseq]->name != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_playseq_name")), gui->editor->song->playseqs[gui->editor->playseq]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_playseq_name")), "");

  /* Set window title */
  snprintf(title, 24, "Playing Sequence %d/%d", gui->editor->playseq + 1, gui->editor->song->numplayseqs);
  gtk_window_set_title(GTK_WINDOW(glade_xml_get_widget(gui->xml, "dialog_playseq")), title);

  /* Make the delete position button sensitive/insensitive */
  if (gui->editor->song->playseqs[gui->editor->playseq]->length > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_delete")), FALSE);

  /* Make the next/previous block buttons sensitive/insensitive */
  if (gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position] > 0)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_prev")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_prev")), FALSE);
  if (gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position] < gui->editor->song->numblocks - 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_next")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_next")), FALSE);

  /* Make the delete playing sequence button sensitive/insensitive */
  if (gui->editor->song->numplayseqs > 1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_deletethis")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "button_playseq_deletethis")), FALSE);
}

void gui_playseq_refresh(struct gui *gui, gboolean recreate)
{
  /* Refresh the playing sequence window widgets */
  int i;
  char playseqpos[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(playseqpos, 10, "%d", gui->editor->position);

  if (recreate) {
    /* Recreate the playing sequence */
    gtk_list_store_clear(gui->store_playseq);
    for (i = 0; i < gui->editor->song->playseqs[gui->editor->playseq]->length; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_playseq, &iter);
      gtk_list_store_set(gui->store_playseq, &iter, 0, i + 1, 1, gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i] + 1, 2, gui->editor->song->blocks[gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i]]->name, -1);
    }

  }
  /* Move back to the previously selected position */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_playseq"));
  treepath = gtk_tree_path_new_from_string(playseqpos);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  gui_playseq_info_refresh(gui);
}

/* Refresh section list buttons */
void gui_sectionlist_buttons_refresh(struct gui *gui)
{
  /* Make the delete button sensitive/insensitive */
  if (gui->editor->song->numsections > 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_delete"), FALSE);

  /* Make the next/previous buttons sensitive/insensitive */
  if (gui->editor->song->sections[gui->editor->section] > 0)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_prev"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_prev"), FALSE);
  if (gui->editor->song->sections[gui->editor->section] < gui->editor->song->numplayseqs - 1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_next"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_sectionlist_next"), FALSE);
}

/* Refresh all section list widgets */
void gui_sectionlist_refresh(struct gui *gui, gboolean recreate)
{
  int i;
  char section[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(section, 10, "%d", gui->editor->section);

  if (recreate) {
    /* Recreate the section list */
    gtk_list_store_clear(gui->store_sectionlist);
    for (i = 0; i < gui->editor->song->numsections; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_sectionlist, &iter);
      gtk_list_store_set(gui->store_sectionlist, &iter, 0, i + 1, 1, gui->editor->song->sections[i] + 1, 2, gui->editor->song->playseqs[gui->editor->song->sections[i]]->name, -1);
    }
  }

  /* Move back to the previously selected section */
  treeview = GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_sectionlist"));
  treepath = gtk_tree_path_new_from_string(section);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview), treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  gui_sectionlist_buttons_refresh(gui);
}

void gui_messagelist_refresh(struct gui *gui, gboolean recreate)
{
  /* Refresh the message list widgets */
  int i;
  char oldmessage[10], message[6 + 8];

  if (gui->editor->song->nummessages == 0)
    gui->messagelist_message = -1;

  snprintf(oldmessage, 10, "%d", gui->messagelist_message);

  if (recreate) {
    /* Recreate the Message list */
    gtk_list_store_clear(gui->store_messagelist);
    for (i = 0; i < gui->editor->song->nummessages; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_messagelist, &iter);
      gtk_list_store_set(gui->store_messagelist, &iter, 0, i, 1, gui->editor->song->messages[i]->name, -1);
    }
  }

  /* Move back to the previously selected message and make widgets sensitive */
  if (gui->messagelist_message != -1) {
    gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "treeview_messagelist"))), gtk_tree_path_new_from_string(oldmessage));

    /* Show the number of the selected message */
    snprintf(message, 6 + 8, "Message %d/%d", gui->messagelist_message + 1, gui->editor->song->nummessages);
    gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "frame_messagelist_controls")), message);

    /* Display the properties of the message */
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_message_length")), gui->editor->song->messages[gui->messagelist_message]->length);
    if(gui->editor->song->messages[gui->messagelist_message]->name != NULL)
      gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_message_name")), gui->editor->song->messages[gui->messagelist_message]->name);
    else
      gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_message_name")), "");

    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "table_messagelist_controls_1"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "checkbutton_message_autostop"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), TRUE);
  } else {
    gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "frame_messagelist_controls")), "Message 0/0");
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "table_messagelist_controls_1"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_send"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_receive"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_load"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_save"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "checkbutton_message_autostop"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "button_message_delete"), FALSE);
  }
}

void gui_trackvolumes_refresh(struct gui *gui)
{
  /* Refresh the track volumes window widgets */
  int i;
  char track[10];
  GtkWidget *table = glade_xml_get_widget(gui->xml, "table_trackvolumes");

  /* Destroy all children */
  gtk_container_foreach(GTK_CONTAINER(table), (GtkCallback) gtk_widget_destroy, NULL);

  gtk_table_resize(GTK_TABLE(table), 3, gui->editor->song->maxtracks);

  /* Create widgets for each channel */
  for (i = 0; i < gui->editor->song->maxtracks; i++) {
    GtkWidget *w, *x;

    /* Track numbers */
    snprintf(track, 10, "Track %d", i + 1);
    x = gtk_label_new(track);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 0, 1, GTK_EXPAND | GTK_FILL, GTK_FILL, 2, 4);

    /* Volume sliders */
    w = (GtkWidget *) gtk_adjustment_new((gui->editor->song->trackvolumes[i]) & 127, 0, 127, 1, 8, 0);
    g_signal_connect(G_OBJECT(w), "value_changed", G_CALLBACK(on_adjustment_trackvolumes_changed), (gpointer) & gui->editor->song->trackvolumes[i]);
    x = gtk_vscale_new(GTK_ADJUSTMENT(w));
    gtk_range_set_inverted(GTK_RANGE(x), TRUE);
    gtk_scale_set_digits(GTK_SCALE(x), 0);
    gtk_table_attach(GTK_TABLE(table), x, i, i + 1, 1, 2, GTK_EXPAND, GTK_EXPAND | GTK_FILL, 2, 4);

    /* Mute buttons */
    w = gtk_check_button_new_with_label("Mute");
    if (gui->editor->song->trackvolumes[i] > 127)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), TRUE);
    g_signal_connect(G_OBJECT(w), "toggled", G_CALLBACK(on_togglebutton_trackvolumes_toggled), (gpointer) & gui->editor->song->trackvolumes[i]);
    gtk_table_attach(GTK_TABLE(table), w, i, i + 1, 2, 3, GTK_EXPAND, GTK_FILL, 2, 4);
  }

  gtk_widget_show_all(table);
}

void gui_preferences_refresh(struct gui *gui)
{
  /* Refresh the preferences window widgets */
  if (gconf_client_get_string(gui->editor->gconfclient, "/apps/tutka/mididevice", NULL) != NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_preferences_midi_device")), gconf_client_get_string(gui->editor->gconfclient, "/apps/tutka/mididevice", NULL));
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "entry_preferences_midi_device")), "");

  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_preferences_scheduling_mode")), gconf_client_get_int(gui->editor->gconfclient, "/apps/tutka/scheduler", NULL)); 
}
