/* song.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <string.h>
#include <libxml/tree.h>
#include "song.h"

/* Allocates a new song structure and initializes it to default values */
struct song *song_alloc(void)
{
  struct song *song;
  int i;

  song = (struct song *)calloc(1, sizeof(struct song));
  song->name = (char *)strdup("Untitled");
  song->tempo = 130;
  song->ticksperline = 6;
  song->numsections = 1;
  song->sections = (unsigned int *)calloc(1, sizeof(unsigned int));
  song->numplayseqs = 1;
  song->playseqs = (struct playseq **)calloc(1, sizeof(struct playseq *));
  song->playseqs[0] = playseq_alloc();
  song->numblocks = 1;
  song->blocks = (struct block **)calloc(1, sizeof(struct block *));
  song->blocks[0] = block_alloc(4, 64, 1);
  song->numinstruments = 1;
  song->instruments = (struct instrument **)calloc(1, sizeof(struct instrument *));
  song->instruments[0] = instrument_alloc();
  song->maxtracks = 4;
  song->trackvolumes = (unsigned int *)calloc(4, sizeof(unsigned int));
  song->mastervolume = 127;
  for (i = 0; i < song->maxtracks; i++)
    song->trackvolumes[i] = 127;

  return (song);
}

/* Frees a song structure and its contents */
void song_free(struct song *song)
{
  if (song == NULL) {
    fprintf(stderr, "song_free() called with null song\n");
    return;
  }

  if (song->name)
    free(song->name);

  if (song->numsections > 0)
    free(song->sections);

  if (song->numplayseqs > 0) {
    int i;

    for (i = 0; i < song->numplayseqs; i++)
      playseq_free(song->playseqs[i]);

    free(song->playseqs);
  }

  if (song->numblocks > 0) {
    int i;

    for (i = 0; i < song->numblocks; i++)
      block_free(song->blocks[i]);

    free(song->blocks);
  }

  if (song->numinstruments > 0) {
    int i;

    for (i = 0; i < song->numinstruments; i++)
      if (song->instruments[i] != NULL)
	instrument_free(song->instruments[i]);

    free(song->instruments);
  }

  if (song->maxtracks > 0)
    free(song->trackvolumes);

  free(song);
}

/* Inserts a new block in the block array in the given position */
void song_insert_block(struct song *song, int pos, int current)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_insert_block() called with null song\n");
    return;
  }

  /* Check block existence */
  if (pos < 0)
    pos = 0;
  else if (pos > song->numblocks)
    pos = song->numblocks;

  song->numblocks++;
  /* Reallocate the block array */
  song->blocks = (struct block **)realloc(song->blocks, song->numblocks * sizeof(struct block *));

  /* Move rest of the blocks onwards */
  for (i = song->numblocks - 1; i > pos; i--)
    song->blocks[i] = song->blocks[i - 1];

  /* Insert a new block similar to the current block */
  song->blocks[pos] = block_alloc(song->blocks[current]->tracks, song->blocks[current]->length, song->blocks[current]->commandpages);

  /* Update playing sequences */
  for (i = 0; i < song->numplayseqs; i++) {
    int j;

    for (j = 0; j < song->playseqs[i]->length; j++)
      if (song->playseqs[i]->blocknumbers[j] >= pos)
	song->playseqs[i]->blocknumbers[j]++;
  }
}

/* Deletes a block from the given position of the block array */
void song_delete_block(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_delete_block() called with null song\n");
    return;
  }

  /* Check block existence */
  if (pos < 0)
    pos = 0;
  else if (pos >= song->numblocks)
    pos = song->numblocks - 1;

  /* Don't delete the last block */
  if (song->numblocks > 1) {
    /* Free the block in question */
    block_free(song->blocks[pos]);
    song->numblocks--;

    /* Move rest of the blocks backwards */
    for (i = pos; i < song->numblocks; i++)
      song->blocks[i] = song->blocks[i + 1];

    /* Reallocate the block array */
    song->blocks = (struct block **)realloc(song->blocks, song->numblocks * sizeof(struct block *));

    /* Update playing sequences */
    for (i = 0; i < song->numplayseqs; i++) {
      int j;

      for (j = 0; j < song->playseqs[i]->length; j++)
	if (song->playseqs[i]->blocknumbers[j] >= pos && song->playseqs[i]->blocknumbers[j] > 0)
	  song->playseqs[i]->blocknumbers[j]--;
    }
  }
}

/* Inserts a new playseq in the playseq array in the given position */
void song_insert_playseq(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_insert_playseq() called with null song\n");
    return;
  }

  /* Check playseq existence */
  if (pos < 0)
    pos = 0;
  else if (pos > song->numplayseqs)
    pos = song->numplayseqs;

  song->numplayseqs++;
  /* Reallocate the playseqs array */
  song->playseqs = (struct playseq **)realloc(song->playseqs, song->numplayseqs * sizeof(struct playseq *));

  /* Move rest of the playseqs onwards */
  for (i = song->numplayseqs - 1; i > pos; i--)
    song->playseqs[i] = song->playseqs[i - 1];

  /* Insert a new playing sequence */
  song->playseqs[pos] = playseq_alloc();

  /* Update sections */
  for (i = 0; i < song->numsections; i++) {
    if (song->sections[i] >= pos)
      song->sections[i]++;
  }
}

/* Deletes a playseq from the given position of the playseq array */
void song_delete_playseq(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_delete_playseq() called with null song\n");
    return;
  }

  /* Check block existence */
  if (pos < 0)
    pos = 0;
  else if (pos >= song->numplayseqs)
    pos = song->numplayseqs - 1;

  /* Don't delete the last playseq */
  if (song->numplayseqs > 1) {
    /* Free the playseq to be deleted */
    playseq_free(song->playseqs[pos]);
    song->numplayseqs--;

    /* Move rest of the playseqs backwards */
    for (i = pos; i < song->numplayseqs; i++)
      song->playseqs[i] = song->playseqs[i + 1];

    /* Reallocate the playseqs array */
    song->playseqs = (struct playseq **)realloc(song->playseqs, song->numplayseqs * sizeof(struct playseq *));

    /* Update section lists */
    for (i = 0; i < song->numsections; i++) {
      if (song->sections[i] >= pos && song->sections[i] > 0)
	song->sections[i]--;
    }
  }
}


/* Inserts a new section in the section array in the given position */
void song_insert_section(struct song *song, int pos)
{
  int i;
  unsigned int old;

  if (song == NULL) {
    fprintf(stderr, "song_insert_section() called with null song\n");
    return;
  }

  /* Check that the value is possible */
  if (pos < 0)
    pos = 0;
  else if (pos > song->numsections)
    pos = song->numsections;

  /* Which playing sequence number to insert */
  if (pos < song->numsections)
    old = song->sections[pos];
  else
    old = song->sections[song->numsections - 1];

  song->numsections++;
  /* Reallocate the sections array */
  song->sections = (unsigned int *)realloc(song->sections, song->numsections * sizeof(unsigned int));

  /* Move rest of the sections onwards */
  for (i = song->numsections - 1; i > pos; i--)
    song->sections[i] = song->sections[i - 1];

  song->sections[pos] = old;
}

/* Deletes a section from the given position of the section array */
void song_delete_section(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_delete_section() called with null song\n");
    return;
  }

  /* Check block existence */
  if (pos < 0)
    pos = 0;
  else if (pos >= song->numsections)
    pos = song->numsections - 1;

  /* Don't delete the last section */
  if (song->numsections > 1) {
    song->numsections--;

    /* Move rest of the sections backwards */
    for (i = pos; i < song->numsections; i++)
      song->sections[i] = song->sections[i + 1];

    /* Reallocate the sections array */
    song->sections = (unsigned int *)realloc(song->sections, song->numsections * sizeof(unsigned int));
  }
}

/* Inserts a new message in the message array in the given position */
void song_insert_message(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_insert_message() called with null song\n");
    return;
  }

  /* Check message existence */
  if (pos < 0)
    pos = 0;
  else if (pos > song->nummessages)
    pos = song->nummessages;

  song->nummessages++;
  /* Reallocate the message array */
  song->messages = (struct message **)realloc(song->messages, song->nummessages * sizeof(struct message *));

  /* Move rest of the messages onwards */
  for (i = song->nummessages - 1; i > pos; i--)
    song->messages[i] = song->messages[i - 1];

  /* Insert a new message similar to the current message */
  song->messages[pos] = (struct message *)calloc(1, sizeof(struct message));
}

/* Deletes a message from the given position of the message array */
void song_delete_message(struct song *song, int pos)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_delete_message() called with null song\n");
    return;
  }

  /* Check message existence */
  if (pos < 0)
    pos = 0;
  else if (pos >= song->nummessages)
    pos = song->nummessages - 1;

  /* Don't delete inexisting messages */
  if (song->nummessages > 0) {
    /* Free the message in question */
    if (song->messages[pos]->data != NULL)
      free(song->messages[pos]->data);
    if (song->messages[pos]->name != NULL)
      free(song->messages[pos]->name);
    free(song->messages[pos]);
    song->nummessages--;

    /* Move rest of the messages backwards */
    for (i = pos; i < song->nummessages; i++)
      song->messages[i] = song->messages[i + 1];

    /* Reallocate the message array */
    song->messages = (struct message **)realloc(song->messages, song->nummessages * sizeof(struct message *));
  }
}

/* Sets a section in the given position to point somewhere */
void song_set_section(struct song *song, int pos, int playseq)
{
  if (song == NULL) {
    fprintf(stderr, "song_set_section() called with null song\n");
    return;
  }

  if (playseq >= 0 && playseq < song->numplayseqs)
    song->sections[pos] = playseq;
}

/* Sets the number of ticks per line for a song */
void song_set_tpl(struct song *song, unsigned int ticksperline)
{
  if (song == NULL) {
    fprintf(stderr, "song_set_tpl() called with null song\n");
    return;
  }

  song->ticksperline = ticksperline;
}

/* Sets the tempo of a song */
void song_set_tempo(struct song *song, unsigned int tempo)
{
  if (song == NULL) {
    fprintf(stderr, "song_set_tempo() called with null song\n");
    return;
  }

  song->tempo = tempo;
}

/* If the maximum number of tracks has changed recreate the track volumes */
int song_check_maxtracks(struct song *song)
{
  int i, oldmax, max = 0;

  if (song == NULL) {
    fprintf(stderr, "song_check_maxtracks() called with null song\n");
    return 0;
  }

  oldmax = song->maxtracks;

  /* Check the maximum number of tracks; */
  for (i = 0; i < song->numblocks; i++)
    if (song->blocks[i]->tracks > max)
      max = song->blocks[i]->tracks;

  /* If the maximum number of tracks as changed... */
  if (oldmax != max) {
    /* Reallocate volume array */
    song->trackvolumes = (unsigned int *)realloc(song->trackvolumes, max * sizeof(unsigned int));

    /* Set new tracks to volume 127 */
    for (i = song->maxtracks; i < max; i++)
      song->trackvolumes[i] = 127;

    song->maxtracks = max;
    return 1;
  } else
    return 0;
}

/* Make sure the instrument exists; add instruments if necessary */
void song_check_instrument(struct song *song, int instrument)
{
  if (song == NULL) {
    fprintf(stderr, "song_check_instrument() called with null song\n");
    return;
  }

  if (instrument >= song->numinstruments || song->instruments[instrument] == NULL) {
    if (instrument >= song->numinstruments) {
      int i;

      song->instruments = realloc(song->instruments, (instrument + 1) * sizeof(struct instrument *));
      for (i = song->numinstruments; i < (instrument + 1); i++)
	song->instruments[i] = instrument_alloc();

      song->numinstruments = instrument + 1;
    }
  }
}

/* Transposes all blocks in a song */
void song_transpose(struct song *song, int instruments, int halfnotes)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_transpose() called with null song\n");
    return;
  }

  for (i = 0; i < song->numblocks; i++)
    block_transpose(song->blocks[i], instruments, halfnotes, 0, 0, song->blocks[i]->tracks - 1, song->blocks[i]->length - 1);
}

/* Expands/shrinks all blocks in a song */
void song_expandshrink(struct song *song, int factor)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_expandshrink() called with null song\n");
    return;
  }

  for (i = 0; i < song->numblocks; i++)
    block_expandshrink(song->blocks[i], factor, 0, 0, song->blocks[i]->tracks - 1, song->blocks[i]->length - 1);
}

/* Changes an instrument to another in all blocks of a song */
void song_changeinstrument(struct song *song, int from, int to, int swap)
{
  int i;

  if (song == NULL) {
    fprintf(stderr, "song_changeinstrument() called with null song\n");
    return;
  }

  for (i = 0; i < song->numblocks; i++)
    block_changeinstrument(song->blocks[i], from, to, swap, 0, 0, song->blocks[i]->tracks - 1, song->blocks[i]->length - 1);
}

/* Loads a song from an XML file */
struct song *song_load(char *filename)
{
  xmlDocPtr doc;
  xmlNodePtr cur;

  /* Build an XML tree from a the file */
  if (!(doc = xmlParseFile(filename)))
    return NULL;

  /* Go ahead and parse the document */
  cur = xmlDocGetRootElement(doc);
  return song_parse(doc, NULL, cur);
}

/* Parses a song element in an XML file */
struct song *song_parse(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
  struct song *song = NULL;

  if ((!xmlStrcmp(cur->name, "song")) && (cur->ns == ns)) {
    char *prop;
    /* Allocate song */
    song = (struct song *)calloc(1, sizeof(struct song));
    song->name = xmlGetProp(cur, "name");
    prop = xmlGetProp(cur, "tempo");
    if (prop != NULL)
      song->tempo = atoi(prop);
    prop = xmlGetProp(cur, "ticksperline");
    if (prop != NULL)
      song->ticksperline = atoi(prop);
    prop = xmlGetProp(cur, "mastervolume");
    if (prop != NULL)
      song->mastervolume = atoi(prop);
    prop = xmlGetProp(cur, "sendsync");
    if (prop != NULL)
      song->sendsync = atoi(prop);

    cur = cur->xmlChildrenNode;
    while (cur != NULL) {
      if ((!xmlStrcmp(cur->name, "blocks")) && (cur->ns == ns)) {
	/* Parse and add all block elements */
	xmlNodePtr temp = cur->xmlChildrenNode;

	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int number = -1;
	    struct block *block = block_parse(doc, ns, temp);

	    if (block != NULL) {
	      prop = xmlGetProp(temp, "number");
	      if (prop != NULL)
		number = atoi(prop);

	      if (number >= song->numblocks) {
		int i;

		song->blocks = realloc(song->blocks, (number + 1) * sizeof(struct block *));
		for (i = song->numblocks; i <= number; i++)
		  song->blocks[i] = NULL;
	      }
	      if (song->blocks[number] != NULL)
		block_free(song->blocks[number]);
	      song->blocks[number] = block;
	      song->numblocks = number + 1;
	    }
	  }

	  temp = temp->next;
	}
      } else if ((!xmlStrcmp(cur->name, "sections")) && (cur->ns == ns)) {
	/* Parse and add all section elements */
	xmlNodePtr temp = cur->xmlChildrenNode;

	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int number = -1;

	    /* The section number is required */
	    if ((!xmlStrcmp(temp->name, "section")) && (temp->ns == ns)) {
	      xmlNodePtr temp2;
	      prop = xmlGetProp(temp, "number");
	      if (prop != NULL)
		number = atoi(prop);

	      if (number >= song->numsections) {
		int i;

		song->sections = realloc(song->sections, (number + 1) * sizeof(unsigned int));
		for (i = song->numsections; i <= number; i++)
		  song->sections[i] = 0;
	      }

	      /* Get playing sequence */
	      temp2 = temp->xmlChildrenNode;
	      if (temp2 != NULL)
		song->sections[number] = atoi(temp2->content);
	      song->numsections = number + 1;
	    } else if (temp->type != XML_COMMENT_NODE)
	      fprintf(stderr, "XML error: expected section, got %s\n", temp->name);
	  }

	  temp = temp->next;
	}
      } else if ((!xmlStrcmp(cur->name, "playingsequences")) && (cur->ns == ns)) {
	/* Parse and add all playingsequence elements */

	xmlNodePtr temp = cur->xmlChildrenNode;
	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int number = -1;
	    struct playseq *playseq = playseq_parse(doc, ns, temp);

	    if (playseq != NULL) {
	      prop = xmlGetProp(temp, "number");
	      if (prop != NULL)
		number = atoi(prop);

	      if (number >= song->numplayseqs) {
		int i;

		song->playseqs = realloc(song->playseqs, (number + 1) * sizeof(struct playseq *));
		for (i = song->numplayseqs; i <= number; i++)
		  song->playseqs[i] = NULL;
	      }
	      if (song->playseqs[number] != NULL)
		playseq_free(song->playseqs[number]);
	      song->playseqs[number] = playseq;
	      song->numplayseqs = number + 1;
	    }
	  }

	  temp = temp->next;
	}
      } else if ((!xmlStrcmp(cur->name, "instruments")) && (cur->ns == ns)) {
	/* Parse and add all instrument elements */
	xmlNodePtr temp = cur->xmlChildrenNode;

	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int number = -1;
	    struct instrument *instrument = instrument_parse(doc, ns, temp);

	    if (instrument != NULL) {
	      prop = xmlGetProp(temp, "number");
	      if (prop != NULL)
		number = atoi(prop);

	      if (number >= song->numinstruments) {
		int i;

		song->instruments = realloc(song->instruments, (number + 1) * sizeof(struct instrument *));
		for (i = song->numinstruments; i <= number; i++)
		  song->instruments[i] = NULL;
	      }
	      if (song->instruments[number] != NULL)
		instrument_free(song->instruments[number]);
	      song->instruments[number] = instrument;
	      song->numinstruments = number + 1;
	    }
	  }

	  temp = temp->next;
	}
      } else if ((!xmlStrcmp(cur->name, "trackvolumes")) && (cur->ns == ns)) {
	/* Parse and add all track volume elements */
	xmlNodePtr temp = cur->xmlChildrenNode;

	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int track = -1;

	    /* The track number is required */
	    if ((!xmlStrcmp(temp->name, "trackvolume")) && (temp->ns == ns)) {
	      xmlNodePtr temp2;
	      prop = xmlGetProp(temp, "track");
	      if (prop != NULL)
		track = atoi(prop);

	      if (track >= song->maxtracks) {
		int i;

		song->trackvolumes = realloc(song->trackvolumes, (track + 1) * sizeof(unsigned int));
		for (i = song->maxtracks; i <= track; i++)
		  song->trackvolumes[i] = 0;
	      }

	      /* Get the volume */
	      temp2 = temp->xmlChildrenNode;
	      if (temp2 != NULL)
		song->trackvolumes[track] = atoi(temp2->content);
	      song->maxtracks = track + 1;
	    } else if (temp->type != XML_COMMENT_NODE)
	      fprintf(stderr, "XML error: expected section, got %s\n", temp->name);
	  }

	  temp = temp->next;
	}
      } else if ((!xmlStrcmp(cur->name, "messages")) && (cur->ns == ns)) {
	/* Parse and add all Message elements */
	xmlNodePtr temp = cur->xmlChildrenNode;

	while (temp != NULL) {
	  if (!(xmlIsBlankNode(temp))) {
	    int number = -1;
	    struct message *message = message_parse(doc, ns, temp);

	    if (message != NULL) {
	      prop = xmlGetProp(temp, "number");
	      if (prop != NULL)
		number = atoi(prop);

	      if (number >= song->nummessages) {
		int i;

		song->messages = realloc(song->messages, (number + 1) * sizeof(struct message *));
		for (i = song->nummessages; i <= number; i++)
		  song->messages[i] = NULL;
	      }
	      if (song->messages[number] != NULL) {
		free(song->messages[number]->data);
		free(song->messages[number]);
	      }
	      song->messages[number] = message;
	      song->nummessages = number + 1;
	    }
	  }

	  temp = temp->next;
	}
      }
      cur = cur->next;
    }
  } else if (cur->type != XML_COMMENT_NODE)
    fprintf(stderr, "XML error: expected song, got %s\n", cur->name);

  return song;
}

/* Saves a song to an XML file */
void song_save(struct song *song, char *filename)
{
  xmlDocPtr doc;
  xmlNodePtr node, subnode;
  int i;
  char c[10];

  if (song == NULL) {
    fprintf(stderr, "song_save() called with null song\n");
    return;
  }

  doc = xmlNewDoc("1.0");
  doc->xmlRootNode = xmlNewDocNode(doc, NULL, "song", NULL);
  xmlSetProp(doc->xmlRootNode, "name", song->name);
  snprintf(c, 10, "%d", song->tempo);
  xmlSetProp(doc->xmlRootNode, "tempo", c);
  snprintf(c, 10, "%d", song->ticksperline);
  xmlSetProp(doc->xmlRootNode, "ticksperline", c);
  snprintf(c, 10, "%d", song->mastervolume);
  xmlSetProp(doc->xmlRootNode, "mastervolume", c);
  snprintf(c, 10, "%d", song->sendsync);
  xmlSetProp(doc->xmlRootNode, "sendsync", c);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "blocks", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all blocks */
  for (i = 0; i < song->numblocks; i++)
    block_save(song->blocks[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "sections", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all sections */
  for (i = 0; i < song->numsections; i++) {
    snprintf(c, 10, "%d", song->sections[i]);
    subnode = xmlNewChild(node, NULL, "section", c);
    snprintf(c, 10, "%d", i);
    xmlSetProp(subnode, "number", c);
    xmlAddChild(node, xmlNewText("\n"));
  }
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "playingsequences", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all playing sequences */
  for (i = 0; i < song->numplayseqs; i++)
    playseq_save(song->playseqs[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "instruments", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all instruments */
  for (i = 0; i < song->numinstruments; i++)
    instrument_save(song->instruments[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "trackvolumes", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all track volumes */
  for (i = 0; i < song->maxtracks; i++) {
    snprintf(c, 10, "%d", song->trackvolumes[i]);
    subnode = xmlNewChild(node, NULL, "trackvolume", c);
    snprintf(c, 10, "%d", i);
    xmlSetProp(subnode, "track", c);
    xmlAddChild(node, xmlNewText("\n"));
  }
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  node = xmlNewChild(doc->xmlRootNode, NULL, "messages", NULL);
  xmlAddChild(node, xmlNewText("\n"));
  /* Add all Message messages */
  for (i = 0; i < song->nummessages; i++)
    message_save(song->messages[i], i, node);
  xmlAddChild(doc->xmlRootNode, xmlNewText("\n\n"));

  /* Use stdout if no filename is specified */
  if (filename == NULL)
    filename = "-";

  xmlSaveFile(filename, doc);
  xmlFreeDoc(doc);
}


void song_print(struct song *song, char *path)
{
  char *notestrings[] = {
    "---",
    "C-1", "C#1", "D-1", "D#1", "E-1", "F-1", "F#1", "G-1", "G#1", "A-1", "A#1", "B-1",
    "C-2", "C#2", "D-2", "D#2", "E-2", "F-2", "F#2", "G-2", "G#2", "A-2", "A#2", "B-2",
    "C-3", "C#3", "D-3", "D#3", "E-3", "F-3", "F#3", "G-3", "G#3", "A-3", "A#3", "B-3",
    "C-4", "C#4", "D-4", "D#4", "E-4", "F-4", "F#4", "G-4", "G#4", "A-4", "A#4", "B-4",
    "C-5", "C#5", "D-5", "D#5", "E-5", "F-5", "F#5", "G-5", "G#5", "A-5", "A#5", "B-5",
    "C-6", "C#6", "D-6", "D#6", "E-6", "F-6", "F#6", "G-6", "G#6", "A-6", "A#6", "B-6",
    "C-7", "C#7", "D-7", "D#7", "E-7", "F-7", "F#7", "G-7", "G#7", "A-7", "A#7", "B-7",
    "C-8", "C#8", "D-8", "D#8", "E-8", "F-8", "F#8", "G-8", "G#8", "A-8", "A#8", "B-8",
    "C-9", "C#9", "D-9", "D#9", "E-9", "F-9", "F#9", "G-9", "G#9", "A-9", "A#9", "B-9",
    "C-A", "C#A", "D-A", "D#A", "E-A", "F-A", "F#A", "G-A", "G#A", "A-A", "A#A", "B-A"
  };
  char *instrumentstrings = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ";
  FILE *file;
  int i, j, k;

  if (path)
    file = fopen(path, "r+");
  else
    file = stdout;

  if (file) {
    fprintf(file, "======= Song: %s\n\n", song->name);
    fprintf(file, "#  Instrument name                               Mc Mpst Vol Trns Hold\n");
    for (i = 0; i < song->numinstruments; i++) {
      fprintf(file, "%.2d %-45s %2d %4d %3d %4d %4d\n", i + 1, song->instruments[i]->name, song->instruments[i]->midichannel, song->instruments[i]->midipreset, song->instruments[i]->defaultvelocity, song->instruments[i]->transpose, song->instruments[i]->hold);
    }

    fprintf(file, "\nDefault Tempo = %d/%d\n\n", song->tempo, song->ticksperline);
    fprintf(file, "Master Volume = %d, Track Volumes:", song->mastervolume);
    for (i = 0; i < song->maxtracks; i++) {
      if (!(i % 8))
	fprintf(file, "\n");
      fprintf(file, "%2d: %3d  ", i + 1, song->trackvolumes[i]);
    }

    fprintf(file, "\n\n======================================================================\n\n");

    for (i = 0; i < song->numplayseqs; i++) {
      struct playseq *playseq;

      playseq = song->playseqs[i];

      fprintf(file, "Playing Sequence List #%d (length = %d): %s", i + 1, playseq->length, playseq->name);
      for (j = 0; j < playseq->length; j++) {
	if (!(j % 8))
	  fprintf(file, "\n");
	fprintf(file, "%3d ", playseq->blocknumbers[j] + 1);
      }
      fprintf(file, "\n\n");
    }

    fprintf(file, "Section List (length = %d):\n", song->numsections);
    for (i = 0; i < song->numsections; i++) {
      fprintf(file, "%.3d: %.3d %s\n", i + 1, song->sections[i] + 1, song->playseqs[song->sections[i]]->name);
    }

    fprintf(file, "\n======================================================================\n\n");

    for (i = 0; i < song->numblocks; i++) {
      struct block *block;
      char datastring[6];
      datastring[5] = 0;

      block = song->blocks[i];

      fprintf(file, "Block #%3d: %3d tracks, %3d lines\n", i + 1, block->tracks, block->length);
      fprintf(file, "=================================\n");

      for (j = 0; j < block->length; j++) {
	fprintf(file, "%.3d ", j);
	for (k = 0; k < block->tracks; k++) {
	  datastring[0] = instrumentstrings[block->notes[(block->tracks * j + k) * 2 + 1] & 0x3f];
	  datastring[1] = instrumentstrings[(block->commands[(block->tracks * j + k) * 2] & 0xf0) >> 4];
	  datastring[2] = instrumentstrings[block->commands[(block->tracks * j + k) * 2] & 0x0f];
	  datastring[3] = instrumentstrings[(block->commands[(block->tracks * j + k) * 2 + 1] & 0xf0) >> 4];
	  datastring[4] = instrumentstrings[block->commands[(block->tracks * j + k) * 2 + 1] & 0x0f];
	  fprintf(file, "%s %s ", notestrings[block->notes[(block->tracks * j + k) * 2] & 0x7f], datastring);
	}
	fprintf(file, "\n");
      }

      fprintf(file, "\n");
    }
  }
}
