/* callbacks.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <gnome.h>
#include <glade/glade.h>

#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"
#include "gui.h"

void on_menuitem_file_new_activate(GtkMenuItem *menuitem,
				   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  editor_song_open(gui->editor, NULL);
}

void on_menuitem_file_open_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_open");

  if(!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);
  else
    gtk_widget_hide(fs);
}

void on_menuitem_file_save_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_save(gui->editor->song, gui->editor->filename);
}

void on_menuitem_file_saveas_activate(GtkMenuItem *menuitem,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_saveas");

  if(!GTK_WIDGET_VISIBLE(fs))
    gtk_widget_show(fs);
  else
    gtk_widget_hide(fs);
}

void on_menuitem_file_print_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  /* Fix: A file requester is needed here */
  song_print(gui->editor->song, NULL);
}

void on_menuitem_file_exit_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  gtk_main_quit();
}

void on_menuitem_edit_cut_activate(GtkMenuItem *menuitem,
				   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  if(gui->editor->copyarea)
    block_free(gui->editor->copyarea);
  gui->editor->copyarea=block_copy(gui->editor->song->blocks[gui->editor->block],
				   tracker->sel_start_ch,
				   tracker->sel_start_row,
				   tracker->sel_end_ch,
				   tracker->sel_end_row);
  block_clear(gui->editor->song->blocks[gui->editor->block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_menuitem_edit_copy_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  if(gui->editor->copyarea)
    block_free(gui->editor->copyarea);
  gui->editor->copyarea=block_copy(gui->editor->song->blocks[gui->editor->block],
				   tracker->sel_start_ch, tracker->sel_start_row,
				   tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_menuitem_edit_paste_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copyarea,
	      tracker->cursor_ch, tracker->patpos);
  tracker_redraw(tracker);
}

void on_menuitem_edit_clear_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block],
	      tracker->sel_start_ch, tracker->sel_start_row,
	      tracker->sel_end_ch, tracker->sel_end_row);
  tracker_clear_mark_selection(tracker);
}

void on_menuitem_edit_transpose_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")))==GUI_AREA_SELECTION) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")), GUI_AREA_SELECTION);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_edit_changeinstrument_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")))==GUI_AREA_SELECTION) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")), GUI_AREA_SELECTION);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_edit_expandshrink_activate(GtkMenuItem *menuitem,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")))==GUI_AREA_SELECTION) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")), GUI_AREA_SELECTION);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_scale_tempo_value_changed(GtkRange *range,
				  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_set_tempo(gui->editor->song, (unsigned char)gtk_range_get_value(range));
  editor_set_tempo(gui->editor);
}

void on_scale_tpl_value_changed(GtkRange *range,
				gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_set_tpl(gui->editor->song, (unsigned char)gtk_range_get_value(range));
  editor_set_tempo(gui->editor);
}

void on_menuitem_song_trackvolumes_activate(GtkMenuItem *menuitem,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_trackvolumes");

  if(!GTK_WIDGET_VISIBLE(window)) {
    int oldmax=gui->editor->song->maxtracks;

    if(song_check_maxtracks(gui->editor->song)) {
      player_trackstatus_create(gui->editor->player, oldmax);
      gui_trackvolumes_refresh(gui);
    }
    gtk_widget_show(window);
  } else
    gtk_widget_hide(window);
}

void on_menuitem_song_transpose_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")))==GUI_AREA_SONG) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")), GUI_AREA_SONG);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_song_expandshrink_activate(GtkMenuItem *menuitem,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")))==GUI_AREA_SONG) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")), GUI_AREA_SONG);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_song_changeinstrument_activate(GtkMenuItem *menuitem,
						gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")))==GUI_AREA_SONG) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")), GUI_AREA_SONG);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_song_properties_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml,
					 "dialog_songproperties");

  if(!GTK_WIDGET_VISIBLE(window)) {
    gui_song_refresh(gui);
    gtk_widget_show(window);
  } else
    gtk_widget_hide(window);
}

void on_menuitem_block_cut_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  if(gui->editor->copyblock)
    block_free(gui->editor->copyblock);
  gui->editor->copyblock=block_copy(gui->editor->song->blocks[gui->editor->block],
				    0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  block_clear(gui->editor->song->blocks[gui->editor->block],
	      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_menuitem_block_copy_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->copyblock)
    block_free(gui->editor->copyblock);
  gui->editor->copyblock=block_copy(gui->editor->song->blocks[gui->editor->block],
				    0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
}

void on_menuitem_block_paste_activate(GtkMenuItem *menuitem,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copyblock,
	      0, 0);
  tracker_redraw(tracker);
}

void on_menuitem_block_selectall_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  tracker->sel_start_ch=0;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=gui->editor->song->blocks[gui->editor->block]->tracks-1;
  tracker->sel_end_row=gui->editor->song->blocks[gui->editor->block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_menuitem_block_clear_activate(GtkMenuItem *menuitem,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block],
	      0, 0, gui->editor->song->blocks[gui->editor->block]->tracks-1, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_menuitem_block_transpose_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")))==GUI_AREA_BLOCK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")), GUI_AREA_BLOCK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_block_expandshrink_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")))==GUI_AREA_BLOCK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")), GUI_AREA_BLOCK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_block_changeinstrument_activate(GtkMenuItem *menuitem,
						 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")))==GUI_AREA_BLOCK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")), GUI_AREA_BLOCK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_block_list_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  on_button_block_clicked(NULL, user_data);
}

void on_menuitem_track_cut_activate(GtkMenuItem *menuitem,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  if(gui->editor->copytrack)
    block_free(gui->editor->copytrack);
  gui->editor->copytrack=block_copy(gui->editor->song->blocks[gui->editor->block],
				    tracker->cursor_ch, 0,
				    tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  block_clear(gui->editor->song->blocks[gui->editor->block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_menuitem_track_copy_activate(GtkMenuItem *menuitem,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  if(gui->editor->copytrack)
    block_free(gui->editor->copytrack);
  gui->editor->copytrack=block_copy(gui->editor->song->blocks[gui->editor->block],
				    tracker->cursor_ch, 0,
				    tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
}

void on_menuitem_track_paste_activate(GtkMenuItem *menuitem,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_paste(gui->editor->song->blocks[gui->editor->block], gui->editor->copytrack,
	      tracker->cursor_ch, 0);
  tracker_redraw(tracker);
}

void on_menuitem_track_selectall_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  tracker->sel_start_ch=tracker->cursor_ch;
  tracker->sel_start_row=0;
  tracker->sel_end_ch=tracker->cursor_ch;
  tracker->sel_end_row=gui->editor->song->blocks[gui->editor->block]->length-1;
  tracker->inSelMode=FALSE;
  tracker_redraw(tracker);
}

void on_menuitem_track_clear_activate(GtkMenuItem *menuitem,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  block_clear(gui->editor->song->blocks[gui->editor->block], tracker->cursor_ch, 0,
	      tracker->cursor_ch, gui->editor->song->blocks[gui->editor->block]->length-1);
  tracker_redraw(tracker);
}

void on_menuitem_track_transpose_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_transpose");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")))==GUI_AREA_TRACK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")), GUI_AREA_TRACK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_track_expandshrink_activate(GtkMenuItem *menuitem,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")))==GUI_AREA_TRACK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")), GUI_AREA_TRACK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_track_changeinstrument_activate(GtkMenuItem *menuitem,
						 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  if(gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")))==GUI_AREA_TRACK) {
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
    else
      gtk_widget_hide(window);
  } else {
    gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")), GUI_AREA_TRACK);
    if(!GTK_WIDGET_VISIBLE(window))
      gtk_widget_show(window);
  }
}

void on_menuitem_song_sysexlist_activate(GtkMenuItem *menuitem,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_sysex");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_menuitem_midi_killallnotes_activate(GtkMenuItem *menuitem,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  player_stop_all_notes(gui->editor->player);
}

void on_menuitem_midi_resetpitch_activate(GtkMenuItem *menuitem,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  player_reset_pitch(gui->editor->player);
}

void on_menuitem_midi_recordcontrollers_activate(GtkMenuItem *menuitem,
						 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  if(gtk_check_menu_item_get_active(GTK_CHECK_MENU_ITEM(glade_xml_get_widget(gui->xml, "menuitem_midi_recordcontrollers"))))
    gui->editor->recordcontrollers=1;
  else
    gui->editor->recordcontrollers=0;
}

void on_menuitem_settings_preferences_activate(GtkMenuItem *menuitem,
					       gpointer user_data) {
  /* FIX: implement */
  fprintf(stderr, "Preferences not implemented yet\n");
}

void on_menuitem_about_activate(GtkMenuItem *menuitem,
				gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gchar *authors[] = { "Vesa Halttunen <vesuri@jormas.com>", NULL };

  if(gui->window_about!=NULL) {
    gdk_window_show(gui->window_about->window);
    gdk_window_raise(gui->window_about->window);
    return;
  }

  gui->window_about=gnome_about_new("Tutka", VERSION,
				    "Copyright \xc2\xa9 2002-2003 Vesa Halttunen",
				    "http://www.freesoftware.fsf.org/tutka/",
				    (const gchar **)authors,
				    (const gchar **)authors,
				    NULL,
				    NULL);

  gtk_window_set_destroy_with_parent(GTK_WINDOW(gui->window_about), TRUE);
  g_signal_connect(G_OBJECT(gui->window_about), "destroy",
		   G_CALLBACK(gtk_widget_destroyed), &gui->window_about);
  gtk_widget_show(gui->window_about);
}


GtkWidget* create_tracker(gchar *widget_name, gchar *string1,
			  gchar *string2, gint int1, gint int2) {
  return tracker_new();
}

void on_button_playsong_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  player_start(gui->editor->player, MODE_PLAY_SONG,
	       gui->editor->section, gui->editor->position,
	       gui->editor->block, 0);
  gui_statusbar_refresh(gui);
}

void on_button_playblock_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  player_start(gui->editor->player, MODE_PLAY_BLOCK,
	       gui->editor->section, gui->editor->position,
	       gui->editor->block, 0);
  gui_statusbar_refresh(gui);
}

void on_button_contsong_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->player->mode!=MODE_PLAY_SONG) {
    player_start(gui->editor->player, MODE_PLAY_SONG,
		 gui->editor->section, gui->editor->position,
		 gui->editor->block, 1);
    gui_statusbar_refresh(gui);
  }
}

void on_button_contblock_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->player->mode!=MODE_PLAY_BLOCK) {
    player_start(gui->editor->player, MODE_PLAY_BLOCK,
		 gui->editor->section, gui->editor->position,
		 gui->editor->block, 1);
    gui_statusbar_refresh(gui);
  }
}

void on_button_stop_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  player_stop(gui->editor->player);
}

void on_scale_volume_value_changed(GtkRange *range,
				   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->defaultvelocity=(unsigned char)gtk_range_get_value(range);
}

void on_scale_transpose_value_changed(GtkRange *range,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->transpose=(unsigned char)gtk_range_get_value(range);
}

void on_scale_hold_value_changed(GtkRange *range,
				 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->hold=(unsigned char)gtk_range_get_value(range);
}

void on_scale_midich_value_changed(GtkRange *range,
				   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->song->instruments[gui->editor->instrument]->midichannel=(unsigned char)gtk_range_get_value(range)-1;
}

void on_button_properties_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_instrumentproperties");
  
  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_checkbutton_edit_toggled(GtkToggleButton *togglebutton,
				 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->edit=1;
  else
    gui->editor->edit=0;
}

void on_checkbutton_chord_toggled(GtkToggleButton *togglebutton,
				  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->chord=1;
  else
    gui->editor->chord=0;
}

void on_spinbutton_space_changed(GtkSpinButton *spinbutton,
				 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  gui->editor->space=gtk_spin_button_get_value(spinbutton);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "button_properties"));
}

void on_button_section_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_sectionlist");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_button_position_clicked(GtkButton *button,
				gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_playseq");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_button_block_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_blocklist");

  if(!GTK_WIDGET_VISIBLE(window))
    gtk_widget_show(window);
  else
    gtk_widget_hide(window);
}

void on_button_trackvolumes_close_clicked(GtkButton *button,
					  gpointer user_data) {
  on_menuitem_song_trackvolumes_activate(NULL, user_data);
}

gboolean on_label_timer_button_press_event(GtkWidget *widget,
					   GdkEventButton *event,
					   gpointer user_data) {
  /* FIX: implement this */

  return FALSE;
}

void on_selection_sysexlist_changed(GtkTreeSelection *selection,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter))
    gtk_tree_model_get(model, &iter, 0, &gui->sysexlist_sysex, -1);

  gui_sysexlist_refresh(gui, FALSE);
}

void on_button_sysex_appendnew_clicked(GtkButton *button,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_sysex(gui->editor->song, gui->editor->song->numsysexes);
  gui_sysexlist_refresh(gui, TRUE);
}

void on_button_sysex_delete_clicked(GtkButton *button,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    song_delete_sysex(gui->editor->song, gui->sysexlist_sysex);
    gui_sysexlist_refresh(gui, TRUE);
  }
}

void on_button_sysex_insertnew_clicked(GtkButton *button,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_sysex(gui->editor->song, gui->sysexlist_sysex);
  if(gui->sysexlist_sysex==-1)
    gui->sysexlist_sysex=0;
  gui_sysexlist_refresh(gui, TRUE);
}

void on_button_sysex_send_clicked(GtkButton *button,
				  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    midi_system_exclusive(gui->editor->player->midi,
			  gui->editor->song->sysexes[gui->sysexlist_sysex]->data,
			  gui->editor->song->sysexes[gui->sysexlist_sysex]->length);
  }
}

void on_button_sysex_receive_clicked(GtkButton *button,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    midi_read_system_exclusive(gui->editor->player->midi,
			       gui->editor->song->sysexes[gui->sysexlist_sysex],
			       gui->sysexlist_autostop);
  }
}

void on_checkbutton_sysex_autostop_toggled(GtkToggleButton *togglebutton,
					   gpointer user_data) {
  
}

void on_selection_sectionlist_changed(GtkTreeSelection *selection,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value=0;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_section(gui->editor, value);
}

void on_button_sectionlist_next_clicked(GtkButton *button,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  if(gui->editor->song->sections[gui->editor->section]<gui->editor->song->numplayseqs-1)
    gui->editor->song->sections[gui->editor->section]++;

  editor_refresh_playseq_and_block(gui->editor);
}

void on_button_sectionlist_delete_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_delete_section(gui->editor->song, gui->editor->section);
  if(gui->editor->section>=gui->editor->song->numsections)
    gui->editor->section=gui->editor->song->numsections-1;

  editor_refresh_playseq_and_block(gui->editor);
  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_sectionlist_append_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_insert_section(gui->editor->song, gui->editor->song->numsections);

  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_sectionlist_prev_clicked(GtkButton *button,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->editor->song->sections[gui->editor->section]>0)
    gui->editor->song->sections[gui->editor->section]--;

  editor_refresh_playseq_and_block(gui->editor);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_button_sectionlist_insert_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_insert_section(gui->editor->song, gui->editor->section);

  gui_sectionlist_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_selection_playseq_changed(GtkTreeSelection *selection,
				  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value=0;
  
  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_position(gui->editor, value);
}

void on_button_playseq_next_clicked(GtkButton *button,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  unsigned int playseq=gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position];

  if(playseq<gui->editor->song->numblocks-1)
    playseq_set(gui->editor->song->playseqs[gui->editor->playseq],
		gui->editor->position, playseq+1);

  gui_playseq_refresh(gui, TRUE);
}

void on_button_playseq_delete_clicked(GtkButton *button,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_delete(gui->editor->song->playseqs[gui->editor->playseq],
		 gui->editor->position);
  editor_refresh_playseq_and_block(gui->editor);
  
  if(gui->editor->position>=gui->editor->song->playseqs[gui->editor->playseq]->length)
    gui->editor->position=gui->editor->song->playseqs[gui->editor->playseq]->length-1;
  
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseq_append_clicked(GtkButton *button,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_insert(gui->editor->song->playseqs[gui->editor->playseq],
		 gui->editor->song->playseqs[gui->editor->playseq]->length);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseq_prev_clicked(GtkButton *button,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  unsigned int playseq=gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position];

  if(playseq>0)
    playseq_set(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position,
		playseq-1);

  gui_playseq_refresh(gui, TRUE);
}

void on_button_playseq_insert_clicked(GtkButton *button,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  playseq_insert(gui->editor->song->playseqs[gui->editor->playseq], gui->editor->position);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_playseq_insertnew_clicked(GtkButton *button,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_playseq(gui->editor->song, gui->editor->playseq);
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_button_playseq_appendnew_clicked(GtkButton *button,
					 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_playseq(gui->editor->song, gui->editor->song->numplayseqs);
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_button_playseq_deletethis_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  song_delete_playseq(gui->editor->song, gui->editor->playseq);
  
  editor_refresh_playseq_and_block(gui->editor);
  
  gui_info_refresh(gui);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
}

void on_selection_blocklist_changed(GtkTreeSelection *selection,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkTreeIter iter;
  GtkTreeModel *model;
  int value=0;

  /* Get the selected row */
  if(gtk_tree_selection_get_selected(selection, &model, &iter)) {
    gtk_tree_model_get(model, &iter, 0, &value, -1);
    value--;
  }

  editor_set_block(gui->editor, value);
}

void on_spinbutton_blocklist_tracks_changed(GtkEditable *editable,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  int oldmax=gui->editor->song->maxtracks;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");
  
  block_set_tracks(gui->editor->song->blocks[gui->editor->block],
		   (unsigned int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(editable)));
  tracker_set_num_channels(tracker, (int)gtk_spin_button_get_value(GTK_SPIN_BUTTON(editable)));
  tracker_redraw(tracker);
  
  if(song_check_maxtracks(gui->editor->song)) {
    player_trackstatus_create(gui->editor->player, oldmax);
    gui_trackvolumes_refresh(gui);
  }
}

void on_spinbutton_blocklist_length_changed(GtkEditable *editable,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");
  
  block_set_length(gui->editor->song->blocks[gui->editor->block],
		   (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
  tracker->curpattern=gui->editor->song->blocks[gui->editor->block];
  tracker_redraw(tracker);
}

void on_spinbutton_blocklist_commandpages_changed(GtkEditable *editable,
						  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  block_set_commandpages(gui->editor->song->blocks[gui->editor->block],
			 (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
  if(gui->editor->cmdpage>=gui->editor->song->blocks[gui->editor->block]->commandpages) {
    while(gui->editor->cmdpage>=gui->editor->song->blocks[gui->editor->block]->commandpages)
      gui->editor->cmdpage--;
    gui_tracker_refresh(gui);
  }
  gui_info_refresh(gui);
  gui_window_title_refresh(gui);  
}

void on_button_blocklist_insertnew_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_block(gui->editor->song,
		    gui->editor->block, gui->editor->block);
  gui_tracker_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_blocklist_appendnew_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  song_insert_block(gui->editor->song, gui->editor->song->numblocks,
		    gui->editor->block);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);
}

void on_button_blocklist_delete_clicked(GtkButton *button,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  unsigned int line=gui->editor->line;

  song_delete_block(gui->editor->song, gui->editor->block);
  
  if(gui->editor->block>=gui->editor->song->numblocks)
    gui->editor->block=gui->editor->song->numblocks-1;

  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_info_refresh(gui);

  /* Go back to the line the editor was on */
  gui->editor->line=line;
  gui_tracker_refresh(gui);
}

gboolean on_entry_blocklist_focus_out_event(GtkWidget *widget,
					    GdkEventFocus *event,
					    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));
  
  if(gui->editor->song->blocks[gui->editor->block]->name)
    free(gui->editor->song->blocks[gui->editor->block]->name);
  
  gui->editor->song->blocks[gui->editor->block]->name=(char *)strdup(name);
  
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);

  return FALSE;
}

gboolean on_entry_sysex_name_focus_out_event(GtkWidget *widget,
					     GdkEventFocus *event,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));
    
    if(gui->editor->song->sysexes[gui->sysexlist_sysex]->name)
      free(gui->editor->song->sysexes[gui->sysexlist_sysex]->name);
    
    gui->editor->song->sysexes[gui->sysexlist_sysex]->name=(char *)strdup(name);
    gui_sysexlist_refresh(gui, TRUE);
  }

  return FALSE;
}

gboolean on_entry_playseq_name_focus_out_event(GtkWidget *widget,
					       GdkEventFocus *event,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->playseqs[gui->editor->playseq]->name)
    free(gui->editor->song->playseqs[gui->editor->playseq]->name);
  
  gui->editor->song->playseqs[gui->editor->playseq]->name=(char *)strdup(name);
  
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);

  return FALSE;
}

gboolean on_entry_instrumentproperties_name_focus_out_event(GtkWidget *widget, GdkEventFocus *event, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->instruments[gui->editor->instrument]->name!=NULL)
    free(gui->editor->song->instruments[gui->editor->instrument]->name);

  gui->editor->song->instruments[gui->editor->instrument]->name=strdup(name);

  gui_instrument_refresh(gui, FALSE);

  return FALSE;
}

void on_button_sysex_close_clicked(GtkButton *button, gpointer user_data) {
  on_menuitem_song_sysexlist_activate(NULL, user_data);
}

void on_button_saveas_ok_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_saveas");

  /* Store the filename */
  if(gui->editor->filename!=NULL)
    free(gui->editor->filename);
  gui->editor->filename=strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));
  
  /* Save... */
  song_save(gui->editor->song, gui->editor->filename);
  
  /* ...and close the file selector */
  gtk_widget_hide(fs);
}

void on_button_open_ok_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_open");

  editor_song_open(gui->editor, (char *)gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs)));

  gtk_widget_hide(fs);
}

void on_button_open_cancel_clicked(GtkButton *button,
				   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_open");

  gtk_widget_hide(fs);
}

void on_button_saveas_cancel_clicked(GtkButton *button,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *fs=glade_xml_get_widget(gui->xml, "fileselection_saveas");

  gtk_widget_hide(fs);
}

void on_spinbutton_sysex_length_changed(GtkEditable *editable,
					gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1)
    sysex_set_length(gui->editor->song->sysexes[gui->sysexlist_sysex],
		     (unsigned int)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value);
}

gboolean on_entry_songname_focus_out_event(GtkWidget *widget,
					   GdkEventFocus *event,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  char *name=(char *)gtk_entry_get_text(GTK_ENTRY(widget));

  if(gui->editor->song->name!=NULL)
    free(gui->editor->song->name);

  gui->editor->song->name=strdup(name);

  gui_window_title_refresh(gui);

  return FALSE;
}

void on_button_songproperties_close_clicked(GtkButton *button,
					    gpointer user_data) {
  on_menuitem_song_properties_activate(NULL, user_data);
}

void on_spinbutton_instrument_changed(GtkEditable *editable,
				      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  /* Select an instrument if an instrument selection key was pressed */
  gui->editor->instrument=(unsigned char)gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(editable))->value-1;

  /* Make sure the instrument exists */
  song_check_instrument(gui->editor->song, gui->editor->instrument);
  gui_instrument_refresh(gui, FALSE);

  /* Activate another widget to prevent typing in the spin button */
  gtk_widget_grab_focus(glade_xml_get_widget(gui->xml, "button_properties"));
}

void on_adjustment_trackvolumes_changed(GtkAdjustment *adjustment,
					gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  *data=((*data)&128)|((unsigned char)adjustment->value);
}

void on_togglebutton_trackvolumes_toggled(GtkToggleButton *togglebutton,
					  gpointer user_data) {
  unsigned char *data=(unsigned char *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    *data|=128;
  else
    *data&=127;
}

void on_optionmenu_keyboard_changed(GtkOptionMenu *optionmenu,
				    gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;

  gui->editor->octave=gtk_option_menu_get_history(optionmenu);
}

void on_button_transpose_close_clicked(GtkButton *button,
				       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_transpose");

  gtk_widget_hide(window);
}

void on_button_transpose_clicked(GtkButton *button,
				 gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_area")));
  int instruments=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_instruments")));
  int mode=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_transpose_mode")));

  int instrument=-1, halfnotes=0;

  switch(mode) {
  case 0:
    halfnotes=12;
    break;
  case 1:
    halfnotes=1;
    break;
  case 2:
    halfnotes=-1;
    break;
  case 3:
    halfnotes=-12;
    break;
  default:
    break;
  }

  if(instruments==1)
    instrument=gui->editor->instrument;

  switch(area) {
  case GUI_AREA_SONG:
    song_transpose(gui->editor->song, instrument, halfnotes);
    break;
  case GUI_AREA_BLOCK:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes, 0, 0,
		    gui->editor->song->blocks[gui->editor->block]->tracks-1,
		    gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_TRACK:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes,
		    tracker->cursor_ch, 0, tracker->cursor_ch,
		    gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_SELECTION:
    block_transpose(gui->editor->song->blocks[gui->editor->block],
		    instrument, halfnotes,
		    tracker->sel_start_ch, tracker->sel_start_row,
		    tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_expandshrink_close_clicked(GtkButton *button,
					  gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_expandshrink");

  gtk_widget_hide(window);
}

void on_button_expandshrink_shrink_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")));
  int factor=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_expandshrink_factor")));

  if(factor<2)
    return;

  switch(area) {
  case GUI_AREA_SONG:
    song_expandshrink(gui->editor->song, -factor);
    break;
  case GUI_AREA_BLOCK:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], -factor,
		       0, 0,
		       gui->editor->song->blocks[gui->editor->block]->tracks-1,
		       gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_TRACK:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], -factor,
		       tracker->cursor_ch, 0, tracker->cursor_ch,
		       gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_SELECTION:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], -factor,
		       tracker->sel_start_ch, tracker->sel_start_row,
		       tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_expandshrink_expand_clicked(GtkButton *button,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_expandshrink_area")));
  int factor=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_expandshrink_factor")));

  if(factor<2)
    return;

  switch(area) {
  case GUI_AREA_SONG:
    song_expandshrink(gui->editor->song, factor);
    break;
  case GUI_AREA_BLOCK:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], factor,
		       0, 0,
		       gui->editor->song->blocks[gui->editor->block]->tracks-1,
		       gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_TRACK:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], factor,
		       tracker->cursor_ch, 0, tracker->cursor_ch,
		       gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_SELECTION:
    block_expandshrink(gui->editor->song->blocks[gui->editor->block], factor,
		       tracker->sel_start_ch, tracker->sel_start_row,
		       tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  tracker_redraw(tracker);
}

void on_button_changeinstrument_close_clicked(GtkButton *button,
					      gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  GtkWidget *window=glade_xml_get_widget(gui->xml, "dialog_changeinstrument");

  gtk_widget_hide(window);
}

void on_button_changeinstrument_swap_clicked(GtkButton *button,
					     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")));
  int from=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_from")));
  int to=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_to")));

  switch(area) {
  case GUI_AREA_SONG:
    song_changeinstrument(gui->editor->song, from, to, 1);
    break;
  case GUI_AREA_BLOCK:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 1, 0, 0,
			   gui->editor->song->blocks[gui->editor->block]->tracks-1,
			   gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_TRACK:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 1,
			   tracker->cursor_ch, 0, tracker->cursor_ch,
			   gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_SELECTION:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 1,
			   tracker->sel_start_ch, tracker->sel_start_row,
			   tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  song_check_instrument(gui->editor->song, from);
  song_check_instrument(gui->editor->song, to);

  tracker_redraw(tracker);
}

void on_button_changeinstrument_change_clicked(GtkButton *button,
					       gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "tracker");

  int area=gtk_option_menu_get_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "optionmenu_changeinstrument_area")));
  int from=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_from")));
  int to=gtk_spin_button_get_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "spinbutton_changeinstrument_to")));

  switch(area) {
  case GUI_AREA_SONG:
    song_changeinstrument(gui->editor->song, from, to, 0);
    break;
  case GUI_AREA_BLOCK:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 0, 0, 0,
			   gui->editor->song->blocks[gui->editor->block]->tracks-1,
			   gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_TRACK:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 0,
			   tracker->cursor_ch, 0, tracker->cursor_ch,
			   gui->editor->song->blocks[gui->editor->block]->length-1);
    break;
  case GUI_AREA_SELECTION:
    block_changeinstrument(gui->editor->song->blocks[gui->editor->block],
			   from, to, 0,
			   tracker->sel_start_ch, tracker->sel_start_row,
			   tracker->sel_end_ch, tracker->sel_end_row);
    break;
  default:
    break;
  }

  song_check_instrument(gui->editor->song, to);

  tracker_redraw(tracker);
}

void on_checkbutton_sendsync_toggled(GtkToggleButton *togglebutton,
				     gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gtk_toggle_button_get_active(togglebutton))
    gui->editor->song->sendsync=1;
  else
    gui->editor->song->sendsync=0;
}

void on_checkbutton_sysex_autosend_toggled(GtkToggleButton *togglebutton,
					   gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  if(gui->sysexlist_sysex!=-1) {
    if(gtk_toggle_button_get_active(togglebutton))
      sysex_set_autosend(gui->editor->song->sysexes[gui->sysexlist_sysex], 1);
    else
      sysex_set_autosend(gui->editor->song->sysexes[gui->sysexlist_sysex], 0);
  }
}

void on_button_trackvolumes_mute_clicked(GtkButton *button, gpointer user_data) {
  struct gui *gui=(struct gui *)user_data;
  int i, unmute=1;

  /* Check if all tracks are already muted */
  for(i=0; i<gui->editor->song->maxtracks; i++)
    if(gui->editor->song->trackvolumes[i]<128) {
      unmute=0;
      break;
    }

  if(unmute)
    for(i=0; i<gui->editor->song->maxtracks; i++)
      gui->editor->song->trackvolumes[i]&=127;
  else
    for(i=0; i<gui->editor->song->maxtracks; i++)
      gui->editor->song->trackvolumes[i]|=128;

  gui_trackvolumes_refresh(gui);
}
