/* gui.c
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <math.h>
#include <gmodule.h>
#include <gnome.h>
#include <glade/glade.h>

#include "gui.h"
#include "callbacks.h"
#include "editor.h"
#include "player.h"
#include "trackerwidget.h"

#define new_column gtk_tree_view_column_new_with_attributes

gboolean gui_quit=FALSE;

/* Return the length of an integer in string representation */
static int intlen(int i) {
  if(i==0)
    return 1;

  if(i<0)
    return (int)log10((double)-i)+2;

  return (int)log10((double)i)+1;
}

/* Connects signals to objects and passes the gui struct on as user_data */
void gui_gladexmlconnectfunc(const gchar *handler_name, GObject *object,
			     const gchar *signal_name,
			     const gchar *signal_data, GObject *connect_object,
			     gboolean after, gpointer user_data) {
  GCallback func;

  /* Use the gmodule interface to resolve the function name to a pointer */
  if(!g_module_symbol(g_module_open(NULL, 0), handler_name, (gpointer *)&func))
    g_warning("could not find signal handler '%s'.", handler_name);

  /* Connect the signal */
  if(after)
    g_signal_connect_after(object, signal_name, func, user_data);
  else
    g_signal_connect(object, signal_name, func, user_data);

};

void gtk_main_quit() {
  gui_quit=TRUE;
}

/* Initializes a GUI */
struct gui *gui_open(struct editor *editor, char *filename) {
  struct gui *gui=(struct gui *)calloc(1, sizeof(struct gui));
  GtkCellRenderer *renderer;
  GtkTreeView *treeview;
  GtkTreeSelection *select;

  gui->editor=editor;

  /* Mutex for the refreshing function */
  gui->refresh_mutex=g_mutex_new();

  /* Load the GUI from a GLADE file and connect signals */
  gui->xml=glade_xml_new(filename, NULL, NULL);
  glade_xml_signal_autoconnect_full(gui->xml, gui_gladexmlconnectfunc, gui);
  gui->window_main=glade_xml_get_widget(gui->xml, "gui_window_main");
  gtk_widget_add_events(gui->window_main, GDK_KEY_RELEASE_MASK);

  /* Create the MVC classes for various list view widgets */
  renderer=gtk_cell_renderer_text_new();

  /* Section list */
  gui->store_sectionlist=gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT,
					    G_TYPE_STRING);
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml,
					      "gui_treeview_sectionlist"));
  select=gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT (select), "changed",
		   G_CALLBACK(on_gui_selection_sectionlist_changed),
		   gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_sectionlist));
  gtk_tree_view_append_column(treeview,
			      new_column("Section", renderer,
					 "text", 0, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Playing Sequence", renderer,
					 "text", 1, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Playing Sequence Name", renderer,
					 "text", 2, NULL));

  /* Playing Sequence */
  gui->store_playseq=gtk_list_store_new(3, G_TYPE_INT, G_TYPE_INT,
					G_TYPE_STRING);
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_playseq"));
  select=gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT (select), "changed",
		   G_CALLBACK(on_gui_selection_playseq_changed),
		   gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_playseq));
  gtk_tree_view_append_column(treeview,
			      new_column("Position", renderer,
					 "text", 0, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Block Number", renderer,
					 "text", 1, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Block Name", renderer,
					 "text", 2, NULL));

  /* Block List */
  gui->store_blocklist=gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_blocklist"));
  select=gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT (select), "changed",
		   G_CALLBACK(on_gui_selection_blocklist_changed),
		   gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_blocklist));
  gtk_tree_view_append_column(treeview,
			      new_column("Block Number", renderer,
					 "text", 0, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Block Name", renderer,
					 "text", 1, NULL));

  /* SysEx List */
  gui->store_sysexlist=gtk_list_store_new(2, G_TYPE_INT, G_TYPE_STRING);
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_sysexlist"));
  select=gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_set_mode(select, GTK_SELECTION_SINGLE);
  g_signal_connect(G_OBJECT (select), "changed",
		   G_CALLBACK(on_gui_selection_sysexlist_changed),
		   gui);
  gtk_tree_view_set_model(treeview, GTK_TREE_MODEL(gui->store_sysexlist));
  gtk_tree_view_append_column(treeview,
			      new_column("Sysex Number", renderer,
					 "text", 0, NULL));
  gtk_tree_view_append_column(treeview,
			      new_column("Sysex Name", renderer,
					 "text", 1, NULL));

  /* Set the keyboard octave */
  gtk_option_menu_set_history(GTK_OPTION_MENU(glade_xml_get_widget(gui->xml, "gui_optionmenu_keyboard")), 3);
  gui->editor->octave=3;

  /* Make some widgets that are not usable yet insensitive */
  gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml,
						"gui_menuitem_file_print"),
			   FALSE);
  gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_menuitem_settings_preferences"), FALSE);

  return gui;
}

/* Closes a GUI */
void gui_close(struct gui *gui) {
  g_mutex_free(gui->refresh_mutex);

  free(gui);
}

/* Refreshes all parts of the GUI */
void gui_refresh_all(struct gui *gui) {
  /* Reset the tracker */
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  tracker_reset(tracker);
  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_num_channels(tracker,
			   gui->editor->song->blocks[gui->editor->block]->tracks);

  /* Refresh the GUI */
  gui_info_refresh(gui);
  gui_tempo_refresh(gui);
  gui_blocklist_refresh(gui, TRUE);
  gui_playseq_refresh(gui, TRUE);
  gui_sectionlist_refresh(gui, TRUE);
  gui_instrument_refresh(gui, TRUE);
  gui_sysexlist_refresh(gui);
  gui_trackvolumes_refresh(gui);
  gui_song_refresh(gui);

  /* Show the main window */
  gtk_widget_show(gui->window_main);
  gtk_widget_show(GTK_WIDGET(tracker));

  /* FIX: Tracker widget kludge */
  tracker->cursor_ch=0;
}
 
Tracker *gui_get_tracker(struct gui *gui) {
  return (Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");
}

int gui_get_tracker_position(struct gui *gui) {
  Tracker *tracker=(Tracker *)glade_xml_get_widget(gui->xml, "gui_tracker");

  return tracker->patpos;
}

/* Set the refresh mask */
void gui_set_refresh(struct gui *gui, unsigned int mask) {
  g_mutex_lock(gui->refresh_mutex);
  gui->refresh_mask|=mask;
  g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh all those parts of the GUI that need to be refreshed */
void gui_refresh(struct gui *gui) {
  g_mutex_lock(gui->refresh_mutex);
  if(gui->refresh_mask&GUI_REFRESH_INSTRUMENT)
    gui_instrument_refresh(gui, FALSE);
  if(gui->refresh_mask&GUI_REFRESH_TEMPO)
    gui_tempo_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_INFO)
    gui_info_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_TIMER)
    gui_timer_refresh(gui, gui->editor->time);
  if(gui->refresh_mask&GUI_REFRESH_BLOCKLIST)
    gui_blocklist_refresh(gui, FALSE);
  if(gui->refresh_mask&GUI_REFRESH_BLOCKLIST_INFO)
    gui_blocklist_info_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_PLAYSEQ)
    gui_playseq_refresh(gui, FALSE);
  if(gui->refresh_mask&GUI_REFRESH_PLAYSEQ_INFO)
    gui_playseq_info_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_SECTIONLIST)
    gui_sectionlist_refresh(gui, FALSE);
  if(gui->refresh_mask&GUI_REFRESH_SECTIONLIST_BUTTONS)
    gui_sectionlist_buttons_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_SYSEXLIST)
    gui_sysexlist_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_TRACKVOLUMES)
    gui_trackvolumes_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_SONG)
    gui_song_refresh(gui);
  if(gui->refresh_mask&GUI_REFRESH_TRACKER)
    gui_tracker_refresh(gui);
  
  gui->refresh_mask=0;
  g_mutex_unlock(gui->refresh_mutex);
}

/* Refresh the tracker */
void gui_tracker_refresh(struct gui *gui) {
  Tracker *tracker=gui_get_tracker(gui);

  tracker_set_pattern(tracker, gui->editor->song->blocks[gui->editor->block]);
  tracker_set_patpos(tracker, gui->editor->line);
  tracker_set_cmdpage(tracker, gui->editor->cmdpage);
}

/* Refresh the song properties window widgets */
void gui_song_refresh(struct gui *gui) {
  if(gui->editor->song->name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_songname")), gui->editor->song->name);

  if(gui->editor->song->sendsync)
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "gui_checkbutton_sendsync")), TRUE);
  else
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(glade_xml_get_widget(gui->xml, "gui_checkbutton_sendsync")), FALSE);
}

void gui_instrument_refresh(struct gui *gui, gboolean refresh_spin) {
  char instrument[3];
  snprintf(instrument, 3, "%.2d", gui->editor->instrument+1);

  /* Refresh the main window widgets */
  /* Only refresh the spin button if necessary to avoid forever loops */
  if(refresh_spin)
    gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "gui_spinbutton_instrument")), gui->editor->instrument+1);
  if(gui->editor->song->instruments[gui->editor->instrument]->name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_instrument")), gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_instrument")), "");

  /* Refresh the instrument window widgets */
  if(gui->editor->song->instruments[gui->editor->instrument]->name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_instrumentproperties_name")), gui->editor->song->instruments[gui->editor->instrument]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_instrumentproperties_name")), "");
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_instrumentproperties_volume"))), gui->editor->song->instruments[gui->editor->instrument]->defaultvelocity);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_instrumentproperties_transpose"))), gui->editor->song->instruments[gui->editor->instrument]->transpose);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_instrumentproperties_hold"))), gui->editor->song->instruments[gui->editor->instrument]->hold);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_instrumentproperties_midich"))), gui->editor->song->instruments[gui->editor->instrument]->midichannel+1);
}

void gui_tempo_refresh(struct gui *gui) {
  int l=46+strlen(gui->editor->song->name)+intlen(gui->editor->block+1)+intlen(gui->editor->song->numblocks)+intlen(1)+intlen(gui->editor->song->blocks[gui->editor->block]->commandpages)+intlen(gui->editor->song->tempo)+intlen(gui->editor->song->ticksperline)+1;
  char *title=(char *)calloc(l, sizeof(char));

  /* Refresh the song properties window widgets */
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_tempo"))), gui->editor->song->tempo);
  gtk_adjustment_set_value(gtk_range_get_adjustment(GTK_RANGE(glade_xml_get_widget(gui->xml, "gui_scale_tpl"))), gui->editor->song->ticksperline);

  /* Refresh the main window widgets if the window exists */
  if(gui->window_main!=NULL && GTK_IS_WINDOW(gui->window_main)) {
    snprintf(title, l,
	     "Tutka: %s - Block %d/%d - Cmd Page %d/%d - Tempo %d - TPL %d",
	     gui->editor->song->name,
	     gui->editor->block+1, gui->editor->song->numblocks, 1,
	     gui->editor->song->blocks[gui->editor->block]->commandpages,
	     gui->editor->song->tempo, gui->editor->song->ticksperline);
    gtk_window_set_title(GTK_WINDOW(gui->window_main), title);
  }

  free(title);
}

void gui_info_refresh(struct gui *gui) {
  char section[9+8], playseq[10+8], block[7+10+32];
  int l=46+strlen(gui->editor->song->name)+intlen(gui->editor->block+1)+intlen(gui->editor->song->numblocks)+intlen(1)+intlen(gui->editor->song->blocks[gui->editor->block]->commandpages)+intlen(gui->editor->song->tempo)+intlen(gui->editor->song->ticksperline)+1;
  char *title=(char *)calloc(l, sizeof(char));

  /* Refresh the main window widgets if the window exists */
  if(gui->window_main!=NULL && GTK_IS_WINDOW(gui->window_main)) {
    GtkLabel *status=GTK_LABEL(glade_xml_get_widget(gui->xml, 
						    "gui_label_status"));

    snprintf(section, 8+8, "Section %d/%d", gui->editor->section+1,
	     gui->editor->song->numsections);
    snprintf(playseq, 9+8, "Position %d/%d", gui->editor->position+1,
	     gui->editor->song->playseqs[gui->editor->playseq]->length);
    if(gui->editor->song->blocks[gui->editor->block]->name!=NULL)
      snprintf(block, 6+10+32, "Block %d/%d: %s", gui->editor->block+1,
	       gui->editor->song->numblocks,
	       gui->editor->song->blocks[gui->editor->block]->name);
    else
      snprintf(block, 6+10+32, "Block %d/%d: ", gui->editor->block+1,
	       gui->editor->song->numblocks);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml,
						      "gui_label_section")),
		       section);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml,
						      "gui_label_position")),
		       playseq);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml,
						      "gui_label_block")),
		       block);
    
    /* Window title bar */
    snprintf(title, l,
	     "Tutka: %s - Block %d/%d - Cmd Page %d/%d - Tempo %d - TPL %d",
	     gui->editor->song->name,
	     gui->editor->block+1, gui->editor->song->numblocks,
	     gui->editor->cmdpage+1,
	     gui->editor->song->blocks[gui->editor->block]->commandpages,
	     gui->editor->song->tempo, gui->editor->song->ticksperline);
    gtk_window_set_title(GTK_WINDOW(gui->window_main), title); 

    /* Status bar */
    switch(gui->editor->player->mode) {
    case MODE_PLAY_SONG:
      gtk_label_set_text(status, "Playing song");
      break;
    case MODE_PLAY_BLOCK:
      gtk_label_set_text(status, "Playing block");
      break;
    case MODE_IDLE:
      gtk_label_set_text(status, "Stopped");
      break;
    }
  }
  free(title);
}

void gui_timer_refresh(struct gui *gui, unsigned int seconds) {
  char time[6];

  /* Refresh the main window widgets if the window exists */
  if(gui->window_main!=NULL && GTK_IS_WINDOW(gui->window_main)) {
    snprintf(time, 6, "%.2d:%.2d", seconds/60, seconds%60);
    gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "gui_label_timer")), time);
  }
}

/* Updates the current block information in the block list window */
void gui_blocklist_info_refresh(struct gui *gui) {
  GtkEntry *name=GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_blocklist"));
  char label[6+8];
  
  /* Show the number of the selected block */
  snprintf(label, 6+8, "Block %d/%d",
	   gui->editor->block+1, gui->editor->song->numblocks);
  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml,
						     "gui_frame_blocklist")),
		      label);
  
  /* Show the name of the selected block */
  if(gui->editor->song->blocks[gui->editor->block]->name!=NULL)
    gtk_entry_set_text(name, gui->editor->song->blocks[gui->editor->block]->name);
  else
    gtk_entry_set_text(name, "");
  
  /* Show the properties of the selected block */
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "gui_spinbutton_blocklist_tracks"))),
			   gui->editor->song->blocks[gui->editor->block]->tracks);
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "gui_spinbutton_blocklist_length"))),
			   gui->editor->song->blocks[gui->editor->block]->length);
  gtk_adjustment_set_value(gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(glade_xml_get_widget(gui->xml, "gui_spinbutton_blocklist_commandpages"))),
			   gui->editor->song->blocks[gui->editor->block]->commandpages);

  /* Make the delete button sensitive/insensitive */
  if(gui->editor->song->numblocks>1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_blocklist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_blocklist_delete"), FALSE);
  
}

/* Recreates and refreshes the block list window contents */
void gui_blocklist_refresh(struct gui *gui, gboolean recreate) {
  int i;
  char block[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(block, 10, "%d", gui->editor->block);

  if(recreate) {
    /* Recreate the block list */
    gtk_list_store_clear(gui->store_blocklist);
    for(i=0; i<gui->editor->song->numblocks; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_blocklist, &iter);
      gtk_list_store_set(gui->store_blocklist, &iter,
			 0, i+1, 1, gui->editor->song->blocks[i]->name, -1);
    }
  }

  /* Move back to the previously selected block */
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_blocklist"));
  treepath=gtk_tree_path_new_from_string(block);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview),
				 treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  gui_blocklist_info_refresh(gui);
}

/* Updates the current playseq information in the playing sequence window */
void gui_playseq_info_refresh(struct gui *gui) {
  char position[9+8], block[6+8], title[24];

  /* Show current block */
  snprintf(position, 9+8, "Position %d/%d", gui->editor->position+1,
	   gui->editor->song->playseqs[gui->editor->playseq]->length);
  snprintf(block, 6+8, "Block %d/%d", gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position]+1, gui->editor->song->numblocks);
  gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "gui_frame_playseq")), position);
  gtk_label_set_text(GTK_LABEL(glade_xml_get_widget(gui->xml, "gui_label_playseq_block")), block);

  /* Show playing sequence name */
  if(gui->editor->song->playseqs[gui->editor->playseq]->name!=NULL)
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_playseq_name")), gui->editor->song->playseqs[gui->editor->playseq]->name);
  else
    gtk_entry_set_text(GTK_ENTRY(glade_xml_get_widget(gui->xml, "gui_entry_playseq_name")), "");
    
  /* Set window title */
  snprintf(title, 24, "Playing Sequence %d/%d", gui->editor->playseq+1,
	   gui->editor->song->numplayseqs);
  gtk_window_set_title(GTK_WINDOW(glade_xml_get_widget(gui->xml, "gui_dialog_playseq")), title);

  /* Make the delete position button sensitive/insensitive */
  if(gui->editor->song->playseqs[gui->editor->playseq]->length>1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_delete")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_delete")), FALSE);

  /* Make the next/previous block buttons sensitive/insensitive */
  if(gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position]>0)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_prev")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_prev")), FALSE);
  if(gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[gui->editor->position]<gui->editor->song->numblocks-1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_next")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_next")), FALSE);

  /* Make the delete playing sequence button sensitive/insensitive */
  if(gui->editor->song->numplayseqs>1)
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_deletethis")), TRUE);
  else
    gtk_widget_set_sensitive(GTK_WIDGET(glade_xml_get_widget(gui->xml, "gui_button_playseq_deletethis")), FALSE);
}

void gui_playseq_refresh(struct gui *gui, gboolean recreate) {
  /* Refresh the playing sequence window widgets */
  int i;
  char playseqpos[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(playseqpos, 10, "%d", gui->editor->position);

  if(recreate) {
    /* Recreate the playing sequence */
    gtk_list_store_clear(gui->store_playseq);
    for(i=0; i<gui->editor->song->playseqs[gui->editor->playseq]->length; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_playseq, &iter);
      gtk_list_store_set(gui->store_playseq, &iter,
			 0, i+1, 1, gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i]+1, 2, gui->editor->song->blocks[gui->editor->song->playseqs[gui->editor->playseq]->blocknumbers[i]]->name, -1);
    }
    
  }
  /* Move back to the previously selected position */
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_playseq"));
  treepath=gtk_tree_path_new_from_string(playseqpos);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview),
				 treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);
  
  gui_playseq_info_refresh(gui);
}

/* Refresh section list buttons */
void gui_sectionlist_buttons_refresh(struct gui *gui) {
  /* Make the delete button sensitive/insensitive */
  if(gui->editor->song->numsections>1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_delete"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_delete"), FALSE);

  /* Make the next/previous buttons sensitive/insensitive */
  if(gui->editor->song->sections[gui->editor->section]>0)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_prev"), TRUE);
  else
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_prev"), FALSE);
  if(gui->editor->song->sections[gui->editor->section]<gui->editor->song->numplayseqs-1)
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_next"), TRUE);
  else
   gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sectionlist_next"), FALSE);
}

/* Refresh all section list widgets */
void gui_sectionlist_refresh(struct gui *gui, gboolean recreate) {
  int i;
  char section[10];
  GtkTreeView *treeview;
  GtkTreePath *treepath;
  snprintf(section, 10, "%d", gui->editor->section);
  
  if(recreate) {
    /* Recreate the section list */
    gtk_list_store_clear(gui->store_sectionlist);
    for(i=0; i<gui->editor->song->numsections; i++) {
      GtkTreeIter iter;
      gtk_list_store_append(gui->store_sectionlist, &iter);
      gtk_list_store_set(gui->store_sectionlist, &iter, 0, i+1,
			 1, gui->editor->song->sections[i]+1, 2,
			 gui->editor->song->playseqs[gui->editor->song->sections[i]]->name,
			 -1);
    }
  }

  /* Move back to the previously selected section */
  treeview=GTK_TREE_VIEW(glade_xml_get_widget(gui->xml,
					      "gui_treeview_sectionlist"));
  treepath=gtk_tree_path_new_from_string(section);
  gtk_tree_selection_select_path(gtk_tree_view_get_selection(treeview),
				 treepath);
  gtk_tree_view_scroll_to_cell(treeview, treepath, NULL, TRUE, 0.5, 0);
  gtk_tree_view_set_cursor(treeview, treepath, NULL, FALSE);

  gui_sectionlist_buttons_refresh(gui);
}

void gui_sysexlist_refresh(struct gui *gui) {
  /* Refresh the sysex list widgets */
  int i;
  char oldsysex[10], sysex[6+8];

  if(gui->editor->song->numsysexes==0)
    gui->sysexlist_sysex=-1;

  snprintf(oldsysex, 10, "%d", gui->sysexlist_sysex);
  
  /* Recreate the SysEx list */
  gtk_list_store_clear(gui->store_sysexlist);
  for(i=0; i<gui->editor->song->numsysexes; i++) {
    GtkTreeIter iter;
    gtk_list_store_append(gui->store_sysexlist, &iter);
    gtk_list_store_set(gui->store_sysexlist, &iter,
		       0, i, 1, gui->editor->song->sysexes[i]->name, -1);
  }
  
  /* Move back to the previously selected SysEx and make widgets sensitive */
  if(gui->sysexlist_sysex!=-1) {
    gtk_tree_selection_select_path(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(gui->xml, "gui_treeview_sysexlist"))), gtk_tree_path_new_from_string(oldsysex));

    /* Show the number of the selected SysEx message */
    snprintf(sysex, 6+8, "SysEx %d/%d", gui->sysexlist_sysex+1,
	     gui->editor->song->numsysexes);
    gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "gui_frame_sysex_controls")), sysex);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_table_sysex1"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_send"), TRUE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_receive"), TRUE);
    /* gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_checkbutton_sysex_autostop"), TRUE); */
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_delete"), TRUE);
  } else {
    gtk_frame_set_label(GTK_FRAME(glade_xml_get_widget(gui->xml, "gui_frame_sysex_controls")), "SysEx 0/0");
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_table_sysex1"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_send"), FALSE);
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_receive"), FALSE);
    /* gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_checkbutton_sysex_autostop"), FALSE); */
    gtk_widget_set_sensitive(glade_xml_get_widget(gui->xml, "gui_button_sysex_delete"), FALSE);
  }
}

void gui_trackvolumes_refresh(struct gui *gui) {
  /* Refresh the track volumes window widgets */
  int i;
  char track[4];
  GtkWidget *w1, *w2, *w3, *w4;
  GtkWidget *hbox=glade_xml_get_widget(gui->xml, "gui_hbox_trackvolumes");
  
  /* Destroy all children */
  gtk_container_foreach(GTK_CONTAINER(hbox), (GtkCallback)gtk_widget_destroy,
			NULL);
  
  for(i=0; i<gui->editor->song->maxtracks; i++) {
    w1=gtk_vbox_new(FALSE, 0);
    gtk_box_pack_start(GTK_BOX(hbox), w1, TRUE, TRUE, 4);
    
    /* Volume sliders */
    w2=(GtkWidget *)gtk_adjustment_new((gui->editor->song->trackvolumes[i])&127,
				       0, 127, 1, 1, 0);
    g_signal_connect(G_OBJECT(w2), "value_changed",
		     G_CALLBACK(on_gui_adjustment_trackvolumes_changed),
		     (gpointer)&gui->editor->song->trackvolumes[i]);
    w3=gtk_vscale_new(GTK_ADJUSTMENT(w2));
    gtk_range_set_inverted(GTK_RANGE(w3), TRUE);
    gtk_scale_set_digits(GTK_SCALE(w3), 0);
    gtk_box_pack_start(GTK_BOX(w1), w3, TRUE, TRUE, 4);
    
    /* Mute buttons */
    snprintf(track, 4, "%d", i+1);
    w4=gtk_check_button_new_with_label(track);
    if(gui->editor->song->trackvolumes[i]>127)
      gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w4), TRUE);
    g_signal_connect(G_OBJECT(w4), "toggled",
		       G_CALLBACK(on_gui_togglebutton_trackvolumes_toggled),
		       (gpointer)&gui->editor->song->trackvolumes[i]);
    gtk_box_pack_start(GTK_BOX(w1), w4, FALSE, FALSE, 4);
  }
  
  gtk_widget_show_all(hbox);
}
