/* player.h
 *
 * Copyright 2002-2003 Vesa Halttunen
 *
 * This file is part of Tutka.
 *
 * Tutka is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Tutka is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Tutka; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef _PLAYER_H
#define _PLAYER_H

#include "song.h"
#include "editor.h"

/* Player mode */
enum {
  MODE_IDLE,
  MODE_PLAY_SONG,
  MODE_PLAY_BLOCK
};

/* Track status values */
enum {
  TRACK_STATUS_MIDI_CHANNEL,
  TRACK_STATUS_NOTE,
  TRACK_STATUS_VOLUME,
  TRACK_STATUS_HOLD,
  TRACK_STATUS_PREVIOUS_COMMAND,
  TRACK_STATUS_NUMVALUES
};

/* 128 MIDI controllers plus aftertouch, channel pressure and pitch wheel */
enum {
  VALUES=(128+3),
  VALUES_AFTERTOUCH=128,
  VALUES_CHANNEL_PRESSURE=129,
  VALUES_PITCH_WHEEL=130
};

struct player {
  /* Current location in song */
  unsigned int section, playseq, position, block, line, tick;
  /* The song currently being played */
  struct song *song;
  /* Player mode */
  unsigned int mode;
  /* Status of tracks; notes playing */
  char *trackstatus;
  /* MIDI controller values; one for each controller on each channel */
  unsigned char midicontrollervalues[16*VALUES];
  /* For measuring how long the song has been playing */
  struct timeval playingstarted, playedsofar;
  /* The initialization status of the player thread */
  int threadstatus;
  /* Player thread pointer */
  pthread_t thread;
  /* MIDI device file descriptor */
  int midi;
  struct editor *editor;
};

/* Creates a player for a song */
struct player *player_open(struct song *, struct editor *);
/* Closes a player for a song */
void player_close(struct player *);

/* Starts the player thread */
void player_start(struct player *, unsigned int, int, int, int, int);
/* Kills the player thread */
void player_stop(struct player *);

/* Plays a note using given instrument on a given channel */
void player_play_note(struct player *, unsigned int, unsigned char,
		      unsigned char, unsigned char);
/* Stops notes playing at the moment */
void player_stop_notes(struct player *);
/* Stops all notes */
void player_stop_all_notes(struct player *);
/* Resets the pitch wheel on all channels */
void player_reset_pitch(struct player *);
/* Reallocate track status array */
void player_trackstatus_create(struct player *, int);

#endif
