// ship.h - declarations of ship and weapons
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef SHIP_H
#define SHIP_H

#include <list>
#include <string>

#include "ball.h"
#include "controller.h"

// needed for balls::ship
namespace Game { class game; }

namespace balls {
	void ship_register(factory_map& factories);

	class weapon;

	// The ship implementation. Takes care of controller (such
	// as keyboard), etc.
	class ship : public ball, public graphics::deep_drawable {
		friend class Game::game;
		friend class weapon;
	public:
		ship(const vect3d& ll, const vect3d& vv, Game::game* gg, control::controller* controller);
		~ship();

		void add_score(int i) { score += i;}

		void add_strength(double d);
		void add_shield(double d);
		void add_ammo(double d);
		void add_weapon(weapon* w);

		void move(double t);
		void do_collide(const ball* other);
		double damage() const;

	protected:
		int score;
		double strength, shield, ammo;

		double min_zvel, max_zvel;         // min and max z velocities
		double zaccel, zdecel;             // z acceleration/decelaration
		double xyvel;                      // velocity for x/y movement
		double max_str, max_shld, max_amm; // max strength...
		double depth_scr;                  // score for a depth of 1.0
		int depth_scr_till_now;            // score added for depth till now
		control::controller* control;      // controller for movement
		std::list<weapon*> weapons;       // available weapons
		std::list<weapon*>::iterator cur_weap;
		bool changed_weap_last_turn;
		double x_pending, y_pending;	// for mouse x and y not to be lost
		double last_accel, last_decel;  // acceleration holds for accel_hold ms

	private:
		ship(const ship&);
		ship& operator=(const ship&);

	};

	inline bool is_ship(const ball* b)
	{
		return dynamic_cast<const ship*>(b) != 0;
	}

	// abstract base class to hold information about weapon
	class weapon {
	public:
		weapon(ship* sp) : s(sp) { }
		virtual void charge(double t) = 0;
		virtual void fire() = 0;
		virtual const graphics::image* get_image() const = 0;
		virtual const std::string get_name() const = 0;
	protected:
		double& ammo() { return s->ammo; }
		ship* s;
	private:
		weapon(const weapon&);
		weapon& operator=(const weapon&);
	};

	// base class for pickups
	class pickup : public balls::ball {
	public:
		pickup(double r, double m, const vect3d& ll, const vect3d& vv, Game::game* gg)
			: ball(r, m, ll, vv, gg), dv(0, 0, 0) { }
		
		void move(double t);
		void do_collide(const ball* other);
		double damage() const;
		
		virtual void pick(ship* s) const = 0;
	protected:
		vect3d dv;
	};
	
	inline bool is_pickup(const balls::ball* b)
	{
		return dynamic_cast<const pickup*>(b) != 0;
	}

} // namespace balls

#endif // !SHIP_H

// Local Variables:
// mode: c++
// End:
