// ship.cc - functions required by ship
//
// Copyright (C) 2000, 2001, 2004 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "audio.h"
#include "ball.h"
#include "controller.h"
#include "game.h"
#include "graphics.h"
#include "ship.h"

namespace {
	using misc::vectx;
	using misc::vecty;
	using misc::vectz;
	using misc::vect2d;
	using misc::vect3d;

	const double accel_hold = 200;

	class ship_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		ship_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	class def_weapon : public balls::weapon {
	public:
		def_weapon(balls::ship* sp);
		void charge(double t);
		void fire();
		const graphics::image* get_image() const { return im; }
		const std::string get_name() const { return "def_weapon"; }
	private:
		double ch;
		double req_charge;
		double req_ammo;
		double extra_vel;
		const graphics::image* im;
		const audio::sound* snd;

		def_weapon(const def_weapon&);
		def_weapon& operator=(const def_weapon&);
	};

	class fast_weapon : public balls::weapon {
	public:
		fast_weapon(balls::ship* sp);
		void charge(double t);
		void fire();
		const graphics::image* get_image() const { return im; }
		const std::string get_name() const { return "fast_weapon"; }
	private:
		double ch;
		double req_charge;
		double req_ammo;
		double extra_vel;
		const graphics::image* im;
		const audio::sound* snd;

		fast_weapon(const fast_weapon&);
		fast_weapon& operator=(const fast_weapon&);
	};

	class strong_weapon : public balls::weapon {
	public:
		strong_weapon(balls::ship* sp);
		void charge(double t);
		void fire();
		const graphics::image* get_image() const { return im; }
		const std::string get_name() const { return "strong_weapon"; }
	private:
		double ch;
		double req_charge;
		double req_ammo;
		double extra_vel;
		const graphics::image* im;
		const audio::sound* snd;

		strong_weapon(const strong_weapon&);
		strong_weapon& operator=(const strong_weapon&);
	};

	class strong_fast_weapon : public balls::weapon {
	public:
		strong_fast_weapon(balls::ship* sp);
		void charge(double t);
		void fire();
		const graphics::image* get_image() const { return im; }
		const std::string get_name() const { return "strong_fast_weapon"; }
	private:
		double ch;
		double req_charge;
		double req_ammo;
		double extra_vel;
		const graphics::image* im;
		const audio::sound* snd;

		strong_fast_weapon(const strong_fast_weapon&);
		strong_fast_weapon& operator=(const strong_fast_weapon&);
	};

	// class normal_bullet
	// defines things in which a normal bullet differs from base
	class normal_bullet : public balls::ship_bullet, public graphics::deep_drawable {
	public:
		normal_bullet(const vect3d& ll, const vect3d& vv,
			      Game::game* gg, balls::ship* s);
	};

	// class strong_bullet
	class strong_bullet : public balls::ship_bullet, public graphics::deep_drawable {
	public:
		strong_bullet(const vect3d& ll, const vect3d& vv,
			      Game::game* gg, balls::ship* s);
	};

	class strength_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		strength_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class strength_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new strength_pickup(l, v, g)); }
	};

	class shield_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		shield_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class shield_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new shield_pickup(l, v, g)); }
	};

	class weap_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class weap_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new weap_pickup(l, v, g)); }
	};

	class fast_weap_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		fast_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class fast_weap_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new fast_weap_pickup(l, v, g)); }
	};

	class strong_weap_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		strong_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class strong_weap_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new strong_weap_pickup(l, v, g)); }
	};

	class strong_fast_weap_pickup : public balls::pickup, public graphics::deep_drawable {
	public:
		strong_fast_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void pick(balls::ship* s) const;
	private:
		double contents;
	};

	class strong_fast_weap_pickup_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g)
		{ g->swallow(new strong_fast_weap_pickup(l, v, g)); }
	};
} // namespace

void balls::ship_bullet::add_score(int i) const
{
	firer->add_score(i);
}

balls::ship::ship(const vect3d& ll, const vect3d& vv, Game::game* gg,
		  control::controller* controller)
	: ball(gg->get_bd().sub("ship", "radius"),
	       gg->get_bd().sub("ship", "mass"),
	       ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["ship"],
				  &gg->get_mapp()),
	  score(0),
	  strength(gg->get_bd().sub("ship", "strength")),
	  shield(gg->get_bd().sub("ship", "shield")),
	  ammo(gg->get_bd().sub("ship", "ammo")),
	  min_zvel(gg->get_bd().sub("ship", "minvelz")),
	  max_zvel(gg->get_bd().sub("ship", "maxvelz")),
	  zaccel(gg->get_bd().sub("ship", "accelz")),
	  zdecel(gg->get_bd().sub("ship", "decelz")),
	  xyvel(gg->get_bd().sub("ship", "velxy")),
	  max_str(gg->get_bd().sub("ship", "smax")),
	  max_shld(gg->get_bd().sub("ship", "hmax")),
	  max_amm(gg->get_bd().sub("ship", "amax")),
	  depth_scr(gg->get_bd().sub("ship", "depth_scr")),
	  depth_scr_till_now(0),
	  control(controller),
	  changed_weap_last_turn(false),
	  x_pending(0),
	  y_pending(0),
	  last_accel(-1),
	  last_decel(-1)
{
	cur_weap = weapons.begin();
}

balls::ship::~ship()
{
	for (std::list<weapon*>::iterator i = weapons.begin();
	     i != weapons.end(); ++i) {

		delete *i;
	}
}

void balls::ship::add_strength(double d)
{
	double req = max_str - strength;
	strength += req > 0 ? (req < d ? req : d) : 0;
}

void balls::ship::add_shield(double d)
{
	double req = max_shld - shield;
	shield += req > 0 ? (req < d ? req : d) : 0;
}

void balls::ship::add_ammo(double d)
{
	double req = max_amm - ammo;
	ammo += req > 0 ? (req < d ? req : d) : 0;
}

void balls::ship::add_weapon(weapon* w)
{
	std::list<weapon*>::iterator i;
	for (i = weapons.begin(); i != weapons.end(); ++i) {
		if ((*i)->get_name() == w->get_name()) {
			delete w;
			return;
		}
	}
	weapons.push_back(w);
	cur_weap = weapons.end();
	-- cur_weap;
}

// move the ship
void balls::ship::move(double t)
{
	v.setx(0.0).sety(0.0);

	bool left = control->left(), right = control->right();
	bool up = control->up(), down = control->down();
	bool accel = control->accel() || control->accel(control->click);
	bool decel = control->decel() || control->decel(control->click);;
	double accel_e, decel_e;
	bool fire = control->fire(), weapon = control->weapon();
	int x_pix = control->get_x_pix(), y_pix = control->get_y_pix();

	// if mouse motion and mouse pending opposite, cancel pending
	if ((x_pix >= 0 && x_pending < 0) || (x_pix <= 0 && x_pending > 0))
		x_pending = 0;
	if ((y_pix >= 0 && y_pending < 0) || (y_pix <= 0 && y_pending > 0))
		y_pending = 0;
	// add new mouse motion to pending mouse motion
	x_pending += g->get_mapp().iprojx(x_pix, 0);
	y_pending += g->get_mapp().iprojy(y_pix, 0);

	// move left
	if (left && !right)
		v.setx(-xyvel);
	// move right
	if (right && !left)
		v.setx(xyvel);
	// move up
	if (up && !down)
		v.sety(-xyvel);
	// move down
	if (down && !up)
		v.sety(xyvel);
	// check for analogue (mouse) motion
	v.x() += x_pending / t;
	v.y() += y_pending / t;

	// check if too fast
	if (misc::abs(v.xy()) > xyvel) {
		v.xy() *= xyvel / misc::abs(v.xy());
	}

	if (accel) {
		accel_e = zaccel * t;
		if (last_accel < 0)
			last_accel = t;
		else if (last_accel < accel_hold)
			last_accel += t;
	} else if (last_accel > 0) {
		if (last_accel < accel_hold - t) {
			accel_e = zaccel * t;
			last_accel += t;
		} else if (last_accel < accel_hold) {
			accel_e = zaccel * (accel_hold - last_accel);
			last_accel = -1;
		} else {
			accel_e = 0;
			last_accel = -1;
		}
	} else {
		accel_e = 0;
	}

	if (decel) {
		decel_e = zaccel * t;
		if (last_decel < 0)
			last_decel = t;
		else if (last_decel < accel_hold)
			last_decel += t;
	} else if (last_decel > 0) {
		if (last_decel < accel_hold - t) {
			decel_e = zaccel * t;
			last_decel += t;
		} else if (last_decel < accel_hold) {
			decel_e = zaccel * (accel_hold - last_decel);
			last_decel = -1;
		} else {
			decel_e = 0;
			last_decel = -1;
		}
	} else {
		decel_e = 0;
	}

	// accelerate
	if (accel_e > 0 && decel_e <= 0)
		v += vectz(accel_e);
	// decelerate
	if (decel_e > 0 && accel_e <= 0)
		v -= vectz(decel_e);

	if (v.z() < min_zvel)
		v.setz(min_zvel);
	else if (v.z() > max_zvel)
		v.setz(max_zvel);

	l += v * t;

	// reduce x_ and y_pending accordingly
	if ((v.x() * t > 0 && x_pending > v.x() * t) || (v.x() * t < 0 && x_pending < v.x() * t))
		x_pending -= v.x() * t;
	else
		x_pending = 0;
	if ((v.y() * t > 0 && y_pending > v.y() * t) || (v.y() * t < 0 && y_pending < v.y() * t))
		y_pending -= v.y() * t;
	else
		y_pending = 0;

	if (hit_wall(this))
		get_inside_wall(this);
	if (strength <= 0) {
		die();

		const rec::record& r = g->get_bd().sub("ship");
		int frags = r["frags"];
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["velxy"]));
			ball* frag = new ship_frag(floc, fvel, g);
			g->swallow(frag);
		}
		g->get_sounds()["ship"]->play(1.0);
	}
	// toggle weapon
	if (changed_weap_last_turn) {
		if (!weapon)
			changed_weap_last_turn = false;
	}
	else {
		if (weapon) {
			changed_weap_last_turn = true;
			// if previous weapon is valid, move to next weapon
			if (cur_weap != weapons.end())
				++cur_weap;
			// if previous weapon is invalid or we have
			// reached end of list, move to beginning of
			// list. Note that if there are no weapons,
			// cur_weap == weapons->end() both before and
			// after.
			if (cur_weap == weapons.end())
				cur_weap = weapons.begin();
		}
	}

	// fire
	if (cur_weap != weapons.end()) {
		if (fire)
			(*cur_weap)->fire();
		(*cur_weap)->charge(t);
	}

	// add score for depth
	int req_score = int(l.z() * depth_scr);
	add_score(req_score - depth_scr_till_now);
	depth_scr_till_now = req_score;
}

void balls::ship::do_collide(const ball* other)
{
	double d = other -> damage();
	double shield_loss = d * 2 / 3;
	if (shield_loss > shield)
		shield_loss = shield;
	shield -= shield_loss;
	strength -= d - shield_loss;

	const pickup* p = dynamic_cast<const pickup*>(other);
	if (p)
		p->pick(this);
}

// damage caused by a ship
double balls::ship::damage() const
{
	return mas;
}

ship_frag::ship_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("ship_frag", "radius"),
		 gg->get_bd().sub("ship_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("ship_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["ship_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

def_weapon::def_weapon(balls::ship* sp)
	: weapon(sp),
	  ch(0),
	  req_charge(sp->gam()->get_bd().sub("def_weapon", "req_charge")),
	  req_ammo(sp->gam()->get_bd().sub("def_weapon", "req_ammo")),
	  extra_vel(sp->gam()->get_bd().sub("def_weapon", "extra_vel")),
	  im(sp->gam()->get_wicons()["def_weapon"]),
	  snd(sp->gam()->get_sounds()["def_weapon"])
{
}

void def_weapon::charge(double t)
{
	ch += t;
}

void def_weapon::fire()
{
	if (ch < req_charge || ammo() < req_ammo)
		return;
	ch = 0;
	ammo() -= req_ammo;
	vect3d bloc(s->loc()), bvel(vectz(s->vel().z()));
	bloc += vectz(s->radius() * 3);
	bvel += vectz(extra_vel);
	balls::ball* bullet = new normal_bullet(bloc, bvel, s->gam(), s);
	s->gam()->swallow(bullet);
	snd->play(1.0);
}

fast_weapon::fast_weapon(balls::ship* sp)
	: weapon(sp),
	  ch(0),
	  req_charge(sp->gam()->get_bd().sub("fast_weapon", "req_charge")),
	  req_ammo(sp->gam()->get_bd().sub("fast_weapon", "req_ammo")),
	  extra_vel(sp->gam()->get_bd().sub("fast_weapon", "extra_vel")),
	  im(sp->gam()->get_wicons()["fast_weapon"]),
	  snd(sp->gam()->get_sounds()["fast_weapon"])
{
}

void fast_weapon::charge(double t)
{
	ch += t;
}

void fast_weapon::fire()
{
	if (ch < req_charge || ammo() < req_ammo)
		return;
	ch = 0;
	ammo() -= req_ammo;
	vect3d bloc(s->loc()), bvel(vectz(s->vel().z()));
	bloc += vectz(s->radius() * 3);
	bvel += vectz(extra_vel);
	balls::ball* bullet = new normal_bullet(bloc, bvel, s->gam(), s);
	s->gam()->swallow(bullet);
	snd->play(1.0);
}

strong_weapon::strong_weapon(balls::ship* sp)
	: weapon(sp),
	  ch(0),
	  req_charge(sp->gam()->get_bd().sub("strong_weapon", "req_charge")),
	  req_ammo(sp->gam()->get_bd().sub("strong_weapon", "req_ammo")),
	  extra_vel(sp->gam()->get_bd().sub("strong_weapon", "extra_vel")),
	  im(sp->gam()->get_wicons()["strong_weapon"]),
	  snd(sp->gam()->get_sounds()["strong_weapon"])
{
}

void strong_weapon::charge(double t)
{
	ch += t;
}

void strong_weapon::fire()
{
	if (ch < req_charge || ammo() < req_ammo)
		return;
	ch = 0;
	ammo() -= req_ammo;
	vect3d bloc(s->loc()), bvel(vectz(s->vel().z()));
	bloc += vectz(s->radius() * 3);
	bvel += vectz(extra_vel);
	balls::ball* bullet = new strong_bullet(bloc, bvel, s->gam(), s);
	s->gam()->swallow(bullet);
	snd->play(1.0);
}

strong_fast_weapon::strong_fast_weapon(balls::ship* sp)
	: weapon(sp),
	  ch(0),
	  req_charge(sp->gam()->get_bd().sub("strong_fast_weapon", "req_charge")),
	  req_ammo(sp->gam()->get_bd().sub("strong_fast_weapon", "req_ammo")),
	  extra_vel(sp->gam()->get_bd().sub("strong_fast_weapon", "extra_vel")),
	  im(sp->gam()->get_wicons()["strong_fast_weapon"]),
	  snd(sp->gam()->get_sounds()["strong_fast_weapon"])
{
}

void strong_fast_weapon::charge(double t)
{
	ch += t;
}

void strong_fast_weapon::fire()
{
	if (ch < req_charge || ammo() < req_ammo)
		return;
	ch = 0;
	ammo() -= req_ammo;
	vect3d bloc(s->loc()), bvel(vectz(s->vel().z()));
	bloc += vectz(s->radius() * 3);
	bvel += vectz(extra_vel);
	balls::ball* bullet = new strong_bullet(bloc, bvel, s->gam(), s);
	s->gam()->swallow(bullet);
	snd->play(1.0);
}

normal_bullet::normal_bullet(const vect3d& ll, const vect3d& vv,
                                    Game::game* gg, balls::ship* s)
	: ship_bullet(gg->get_bd().sub("normal_bullet", "radius"),
		      gg->get_bd().sub("normal_bullet", "mass"),
		      ll, vv, gg,
		      gg->get_bd().sub("normal_bullet", "damage"), s),
	  graphics::deep_drawable(gg->get_gallery()["normal_bullet"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

strong_bullet::strong_bullet(const vect3d& ll, const vect3d& vv,
                                    Game::game* gg, balls::ship* s)
	: ship_bullet(gg->get_bd().sub("strong_bullet", "radius"),
		      gg->get_bd().sub("strong_bullet", "mass"),
		      ll, vv, gg,
		      gg->get_bd().sub("strong_bullet", "damage"), s),
	  graphics::deep_drawable(gg->get_gallery()["strong_bullet"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

void balls::pickup::move(double t)
{
	v += dv;
	dv.set();

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}
}

void balls::pickup::do_collide(const ball* other)
{
	if (is_ship(other))
		die();
	dv += dvel_collision(this, other);
}

double balls::pickup::damage() const
{
	return 0;
}

strength_pickup::strength_pickup(const vect3d& ll, const vect3d& vv,
                                        Game::game* gg)
	: pickup(gg->get_bd().sub("strength_pickup", "radius"),
		 gg->get_bd().sub("strength_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["strength_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("strength_pickup", "contents"))
{
}

void strength_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_strength(contents);
}

shield_pickup::shield_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: pickup(gg->get_bd().sub("shield_pickup", "radius"),
		 gg->get_bd().sub("shield_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["shield_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("shield_pickup", "contents"))
{
}

void shield_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_shield(contents);
}

weap_pickup::weap_pickup(const vect3d& ll, const vect3d& vv,
                                Game::game* gg)
	: pickup(gg->get_bd().sub("weap_pickup", "radius"),
		 gg->get_bd().sub("weap_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["weap_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("weap_pickup", "contents"))
{
}

void weap_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_weapon(new def_weapon(s));
	s->add_ammo(contents);
}

fast_weap_pickup::fast_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: pickup(gg->get_bd().sub("fast_weap_pickup", "radius"),
		 gg->get_bd().sub("fast_weap_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["fast_weap_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("fast_weap_pickup", "contents"))
{
}

void fast_weap_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_weapon(new fast_weapon(s));
	s->add_ammo(contents);
}

strong_weap_pickup::strong_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: pickup(gg->get_bd().sub("strong_weap_pickup", "radius"),
		 gg->get_bd().sub("strong_weap_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["strong_weap_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("strong_weap_pickup", "contents"))
{
}

void strong_weap_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_weapon(new strong_weapon(s));
	s->add_ammo(contents);
}

strong_fast_weap_pickup::strong_fast_weap_pickup(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: pickup(gg->get_bd().sub("strong_fast_weap_pickup", "radius"),
		 gg->get_bd().sub("strong_fast_weap_pickup", "mass"),
		 ll, vv, gg),
	  graphics::deep_drawable(gg->get_gallery()["strong_fast_weap_pickup"],
				  &gg->get_mapp()),
	  contents(gg->get_bd().sub("strong_fast_weap_pickup", "contents"))
{
}

void strong_fast_weap_pickup::pick(balls::ship* s) const
{
	g->get_sounds()["pickup"]->play(1.0);
	s->add_weapon(new strong_fast_weapon(s));
	s->add_ammo(contents);
}

void balls::ship_register(factory_map& factories)
{
	factory_register(factories, "weap_pickup",
			 new weap_pickup_factory);
	factory_register(factories, "fast_weap_pickup",
			 new fast_weap_pickup_factory);
	factory_register(factories, "shield_pickup",
			 new shield_pickup_factory);
	factory_register(factories, "strength_pickup",
			 new strength_pickup_factory);
	factory_register(factories, "strong_fast_weap_pickup",
			 new strong_fast_weap_pickup_factory);
	factory_register(factories, "strong_weap_pickup",
			 new strong_weap_pickup_factory);
}

// Local Variables:
// mode: c++
// End:
