// sdl_timer.cc - various functions for the timer with SDL
//
// Copyright (C) 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "SDL.h"

#include "sdl_timer.h"

sdl::timer::timer(double cycle, double drawmin, double drawmax)
	: cycle_dur(cycle), draw_min(drawmin), draw_max(drawmax),
	  paused(true)
{
	reset();
}

tim::timer::item sdl::timer::next()
{
	double cur_time = SDL_GetTicks() - first_time;

	// if last item was a play and we NEED to draw
	if (last_time[draw] < last_time[play]
	    && last_time[draw] + draw_max < cur_time) {
		reset();
		return draw;
	}

	double play_due = last_time[play] + cycle_dur;
	double draw_due = last_time[draw] + draw_min;

	// if a play is overdue, we play
	if (play_due < cur_time) {
		last_time[play] += cycle_dur;
		return play;
	}

	// otherwise, if a draw is overdue, we draw
	if (draw_due < cur_time) {
		last_time[draw] = cur_time;
		return draw;
	}

	// what is next, draw or play?
	if (play_due < draw_due) {
		sleep(play_due - cur_time);
		last_time[play] += cycle_dur;
		return play;
	}
	else {
		sleep(draw_due - cur_time);
		cur_time = SDL_GetTicks() - first_time;
		last_time[draw] = cur_time;
		return draw;
	}
}

void sdl::timer::start()
{
	if (paused == true) {
		reset();
		paused = false;
	}
}

void sdl::timer::pause()
{
	last_time[no_of_items] = SDL_GetTicks() - first_time;
	paused = true;
}

void sdl::timer::reset()
{
	for (int i = 0; i <= no_of_items; ++i)
		last_time[i] = 0;
	first_time = SDL_GetTicks();
}

void sdl::timer::sleep(double msec)
{
	Uint32 s = SDL_GetTicks();
	Uint32 m = Uint32(msec);

	for (Uint32 t = SDL_GetTicks() - s; t < m; t = SDL_GetTicks())
		SDL_Delay(m - t);
}

// Local Variables:
// mode: c++
// End:
