// sdl_graphics.cc - functions from graphics using SDL
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <algorithm>
#include <cmath>
#include <utility>
#include <string>
#include "SDL.h"
#include "SDL_image.h"

#include "graphics.h"
#include "misc.h"
#include "sdl_graphics.h"

#define D4(command, n)				\
do {						\
	int i = (n + 3) / 4;		\
	switch (n % 4) {		\
	case 0:	do {	command;	\
	case 3:		command;	\
	case 2:		command;	\
	case 1:		command;	\
		} while (--i > 0);	\
	}				\
} while (0)

namespace sdl {

	// Set a pixel at *dst and move dst to next pixel
	inline void setpixm1(Uint8*& dst, Uint32 col)
	{
		*(dst++) = col;
	}

	inline void setpixm2(Uint8*& dst, Uint32 col)
	{
		* reinterpret_cast<Uint16*>(dst) = col;
		dst += 2;
	}

	inline void setpixm3(Uint8*& dst, Uint32 col)
	{
		if(SDL_BYTEORDER == SDL_BIG_ENDIAN) {
			dst[0] = (col >> 16) & 0xff;
			dst[1] = (col >> 8) & 0xff;
			dst[2] = col & 0xff;
		} else {
			dst[0] = col & 0xff;
			dst[1] = (col >> 8) & 0xff;
			dst[2] = (col >> 16) & 0xff;
		}
		dst += 3;
	}

	inline void setpixm4(Uint8*& dst, Uint32 col)
	{
		* reinterpret_cast<Uint32*>(dst) = col;
		dst += 4;
	}

	inline void setpixm(Uint8*& dst, Uint32 col, int bypp)
	{
		switch (bypp) {
		case 1:
			setpixm1(dst, col);
			break;
		case 2:
			setpixm2(dst, col);
			break;
		case 3:
			setpixm3(dst, col);
			break;
		case 4:
			setpixm4(dst, col);
			break;
		default:
			; // should not happen
		}
	}

	// Get the pixel at *dst and move dst to next pixel
	inline void getpixm1(const Uint8*& dst, Uint32& col)
	{
		col = *(dst++);
	}

	inline void getpixm2(const Uint8*& dst, Uint32& col)
	{
		col = * reinterpret_cast<const Uint16*>(dst);
		dst += 2;
	}

	inline void getpixm3(const Uint8*& dst, Uint32& col)
	{
		if (SDL_BYTEORDER == SDL_BIG_ENDIAN)
			col = dst[0] << 16 | dst[1] << 8 | dst[2];
		else
			col = dst[0] | dst[1] << 8 | dst[2] << 16;
		dst += 3;
	}

	inline void getpixm4(const Uint8*& dst, Uint32& col)
	{
		col = * reinterpret_cast<const Uint32*>(dst);
		dst += 4;
	}

	inline void getpixm(const Uint8*& dst, Uint32& col, int bypp)
	{
		switch (bypp) {
		case 1:
			getpixm1(dst, col);
			break;
		case 2:
			getpixm2(dst, col);
			break;
		case 3:
			getpixm3(dst, col);
			break;
		case 4:
			getpixm4(dst, col);
			break;
		default:
			; // should not happen
		}
	}

	// Set pixels at *dst and move dst to next pixel
	inline void setpixelsm1(Uint8*& dst, Uint32 col, int n)
	{
		D4(*(dst++) = col, n);
	}

	inline void setpixelsm2(Uint8*& dst, Uint32 col, int n)
	{
		Uint16* dstt = reinterpret_cast<Uint16*>(dst);
		D4(*(dstt++) = col, n);
		dst = reinterpret_cast<Uint8*>(dstt);
	}

	inline void setpixelsm3(Uint8*& dst, Uint32 col, int n)
	{
		Uint8* srcp = reinterpret_cast<Uint8*>(&col);

		if (SDL_BYTEORDER == SDL_BIG_ENDIAN)
			col <<= 8;

		D4(	*(dst++) = srcp[0];
			*(dst++) = srcp[1];
			*(dst++) = srcp[2], n);
	}

	inline void setpixelsm4(Uint8*& dst, Uint32 col, int n)
	{
		Uint32* dstt = reinterpret_cast<Uint32*>(dst);
		D4(*(dstt++) = col, n);
		dst = reinterpret_cast<Uint8*>(dstt);
	}

	inline void setpixelsm(Uint8*& dst, Uint32 col, int bypp, int n)
	{
		switch (bypp) {
		case 1:
			setpixelsm1(dst, col, n);
			break;
		case 2:
			setpixelsm2(dst, col, n);
			break;
		case 3:
			setpixelsm3(dst, col, n);
			break;
		case 4:
			setpixelsm4(dst, col, n);
			break;
		default:
			; // should not happen
		}
	}

	// Wrappers for above functions not to move dst.
	// (Only difference is that these take Uint8* instead of Uint8*&)
	inline void setpix(Uint8* dst, Uint32 col, int bypp)
        { setpixm(dst, col, bypp); }
	inline void getpix(const Uint8* dst, Uint32& col, int bypp)
        { getpixm(dst, col, bypp); }
	inline void setpixels(Uint8* dst, Uint32 col, int bypp, int n)
        { setpixelsm(dst, col, bypp, n); }

	// Make an image from a SDL_Surface. New size is w * h, and colours
	// are amplified by bright such that a bright value of 0 would
	// give a black image. The image is stored with the given display
	// format for fast blitting.
	class surface : public graphics::raw_image {
	public:
		surface(const std::string& filename);
		surface(const SDL_Surface* src, const SDL_PixelFormat* format,
			int w, int h, double bright);
		surface(const surface& i);
		~surface();

		surface& operator=(const surface& i);

		const SDL_Surface* access() const { return s; }

		int w() const { return s->w; }
		int h() const { return s->h; }
	private:
		int* ref_count;
		SDL_Surface* s;
	};

	// An implementation of Abstract Base Class (ABC) graphics::image.
	class image : public graphics::image {
	public:
		image(const surface* src, const SDL_PixelFormat* format,
		      int w, int h, double bright, tsurface* dst);

		void draw(int x, int y) const;

		const SDL_Surface* access() const { return s.access(); }

	private:
		surface s;
		tsurface* ts;
	};

	// An implementation of Abstract Base Class (ABC) graphics::deep_image.
	// It has protected member function geti(...) to get the image
	// corresponding to a certain depth.
	class deep_image : public graphics::deep_image {
	public:
		template <typename In1, typename In2, typename In3>
		deep_image(const surface* src, const SDL_PixelFormat* format,
			   int n, In1 w, In2 h, In3 bright, tsurface* dst);

		void draw(const misc::vect3d& loc, const graphics::mapper& m) const;

	protected:
		surface& geti(double depth)
		{ return arr[int(size * depth / tunnel_dpth)]; }
		const surface& geti(double depth) const
		{ return arr[int(size * depth / tunnel_dpth)]; }
	private:
		int size;
		std::vector<surface> arr;
		tsurface* ts;
	};

	class dial : public graphics::dial {
	public:
		dial(tsurface* t, const graphics::raw_image* src, double amin,
		     double amax, int w, int h);

		void draw(int x, int y, double actual, double min, double max) const;
	private:
		tsurface* ts;
		int bypp, pitch;
		double Amin, range;
		int W, H;
		Uint32 aminc, currc, maxc, amaxc;
	};

	// draw the tunnel
	class tunnel_drawable : public graphics::tunnel_drawable {
	public:
		tunnel_drawable(tsurface* t, const graphics::raw_image* src,
				const graphics::mapper* m, int r);
		~tunnel_drawable();

		void move(double d);
	protected:
		struct rgb { Uint8 r, g, b; };

		tsurface* ts;
		const graphics::mapper* mp;

		int rings, rx, ry;
		unsigned char* timage;
		int ncols;
		rgb* cols;
		double* brights;
		int first_col;
		Uint32 back_col;
		Uint32 center_col;

		double required_depth;
		double absolute_depth;
		double current_depth;

		static void write(unsigned char*& i, int n)
		{ while (n >= 250) { *(i++) = 250; n -= 250; } *(i++) = n; }
		static int read(const unsigned char*& i)
		{ int n = 0; n += *i; while (*(i++) == 250) { n += *i; } return n; }
		static int readr(const unsigned char*& i)
		{ int n = 0; n += *--i; while (*(i-1)==250) { n+=250;--i; } return n; }

		int last_col;
		void next_col();

	private:
		tunnel_drawable(const tunnel_drawable&);
		tunnel_drawable& operator=(const tunnel_drawable&);
	};

	class tunnel_drawable1 : public tunnel_drawable {
	public:
		tunnel_drawable1(tsurface* t,
				 const graphics::raw_image* src,
				 const graphics::mapper* m, int r)
			: tunnel_drawable(t, src, m, r) { }

		void draw(const misc::vect3d& loc) const;
	};

	class tunnel_drawable2 : public tunnel_drawable {
	public:
		tunnel_drawable2(tsurface* t,
				 const graphics::raw_image* src,
				 const graphics::mapper* m, int r)
			: tunnel_drawable(t, src, m, r) { }

		void draw(const misc::vect3d& loc) const;
	};

	class tunnel_drawable3 : public tunnel_drawable {
	public:
		tunnel_drawable3(tsurface* t,
				 const graphics::raw_image* src,
				 const graphics::mapper* m, int r)
			: tunnel_drawable(t, src, m, r) { }

		void draw(const misc::vect3d& loc) const;
	};

	class tunnel_drawable4 : public tunnel_drawable {
	public:
		tunnel_drawable4(tsurface* t,
				 const graphics::raw_image* src,
				 const graphics::mapper* m, int r)
			: tunnel_drawable(t, src, m, r) { }

		void draw(const misc::vect3d& loc) const;
	};

} // namespace sdl

// constructor for sdl::deep_image. Takes an oimage& containing the
// source, the number of depths in the deep_image, the widths and heights
// of all the images, the amplification to brightness of all the images,
// and the tsurface on which to be drawn.
// n is the number of depths and w, h, bright are input iterators that
// are assumed to have n items each.
template <typename In1, typename In2, typename In3>
sdl::deep_image::deep_image(const surface* src, const SDL_PixelFormat* format,
                            int n, In1 w, In2 h, In3 bright, tsurface* dst)
	: size(n), ts(dst)
{
	// avoid reallocation
	arr.reserve(size);

	for (int i = 0; i < size; ++i) {
		// create images from the source.
		arr.push_back(surface(src->access(), format, *w, *h, *bright));
		++w;
		++h;
		++bright;
	}
}

// Reads an SDL_Surface from a file, and leaves it in its format.
sdl::surface::surface(const std::string& filename)
	: ref_count(new int(1)),
	  s(IMG_Load(filename.c_str()))
{
	if (s == 0)
		throw std::string("unable to load image: ") + filename;
}

namespace {
	void do_bright(const SDL_Surface* src, SDL_Surface* dst, int br)
	{
		int w = src->w;
		int h = src->h;
		const Uint8* srcp = reinterpret_cast<const Uint8*>(src->pixels);
		int srcpitch = src->pitch;
		int srcskip = srcpitch - (w << 2);
		Uint8* dstp = reinterpret_cast<Uint8*>(dst->pixels);
		int dstpitch = dst->pitch;
		int dstskip = dstpitch - (w << 2);

		while (h--) {
			if (br == 0x100) {
				std::memcpy(dstp, srcp, w << 2);
				srcp += srcpitch;
				dstp += dstpitch;
			}
			else {
				D4({
					Uint32 s = *reinterpret_cast<const Uint32*>(srcp);
					*reinterpret_cast<Uint32*>(dstp)
						= (s & 0xff000000
						   | ((((s & 0xff00ff) * br) >> 8) & 0xff00ff)
						   | ((((s & 0xff00) * br) >> 8) & 0xff00));
					srcp += 4;
					dstp += 4;
				}, w);
				srcp += srcskip;
				dstp += dstskip;
			}
		}
	}
	
	void scale_width_bright(const SDL_Surface* src, SDL_Surface* dst, int br)
	{
		int srcw = src->w;
		int h = src->h;
		const Uint8* srcp = reinterpret_cast<const Uint8*>(src->pixels);
		int srcpitch = src->pitch;
		int dstw = dst->w;
		Uint8* dstp = reinterpret_cast<Uint8*>(dst->pixels);
		int dstpitch = dst->pitch;

		Uint32* buf = new Uint32[h * 4];
		std::memset(buf, 0, h * 4 * sizeof(Uint32));

		int wt;
		int index = srcw;
		int lower = 0, rem = 0;
		int nlower = srcw / dstw, nrem = srcw % dstw;

		for (int i = 0; i < srcw;) {
			if (i == nlower)
				wt = nrem;
			else
				wt = dstw;
			if (i == lower)
				wt -= rem;

			if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
				const Uint8* srcp2 = srcp;
				Uint32* bufp = buf;
				D4({
					int wt2 = srcp2[0] * wt;
					*(bufp++) += wt2;
					*(bufp++) += srcp2[1] * wt2;
					*(bufp++) += srcp2[2] * wt2;
					*(bufp++) += srcp2[3] * wt2;
					srcp2 += srcpitch;
				}, h);
			}
			else {
				const Uint8* srcp2 = srcp;
				Uint32* bufp = buf;
				D4({
					int wt2 = srcp2[3] * wt;
					*(bufp++) += wt2;
					*(bufp++) += srcp2[0] * wt2;
					*(bufp++) += srcp2[1] * wt2;
					*(bufp++) += srcp2[2] * wt2;
					srcp2 += srcpitch;
				}, h);
			}
			if (i == nlower || (i == nlower-1 && nrem == 0)) {
				if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
					Uint8* dstp2 = dstp;
					Uint32* bufp = buf;
					D4({
						int saw = bufp[0];
						if (saw) {
							dstp2[0] = saw / srcw;
							dstp2[1] = (bufp[1] / saw * br) >> 8;
							dstp2[2] = (bufp[2] / saw * br) >> 8;
							dstp2[3] = (bufp[3] / saw * br) >> 8;
						}
						else {
							*reinterpret_cast<Uint32*>(dstp2) = 0;
						}
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						dstp2 += dstpitch;
					}, h);
					dstp += 4;
				}
				else {
					Uint8* dstp2 = dstp;
					Uint32* bufp = buf;
					D4({
						int saw = bufp[0];
						if (saw) {
							dstp2[3] = saw / srcw;
							dstp2[0] = (bufp[1] / saw * br) >> 8;
							dstp2[1] = (bufp[2] / saw * br) >> 8;
							dstp2[2] = (bufp[3] / saw * br) >> 8;
						}
						else {
							*reinterpret_cast<Uint32*>(dstp2) = 0;
						}
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						dstp2 += dstpitch;
					}, h);
					dstp += 4;
				}

				if (i != nlower) {
					++i;
					srcp += 4;
				}

				index += srcw;
				lower = nlower;
				rem = nrem;
				nlower = index / dstw;
				nrem = index % dstw;
			}
			else if (i != nlower) {
				++i;
				srcp += 4;
			}
		}

		delete[] buf;
	}
	
	void scale_height_bright(const SDL_Surface* src, SDL_Surface* dst, int br)
	{
		int w = src->w;
		int srch = src->h;
		const Uint8* srcp = reinterpret_cast<const Uint8*>(src->pixels);
		int srcpitch = src->pitch;
		int dsth = dst->h;
		Uint8* dstp = reinterpret_cast<Uint8*>(dst->pixels);
		int dstpitch = dst->pitch;

		Uint32* buf = new Uint32[w * 4];
		std::memset(buf, 0, w * 4 * sizeof(Uint32));

		int wt;
		int index = srch;
		int lower = 0, rem = 0;
		int nlower = srch / dsth, nrem = srch % dsth;

		for (int i = 0; i < srch;) {
			if (i == nlower)
				wt = nrem;
			else
				wt = dsth;
			if (i == lower)
				wt -= rem;

			if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
				const Uint8* srcp2 = srcp;
				Uint32* bufp = buf;
				D4({
					int wt2 = srcp2[0] * wt;
					*(bufp++) += wt2;
					*(bufp++) += srcp2[1] * wt2;
					*(bufp++) += srcp2[2] * wt2;
					*(bufp++) += srcp2[3] * wt2;
					srcp2 += 4;
				}, w);
			}
			else {
				const Uint8* srcp2 = srcp;
				Uint32* bufp = buf;
				D4({
					int wt2 = srcp2[3] * wt;
					*(bufp++) += wt2;
					*(bufp++) += srcp2[0] * wt2;
					*(bufp++) += srcp2[1] * wt2;
					*(bufp++) += srcp2[2] * wt2;
					srcp2 += 4;
				}, w);
			}
			if (i == nlower || (i == nlower-1 && nrem == 0)) {
				if (SDL_BYTEORDER == SDL_BIG_ENDIAN) {
					Uint8* dstp2 = dstp;
					Uint32* bufp = buf;
					D4({
						int saw = bufp[0];
						if (saw) {
							dstp2[0] = saw / srch;
							dstp2[1] = (bufp[1] / saw * br) >> 8;
							dstp2[2] = (bufp[2] / saw * br) >> 8;
							dstp2[3] = (bufp[3] / saw * br) >> 8;
						}
						else {
							*reinterpret_cast<Uint32*>(dstp2) = 0;
						}
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						dstp2 += 4;
					}, w);
					dstp += dstpitch;
				}
				else {
					Uint8* dstp2 = dstp;
					Uint32* bufp = buf;
					D4({
						int saw = bufp[0];
						if (saw) {
							dstp2[3] = saw / srch;
							dstp2[0] = (bufp[1] / saw * br) >> 8;
							dstp2[1] = (bufp[2] / saw * br) >> 8;
							dstp2[2] = (bufp[3] / saw * br) >> 8;
						}
						else {
							*reinterpret_cast<Uint32*>(dstp2) = 0;
						}
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						*(bufp++) = 0;
						dstp2 += 4;
					}, w);
					dstp += dstpitch;
				}

				if (i != nlower) {
					srcp += srcpitch;
					++i;
				}

				index += srch;
				lower = nlower;
				rem = nrem;
				nlower = index / dsth;
				nrem = index % dsth;
			}
			else if (i != nlower) {
				++i;
				srcp += srcpitch;
			}
		}
		
		delete[] buf;
	}
} // namespace

// Create a surface with the given pixel format from src. It is scaled
// such that it has width w and height h, and its brightness is
// amplified by bright.
// Also initializes a reference count to avoid copying, since once image
// is created, it cannot be modified.
sdl::surface::surface(const SDL_Surface* src, const SDL_PixelFormat* format,
                      int w, int h, double bright)
	: ref_count(new int(1))
{
	int br = int(0x100 * bright), srcw = src->w, srch = src->h;
	if (format->Amask!= 0xff000000
	    || (format->Rmask | format->Gmask | format->Bmask) != 0xffffff
	    || (format->Rmask != 0xff00
		&& format->Gmask != 0xff00
		&& format->Bmask !=0xff00)) {

		throw "cannot scale this surface";
	}

	s = SDL_CreateRGBSurface(0, w, h, format->BitsPerPixel,
				 format->Rmask, format->Gmask,
				 format->Bmask, format->Amask);
	if (s == 0)
		throw "unable to create video surface";

	if (w == 0 || h == 0)
		return;

	SDL_Surface* step0;
	
	if (src->format->Amask == format->Amask
	    && src->format->Rmask == format->Rmask
	    && src->format->Gmask == format->Gmask
	    && src->format->Bmask == format->Bmask) {
		step0 = 0;
	}
	else {
		step0 = SDL_ConvertSurface(const_cast<SDL_Surface*>(src),
					   const_cast<SDL_PixelFormat*>(format),
					   0);
		if (step0 == 0)
			throw "unable to convert surface";
	}

	SDL_Surface* step1;
	if (w == srcw || h == srch) {
		step1 = 0;
	}
	else {
		step1 = SDL_CreateRGBSurface(0, w, srch, format->BitsPerPixel,
					     format->Rmask, format->Gmask,
					     format->Bmask, format->Amask);
		if (step1 == 0)
			throw "unable to create surface";
	}
						  
	if (w == srcw && h == srch) {
		do_bright(step0 == 0 ? src : step0, s, br);
	}
	else if (w == srcw) {
		scale_height_bright(step0 == 0 ? src : step0, s, br);
	}
	else if (h == srch) {
		scale_width_bright(step0 == 0 ? src : step0, s, br);
	}
	else {
		scale_width_bright(step0 == 0 ? src : step0, step1, 0x100);
		scale_height_bright(step1, s, br);
		SDL_FreeSurface(step1);
	}
	if (step0 != 0)
		SDL_FreeSurface(step0);

	if (SDL_SetAlpha(s, SDL_SRCALPHA | SDL_RLEACCEL, 0) != 0)
		throw "error converting video surface to alpha";
}

// Copy constructor increments reference count.
sdl::surface::surface(const surface& i)
	: ref_count(i.ref_count), s(i.s)
{
	++ *ref_count;
}

// Copies an image and increments reference count accordingly.
sdl::surface& sdl::surface::operator=(const surface& i)
{
	// make sure we are not assigning an image to itself
	if (&i == this)
		return *this;

	// first decrement count and destroy if it becomes zero.
	-- *ref_count;
	if (*ref_count == 0) {
		delete ref_count;
		SDL_FreeSurface(s);
	}

	// perform the copying and increment the reference count
	ref_count = i.ref_count;
	s = i.s;
	++ *ref_count;

	return *this;
}

// decrease reference count and destroy if count becomes zero
sdl::surface::~surface()
{
	-- *ref_count;
	if (*ref_count == 0) {
		SDL_FreeSurface(s);
		delete ref_count;
	}
}

sdl::image::image(const surface* src, const SDL_PixelFormat* format,
                  int w, int h, double bright, tsurface* dst)
	: s(src->access(), format, w, h, bright),
	  ts(dst)
{
}

// draw the image onto its target surface
void sdl::image::draw(int x, int y) const
{
	// src is the source of the blit and t is the target
	const SDL_Surface* src = s.access();
	SDL_Surface* t = ts->access();

	SDL_Rect srcrect;
	SDL_Rect dstrect;
	srcrect.x = srcrect.y = 0;
	srcrect.w = src->w;
	srcrect.h = src->h;
	dstrect.x = x;
	dstrect.y = y;
	dstrect.w = srcrect.w;
	dstrect.h = srcrect.h;

	// use LowerBlit because we have the correct rectangles, so no
	// need for checking, clipping, etc.
	if (SDL_BlitSurface(const_cast<SDL_Surface*>(src), &srcrect, t, &dstrect)
	    != 0)
		throw "unable to blit video surface";
}

// draw the deep image onto its target surface
void sdl::deep_image::draw(const misc::vect3d& loc,
                           const graphics::mapper& m) const
{
	double z = loc.z() - m.get_depth();
	// make sure depth is within visible range
	if (z < 0 || z >= tunnel_dpth)
		return;

	// src is the source of the blit and s is the target
	const SDL_Surface* src = geti(z).access();
	SDL_Surface* s = ts->access();

	SDL_Rect srcrect;
	SDL_Rect dstrect;
	srcrect.x = srcrect.y = 0;
	srcrect.w = src->w;
	srcrect.h = src->h;
	// use mapper m to translate loc into the location on the target
	dstrect.x = m.mapx(loc.x(), loc.z()) - src->w / 2;
	dstrect.y = m.mapy(loc.y(), loc.z()) - src->h / 2;
	dstrect.w = srcrect.w;
	dstrect.h = srcrect.h;

	if (SDL_BlitSurface(const_cast<SDL_Surface*>(src), &srcrect, s, &dstrect)
	    != 0)
		throw "unable to blit video surface";
}

sdl::dial::dial(tsurface* t, const graphics::raw_image* src, double amin,
                double amax, int w, int h)
	: ts(t), bypp(t->access()->format->BytesPerPixel),
	  pitch(t->access()->pitch),
	  Amin(amin), range(amax - amin), W(w), H(h)
{
	image* i = static_cast<image*>(ts->make_image(src));
	const Uint8* p = static_cast<Uint8*>(i->access()->pixels);
	int bypp = i->access()->format->BytesPerPixel;

	Uint32 col;
	Uint8 r, g, b, a;

	getpixm(p, col, bypp);
	SDL_GetRGBA(col, i->access()->format, &r, &g, &b, &a);
	aminc = SDL_MapRGBA(t->access()->format, r, g, b, a);

	getpixm(p, col, bypp);
	SDL_GetRGBA(col, i->access()->format, &r, &g, &b, &a);
	currc = SDL_MapRGBA(t->access()->format, r, g, b, a);

	getpixm(p, col, bypp);
	SDL_GetRGBA(col, i->access()->format, &r, &g, &b, &a);
	maxc = SDL_MapRGBA(t->access()->format, r, g, b, a);

	getpixm(p, col, bypp);
	SDL_GetRGBA(col, i->access()->format, &r, &g, &b, &a);
	amaxc = SDL_MapRGBA(t->access()->format, r, g, b, a);

	delete i;
}

void sdl::dial::draw(int x, int y, double actual, double min, double max) const
{
	// check ranges
	if (min < Amin)
		min = Amin;
	if (actual < min)
		actual = min;
	if (max > range + Amin)
		max = range + Amin;
	if (actual > max)
		actual = max;

	Uint8* line = new Uint8[W * bypp];
	Uint8* ptr = line;

	int imin = int((min - Amin) * W / range);
	int iactual = int((actual - Amin) * W / range);
	int imax = int((max - Amin) * W / range);

	if (imin)
		setpixelsm(ptr, aminc, bypp, imin);
	if (iactual - imin)
		setpixelsm(ptr, currc, bypp, iactual - imin);
	if (imax - iactual)
		setpixelsm(ptr, maxc, bypp, imax - iactual);
	if (W - imax)
		setpixelsm(ptr, amaxc, bypp, W - imax);

	SDL_Surface* s = ts->access();
	if (SDL_LockSurface(s) != 0)
		throw "error locking video surface";

	Uint8* dst = static_cast<Uint8*>(s->pixels) + s->pitch * y + bypp * x;
	for (int i = 0; i < H; ++i) {
		Uint8* linep = line;
		Uint8* dstp = dst;
		for (int j = 0; j < W*bypp/4; ++j) {
			*(dstp++) = *(linep++);
			*(dstp++) = *(linep++);
			*(dstp++) = *(linep++);
			*(dstp++) = *(linep++);
		}
		switch (W*bypp % 4) {
		case 3:
			*(dstp++) = *(linep++);
		case 2:
			*(dstp++) = *(linep++);
		case 1:
			*(dstp++) = *(linep++);
		case 0:
			;
		}
		dst += pitch;
	}

	SDL_UnlockSurface(s);

	delete[] line;
}

sdl::tsurface::tsurface(SDL_Surface* sur)
	: s(sur)
{
	format.palette = 0;
	format.BitsPerPixel = 32;
	format.BytesPerPixel = 8;
	format.Amask = 0xff000000;
	format.Ashift = 24;
	format.Gmask = 0x0000ff00;
	format.Gshift = 8;
	format.Aloss = format.Rloss = format.Gloss = format.Bloss =0;
	if (s->format->Rmask > s->format->Bmask) {
		format.Rmask = 0x00ff0000;
		format.Rshift = 16;
		format.Bmask = 0x000000ff;
		format.Bshift = 0;
	}
	else {
		format.Bmask = 0x00ff0000;
		format.Bshift = 16;
		format.Rmask = 0x000000ff;
		format.Rshift = 0;
	}
}

void sdl::tsurface::flip()
{
	if (SDL_Flip(s) != 0)
		throw "error flipping video surfaces";
}

void sdl::tsurface::fullscreen(bool what)
{
	bool cur = s->flags & SDL_FULLSCREEN;
	if (cur == what)
		return;
	SDL_WM_ToggleFullScreen(s);
}

void sdl::tsurface::grab_input(bool what)
{
	SDL_WM_GrabInput(what ? SDL_GRAB_ON : SDL_GRAB_OFF);
}

void sdl::tsurface::show_cursor(bool what)
{
	SDL_ShowCursor(what ? SDL_ENABLE : SDL_DISABLE);
}

void sdl::tsurface::set_gamma(double r, double g, double b)
{
	SDL_SetGamma(r, g, b);
}

// clear the surface with the given color
void sdl::tsurface::clear(int r, int g, int b)
{
	Uint32 col = SDL_MapRGB(s->format, r, g, b);
	SDL_Rect rect;
	rect.x = rect.y = 0;
	rect.w = w();
	rect.h = h();
	if (SDL_FillRect(s, &rect, col) != 0)
		throw "error clearing video surface";
}

void sdl::tsurface::clear_rect(int x, int y, int w, int h, int r, int g, int b)
{
	Uint32 col = SDL_MapRGB(s->format, r, g, b);
	SDL_Rect rect;
	rect.x = x;
	rect.y = y;
	rect.w = w;
	rect.h = h;
	if (SDL_FillRect(s, &rect, col) != 0)
		throw "error clearing video surface";
}

void sdl::tsurface::clip_rect(int x, int y, int w, int h)
{
	SDL_Rect rect;
	rect.x = x;
	rect.y = y;
	rect.w = w;
	rect.h = h;
	SDL_SetClipRect(s, &rect);
}

void sdl::tsurface::unclip()
{
	SDL_SetClipRect(s, 0);
}

// Make an image from a file.
graphics::raw_image* sdl::tsurface::make_raw_image(const std::string& filename)
{
	return new surface(filename);
}

// Make an image that renders onto this surface from a file.
graphics::image* sdl::tsurface::make_image(const graphics::raw_image* src)
{
	// create the image
	return new image(static_cast<const surface*>(src), &format,
			 src->w(), src->h(), 1.0, this);
}

graphics::image* sdl::tsurface::make_image(const graphics::raw_image* src,
                                           int w, int h, double bright)
{
	// create the image
	return new image(static_cast<const surface*>(src), &format,
			 w, h, bright, this);
}

// Make a deep image that renders onto this surface from a file. The
// number of depths is given, as well as a projector and the width and
// height of the object. These (w and h) are projected using p.
graphics::deep_image* sdl::tsurface::make_deep_image
(const graphics::raw_image* src, int n, const graphics::projector& p,
 double w, double h)
{
	// First create three vectors containg the widths, heights and
	// brightness factors for all n images.
	std::vector<int> ww(n);
	std::vector<int> hh(n);
	std::vector<double> bb(n);
	for (int i = 0; i < n; ++i) {
		double depth = tunnel_dpth * i / n;
		ww[i] = p.projx(w, depth);
		hh[i] = p.projy(h, depth);
		bb[i] = p.projcol(depth);
	}

	// create the deep_image
	return new deep_image(static_cast<const surface*>(src),
			      &format, n, ww.begin(),
			      hh.begin(), bb.begin(), this);
}

graphics::tunnel_drawable* sdl::tsurface::make_tunnel
(const graphics::raw_image* src, const graphics::mapper* m)
{
	switch (s->format->BytesPerPixel) {
        case 1:
		return new tunnel_drawable1(this, src, m, 1024);
        case 2:
		return new tunnel_drawable2(this, src, m, 1024);
        case 3:
		return new tunnel_drawable3(this, src, m, 1024);
        case 4:
		return new tunnel_drawable4(this, src, m, 1024);
        default:
		return 0; // should not get here
	};
}

graphics::dial* sdl::tsurface::make_dial(const graphics::raw_image* src,
                                         double amin, double amax, int w, int h)
{
	return new dial(this, src, amin, amax, w, h);
}

sdl::tunnel_drawable::tunnel_drawable(tsurface* t,
                                      const graphics::raw_image* src,
                                      const graphics::mapper* m, int r)
	: ts(t), mp(m), rings(r)
{
	rx = m->projx(tunnel_r, 0);
	ry = m->projy(tunnel_r, 0);

	// worst case for one line - all in one colour
	int pitch = rings + 2 + rx / 250;
	timage = new unsigned char[pitch * ry];

	brights = new double[rings];

	std::vector<graphics::line_ellipse> ellipses(rings + 1);

	for (int i = 0; i < rings + 1; ++i) {
		double depth = tunnel_dpth * i / rings;
		int X = m->projx(tunnel_r, depth);
		int Y = m->projy(tunnel_r, depth);

		ellipses[i].reset(X, Y);
	}

	unsigned char* im = timage;
	int inside1, inside2;

	for (int i = ry; i > 0; --i) {
		// first ellipse can always be nexted
		ellipses[0].next();

		// background
		write(im, rx - ellipses[0].x());
		inside1 = ellipses[0].x();

		// all rings
		for (int j = 1; j <= rings; ++j) {
			if (i <= ellipses[j].y()) {
				inside2 = ellipses[j].x();
				ellipses[j].next();
				if (ellipses[j].x() == inside2)
					--inside2;
			}
			else {
				inside2 = 0;
			}
			write(im, inside1 - inside2);
			inside1 = inside2;
		}

		// center
		write(im, inside1);
	}

	// assign cols and brights
	first_col = 0;
	const sdl::surface* sr = static_cast<const sdl::surface*>(src);
	const Uint8* p = reinterpret_cast<const Uint8*>(sr->access()->pixels);
	int bypp = sr->access()->format->BytesPerPixel;
	Uint32 col;
	ncols = sr->w();
	cols = new rgb[ncols];
	for (int i = 0; i < ncols; ++i) {
		Uint8 a;
		rgb& cur = cols[i];
		getpixm(p, col, bypp);
		SDL_GetRGBA(col, sr->access()->format, &cur.r, &cur.g, &cur.b, &a);
	}
	center_col = back_col = SDL_MapRGB(ts->access()->format, 0, 0, 0);

	for (int i = 0; i < rings; ++i) {
		brights[i] = m->projcol(tunnel_dpth * i / rings);
	}

	required_depth = tunnel_dpth / rings;
	current_depth = absolute_depth = 0;
}

sdl::tunnel_drawable::~tunnel_drawable()
{
	delete[] brights;
	delete[] cols;
	delete[] timage;
}

void sdl::tunnel_drawable1::draw(const misc::vect3d& loc) const
{
	SDL_Surface* s = ts->access();

	// prepare colors
	std::vector<Uint32> colors(rings);
	int col_index = first_col;
	for (int i = 0; i < rings; ++i) {
		rgb col = cols[col_index];
		if (++col_index == int(ncols))
			col_index = 0;
		double bright = brights[i];
		col.r = Uint8(col.r * bright);
		col.g = Uint8(col.g * bright);
		col.b = Uint8(col.b * bright);
		colors[i] = SDL_MapRGB(s->format, col.r, col.g, col.b);
	}

	int cx = mp->mapx(loc.x(), absolute_depth);
	int cy = mp->mapy(loc.y(), absolute_depth);

	const int pitch = s->pitch;
	const int bypp = 1;

	if (SDL_LockSurface(s) != 0)
		throw "error locking video surface";

	// get top left corner
	Uint8* pyup = static_cast<Uint8*>(s->pixels)
		+ (cy - ry) * pitch + (cx - rx) * bypp;
	// get bottom left corner
	Uint8* pydn = static_cast<Uint8*>(s->pixels)
		+ (cy + ry - 1) * pitch + (cx - rx) * bypp;

	// start at beginning of image
	const unsigned char* im = timage;

	for (int i = 0; i < ry; ++i) {
		Uint8* pup = pyup;
		Uint8* pdn = pydn;

		// how much pixels do I draw?
		int n;

		// draw background
		if ((n = read(im))) {
			setpixelsm1(pup, back_col, n);
			setpixelsm1(pdn, back_col, n);
		}

		// draw rings
		if ((n = read(im))) {
			setpixelsm1(pup, colors[0], n);
			setpixelsm1(pdn, colors[0], n);
		}
		for (int j = 1; j < rings; ++j) {
			if ((n = read(im))) {
				setpixelsm1(pup, colors[j], n);
				setpixelsm1(pdn, colors[j], n);
			}
		}

		// draw center, and record position in im for right tunnel
		const unsigned char* imr = im;
		if ((n = read(im))) {
			n *= 2;
			setpixelsm1(pup, center_col, n);
			setpixelsm1(pdn, center_col, n);
		}

		// draw right part

		// draw rings
		for (int j = rings - 1; j > 0; --j) {
			if ((n = readr(imr))) {
				setpixelsm1(pup, colors[j], n);
				setpixelsm1(pdn, colors[j], n);
			}
		}
		if ((n = readr(imr))) {
			setpixelsm1(pup, colors[0], n);
			setpixelsm1(pdn, colors[0], n);
		}

		// draw background
		if ((n = readr(imr))) {
			setpixelsm1(pup, back_col, n);
			setpixelsm1(pdn, back_col, n);
		}

		// next line
		pyup += pitch;
		pydn -= pitch;
	}

	SDL_UnlockSurface(ts->access());
}

void sdl::tunnel_drawable2::draw(const misc::vect3d& loc) const
{
	SDL_Surface* s = ts->access();

	// prepare colors
	std::vector<Uint32> colors(rings);
	int col_index = first_col;
	for (int i = 0; i < rings; ++i) {
		rgb col = cols[col_index];
		if (++col_index == int(ncols))
			col_index = 0;
		double bright = brights[i];
		col.r = Uint8(col.r * bright);
		col.g = Uint8(col.g * bright);
		col.b = Uint8(col.b * bright);
		colors[i] = SDL_MapRGB(s->format, col.r, col.g, col.b);
	}

	int cx = mp->mapx(loc.x(), absolute_depth);
	int cy = mp->mapy(loc.y(), absolute_depth);

	const int pitch = s->pitch;
	const int bypp = 2;

	if (SDL_LockSurface(s) != 0)
		throw "error locking video surface";

	// get top left corner
	Uint8* pyup = static_cast<Uint8*>(s->pixels)
		+ (cy - ry) * pitch + (cx - rx) * bypp;
	// get bottom left corner
	Uint8* pydn = static_cast<Uint8*>(s->pixels)
		+ (cy + ry - 1) * pitch + (cx - rx) * bypp;

	// start at beginning of image
	const unsigned char* im = timage;

	for (int i = 0; i < ry; ++i) {
		Uint8* pup = pyup;
		Uint8* pdn = pydn;

		// how much pixels do I draw?
		int n;

		// draw background
		if ((n = read(im))) {
			setpixelsm2(pup, back_col, n);
			setpixelsm2(pdn, back_col, n);
		}

		// draw rings
		if ((n = read(im))) {
			setpixelsm2(pup, colors[0], n);
			setpixelsm2(pdn, colors[0], n);
		}
		for (int j = 1; j < rings; ++j) {
			if ((n = read(im))) {
				setpixelsm2(pup, colors[j], n);
				setpixelsm2(pdn, colors[j], n);
			}
		}

		// draw center, and record position in im for right tunnel
		const unsigned char* imr = im;
		if ((n = read(im))) {
			n *= 2;
			setpixelsm2(pup, center_col, n);
			setpixelsm2(pdn, center_col, n);
		}

		// draw right part

		// draw rings
		for (int j = rings - 1; j > 0; --j) {
			if ((n = readr(imr))) {
				setpixelsm2(pup, colors[j], n);
				setpixelsm2(pdn, colors[j], n);
			}
		}
		if ((n = readr(imr))) {
			setpixelsm2(pup, colors[0], n);
			setpixelsm2(pdn, colors[0], n);
		}

		// draw background
		if ((n = readr(imr))) {
			setpixelsm2(pup, back_col, n);
			setpixelsm2(pdn, back_col, n);
		}

		// next line
		pyup += pitch;
		pydn -= pitch;
	}

	SDL_UnlockSurface(ts->access());
}

void sdl::tunnel_drawable3::draw(const misc::vect3d& loc) const
{
	SDL_Surface* s = ts->access();

	// prepare colors
	std::vector<Uint32> colors(rings);
	int col_index = first_col;
	for (int i = 0; i < rings; ++i) {
		rgb col = cols[col_index];
		if (++col_index == int(ncols))
			col_index = 0;
		double bright = brights[i];
		col.r = Uint8(col.r * bright);
		col.g = Uint8(col.g * bright);
		col.b = Uint8(col.b * bright);
		colors[i] = SDL_MapRGB(s->format, col.r, col.g, col.b);
	}

	int cx = mp->mapx(loc.x(), absolute_depth);
	int cy = mp->mapy(loc.y(), absolute_depth);

	const int pitch = s->pitch;
	const int bypp = 3;

	if (SDL_LockSurface(s) != 0)
		throw "error locking video surface";

	// get top left corner
	Uint8* pyup = static_cast<Uint8*>(s->pixels)
		+ (cy - ry) * pitch + (cx - rx) * bypp;
	// get bottom left corner
	Uint8* pydn = static_cast<Uint8*>(s->pixels)
		+ (cy + ry - 1) * pitch + (cx - rx) * bypp;

	// start at beginning of image
	const unsigned char* im = timage;

	for (int i = 0; i < ry; ++i) {
		Uint8* pup = pyup;
		Uint8* pdn = pydn;

		// how much pixels do I draw?
		int n;

		// draw background
		if ((n = read(im))) {
			setpixelsm3(pup, back_col, n);
			setpixelsm3(pdn, back_col, n);
		}

		// draw rings
		if ((n = read(im))) {
			setpixelsm3(pup, colors[0], n);
			setpixelsm3(pdn, colors[0], n);
		}
		for (int j = 1; j < rings; ++j) {
			if ((n = read(im))) {
				setpixelsm3(pup, colors[j], n);
				setpixelsm3(pdn, colors[j], n);
			}
		}

		// draw center, and record position in im for right tunnel
		const unsigned char* imr = im;
		if ((n = read(im))) {
			n *= 2;
			setpixelsm3(pup, center_col, n);
			setpixelsm3(pdn, center_col, n);
		}

		// draw right part

		// draw rings
		for (int j = rings - 1; j > 0; --j) {
			if ((n = readr(imr))) {
				setpixelsm3(pup, colors[j], n);
				setpixelsm3(pdn, colors[j], n);
			}
		}
		if ((n = readr(imr))) {
			setpixelsm3(pup, colors[0], n);
			setpixelsm3(pdn, colors[0], n);
		}

		// draw background
		if ((n = readr(imr))) {
			setpixelsm3(pup, back_col, n);
			setpixelsm3(pdn, back_col, n);
		}

		// next line
		pyup += pitch;
		pydn -= pitch;
	}

	SDL_UnlockSurface(ts->access());
}

void sdl::tunnel_drawable4::draw(const misc::vect3d& loc) const
{
	SDL_Surface* s = ts->access();

	// prepare colors
	std::vector<Uint32> colors(rings);
	int col_index = first_col;
	for (int i = 0; i < rings; ++i) {
		rgb col = cols[col_index];
		if (++col_index == int(ncols))
			col_index = 0;
		double bright = brights[i];
		col.r = Uint8(col.r * bright);
		col.g = Uint8(col.g * bright);
		col.b = Uint8(col.b * bright);
		colors[i] = SDL_MapRGB(s->format, col.r, col.g, col.b);
	}

	int cx = mp->mapx(loc.x(), absolute_depth);
	int cy = mp->mapy(loc.y(), absolute_depth);

	const int pitch = s->pitch;
	const int bypp = 4;

	if (SDL_LockSurface(s) != 0)
		throw "error locking video surface";

	// get top left corner
	Uint8* pyup = static_cast<Uint8*>(s->pixels)
		+ (cy - ry) * pitch + (cx - rx) * bypp;
	// get bottom left corner
	Uint8* pydn = static_cast<Uint8*>(s->pixels)
		+ (cy + ry - 1) * pitch + (cx - rx) * bypp;

	// start at beginning of image
	const unsigned char* im = timage;

	for (int i = 0; i < ry; ++i) {
		Uint8* pup = pyup;
		Uint8* pdn = pydn;

		// how much pixels do I draw?
		int n;

		// draw background
		if ((n = read(im))) {
			setpixelsm4(pup, back_col, n);
			setpixelsm4(pdn, back_col, n);
		}

		// draw rings
		if ((n = read(im))) {
			setpixelsm4(pup, colors[0], n);
			setpixelsm4(pdn, colors[0], n);
		}
		for (int j = 1; j < rings; ++j) {
			if ((n = read(im))) {
				setpixelsm4(pup, colors[j], n);
				setpixelsm4(pdn, colors[j], n);
			}
		}

		// draw center, and record position in im for right tunnel
		const unsigned char* imr = im;
		if ((n = read(im))) {
			n *= 2;
			setpixelsm4(pup, center_col, n);
			setpixelsm4(pdn, center_col, n);
		}

		// draw right part

		// draw rings
		for (int j = rings - 1; j > 0; --j) {
			if ((n = readr(imr))) {
				setpixelsm4(pup, colors[j], n);
				setpixelsm4(pdn, colors[j], n);
			}
		}
		if ((n = readr(imr))) {
			setpixelsm4(pup, colors[0], n);
			setpixelsm4(pdn, colors[0], n);
		}

		// draw background
		if ((n = readr(imr))) {
			setpixelsm4(pup, back_col, n);
			setpixelsm4(pdn, back_col, n);
		}

		// next line
		pyup += pitch;
		pydn -= pitch;
	}

	SDL_UnlockSurface(ts->access());
}

void sdl::tunnel_drawable::move(double d)
{
	current_depth += d - absolute_depth;
	absolute_depth = d;
	while (current_depth >= required_depth) {
		current_depth -= required_depth;
		next_col();
	}
}

void sdl::tunnel_drawable::next_col()
{
	if (++first_col == int(ncols))
		first_col = 0;
}

// Local Variables:
// mode: c++
// End:
