// sdl_controller.cc - controller functions using SDL
//
// Copyright (C) 2000 - 2002, 2004 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <map>
#include <string>
#include "SDL.h"

#include "misc.h"
#include "sdl_controller.h"

namespace {
	// only passes on key events.
	int filter_events(const SDL_Event* event)
	{
		switch (event->type) {
		case SDL_KEYDOWN:
		case SDL_KEYUP:
		case SDL_MOUSEBUTTONDOWN:
		case SDL_MOUSEBUTTONUP:
		case SDL_MOUSEMOTION:
			return 1;
		default:
			;
		}
		return 0;
	}

	void reg_key(std::map<std::string, int>& m, const char* c, int i)
	{
		m[std::string(c)] = i;
	}

	int name_to_num(const std::string& s)
	{
		static int inited = false;
		static std::map<std::string, int> k;

		if (!inited) {
			inited = true;
			reg_key(k, "backspace", SDLK_BACKSPACE);
			reg_key(k, "tab", SDLK_TAB);
			reg_key(k, "clear", SDLK_CLEAR);
			reg_key(k, "return", SDLK_RETURN);
			reg_key(k, "pause", SDLK_PAUSE);
			reg_key(k, "escape", SDLK_ESCAPE);
			reg_key(k, "space", SDLK_SPACE);
			reg_key(k, "!", SDLK_EXCLAIM);
			reg_key(k, "\"", SDLK_QUOTEDBL);
			reg_key(k, "#", SDLK_HASH);
			reg_key(k, "$", SDLK_DOLLAR);
			reg_key(k, "&", SDLK_AMPERSAND);
			reg_key(k, "\'", SDLK_QUOTE);
			reg_key(k, "(", SDLK_LEFTPAREN);
			reg_key(k, ")", SDLK_RIGHTPAREN);
			reg_key(k, "*", SDLK_ASTERISK);
			reg_key(k, "+", SDLK_PLUS);
			reg_key(k, ",", SDLK_COMMA);
			reg_key(k, "-", SDLK_MINUS);
			reg_key(k, ".", SDLK_PERIOD);
			reg_key(k, "/", SDLK_SLASH);
			reg_key(k, "0", SDLK_0);
			reg_key(k, "1", SDLK_1);
			reg_key(k, "2", SDLK_2);
			reg_key(k, "3", SDLK_3);
			reg_key(k, "4", SDLK_4);
			reg_key(k, "5", SDLK_5);
			reg_key(k, "6", SDLK_6);
			reg_key(k, "7", SDLK_7);
			reg_key(k, "8", SDLK_8);
			reg_key(k, "9", SDLK_9);
			reg_key(k, ":", SDLK_COLON);
			reg_key(k, ";", SDLK_SEMICOLON);
			reg_key(k, "<", SDLK_LESS);
			reg_key(k, "=", SDLK_EQUALS);
			reg_key(k, ">", SDLK_GREATER);
			reg_key(k, "?", SDLK_QUESTION);
			reg_key(k, "@", SDLK_AT);
			reg_key(k, "[", SDLK_LEFTBRACKET);
			reg_key(k, "\\", SDLK_BACKSLASH);
			reg_key(k, "]", SDLK_RIGHTBRACKET);
			reg_key(k, "^", SDLK_CARET);
			reg_key(k, "_", SDLK_UNDERSCORE);
			reg_key(k, "`", SDLK_BACKQUOTE);
			reg_key(k, "a", SDLK_a);
			reg_key(k, "b", SDLK_b);
			reg_key(k, "c", SDLK_c);
			reg_key(k, "d", SDLK_d);
			reg_key(k, "e", SDLK_e);
			reg_key(k, "f", SDLK_f);
			reg_key(k, "g", SDLK_g);
			reg_key(k, "h", SDLK_h);
			reg_key(k, "i", SDLK_i);
			reg_key(k, "j", SDLK_j);
			reg_key(k, "k", SDLK_k);
			reg_key(k, "l", SDLK_l);
			reg_key(k, "m", SDLK_m);
			reg_key(k, "n", SDLK_n);
			reg_key(k, "o", SDLK_o);
			reg_key(k, "p", SDLK_p);
			reg_key(k, "q", SDLK_q);
			reg_key(k, "r", SDLK_r);
			reg_key(k, "s", SDLK_s);
			reg_key(k, "t", SDLK_t);
			reg_key(k, "u", SDLK_u);
			reg_key(k, "v", SDLK_v);
			reg_key(k, "w", SDLK_w);
			reg_key(k, "x", SDLK_x);
			reg_key(k, "y", SDLK_y);
			reg_key(k, "z", SDLK_z);
			reg_key(k, "delete", SDLK_DELETE);
			reg_key(k, "k0", SDLK_KP0);
			reg_key(k, "k1", SDLK_KP1);
			reg_key(k, "k2", SDLK_KP2);
			reg_key(k, "k3", SDLK_KP3);
			reg_key(k, "k4", SDLK_KP4);
			reg_key(k, "k5", SDLK_KP5);
			reg_key(k, "k6", SDLK_KP6);
			reg_key(k, "k7", SDLK_KP7);
			reg_key(k, "k8", SDLK_KP8);
			reg_key(k, "k9", SDLK_KP9);
			reg_key(k, "k.", SDLK_KP_PERIOD);
			reg_key(k, "k/", SDLK_KP_DIVIDE);
			reg_key(k, "k*", SDLK_KP_MULTIPLY);
			reg_key(k, "k-", SDLK_KP_MINUS);
			reg_key(k, "k+", SDLK_KP_PLUS);
			reg_key(k, "kenter", SDLK_KP_ENTER);
			reg_key(k, "k=", SDLK_KP_EQUALS);
			reg_key(k, "up", SDLK_UP);
			reg_key(k, "down", SDLK_DOWN);
			reg_key(k, "right", SDLK_RIGHT);
			reg_key(k, "left", SDLK_LEFT);
			reg_key(k, "insert", SDLK_INSERT);
			reg_key(k, "home", SDLK_HOME);
			reg_key(k, "end", SDLK_END);
			reg_key(k, "pageup", SDLK_PAGEUP);
			reg_key(k, "pagedown", SDLK_PAGEDOWN);
			reg_key(k, "f1", SDLK_F1);
			reg_key(k, "f2", SDLK_F2);
			reg_key(k, "f3", SDLK_F3);
			reg_key(k, "f4", SDLK_F4);
			reg_key(k, "f5", SDLK_F5);
			reg_key(k, "f6", SDLK_F6);
			reg_key(k, "f7", SDLK_F7);
			reg_key(k, "f8", SDLK_F8);
			reg_key(k, "f9", SDLK_F9);
			reg_key(k, "f10", SDLK_F10);
			reg_key(k, "f11", SDLK_F11);
			reg_key(k, "f12", SDLK_F12);
			reg_key(k, "f13", SDLK_F13);
			reg_key(k, "f14", SDLK_F14);
			reg_key(k, "f15", SDLK_F15);
			reg_key(k, "numlock", SDLK_NUMLOCK);
			reg_key(k, "capslock", SDLK_CAPSLOCK);
			reg_key(k, "scrollock", SDLK_SCROLLOCK);
			reg_key(k, "rshift", SDLK_RSHIFT);
			reg_key(k, "lshift", SDLK_LSHIFT);
			reg_key(k, "rctrl", SDLK_RCTRL);
			reg_key(k, "lctrl", SDLK_LCTRL);
			reg_key(k, "ralt", SDLK_RALT);
			reg_key(k, "lalt", SDLK_LALT);
			reg_key(k, "rmeta", SDLK_RMETA);
			reg_key(k, "lmeta", SDLK_LMETA);
			reg_key(k, "lsuper", SDLK_LSUPER);
			reg_key(k, "rsuper", SDLK_RSUPER);
			reg_key(k, "mode", SDLK_MODE);
			reg_key(k, "compose", SDLK_COMPOSE);
			reg_key(k, "help", SDLK_HELP);
			reg_key(k, "print", SDLK_PRINT);
			reg_key(k, "sysreq", SDLK_SYSREQ);
			reg_key(k, "break", SDLK_BREAK);
			reg_key(k, "menu", SDLK_MENU);
			reg_key(k, "power", SDLK_POWER);
			reg_key(k, "euro", SDLK_EURO);
			reg_key(k, "mouseleft", SDLK_LAST + SDL_BUTTON_LEFT);
			reg_key(k, "mousemiddle", SDLK_LAST + SDL_BUTTON_MIDDLE);
			reg_key(k, "mouseright", SDLK_LAST + SDL_BUTTON_RIGHT);
			reg_key(k, "mouseup", SDLK_LAST + SDL_BUTTON_WHEELUP);
			reg_key(k, "mousedown", SDLK_LAST + SDL_BUTTON_WHEELDOWN);
		}

		return k[s];
	}
}

sdl::controller::controller()
{
	SDL_SetEventFilter(filter_events);
	for (int i = 0; i < size; ++i) {
		actions[i] = 0;
		state[i] = 0;
	}
}

sdl::controller::~controller()
{
}

void sdl::controller::wait()
{
	if (SDL_WaitEvent(0) == 0)
		throw "error waiting for event";
}

// update action according to which keys have been pressed/released.
void sdl::controller::poll()
{
	// clear press and release flags
	for (int i = 0; i < size; ++i)
		state[i] &= ~(bpress | brel | bclick);

	SDL_Event event;
	x_pix = y_pix = 0;
	while (SDL_PollEvent(&event)) {
		int keysym = event.key.keysym.sym;
		int button = event.button.button;
		
		switch (event.type) {
		case SDL_KEYDOWN:
			state[keysym] |= bpress;
			state[keysym] |= bhold;
			state[keysym] |= bsclick;
			break;
		case SDL_KEYUP:
			state[keysym] |= brel;
			state[keysym] &= ~bhold;
			if (state[keysym] & bsclick)
				state[keysym] |= bclick;
			break;
 		case SDL_MOUSEBUTTONDOWN:
			if (button <= SDL_BUTTON_WHEELDOWN) {
				state[SDLK_LAST + button] |= bpress;
				state[SDLK_LAST + button] |= bhold;
				state[SDLK_LAST + button] |= bsclick;
			}
 			break;
 		case SDL_MOUSEBUTTONUP:
			if (button <= SDL_BUTTON_WHEELDOWN) {
				state[SDLK_LAST + button] |= brel;
				state[SDLK_LAST + button] &= ~bhold;
				if (state[SDLK_LAST + button] & bsclick)
					state[SDLK_LAST + button] |= bclick;
			}
 			break;
		case SDL_MOUSEMOTION:
			x_pix += event.motion.xrel;
			y_pix += event.motion.yrel;
			break;
		default:
			;
		}
	}

	// update action in base class
	action[hold] = action[press] = action[rel] = action[click] = 0;
	for (int i = 0; i < size; ++i) {
		if (state[i] && actions[i]) {
			if (state[i] & bhold) {
				for (unsigned j = bup; j <= bpause; j <<= 1) {
					if (actions[i] & j)
						action[hold] |= j;
				}
			}
			if (state[i] & bpress) {
				for (unsigned j = bup; j <= bpause; j <<= 1) {
					if (actions[i] & j)
						action[press] |= j;
				}
			}
			if (state[i] & brel) {
				for (unsigned j = bup; j <= bpause; j <<= 1) {
					if (actions[i] & j)
						action[rel] |= j;
				}
			}
			if (state[i] & bclick) {
				for (unsigned j = bup; j <= bpause; j <<= 1) {
					if (actions[i] & j)
						action[click] |= j;
				}
			}
		}
	}
}

void sdl::controller::clear()
{
	for (int i = 0; i < size; ++i)
		actions[i] = 0;
}

void sdl::controller::add_up(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bup;
}

void sdl::controller::add_down(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bdown;
}

void sdl::controller::add_left(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bleft;
}

void sdl::controller::add_right(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bright;
}

void sdl::controller::add_weapon(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bweapon;
}

void sdl::controller::add_decel(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bdecel;
}

void sdl::controller::add_fire(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bfire;
}

void sdl::controller::add_accel(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= baccel;
}

void sdl::controller::add_start(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bstart;
}

void sdl::controller::add_quit(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bquit;
}

void sdl::controller::add_pause(const std::string& s)
{
	misc::string_tokener st(s, ' ');
	for (std::string name = st.next(); name != ""; name = st.next())
		actions[name_to_num(name)] |= bpause;
}

void sdl::controller::reset_click()
{
	for (int i = 0; i < size; ++i)
		state[i] &= ~bsclick;
}

// Local Variables:
// mode: c++
// End:
