// record.h - storing of data
//
// Copyright (C) 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef RECORD_H
#define RECORD_H

#include <iostream>
#include <map>
#include <string>
#include <cstring>

namespace rec {

	class item {
	public:
		item();
		item(bool b);
		item(int i);
		item(double d);
		item(const std::string& s);

		operator bool() const;
		operator int() const;
		operator double() const;
		operator std::string() const;

		item& operator=(bool b);
		item& operator=(int i);
		item& operator=(double d);
		item& operator=(const std::string& s);
		item& operator=(const char* c);
	private:
		std::string contents;
	};

	class record {
		typedef std::map<const std::string, item> imap;
		typedef std::map<const std::string, record> rmap;

		imap items;
		rmap records;

	public:
		typedef imap::iterator iiterator;
		typedef imap::const_iterator iconst_iterator;
		typedef rmap::iterator riterator;
		typedef rmap::const_iterator rconst_iterator;

		bool has(const std::string& key) const;
		void add(const std::string& key);
		iiterator find(const std::string& key);
		iconst_iterator find(const std::string& key) const;
		iiterator begin();
		iconst_iterator begin() const;
		iiterator end();
		iconst_iterator end() const;
		item& operator[](const std::string& key);
		const item& operator[](const std::string& key) const;

		bool sub_has(const std::string& key) const;
		void sub_add(const std::string& key);
		riterator sub_find(const std::string& key);
		rconst_iterator sub_find(const std::string& key) const;
		riterator sub_begin();
		rconst_iterator sub_begin() const;
		riterator sub_end();
		rconst_iterator sub_end() const;
		record& sub(const std::string& key);
		const record& sub(const std::string& key) const;

		item& sub(const std::string& r, const std::string& i);
		const item& sub(const std::string& r, const std::string& i) const;

		// now for const char* versions to improve compilation time
		bool has(const char* key) const;
		void add(const char* key);
		iiterator find(const char* key);
		iconst_iterator find(const char* key) const;
		item& operator[](const char* key);
		const item& operator[](const char* key) const;

		bool sub_has(const char* key) const;
		void sub_add(const char* key);
		riterator sub_find(const char* key);
		rconst_iterator sub_find(const char* key) const;
		record& sub(const char* key);
		const record& sub(const char* key) const;

		item& sub(const char* r, const char* i);
		const item& sub(const char* r, const char* i) const;

		std::istream& read_add(std::istream& is);
		std::istream& read(std::istream& is);
		std::ostream& write(std::ostream& os, int ind = 0, int ind_step = 4) const;

		static const record empty;
	};

	std::istream& operator>>(std::istream& is, record& r);
	std::ostream& operator<<(std::ostream& os, const record& r);

	template <typename T>
	class map
	{
		typedef std::map<const std::string, T> m;
		m the_map;
	public:
		typedef typename m::iterator iterator;
		typedef typename m::const_iterator const_iterator;

		bool has(const std::string& key) const
		{ return the_map.find(key) != the_map.end(); }

		void add(const std::string& key)
		{ if (!has(key)) the_map.insert(std::make_pair(key, T())); }

		iterator find(const std::string& key)
		{ return the_map.find(key); }

		const_iterator find(const std::string& key) const
		{ return the_map.find(key); }

		iterator begin()
		{ return the_map.begin(); }

		const_iterator begin() const
		{ return the_map.begin(); }

		iterator end()
		{ return the_map.end(); }

		const_iterator end() const
		{ return the_map.end(); }

		T& operator[](const std::string& key)
		{ iterator i = find(key);
		if (i == end()) throw std::string("key not found: ") + key;
		return i->second; }

		const T& operator[](const std::string& key) const
		{ const_iterator ci = find(key);
		if (ci == end()) throw std::string("key not found: ") + key;
		return ci->second; }
	};

} // namespace rec

#endif // !RECORD_H

// Local Variables:
// mode: c++
// End:
