// record.cc - storing of data
//
// Copyright (C) 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <cctype>
#include <iostream>
#include <map>
#include <utility>
#undef isspace

#include "misc.h"
#include "record.h"

rec::item::item()
	: contents("")
{
}

rec::item::item(bool b)
	: contents(misc::b2s(b))
{
}

rec::item::item(int i)
	: contents(misc::b2s(i))
{
}

rec::item::item(double d)
	: contents(misc::d2s(d))
{
}

rec::item::item(const std::string& s)
	: contents(s)
{
}

rec::item::operator bool() const
{
	return misc::s2b(contents);
}

rec::item::operator int() const
{
	return misc::s2i(contents);
}

rec::item::operator double() const
{
	return misc::s2d(contents);
}

rec::item::operator std::string() const
{
	return contents;
}

rec::item& rec::item::operator=(bool b)
{
	contents = misc::b2s(b);
	return *this;
}

rec::item& rec::item::operator=(int i)
{
	contents = misc::i2s(i);
	return *this;
}

rec::item& rec::item::operator=(double d)
{
	contents = misc::d2s(d);
	return *this;
}

rec::item& rec::item::operator=(const std::string& s)
{
	contents = s;
	return *this;
}

rec::item& rec::item::operator=(const char* c)
{
	contents = c;
	return *this;
}

bool rec::record::has(const std::string& key) const
{
	return items.find(key) != items.end();
}

void rec::record::add(const std::string& key)
{
	if (has(key))
		return;
	items.insert(std::make_pair(key, item()));
}

rec::record::iiterator rec::record::find(const std::string& key)
{
	return items.find(key);
}

rec::record::iconst_iterator rec::record::find(const std::string& key) const
{
	return items.find(key);
}

rec::record::iiterator rec::record::begin()
{
	return items.begin();
}

rec::record::iconst_iterator rec::record::begin() const
{
	return items.begin();
}

rec::record::iiterator rec::record::end()
{
	return items.end();
}

rec::record::iconst_iterator rec::record::end() const
{
	return items.end();
}

rec::item& rec::record::operator[](const std::string& key)
{
	iiterator i = find(key);
	if (i == end())
		throw std::string("key not found: ") + key + " not found";
	return i->second;
}

const rec::item& rec::record::operator[](const std::string& key) const
{
	iconst_iterator ci = find(key);
	if (ci == end())
		throw std::string("key not found: ") + key + " not found";
	return ci->second;
}

bool rec::record::sub_has(const std::string& key) const
{
	return records.find(key) != records.end();
}

void rec::record::sub_add(const std::string& key)
{
	if (sub_has(key))
		return;
	records.insert(std::make_pair(key, rec::record()));
}

rec::record::riterator rec::record::sub_find(const std::string& key)
{
	return records.find(key);
}

rec::record::rconst_iterator rec::record::sub_find(const std::string& key) const
{
	return records.find(key);
}

rec::record::riterator rec::record::sub_begin()
{
	return records.begin();
}

rec::record::rconst_iterator rec::record::sub_begin() const
{
	return records.begin();
}

rec::record::riterator rec::record::sub_end()
{
	return records.end();
}

rec::record::rconst_iterator rec::record::sub_end() const
{
	return records.end();
}

rec::record& rec::record::sub(const std::string& key)
{
	riterator i = sub_find(key);
	if (i == sub_end())
		throw std::string("record key ") + key + " not found";
	return i->second;
}

const rec::record& rec::record::sub(const std::string& key) const
{
	rconst_iterator ci = sub_find(key);
	if (ci == sub_end())
		throw std::string("record key ") + key + " not found";
	return ci->second;
}

rec::item& rec::record::sub(const std::string& r, const std::string& i)
{
	return sub(r)[i];
}

const rec::item& rec::record::sub(const std::string& r, const std::string& i) const
{
	return sub(r)[i];
}

// now for const char* versions to improve compilation time
bool rec::record::has(const char* key) const
{
	return has(std::string(key));
}

void rec::record::add(const char* key)
{
	add(std::string(key));
}

rec::record::iiterator rec::record::find(const char* key)
{
	return find(std::string(key));
}

rec::record::iconst_iterator rec::record::find(const char* key) const
{
	return find(std::string(key));
}

rec::item& rec::record::operator[](const char* key)
{
	return operator[](std::string(key));
}

const rec::item& rec::record::operator[](const char* key) const
{
	return operator[](std::string(key));
}

bool rec::record::sub_has(const char* key) const
{
	return sub_has(std::string(key));
}

void rec::record::sub_add(const char* key)
{
	sub_add(std::string(key));
}

rec::record::riterator rec::record::sub_find(const char* key)
{
	return sub_find(std::string(key));
}

rec::record::rconst_iterator rec::record::sub_find(const char* key) const
{
	return sub_find(std::string(key));
}

rec::record& rec::record::sub(const char* key)
{
	return sub(std::string(key));
}

const rec::record& rec::record::sub(const char* key) const
{
	return sub(std::string(key));
}

rec::item& rec::record::sub(const char* r, const char* i)
{
	return sub(std::string(r), std::string(i));
}

const rec::item& rec::record::sub(const char* r, const char* i) const
{
	return sub(std::string(r), std::string(i));
}

std::istream& rec::record::read_add(std::istream& is)
{
	char c;

	for (; ; ) {
		if (!is.get(c))
			return is;

		// first skip whitespace
		while (std::isspace(c)) {
			if (!is.get(c))
				return is;
		}

		// check for comment
		if (c == ';' || c == '#') {
			while (c != '\n') {
				if (!is.get(c))
					return is;
			}
			continue;
		}

		// read key
		std::string key;
		while (!std::isspace(c) && c != '=') {
			key += c;
			if (!is.get(c))
				throw std::string("end of input before key read: ") + key;
		}
		if (key == "endrecord")
			return is;
		if (key == "record") {
			key = "";
			//  skip whitespace
			while (std::isspace(c)) {
				if (!is.get(c))
					throw std::string("end of input before record name read");
			}

			// read record name
			while (!std::isspace(c) && c != '=') {
				key += c;
				if (!is.get(c))
					throw "end of input before record name read";
			}

			// read `='
			while (c != '=') {
				if (c == '\n' || !std::isspace(c))
					throw std::string("record name contains whitespace: ") + key;
				if (!is.get(c))
					throw std::string("end of input before equals read for record ") + key;
			}
			sub_add(key);

			// read record
			if (!sub(key).read(is))
				return is;
		}
		else { // !record

			// read `='
			while (c != '=') {
				if (c == '\n' || !std::isspace(c))
					throw std::string("record key ") + key + " contains whitespace";
				if (!is.get(c))
					throw "end of input before equals read";
			}
			if (!is.get(c)) {
				add(key);
				(*this)[key] = "";
				return is;
			}

			// skip whitespace
			while (std::isspace(c) && c != '\n') {
				if (!is.get(c)) {
					add(key);
					(*this)[key] = "";
					return is;
				}
			}

			// read contents
			std::string contents;
			while (c != '\n') {
				if (c == '\\') {
					if (!is.get(c)) {
						add(key);
						(*this)[key] = contents;
						return is;
					}
				}
				contents += c;
				if (!is.get(c)) {
					add(key);
					(*this)[key] = contents;
					return is;
				}
			}
			add(key);
			(*this)[key] = contents;
		}
	}

	return is;
}

std::istream& rec::record::read(std::istream& is)
{
	items.clear();
	records.clear();
	return read_add(is);
}

std::ostream& rec::record::write(std::ostream& os, int ind, int ind_step) const
{
	// first write fields
	for (rec::record::imap::const_iterator ci = items.begin();
	     ci != items.end(); ++ci) {

		for (int i = 0; i < ind / 8; ++i)
			os << "        ";
		for (int i = 0; i < ind % 8; ++i)
			os << ' ';
		os << ci->first << " = ";
		std::string s(ci->second);
		for (std::string::const_iterator sci = s.begin();
		     sci != s.end(); ++sci) {

			if (std::isspace(*sci) || (*sci) == '\\')
				os.put('\\');
			os.put(*sci);
		}
		os << '\n';
	}

	// then write records
	for (rmap::const_iterator ci = records.begin();
	     ci != records.end(); ++ci) {

		os << '\n';
		for (int i = 0; i < ind; ++i)
			os << ' ';

		os << "record " << ci->first << " =\n";
		ci->second.write(os, ind + ind_step, ind_step);

		for (int i = 0; i < ind; ++i)
			os << ' ';
		os << "endrecord\n";
	}

	return os;
}

const rec::record rec::record::empty;

std::istream& rec::operator>>(std::istream& is, record& r)
{
	return r.read(is);
}

std::ostream& rec::operator<<(std::ostream& os, const record& r)
{
	return r.write(os);
}

// Local Variables:
// mode: c++
// End:
