// new_ball.cc - to create new balls
//
// Copyright (C) 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <algorithm>
#include <string>
#include <utility>
#include <vector>

#include "ball.h"
#include "game.h"
#include "misc.h"
#include "new_ball.h"
#include "record.h"
#include "ship.h"

balls::level_creator::level_creator(Game::game* gg, const rec::record& levels)
	: g(gg),
	  sp(static_cast<const ship*>(gg->get_ship())),
	  levs(levels),
	  this_at(0),
	  next_at(0),
	  repeat(false)
{
	if (!levs.sub_has("start"))
		throw "no start level";
	this_level = "start";
	next_at += double(levs.sub(this_level)["depth"]);
	next_level = levs.sub(this_level)["next"];
	if (levs.sub(this_level).has("repeat"))
		repeat = bool(levs.sub(this_level)["repeat"]);

	for (rec::record::rconst_iterator ci = levs.sub(this_level).sub_begin();
	     ci != levs.sub(this_level).sub_end(); ++ci) {

		int min = ci->second["min"];
		int max = ci->second["max"];
		for (int i = misc::rndrng(g->rand(), min, max+1); i > 0; --i) {
			double depth = misc::rndrng(g->rand(), this_at, next_at);
			new_balls.push_back(std::make_pair(depth, ci->first));
		}
	}
	std::sort(new_balls.begin(), new_balls.end());
	next = new_balls.begin();
}

void balls::level_creator::move(double t)
{
	check_level();
}

void balls::level_creator::skip_level()
{
	double depth = sp->loc().z();
	next_at = depth;
	repeat = false;
	check_level();
}

void balls::level_creator::check_level()
{
	misc::random& rand = g->rand();
	double depth = sp->loc().z();
	flush_new_balls(depth);

	while (depth >= next_at) {

		this_at = next_at;

		if (!repeat) {
			this_level = next_level;
			next_level = levs.sub(this_level)["next"];
		}
		next_at += double(levs.sub(this_level)["depth"]);
		if (levs.sub(this_level).has("repeat"))
			repeat = bool(levs.sub(this_level)["repeat"]);
		else
			repeat = false;

		new_balls.clear();

		for (rec::record::rconst_iterator ci =levs.sub(this_level).sub_begin();
		     ci != levs.sub(this_level).sub_end(); ++ci) {

			int min = ci->second["min"];
			int max = ci->second["max"];
			for (int i = misc::rndrng(rand, min, max+1);
			     i > 0; --i) {
				double depth = misc::rndrng(rand, this_at, next_at);
				new_balls.push_back(std::make_pair(depth, ci->first));
			}
		}
		std::sort(new_balls.begin(), new_balls.end());
		next = new_balls.begin();
		flush_new_balls(depth);
	}
	flush_new_balls(depth);
}

void balls::level_creator::flush_new_balls(double depth)
{
	misc::random& rand = g->rand();
	while (next != new_balls.end() && next->first <= depth) {
		const rec::record& r = g->get_bd().sub(next->second);
		double radius = r["radius"];
		double speed_xy_ini_max = r["speed_xy_ini_max"];
		double vel_z_ini_min = r["vel_z_ini_min"];
		double vel_z_ini_max = r["vel_z_ini_max"];

		vect3d ll(misc::rndpolar(rand, tunnel_r - radius),
			  depth + tunnel_dpth);
		vect3d vv(misc::rndpm(rand, speed_xy_ini_max),
			  misc::rndpm(rand, speed_xy_ini_max),
			  misc::rndrng(rand, vel_z_ini_min, vel_z_ini_max));
		g->get_factories()[next->second]->new_ball(ll, vv, g);

		++next;
	}
}

// Local Variables:
// mode: c++
// End:
