// graphics.h - interface for graphics and other miscellany
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef GRAPHICS_H
#define GRAPHICS_H

#include <cstddef>
#include <string>
#include <vector>

#include "misc.h"
#include "record.h"

namespace graphics {

	// Projection and mapping

	// projects distances according to the depth.  This also converts
	// from a double to an int, as coords of objects are held as
	// double and they have to be drawn as int.  Also performs color
	// fading.
	//
	// If an object is at distance dist and has width ws and height hs,
	// it will be mapped to width wt and height ht.
	// So   projx = x * (wt / ws) * (dist / (dist + depth))
	//            = x * multx / (dist + depth), multx = wt * dist / ws
	// and  projy = y * multy / (dist + depth), multy = ht * dist / hs
	//
	// members:
	// projx()
	// projy()
	// projcol()
	class projector {
	public:
		projector(double ws, double hs, double dist, int wt, int ht)
			: multx(wt * dist / ws),
			  multy(ht * dist / hs),
			  multb(0.1 / dist),
			  distance(dist)
		{ }

		int projx(double x, double depth) const
		{ return int(x * multx / (distance + depth)); }

		int projy(double y, double depth) const
		{ return int(y * multy / (distance + depth)); }

		double iprojx(int x, double depth) const
		{ return x * (distance + depth) / multx; }

		double iprojy(int y, double depth) const
		{ return y * (distance + depth) / multy; }

		// pass through points (0, 1), (a, b), (1, 0)
		// So for f = [0, a], r = 1 - f * (1-b)/a
		// and for f = [a, 1], r = (1-f) * b/(1-a)
		double projcol(double depth) const
		{ double d = multb * depth;
		const double a = 0.8, b = 0.6;
		if (d < a) return 1 - d * (1-b)/a;
		if (d < 1) return (1-d) * b/(1-a);
		return 0; }

	private:
		double multx, multy, multb, distance;
	};

	// performs the job of a projector and gives a translation as well,
	// so it maps coordinates into screen coordinates.  Also keeps a
	// zero depth so that it is possible to map relative to a specific
	// depth.
	//
	// members:
	// mapx(), mapy(), set_depth(), get_depth()
	class mapper : public projector {
	public:
		mapper(const projector& p, int cx, int cy)
			: projector(p),
			  centrex(cx), centrey(cy),
			  zero_depth(0)
		{ }

		int mapx(double x, double depth) const
		{ return centrex + projx(x, depth - zero_depth); }

		int mapy(double y, double depth) const
		{ return centrey + projy(y, depth - zero_depth); }

		double imapx(int x, double depth) const
		{ return iprojx(x - centrex, depth - zero_depth); }

		double imapy(int y, double depth) const
		{ return iprojy(y - centrey, depth - zero_depth); }

		void set_depth(double depth)
		{ zero_depth = depth; }
		double get_depth() const
		{ return zero_depth; }

	private:
		int centrex, centrey;
		double zero_depth;
	};

	// an image loaded from a file
	class raw_image {
	public:
		virtual ~raw_image() { }

		virtual int w() const = 0;
		virtual int h() const = 0;
	};

	// an image which is renderable onto a target
	// (the image knows its target)
	class image {
	public:
		virtual ~image() { }

		virtual void draw(int x, int y) const = 0;
	};

	// abstract base class deep_image
	//
	// Stores a number of images which can be drawn given a location
	// and a mapper which gives the x,y coords of the surface on which
	// to draw from the location.
	// The implementations should keep track of the target on which they
	// should be drawn.
	class deep_image {
	public:
		virtual ~deep_image() { }

		virtual void draw(const misc::vect3d& loc, const mapper& m) const = 0;
	};

	// draw a range for a number
	//
	// Current min and max are required each time but the absolute
	// min and max should be stored in the class.
	class dial {
	public:
		virtual ~dial() { }

		virtual void draw(int x, int y, double actual,
				  double min, double max) const = 0;
	};

	// a (x_)image_map is a map containing (x_)images with a name key
	typedef rec::map<raw_image*> raw_image_map;
	typedef rec::map<image*> image_map;
	typedef rec::map<deep_image*> deep_image_map;

	// abstract base class tsurface
	//    Target surface on which to draw images, deep_images etc..
	//
	// members:
	// flip(): update the display
	// w(), h(): get the width/height
	// clear(), clear_rect(): clear the surface or part of it with the
	//     given colour
	// clip_rect(), unclip(): clip to a rectangle or remove clipping
	// make_raw_image(): load a raw image from a file
	// make_image(): from a raw image
	// make_deep_image(): from a raw image, given other details
	// make_tunnel_drawable(): to draw the tunnel
	// make_dial(): for this surface
	class tunnel_drawable;
	class tsurface {
	public:
		virtual ~tsurface() { }

		virtual void flip() = 0;
		virtual void fullscreen(bool what = true) = 0;
		virtual void grab_input(bool what = true) = 0;
		virtual void show_cursor(bool what = true) = 0;
		virtual void set_gamma(double r, double g, double b) = 0;
		virtual int w() const = 0;
		virtual int h() const = 0;

		virtual void clear(int r, int g, int b) = 0;
		virtual void clear_rect(int x, int y, int w, int h, int r, int g, int b) = 0;

		virtual void clip_rect(int x, int y, int w, int h) = 0;
		virtual void unclip() = 0;

		virtual raw_image* make_raw_image(const std::string& filename) = 0;
		virtual image* make_image(const raw_image* src) = 0;
		virtual image* make_image(const raw_image* src, int w, int h, double bright = 1.0) = 0;
		virtual deep_image* make_deep_image(const raw_image* src, int n, const projector& p, double w, double h) = 0;
		virtual tunnel_drawable* make_tunnel(const raw_image* src, const mapper* m) = 0;
		virtual dial* make_dial(const raw_image* src, double amin, double amax, int w, int h) = 0;
	};

	// something which can be drawn given its location
	class drawable {
	public:
		virtual ~drawable() { }
		virtual void draw(const misc::vect3d& loc) const = 0;
	};

	// A drawable made from a deep_image. This remembers the mapper
	// and draws the deep_image when required.
	class deep_drawable : public drawable {
	public:
		deep_drawable(const deep_image* i, const mapper* m)
			: di(i), mp(m) { }

		void draw(const misc::vect3d& loc) const
		{ di->draw(loc, *mp); }
	protected:
		const deep_image* di;
		const mapper* mp;
	private:
		deep_drawable(const deep_drawable&);
		deep_drawable& operator=(const deep_drawable&);
	};

	class tunnel_drawable : public drawable {
	public:
		virtual void move(double d) = 0;
	};

	// class ellipse
	//    gives points which are inside or exactly on the quadrant of the
	//    ellipse of radii Rx, Ry. Starts with point (0, Ry) and ends with
	//    point (Rx, 0).
	// member functions
	//    next(): goes to next point, returns false if there isn't
	//    x():    gives current x
	//    y():    gives current y
	class ellipse {
	public:
		ellipse()               { reset(0, 0); }
		ellipse(int Rx, int Ry) { reset(Rx, Ry); }

		void reset(int Rx, int Ry);
		void reset()            { reset(rx, ry); }

		bool next();

		int x() const { return cx; }
		int y() const { return cy; }
	private:
		int rx, ry, cx, cy;
		double f, dfx, dfy, ddfx, ddfy;
		bool second_stage;
	};

	// class line_ellipse
	//    gives the amount of pixels needed to draw each horizontal line
	//    of an ellipse of radii rx, ry. x() gives the number of pixels
	//    and y() gives which line. Starts with (0, ry) indicating that
	//    a line ry up should not be drawn and ends with (rx, 0) so that
	//    in the bottom line, rx pixels are drawn.
	// member functions
	//    next(): goes to next line, returns false if there isn't
	//    x():    gives current x
	//    y():    gives current y
	class line_ellipse {
	public:
		line_ellipse()               { reset (0, 0); }
		line_ellipse(int rx, int ry) { reset (rx, ry); }

		void reset(int rx, int ry)
		{ e.reset(rx, ry); end = false; oldx = -1; read(); }
		void reset()
		{ e.reset(); end = false; oldx = -1; read(); }

		bool next()      { if (end) return false; read(); return true; }

		int x() const { return X; }
		int y() const { return Y; }

	private:
		ellipse e;
		bool end;
		int X, Y, oldx;

		void read();
	};

} // namespace graphics

#endif // !GRAPHICS_H

// Local Variables:
// mode: c++
// End:
