// enemy.cc - functions required by enemy balls
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include <cmath>

#include "audio.h"
#include "ball.h"
#include "enemy.h"
#include "game.h"
#include "graphics.h"
#include "misc.h"
#include "record.h"

namespace {
	using misc::vectx;
	using misc::vecty;
	using misc::vectz;
	using misc::vect2d;
	using misc::vect3d;

	// defines things in which a normal enemy ball differs from base
	// class enemy and makes it a drawable
	class tickler : public balls::enemy, public graphics::deep_drawable {
	public:
		tickler(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
	};

	// when a tickler breaks up, these come up
	class tickler_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		tickler_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	// punch
	//    similar to tickler
	class punch : public balls::enemy, public graphics::deep_drawable {
	public:
		punch(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
	};

	class punch_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		punch_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	class speedy : public balls::enemy, public graphics::deep_drawable {
	public:
		speedy(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
	};

	class speedy_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		speedy_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	class crazy : public balls::enemy, public graphics::deep_drawable {
	public:
		crazy(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
	private:
		double time_change, time_left;
	};

	class crazy_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		crazy_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	class crawler : public balls::enemy, public graphics::deep_drawable {
	public:
		crawler(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
	private:
		double vxy_mag;
		bool clockwise;
	};

	class crawler_frag : public balls::bullet, public graphics::deep_drawable {
	public:
		crawler_frag(const vect3d& ll, const vect3d& vv, Game::game* gg);
	};

	// gives out tickler's
	class tickler_thrower : public balls::enemy, public graphics::deep_drawable {
	public:
		tickler_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_max, fire_vel_z;
	};

	class punch_thrower : public balls::enemy, public graphics::deep_drawable {
	public:
		punch_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_max, fire_vel_z;
	};

	class speedy_thrower : public balls::enemy, public graphics::deep_drawable {
	public:
		speedy_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_max, fire_vel_z;
	};

	class crazy_thrower : public balls::enemy, public graphics::deep_drawable {
	public:
		crazy_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_max, fire_vel_z;
	};

	class crawler_thrower : public balls::enemy, public graphics::deep_drawable {
	public:
		crawler_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_max, fire_vel_z;
	};

	class multics : public balls::enemy, public graphics::deep_drawable {
	public:
		multics(const vect3d& ll, const vect3d& vv, Game::game* gg);
		void move(double t);
		void die();
	private:
		double maxvel_xy, maxvel_z, accel_z, decel_z;
		double charge, req_charge;
		double fire_speed_xy_tickler, fire_speed_xy_punch;
		double fire_speed_xy_speedy, fire_speed_xy_crazy;
		double fire_speed_xy_crawler, fire_vel_z;
	};

	void thrower_fire(const char* fired,
			  balls::ball* thrower,
			  double vel_z, double speed_xy)
	{
		vect3d bv = (thrower->vel()
			     + misc::mkpolar(speed_xy,
					     thrower->gam()->rand()(2*pi))
			     + misc::vectz(vel_z));
		vect3d dir = misc::dir(bv);
		vect3d bl = thrower->loc() + dir * (thrower->radius() + double(thrower->gam()->get_bd().sub(fired, "radius")));
		thrower->gam()->get_factories()[fired]->new_ball(bl, bv, thrower->gam());
	}

	void thrower_frag(const char* fired,
			  balls::ball* thrower,
			  double speed_max)
	{
		vect3d l(thrower->loc()), v(thrower->vel());
		l += rnd3d(thrower->gam()->rand(), thrower->radius() * 2);
		v += rnd3d(thrower->gam()->rand(), speed_max);
		thrower->gam()->get_factories()[fired]->new_ball(l, v, thrower->gam());
	}

	// factories:
	class tickler_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class punch_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class speedy_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class crazy_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class crawler_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class tickler_thrower_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class punch_thrower_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class speedy_thrower_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class crazy_thrower_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class crawler_thrower_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

	class multics_factory : public balls::factory {
	public:
		void new_ball(const vect3d& l, const vect3d& v, Game::game* g);
	};

} // namespace

// factory functions:
void tickler_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new tickler(l, v, g));
}

void punch_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new punch(l, v, g));
}

void speedy_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new speedy(l, v, g));
}

void crazy_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new crazy(l, v, g));
}

void crawler_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new crawler(l, v, g));
}

void tickler_thrower_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new tickler_thrower(l, v, g));
}

void punch_thrower_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new punch_thrower(l, v, g));
}

void speedy_thrower_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new speedy_thrower(l, v, g));
}

void crazy_thrower_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new crazy_thrower(l, v, g));
}

void crawler_thrower_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new crawler_thrower(l, v, g));
}

void multics_factory::new_ball(const vect3d& l, const vect3d& v, Game::game* g)
{
	g->swallow(new multics(l, v, g));
}

tickler::tickler(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("tickler", "radius"),
		gg->get_bd().sub("tickler", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("tickler", "strength"),
		gg->get_bd().sub("tickler", "shield"),
		gg->get_bd().sub("tickler", "score")),
	  graphics::deep_drawable(gg->get_gallery()["tickler"],
				  &gg->get_mapp())
{
}

void tickler::move(double t)
{
	enemy::move(t);
	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("tickler");
		int frags = r["frags"];
		g->get_sounds()["tickler"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["frag_speed_max"]));
			ball* frag = new tickler_frag(floc, fvel, g);
			g->swallow(frag);
		}
	}
}

tickler_frag::tickler_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("tickler_frag", "radius"),
		 gg->get_bd().sub("tickler_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("tickler_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["tickler_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

punch::punch(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("punch", "radius"),
		gg->get_bd().sub("punch", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("punch", "strength"),
		gg->get_bd().sub("punch", "shield"),
		gg->get_bd().sub("punch", "score")),
	  graphics::deep_drawable(gg->get_gallery()["punch"],
				  &gg->get_mapp())
{
}

void punch::move(double t)
{
	enemy::move(t);
	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("punch");
		int frags = r["frags"];
		g->get_sounds()["punch"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["frag_speed_max"]));
			ball* frag = new punch_frag(floc, fvel, g);
			g->swallow(frag);
		}
	}
}

punch_frag::punch_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("punch_frag", "radius"),
		 gg->get_bd().sub("punch_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("punch_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["punch_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

speedy::speedy(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("speedy", "radius"),
		gg->get_bd().sub("speedy", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("speedy", "strength"),
		gg->get_bd().sub("speedy", "shield"),
		gg->get_bd().sub("speedy", "score")),
	  graphics::deep_drawable(gg->get_gallery()["speedy"],
				  &gg->get_mapp())
{
}

void speedy::move(double t)
{
	enemy::move(t);
	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("speedy");
		int frags = r["frags"];
		g->get_sounds()["speedy"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["frag_speed_max"]));
			ball* frag = new speedy_frag(floc, fvel, g);
			g->swallow(frag);
		}
	}
}

speedy_frag::speedy_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("speedy_frag", "radius"),
		 gg->get_bd().sub("speedy_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("speedy_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["speedy_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

crazy::crazy(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("crazy", "radius"),
		gg->get_bd().sub("crazy", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("crazy", "strength"),
		gg->get_bd().sub("crazy", "shield"),
		gg->get_bd().sub("crazy", "score")),
	  graphics::deep_drawable(gg->get_gallery()["crazy"],
				  &gg->get_mapp()),
	  time_change(gg->get_bd().sub("crazy", "timechange")),
	  time_left(time_change)
{
}

void crazy::move(double t)
{
	enemy::move(t);
	time_left -= t;
	if (time_left <= 0) {
		do {
			time_left += time_change;
		} while (time_left <= 0);
		double r = misc::abs(v);
		double a = (g->rand())(pi);
		v = misc::mkpolar(r, a);
	}
	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("crazy");
		int frags = r["frags"];
		g->get_sounds()["crazy"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["frag_speed_max"]));
			ball* frag = new crazy_frag(floc, fvel, g);
			g->swallow(frag);
		}
	}
}

crazy_frag::crazy_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("crazy_frag", "radius"),
		 gg->get_bd().sub("crazy_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("crazy_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["crazy_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

crawler::crawler(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("crawler", "radius"),
		gg->get_bd().sub("crawler", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("crawler", "strength"),
		gg->get_bd().sub("crawler", "shield"),
		gg->get_bd().sub("crawler", "score")),
	  graphics::deep_drawable(gg->get_gallery()["crawler"],
				  &gg->get_mapp()),
	  vxy_mag(misc::abs(vv.xy()))
{
	double lphase = misc::arg(ll);
	double vphase = misc::arg(vv);

	if ((lphase <= vphase && vphase < lphase + pi)
	    || (lphase - 2*pi <= vphase && vphase < lphase - pi))
		clockwise = true;
	else
		clockwise = false;
}

void crawler::move(double t)
{	// Modify vel because of collisions.
	v.z() += dv.z();
	dv.set();
	if ((misc::abs(l.xy()) + radius() * 2) < tunnel_r) {
		vect2d dir_xy = misc::dir(l.xy());
		v.xy() = vxy_mag * dir_xy;
		l += v * t;
	}
	else {
		double angle = vxy_mag * t / misc::abs(l.xy());
		double ra = pi / 2;
		if (!clockwise) {
			angle = -angle;
			ra = -ra;
		}
		l.xy() = misc::mkpolar(misc::abs(l.xy()),
				       misc::arg(l.xy()) + angle);
		v.xy() = misc::mkpolar(vxy_mag, misc::arg(l.xy()) + ra);
	}

	if (hit_wall(this))
		get_inside_wall(this);

	if (strength <= 0) {
		die();

		const rec::record& r = g->get_bd().sub("crawler");
		int frags = r["frags"];
		g->get_sounds()["crawler"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			vect3d floc(l), fvel(v);
			floc += rnd3d(g->rand(), double(r["radius"]) * 2);
			fvel += rnd3d(g->rand(), double(r["frag_speed_max"]));
			ball* frag = new crawler_frag(floc, fvel, g);
			g->swallow(frag);
		}
	}
}

crawler_frag::crawler_frag(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: bullet(gg->get_bd().sub("crawler_frag", "radius"),
		 gg->get_bd().sub("crawler_frag", "mass"),
		 ll, vv, gg,
		 gg->get_bd().sub("crawler_frag", "damage")),
	  graphics::deep_drawable(gg->get_gallery()["crawler_frag"],
				  &gg->get_mapp())
{
	if (hit_wall(this))
		die();
}

tickler_thrower::tickler_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("tickler_thrower", "radius"),
		gg->get_bd().sub("tickler_thrower", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("tickler_thrower", "strength"),
		gg->get_bd().sub("tickler_thrower", "shield"),
		gg->get_bd().sub("tickler_thrower", "score")),
	  graphics::deep_drawable(gg->get_gallery()["tickler_thrower"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("tickler_thrower", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("tickler_thrower", "maxvel_z")),
	  accel_z(gg->get_bd().sub("tickler_thrower", "accel_z")),
	  decel_z(gg->get_bd().sub("tickler_thrower", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("tickler_thrower", "req_charge")),
	  fire_speed_xy_max(gg->get_bd().sub("tickler_thrower", "fire_speed_xy_max")),
	  fire_vel_z(gg->get_bd().sub("tickler_thrower", "fire_vel_z"))
{
}

void tickler_thrower::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		thrower_fire("tickler", this,
			     fire_vel_z, fire_speed_xy_max);
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("tickler_thrower");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["tickler_thrower"]->play(1.0);
		for (int i = 0; i < frags; ++i)
			thrower_frag("tickler", this, frag_speed_max);
		die();
	}
}

void tickler_thrower::die()
{
	g->skip_level();
	ball::die();
}

punch_thrower::punch_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("punch_thrower", "radius"),
		gg->get_bd().sub("punch_thrower", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("punch_thrower", "strength"),
		gg->get_bd().sub("punch_thrower", "shield"),
		gg->get_bd().sub("punch_thrower", "score")),
	  graphics::deep_drawable(gg->get_gallery()["punch_thrower"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("punch_thrower", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("punch_thrower", "maxvel_z")),
	  accel_z(gg->get_bd().sub("punch_thrower", "accel_z")),
	  decel_z(gg->get_bd().sub("punch_thrower", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("punch_thrower", "req_charge")),
	  fire_speed_xy_max(gg->get_bd().sub("punch_thrower", "fire_speed_xy_max")),
	  fire_vel_z(gg->get_bd().sub("punch_thrower", "fire_vel_z"))
{
}

void punch_thrower::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		thrower_fire("punch", this,
			     fire_vel_z, fire_speed_xy_max);
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("punch_thrower");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["punch_thrower"]->play(1.0);
		for (int i = 0; i < frags; ++i)
			thrower_frag("punch", this, frag_speed_max);
		die();
	}
}

void punch_thrower::die()
{
	g->skip_level();
	ball::die();
}

speedy_thrower::speedy_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("speedy_thrower", "radius"),
		gg->get_bd().sub("speedy_thrower", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("speedy_thrower", "strength"),
		gg->get_bd().sub("speedy_thrower", "shield"),
		gg->get_bd().sub("speedy_thrower", "score")),
	  graphics::deep_drawable(gg->get_gallery()["speedy_thrower"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("speedy_thrower", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("speedy_thrower", "maxvel_z")),
	  accel_z(gg->get_bd().sub("speedy_thrower", "accel_z")),
	  decel_z(gg->get_bd().sub("speedy_thrower", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("speedy_thrower", "req_charge")),
	  fire_speed_xy_max(gg->get_bd().sub("speedy_thrower", "fire_speed_xy_max")),
	  fire_vel_z(gg->get_bd().sub("speedy_thrower", "fire_vel_z"))
{
}

void speedy_thrower::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		thrower_fire("speedy", this,
			     fire_vel_z, fire_speed_xy_max);
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("speedy_thrower");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["speedy_thrower"]->play(1.0);
		for (int i = 0; i < frags; ++i)
			thrower_frag("speedy", this, frag_speed_max);
		die();
	}
}

void speedy_thrower::die()
{
	g->skip_level();
	ball::die();
}

crazy_thrower::crazy_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("crazy_thrower", "radius"),
		gg->get_bd().sub("crazy_thrower", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("crazy_thrower", "strength"),
		gg->get_bd().sub("crazy_thrower", "shield"),
		gg->get_bd().sub("crazy_thrower", "score")),
	  graphics::deep_drawable(gg->get_gallery()["crazy_thrower"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("crazy_thrower", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("crazy_thrower", "maxvel_z")),
	  accel_z(gg->get_bd().sub("crazy_thrower", "accel_z")),
	  decel_z(gg->get_bd().sub("crazy_thrower", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("crazy_thrower", "req_charge")),
	  fire_speed_xy_max(gg->get_bd().sub("crazy_thrower", "fire_speed_xy_max")),
	  fire_vel_z(gg->get_bd().sub("crazy_thrower", "fire_vel_z"))
{
}

void crazy_thrower::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		thrower_fire("crazy", this,
			     fire_vel_z, fire_speed_xy_max);
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("crazy_thrower");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["crazy_thrower"]->play(1.0);
		for (int i = 0; i < frags; ++i)
			thrower_frag("crazy", this, frag_speed_max);
		die();
	}
}

void crazy_thrower::die()
{
	g->skip_level();
	ball::die();
}

crawler_thrower::crawler_thrower(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("crawler_thrower", "radius"),
		gg->get_bd().sub("crawler_thrower", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("crawler_thrower", "strength"),
		gg->get_bd().sub("crawler_thrower", "shield"),
		gg->get_bd().sub("crawler_thrower", "score")),
	  graphics::deep_drawable(gg->get_gallery()["crawler_thrower"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("crawler_thrower", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("crawler_thrower", "maxvel_z")),
	  accel_z(gg->get_bd().sub("crawler_thrower", "accel_z")),
	  decel_z(gg->get_bd().sub("crawler_thrower", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("crawler_thrower", "req_charge")),
	  fire_speed_xy_max(gg->get_bd().sub("crawler_thrower", "fire_speed_xy_max")),
	  fire_vel_z(gg->get_bd().sub("crawler_thrower", "fire_vel_z"))
{
}

void crawler_thrower::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		if (g->rand()(2))
			thrower_fire("crawler", this,
				     fire_vel_z, fire_speed_xy_max);
		else
			thrower_fire("speedy", this,
				     fire_vel_z, fire_speed_xy_max);
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("crawler_thrower");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["crawler_thrower"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			if (g->rand()(2))
				thrower_frag("crawler", this, frag_speed_max);
			else
				thrower_frag("speedy", this, frag_speed_max);
		}
		die();
	}
}

void crawler_thrower::die()
{
	g->skip_level();
	ball::die();
}

multics::multics(const vect3d& ll, const vect3d& vv, Game::game* gg)
	: enemy(gg->get_bd().sub("multics", "radius"),
		gg->get_bd().sub("multics", "mass"),
		ll, vv, gg,
		gg->get_bd().sub("multics", "strength"),
		gg->get_bd().sub("multics", "shield"),
		gg->get_bd().sub("multics", "score")),
	  graphics::deep_drawable(gg->get_gallery()["multics"],
				  &gg->get_mapp()),
	  maxvel_xy(gg->get_bd().sub("multics", "maxvel_xy")),
	  maxvel_z(gg->get_bd().sub("multics", "maxvel_z")),
	  accel_z(gg->get_bd().sub("multics", "accel_z")),
	  decel_z(gg->get_bd().sub("multics", "decel_z")),
	  charge(0),
	  req_charge(gg->get_bd().sub("multics", "req_charge")),
	  fire_speed_xy_tickler(gg->get_bd().sub("multics", "fire_speed_xy_tickler")),
	  fire_speed_xy_punch(gg->get_bd().sub("multics", "fire_speed_xy_punch")),
	  fire_speed_xy_speedy(gg->get_bd().sub("multics", "fire_speed_xy_speedy")),
	  fire_speed_xy_crazy(gg->get_bd().sub("multics", "fire_speed_xy_crazy")),
	  fire_speed_xy_crawler(gg->get_bd().sub("multics", "fire_speed_xy_crawler")),
	  fire_vel_z(gg->get_bd().sub("multics", "fire_vel_z"))
{
}

void multics::move(double t)
{
	const ball* sp = g->get_ship();

	double reqvz = (sp->loc().z() + tunnel_dpth * 0.5 - loc().z()) / t;

	double dvz = reqvz - v.z();
	if (dvz < 0) {
		if (v.z() < 0)
			v.z() -= accel_z * t;
		else
			v.z() -= decel_z * t;
	}
	else if (dvz > 0) {
		if (v.z() > 0)
			v.z() += accel_z * t;
		else
			v.z() += decel_z * t;
	}

	double z_mag = std::abs(v.z());
	if (z_mag > maxvel_z)
		v.z() *= maxvel_z / z_mag;

	l += v * t;

	if (hit_wall(this)) {
		bounce_wall(this);
		get_inside_wall(this);
	}

	charge += t;
	while (charge >= req_charge) {
		switch (g->rand()(5)) {
		case 0:
			thrower_fire("tickler", this,
				     fire_vel_z, fire_speed_xy_tickler);
			break;
		case 1:
			thrower_fire("punch", this,
				     fire_vel_z, fire_speed_xy_punch);
			break;
		case 2:
			thrower_fire("speedy", this,
				     fire_vel_z, fire_speed_xy_speedy);
			break;
		case 3:
			thrower_fire("crazy", this,
				     fire_vel_z, fire_speed_xy_crazy);
			break;
		case 4:
			thrower_fire("crawler", this,
				     fire_vel_z, fire_speed_xy_crawler);
			break;
		default:
			;
		};
		charge -= req_charge;
	}

	if (strength <= 0) {
		const rec::record& r = g->get_bd().sub("multics");
		int frags = r["frags"];
		double frag_speed_max = r["frag_speed_max"];
		g->get_sounds()["multics"]->play(1.0);
		for (int i = 0; i < frags; ++i) {
			switch (g->rand()(5)) {
			case 0:
				thrower_frag("tickler", this,
					     frag_speed_max);
				break;
			case 1:
				thrower_frag("punch", this,
					     frag_speed_max);
				break;
			case 2:
				thrower_frag("speedy", this,
					     frag_speed_max);
				break;
			case 3:
				thrower_frag("crazy", this,
					     frag_speed_max);
				break;
			case 4:
				thrower_frag("crawler", this,
					     frag_speed_max);
				break;
			default:
				;
			};
		}

		die();
	}
}

void multics::die()
{
	g->skip_level();
	ball::die();
}

void balls::enemy_register(factory_map& factories)
{
	factory_register(factories, "tickler",
			 new tickler_factory);
	factory_register(factories, "tickler_thrower",
			 new tickler_thrower_factory);

	factory_register(factories, "punch",
			 new punch_factory);
	factory_register(factories, "punch_thrower",
			 new punch_thrower_factory);

	factory_register(factories, "speedy",
			 new speedy_factory);
	factory_register(factories, "speedy_thrower",
			 new speedy_thrower_factory);

	factory_register(factories, "crazy",
			 new crazy_factory);
	factory_register(factories, "crazy_thrower",
			 new crazy_thrower_factory);

	factory_register(factories, "crawler",
			 new crawler_factory);
	factory_register(factories, "crawler_thrower",
			 new crawler_thrower_factory);

	factory_register(factories, "multics",
			 new multics_factory);
}

// Local Variables:
// mode: c++
// End:
