// cotroller.h - controller interface
//
// Copyright (C) 2000 - 2002 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef CONTROLLER_H
#define CONTROLLER_H

namespace control {

	// keys are the following:
	// backspace tab clear return pause escape space
	// ! " # $ & ' ( ) * + , - . /
	// 0 1 2 3 4 5 6 7 8 9
	// : ; < = > ? @
	// [ \ ] ^ _ `
	// a b c d e f g h i j k l m n o p q r s t u v w x y z
	// delete
	// k0 k1 k2 k3 k4 k5 k6 k7 k8 k9
	// k. k/ k* k- k+ kenter k=
	// up down right left insert home end pageup pagedown
	// f1 f2 f3 f4 f5 f6 f7 f8 f9 f10 f11 f12 f13 f14 f15
	// numlock capslock scrollock rshift lshift rctrl lctrl ralt lalt
	// rmeta lmeta lsuper rsuper mode compose
	// help print sysreq break menu power euro
	// mouse1 mouse2 mouse3
	

	// abstract base class controller
	//
	// defines numbers for active controls:
	//    bup, bdown, bleft, bright,
	//    accel, decel, fire, weapon
	//    start, quit, pause.
	// protected member:
	//    action: contains the status of the controller (which controls
	//            are active)
	// pure virtual funciton:
	//    wait(): waits for an action
	//    poll(): polls the state and updates action
	// other functions:
	//    get():  returns a logic OR of all the active controls
	//    up(), down(), ..., pause(): true if the respective control is
	//                                 active
	class controller {
	public:
		enum action_t { hold, press, rel, click };

		controller() { action[0] = action[1] = action[2] = action[3] = x_pix = y_pix = 0; }
		virtual ~controller() { }
		virtual void wait() = 0;
		virtual void poll() = 0;

		unsigned get(action_t a = hold) const	{ return action[a]; }
		unsigned get_x_pix() const		{ return x_pix; }
		unsigned get_y_pix() const		{ return y_pix; }
		bool up(action_t a = hold) const	{ return action[a] & bup; }
		bool down(action_t a = hold) const	{ return action[a] & bdown; }
		bool left(action_t a = hold) const	{ return action[a] & bleft; }
		bool right(action_t a = hold) const	{ return action[a] & bright; }
		bool accel(action_t a = hold) const	{ return action[a] & baccel; }
		bool decel(action_t a = hold) const	{ return action[a] & bdecel; }
		bool fire(action_t a = hold) const	{ return action[a] & bfire; }
		bool weapon(action_t a = hold) const	{ return action[a] & bweapon; }
		bool start(action_t a = hold) const	{ return action[a] & bstart; }
		bool quit(action_t a = hold) const	{ return action[a] & bquit; }
		bool pause(action_t a = hold) const	{ return action[a] & bpause; }

		virtual void clear() = 0;
		virtual void add_up(const std::string& s) = 0;
		virtual void add_down(const std::string& s) = 0;
		virtual void add_left(const std::string& s) = 0;
		virtual void add_right(const std::string& s) = 0;
		virtual void add_accel(const std::string& s) = 0;
		virtual void add_decel(const std::string& s) = 0;
		virtual void add_fire(const std::string& s) = 0;
		virtual void add_weapon(const std::string& s) = 0;
		virtual void add_start(const std::string& s) = 0;
		virtual void add_quit(const std::string& s) = 0;
		virtual void add_pause(const std::string& s) = 0;

		virtual void reset_click() = 0;
	protected:
		enum {
			bup = 0x0001,
			bdown = 0x0002,
			bleft = 0x0004,
			bright = 0x0008,
			baccel = 0x0010,
			bdecel = 0x0020,
			bfire = 0x0040,
			bweapon = 0x0080,
			bstart = 0x0100,
			bquit = 0x0200,
			bpause = 0x0400,
			bhold  = 0x1000,
			bpress = 0x2000,
			brel   = 0x4000,
			bclick = 0x8000,
			bsclick = 0x010000
			
		};

		unsigned action[4];
		int x_pix, y_pix;
	};

} // namespace control

#endif // !CONTROLLER_H

// Local Variables:
// mode: c++
// End:
