// ball.h - most stuff having to do with balls
//
// Copyright (C) 2000, 2001 Trevor Spiteri
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef BALL_H
#define BALL_H

#include <string>
#include <typeinfo>

#include "audio.h"
#include "graphics.h"
#include "record.h"
#include "misc.h"

// needed for ball
namespace Game { class game; }

namespace balls {

	using misc::vectx;
	using misc::vecty;
	using misc::vectz;
	using misc::vect2d;
	using misc::vect3d;

	class ball;

	// Abstract base class ball
	//    Things common to all balls, such as radius and mass
	// pure virtual functions:pppp
	//    move(...): move for a given time
	//    do_collide(...): take care of collision with another ball.
	//    damage(): the damage caused to other balls bumping into this.
	class ball {
	public:
		ball(double r, double m, const vect3d& ll, const vect3d& vv, Game::game* gg)
			: dead(false), rad(r), mas(m), l(ll), v(vv), g(gg) { }
		virtual ~ball() { }

		virtual void move(double t) = 0;
		virtual void do_collide(const ball* other) = 0;
		virtual double damage() const = 0;

		virtual void die()		{ dead = true; }

		bool is_dead() const		{ return dead; }
		double radius() const		{ return rad; }
		double mass() const		{ return mas; }
		const vect3d& loc() const	{ return l; }
		const vect3d& vel() const	{ return v; }
		Game::game* gam() const		{ return g; }

		friend void bounce_wall(ball* b);
		friend void get_inside_wall(ball* b);

	protected:
		bool dead;	// set when the ball is broken up
		double rad;
		double mas;
		vect3d l;	// the centre point
		vect3d v;	// velocity after last move
		Game::game* g;
	private:
		ball(const ball&);
		ball& operator=(const ball&);
	};

	// to create some ball into the game
	class factory {
	public:
		virtual ~factory() { }
		virtual void new_ball(const vect3d& l, const vect3d& v, Game::game* g) = 0;
	};
	typedef rec::map<factory*> factory_map;
	void factory_register(factory_map& fm, const std::string& name, factory* f);
	void factory_register(factory_map& fm, const char* name, factory* f);

	// helper functions

	// inline functions
	// check if top of b1 is above top of b2
	inline bool above(const ball* b1, const ball* b2)
	{
		return b1->loc().z() - b1->radius() < b2->loc().z() - b2->radius();
	}

	// friends
	void bounce_wall(ball* b);
	void get_inside_wall(ball* b);

	// others
	bool hit_wall(const ball* b);
	bool collide(const ball* b1, const ball* b2);
	vect3d dvel_collision(const ball* b1, const ball* b2);

	// Now come the balls

	// base class for bullets. Note that constructor takes location
	// and velocity.
	class bullet : public ball {
	public:
		bullet(double r, double m, const vect3d& ll, const vect3d& vv, Game::game* gg, double d)
			: ball(r, m, ll, vv, gg),
			  hit(false), dam(d)
		{ }

		void move(double t);
		void do_collide(const ball* other);
		double damage() const;
	protected:
		bool hit;
		double dam;
	};

	inline bool is_bullet(const ball* b)
	{
		return dynamic_cast<const bullet*>(b) != 0;
	}

	class ship;
	class ship_bullet : public bullet {
	public:
		ship_bullet(double r, double m, const vect3d& ll, const vect3d& vv, Game::game* gg, double d, ship* s)
			: bullet(r, m, ll, vv, gg, d),
			  firer(s)
		{ }

		virtual void add_score(int i) const;
	protected:
		ship* firer;
	private:
		ship_bullet(const ship_bullet&);
		ship_bullet& operator=(const ship_bullet&);
	};

	inline bool is_ship_bullet(const ball* b)
	{
		return dynamic_cast<const ship_bullet*>(b) != 0;
	}

	// base for other enemy balls.
	class enemy : public ball {
	public:
		enemy(double r, double m, const vect3d& ll, const vect3d& vv, Game::game* gg, double str, double shld, int scr)
			: ball(r, m, ll, vv, gg),
			  strength(str), shield(shld), score(scr)
		{ }

		void move(double t);
		void do_collide(const ball* other);
		double damage() const;

	protected:
		double strength, shield;
		int score;
		vect3d dv;
	};

	inline bool is_enemy(const ball* b)
	{
		return dynamic_cast<const enemy*>(b) != 0;
	}

	// drawable balls are derived from graphics::deep_drawable
	inline bool is_drawable(const ball* b)
	{
		return dynamic_cast<const graphics::drawable*>(b) != 0;
	}

	// function object to draw a ball
	struct do_ball_draw {
		void operator()(const ball* b)
		{
			const graphics::drawable* d = dynamic_cast<const graphics::drawable*>(b);
			if (d)
				d -> draw(b->loc());
		}
	};

	// creates balls according to some rule
	class creator {
	public:
		virtual ~creator() { }
		virtual void move(double t) = 0;
		virtual void skip_level() = 0;
	};

} // namespace balls

#endif // !BALL_H

// Local Variables:
// mode: c++
// End:
