#$Id: consumer_core_domain.rb,v 1.8 2007/02/20 21:00:08 sgalles Exp $
#-----------------------------------------------------------------------
#
#TSP Ruby consumer - component for a generic Transport Sampling Protocol.
#
#Copyright (c) 2006 Stephane GALLES
#
#This library is free software; you can redistribute it and/or
#modify it under the terms of the GNU Lesser General Public
#License as published by the Free Software Foundation; either
#version 2.1 of the License, or (at your option) any later version.
#
#This library is distributed in the hope that it will be useful,
#but WITHOUT ANY WARRANTY; without even the implied warranty of
#MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
#Lesser General Public License for more details.
#
#You should have received a copy of the GNU Lesser General Public
#License along with this library; if not, write to the Free Software
#Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#-----------------------------------------------------------------------
#
#Project    : TSP
#Maintainer : stephane.galles@free.fr
#Component  : RubyTsp Consumer
#
#-----------------------------------------------------------------------

require 'value_object_identity'

module TspVersion
  VERSION = 65537
end

module Status
  
  OK = 0
  NOK = 1
  ERROR_PROVIDER_UNREACHABLE = 2
  ERROR_UNKNOWN = 3
  ERROR_SEE_STRING = 4
  ERROR_VERSION = 5
  ERROR_SYMBOLS = 6
  ERROR_SYMBOL_FILTER = 7
  ERROR_NOT_SUPPORTED = 8
  ERROR_NOT_IMPLEMENTED = 9
  ERROR_PGI_UNKNOWN = 10
  ERROR_ASYNC_READ_NOT_ALLOWED = 11
  ERROR_ASYNC_WRITE_NOT_ALLOWED = 12
  ERROR_ASYNC_READ_NOT_SUPPORTED = 13
  ERROR_ASYNC_WRITE_NOT_SUPPORTED = 14
  ERROR_MEMORY_ALLOCATION = 15
  ERROR_INVALID_CHANNEL_ID = 16
  ERROR_NO_MORE_GLU = 17
  ERROR_NO_MORE_SESSION = 18
  ERROR_GLU_START = 19
  ERROR_GLU_INITIALIZE = 20
  ERROR_BAD_REQUEST_ORDER = 21
  ERROR_DATAPOOL_INSTANTIATE = 22
  ERROR_THREAD_CREATE = 23
  ERROR_NOT_INITIALIZED = 24
  ERROR_INVALID_REQUEST = 25
  ERROR_EMPTY_REQUEST_SAMPLE = 26
  ERROR_CUSTOM_BEGIN = 16384
  
  #TODO : faire une exception particulieres pour l'erreur 6
  
  def check_status
    case status
    when Status::OK
      return
    when Status::ERROR_SEE_STRING 
      raise "TSP Error : " + status.to_s + ":" + status_str
    else
      raise "TSP Error : " + status.to_s
    end
  end
  
end

class Datatype
  
  def initialize(id,&decoder)
    @id = id    
    @decoder = decoder      
  end
  
  # Yuck ! Ho man, I hate to do this type of things.
  SYSTEM_IS_BIG_ENDIAN = ([1].pack("I") == [1].pack("N"))
  
  def self.system_big_endian?    
    SYSTEM_IS_BIG_ENDIAN 
  end
  
  attr_reader :id, :decoder
  
  private
  
  DECODER_DOUBLE = lambda {|io,buf| io.read(8,buf); buf.unpack("G")[0]}
  DECODER_FLOAT = lambda {|io,buf| io.read(4,buf); buf.unpack("g")[0]}
  DECODER_INT32 = lambda do |io,buf|
    io.read(4,buf)    
    buf.reverse! unless system_big_endian?
    buf.unpack("i")[0]
  end
  DECODER_UINT32 = lambda {|io,buf| io.read(4,buf); buf.unpack("N")[0]}
  DECODER_INT64 = lambda do |io,buf|
    io.read(8,buf)
    buf.reverse! unless system_big_endian?
    buf.unpack("q")[0]
  end
  DECODER_UINT64 = lambda do |io,buf|
    io.read(8,buf)
    buf.reverse! unless system_big_endian?
    buf.unpack("Q")[0]
  end
  
  
  public
  
  #TODO : we could do a much clever thing : like build the whole unpack string
  #for the whole group
  
  UNKNOWN = self.new(0) {|io,buf| raise "can't decode unknown type"}
  DOUBLE = self.new(1, &DECODER_DOUBLE)
  FLOAT = self.new(1 + 1, &DECODER_FLOAT)
  INT8 = self.new(1 + 2, &DECODER_INT32)
  INT16 = self.new(1 + 3, &DECODER_INT32)
  INT32 = self.new(1 + 4, &DECODER_INT32)
  INT64 = self.new(1 + 5, &DECODER_INT64)
  UINT8 = self.new(1 + 6, &DECODER_UINT32)
  UINT16 = self.new(1 + 7, &DECODER_UINT32)
  UINT32 = self.new(1 + 8, &DECODER_UINT32)
  UINT64 = self.new(1 + 9, &DECODER_UINT64)
  CHAR = self.new(1 + 10, &DECODER_INT32)
  UCHAR = self.new(1 + 11, &DECODER_UINT32)
  RAW = self.new(1 + 12) {|io,buf| raise "not implemented"}  
  
  private
  
  # build the array list of types, by id
  def self.build_type_list 
    datatypes = Array.new
    nonindex_datatypes = constants.collect {|c_name| Datatype.const_get(c_name)}.select {|c| c.kind_of? Datatype}
    nonindex_datatypes.each {|type| datatypes[type.id] = type}
    return datatypes
  end
  
  DATATYPES = self.build_type_list
  
  public
  
  def self.get_datatype_by_id(id)
    DATATYPES[id]
  end
  
end



class CoreRequestOpen
  
  include ValueObjectIdentity
  
  def initialize(version_id=TspVersion::VERSION,argv=[])
    @version_id = version_id    
    @argv = argv        
  end
  
  attr_reader    :version_id,
  :argv
  
  def nb_args
    @argv.length
  end                 
  
end

class CoreAnswerOpen
  
  include Status
  include ValueObjectIdentity
  
  def initialize()
    @version_id = nil    
    @channel_id = nil
    @status = Status::OK
    @status_str = nil
  end
  
  attr_accessor :version_id,
  :channel_id,
  :status,
  :status_str
  
end

class CoreRequestInformation
  
  include ValueObjectIdentity
  
  def initialize(channel_id, version_id=TspVersion::VERSION)
    @channel_id = channel_id
    @version_id = version_id    
  end
  
  attr_accessor    :version_id,
  :channel_id
  
end


class CoreRequestSample
  
  include ValueObjectIdentity
  
  def initialize(channel_id, symbols, version_id=TspVersion::VERSION)
    @channel_id = channel_id
    @version_id = version_id    
    @symbols = symbols    
    @feature0 = 0
    @feature1 = 0    
    @feature2 = 0
    @feature3 = 0
    @consumer_timeout = 0
    
  end
  
  attr_accessor   :version_id,
  :channel_id,
  :feature0,
  :feature1,
  :feature2,
  :feature3,
  :consumer_timeout
  
  
  def each_symbol
    @symbols.each {|symbol| yield symbol}
  end
  
  def nb_symbols
    @symbols.length
  end
  
end

class CoreSymbolInfo
  
  include ValueObjectIdentity
  
  def initialize(name = "", period=1, phase=0, global_index=-1, group_index=0, group_rank=0, type=0)
    @name = name
    @period = period
    @phase = phase
    @global_index = global_index    
    @group_index = group_index
    @group_rank = group_rank
    @type = type
    @dimension=1
    @nelem=0
    @offset=0
  end
  
  attr_accessor  :name,
  :period,
  :phase,
  :global_index,
  :group_index,
  :group_rank,
  :type,
  :dimension  ,
  :nelem,
  :offset  
  
end


class CoreAnswerSample
  
  include Status
  include ValueObjectIdentity
  
  def initialize(symbols)
    @version_id = nil    
    @channel_id = nil
    @provider_timeout = nil
    @group_number = nil
    @base_frequency = nil
    @max_period = nil
    @max_client_number = nil
    @current_client_number = nil
    @status = Status::OK
    @symbols = symbols
  end
  
  attr_accessor :version_id,
  :channel_id,
  :provider_timeout,
  :group_number,
  :base_frequency,
  :max_period,
  :max_client_number,
  :current_client_number,
  :status
  
  def each_symbol
    @symbols.each {|symbol| yield symbol}
  end
  
  def nb_symbols
    @symbols.length
  end
  
end

class CoreRequestSampleInit
  
  include ValueObjectIdentity
  
  def initialize(channel_id, version_id=TspVersion::VERSION)
    @channel_id = channel_id
    @version_id = version_id    
  end
  
  attr_accessor    :version_id,
  :channel_id
  
end

class CoreAnswerSampleInit
  
  include Status
  include ValueObjectIdentity
  
  def initialize()
    @channel_id = nil
    @version_id = nil
    @host = nil
    @port = nil
    @status = Status::OK
  end
  
  attr_accessor  :version_id,
  :channel_id,
  :status
  
  attr_reader    :host,
  :port
  
  def data_address
    @host + ":" + @port.to_s
  end
  
  def data_address=(new_data_address)
    @host, @port = new_data_address.split(':')
    @port = @port.to_i
  end
  
end

class CoreRequestSampleDestroy 
  
  include ValueObjectIdentity
  
  def initialize(channel_id, version_id=TspVersion::VERSION)
    @channel_id = channel_id
    @version_id = version_id    
  end
  
  attr_accessor    :version_id,
  :channel_id
  
end

class CoreAnswerSampleDestroy
  
  include Status
  include ValueObjectIdentity
  
  def initialize()
    @version_id = nil    
    @channel_id = nil
    @status = Status::OK
  end
  
  attr_accessor :version_id,
  :channel_id,
  :status
  
end


class CoreRequestClose
  
  include ValueObjectIdentity
  
  def initialize(channel_id, version_id=TspVersion::VERSION)
    @channel_id = channel_id
    @version_id = version_id    
  end
  
  attr_accessor    :version_id,
  :channel_id
  
end
