// -*- Mode: Java; indent-tabs-mode: nil; c-basic-offset: 4; -*-
/*
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2000-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Ant", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */
//----------------------------------------------------------------------
// $Header: /cvsroot/ant-doxygen/ant_task/src/org/doxygen/tools/DoxygenTask.java,v 1.8.4.8 2004/01/31 01:39:31 akkumar Exp $
//
package org.doxygen.tools;

import org.apache.tools.ant.BuildException;
import org.apache.tools.ant.Task;
import org.apache.tools.ant.taskdefs.PumpStreamHandler;
import org.apache.tools.ant.taskdefs.Execute;

import java.util.ArrayList;
import java.util.Enumeration;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.TreeMap;
import java.util.Vector;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.PrintStream;

/**
 * Ant task for Doxygen.  This task provides a mapping between
 * all current Doxygen configuration file options (as of 1.2.18)
 * and Ant build files. <br/>
 *
 * \par Doxygen and JavaDoc co-existence. <br/>
 *
 * \note This file uses a combination of JavaDoc and Doxygen
 * formatting codes.  Doxygen tags begin with a single back-slash
 * instead of the JavaDoc standard at-sign.  This reduces the number
 * of warnings which JavaDoc issues when parsing a Doxygen documented
 * file while also including that text in JavaDoc output (albeit
 * poorly formatted). <br/><br/>
 *
 * By placing all Doxygen paragraphs before any JavaDoc paragraphs,
 * all text will naturally flow into the body of the main paragraph
 * (for a graceful degradation). <br/><br/>
 *
 * Also, but placing one or two HTML &lt;br&gt; tags at the end of
 * each paragraph, the degradation from Doxygen to JavaDoc will be
 * less severe. <br/><br/>
 *
 * @author <a href="mailto:akkumar@users.sourceforge.net">Karthik Kumar</a>
 * @version $Revision: 1.8.4.8 $
 * @since Ant 1.5
 *
 */
public class DoxygenTask extends Task {

     /**
     * Name of the temporary configuration file that will be created
     * in ${user.home}.
     **/
    public static final String CONFIG_FILE =
        System.getProperty("user.home") + File.separator + ".doxytmp";

   /**
     * Object containing the configuration options of 
     * DoxygenConf
     **/
    private DoxygenConfig conf; 
    
    /**
     * Object responsible for creating Doxygen processes
     * and executing the same.
     **/
    private DoxygenProcess proc;
    

    /**
     * Flag to control the Ant task output verbosity.
     **/
    private boolean verbose = false;

    
    /** 
     * String that contains the version against which compatibility 
     * check is to be performed.
     **/
    private String versionCompatible = null;
    
   /**
     * Name of the configuration file used as a basis for applying
     * Doxygen parameters.  This value may be updated by the user via
     * the configFilename="file" element of the Ant task.
     **/
    private String configFilename = null;

    //---------------------------------
    /**
     * Initializes the Doxygen Configuration options.
     **/
    public DoxygenTask() {
        proc = new DoxygenProcess();
        conf = new DoxygenConfig();
    }


    //----------------------------------------------------------------------
    /** This is the main execution entry-point of the task.
     *
     *  @throws BuildException thrown in case any problem executing
     *  the task.
     *
     */
    public final void execute() throws BuildException {
        proc.checkVersion(versionCompatible);
        System.out.println("Version check complete");
        if (configFilename == null ) {
            configFilename = CONFIG_FILE;
            proc.createConfig(configFilename);
        }
        conf.writeDoxygenConfig(configFilename);  // Then we update it.
        System.out.println("Created Config file");
        proc.executeDoxygenConfig(configFilename);
        
        // No need to delete the config. file at all. 
        //Let it be there for the user to know what was used.
    }

    //---------------------------------------------------------
    /** 
     * @param newVersion Sets the version against which compatibility 
     * check is to be run.
     **/
    public final void setVersionCompatible(String newVersion) {
        versionCompatible = newVersion;
    }


    /**
     *  @return Returns the version against which the compatibility check
     * is to be run. 
     **/
    public final String getVersionCompatible() {
        return versionCompatible;
    }
    
    //======================================================================
    //          Ant Task related elements and nested elements.
    //======================================================================

    //----------------------------------------------------------------------
    /** Core Ant task element that controls the output verbosity of
     *  this Ant task.  By default, verbosity is minimized to allow
     *  for cleaner build logs.  Turn this on, if you need more
     *  detailed diagnostic information. <br/>
     *
     *  \test there is a marked difference in output depending on the
     *  value of verbose.
     *
     *  @param attr from build.xml.
     *
     */
    public final void setVerbose(final boolean attr) { verbose = attr; }


    //----------------------------------------------------------------------
    /** Core Ant task element which specifies an non-generated Doxygen base
     *  configuration file.  All &lt;doxygen&gt; elements and nested elements
     *  that appear in the build.xml file will take precedence over this
     *  base configuration. <br/><br/>
     *
     *  The base configuration file will not be updated.  Doxygen will
     *  be invoked with an amalgamated configuration file which is
     *  derived from either this configuration file or if not
     *  specified, a Doxygen default and then Ant task flavored
     *  almalgamated configuration file. <br/><br/>
     *
     *  \test If the configuration file is set, then the amalgamated
     *  configuration file is retained. <br/>
     *
     *  @param attr from build.xml.
     *
     */
    public final void setConfigFilename(final String attr) {
        configFilename = attr;
    }


    //----------------------------------------------------------------------
    /** Core Ant task element which specifies a path to the Doxygen
     *  executable.  By default the application name "doxygen" is used
     *  which presumes Doxygen is on the path. <br/>
     *
     *  \test Specification of a bogus application name will fail. <br/>
     *
     *  \todo Add a test to see that the specified executable exists. <br/>
     *
     *  @param attr from build.xml.
     *
     */
    public final void setDoxygenPath(final String attr) { 
        proc.setDoxygenPath(attr); 
    }


    //----------------------------------------------------------------------
    /** This method enables nested property tags.  This is the
     *  preferred attribute specification method as it reduces the
     *  acute knowledge that the task has over a given Doxygen
     *  implementation.  However, non-nested attributes may still be
     *  specified. <br/>
     *
     *  \note this method is only called by Ant. <br/>
     *
     *  @return an empty <code>Property</code>.
     *
     */
    public final Property createProperty() {
        Property p = new Property();
        conf.addNestedAttribute(p);
        return p;
    }

    //-----------------------------------------------
    /**
     * @param name Name of the configuration property.
     * @value Value of the configuration property.
     **/
    public final void setConfig(String name, String value) {
        Property p = new Property();
        p.setName(name);
        p.setValue(value);
        conf.addNestedAttribute(p);
    }

    //-------------------------------------------------
    /**
     * @return Returns the Task Attributes of the Doxygen 
     * config. property file.
     **/    
    public Map getTaskAttributes() {
        return conf.getTaskAttributes();
    }
    //----------------------------------------------------------------------
    /** This method allows a single point of control over
     *  the task output verbosity. <br/>
     *
     *  \test message output is conditional.  Some messages are always
     *  output, while others are only sometimes output. <br/>
     *
     *  @param terseMessage is a flag that indicates the calling code
     *  wants this message to appear regardless of the current
     *  verbosity level.
     *
     *  @param theMessage to be output, guard condition permitting.
     *
     */
    public final void activityLog(final boolean terseMessage,
                                  final String theMessage) {
        if  (theMessage != null) {
            if  (terseMessage || verbose) {
                System.out.println(theMessage);
            }
        }
    }

    //----------------------------------------------------------------
    /** This inner class implements a nested Ant XML property
     *  container.  This class is used for both nested and non-nested
     *  versioned properties.
     *
     */
    public static class Property {

        /** This is the Doxygen configuration file parameter name. */
        private String key;

        /** This is the Doxygen configuration file parameter value. */
        private String val;

        //------------------------------------------------------------
        /** This accessor method retrieves the current Doxygen
         *  parameter name.
         *
         *  @return a <code>String</code> value containing the
         *  property key name.
         */
        public final String getName()  { return key; }


        //------------------------------------------------------------
        /** This access method retrieves the current Doxygen parameter
         *  value.
         *
         *  @return a <code>String</code> value containing the
         *  property value.
         *
         */
        public final String getValue() { return val; }


        //------------------------------------------------------------
        /** This method sets the Doxygen parameter name.  This is the
         *  actual parameter name as seen in the Doxygen configuration
         *  file.
         *
         *  @param theNewName of the property key.
         *
         */
        public final void setName(final String theNewName) { key = theNewName; }


        //------------------------------------------------------------
        /** This method sets the value of the Doxygen parameter.  In a
         *  nested element context, the value is a String.  In the
         *  &lt;doxygen&gt; task context, typed values must be coerced
         *  to String form before being set as a Property.
         *
         *  @param theNewValue of the property.
         *
         */
        public final void setValue(final String theNewValue) {
            if  (theNewValue.toLowerCase().equals("true")) {
                val = "YES";
            } else if (theNewValue.toLowerCase().equals("false")) {
                val = "NO";
            } else { val = theNewValue; }
        }


        //------------------------------------------------------------
        /** This method returns a String representation of this
         *  instance.
         *
         *  @return a <code>String</code> containing identifying
         *  information about this instance.
         *
         */
        public final String toString() {
            return "Property=["
                + "key={" + key + "},"
                + "value={" + val + "},"
                + "]";
        }
    }
}
