# smi_parser.rb: parses a smidump'ed MIB
# Copyright (C) 2006,2007  Frederik Deweerdt <frederik.deweerdt@gmail.com>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software Foundation,
# Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.

require "rexml/document"
require "rexml/element"

# As of ruby 1.8.6, the Range.max method is extremly slow probably because
# it's supposed to work with other classes than just integer this little
# wrapper class saves minutes(!) when generating code for intergers whose
# ranges are large
class DadiRange < Range
  def initialize(min, max)
    super(min, max)
    @_min = min
    @_max = max
  end#initialize

  def min
    return @_min
  end#min

  def max
    return @_max
  end#max
	def length
		return (max - min)+1
	end#length
end#class DadiRange

class SmiParser
	def initialize(filename)
		@filename = filename
		@file = File.new(filename)
    begin
      @doc = REXML::Document.new(@file)
    rescue
			raise "Cannot parse #{filename}:#{$!}"
    end
		@datatree = DataTree.new
		@type_mapper = { "Integer32" => "int",
				"Unsigned32" => "uint",
				"Gauge32" => "uint",
				"Counter32" => "uint",
				"TruthValue" => "int",
				"enum" => "int",
				"OctetString" => "chartab",
				"DisplayString" => "chartab",
				"IpAddress" => "ipaddr",
				"MacAddress" => "macaddr" ,
				"InetAddressType" => "uint" ,
				"InetAddress" => "chartab" ,
				"InetPortNumber" => "uint" ,
				"Counter64" => "uint64_t" }
		@type_sizer = { "MacAddress" => 6,
				"int" => "sizeof(int)",
				"uint" => "sizeof(unsigned int)",
				"DisplayString" => 255,
				"OctetString" => 255,
				"IpAddress" => 4 ,
				"InetAddressType" => 4 ,
				"InetAddress" => 255 ,
				"InetPortNumber" => 4 ,
				"Counter64" => "sizeof(uint64_t)" }
	end#initialize

	def parseRange(el_range)
		el_range.elements["syntax"].each_element() do
			|e|
			case e.name
				when "typedef"
					rel = e.elements["range"]
					if rel != nil then
						return DadiRange.new(rel.attributes["min"].to_i,
								     rel.attributes["max"].to_i)
					end#if
			end#case
		end#each_element
		return nil
	end#parseRange

	def parseTable(el_table)
		t = Table.new(el_table.attributes["name"])
		t["oid"] = el_table.attributes["oid"]
		index_name = ""
		el_table.get_elements("row/linkage/index").each { |e|
			index_name = e.attributes["name"]
		}
		el_table.elements["row"].each_element("column") do
			|e|
			if e.attributes["name"] == index_name
			then
				r = parseRange(e)
				t["index_name"] = index_name
				t["range"] = r
				t["length"] = r.length
				next
			end#if
			s = parseScalar(e)
			t.addCol(s)
		end#each_element("column")
		return t
	end#parseTable

	def parseScalar(e)
		s = Scalar.new(e.attributes["name"])
		s["range"] = parseRange(e)
		s["length"] = s["range"].length if s["range"] != nil
		s["oid"] =  e.attributes["oid"]
		s["access"] = e.attributes["access"] if e.attributes["access"] != nil
		s["status"] =  e.attributes["status"]

		e.each_element("syntax") do
			|el|
			el.each_element() do
				|ele|
				case ele.name
					when "typedef"
						ret = parseTypedefs(el, s["name"])
						if ret != nil then
							s["type"] = @type_mapper.fetch(ret)
							s["real_type"] = "#{ret}"
						else
							s["type"] = @type_mapper.fetch("enum")
							s["real_type"] = "int"
						end#if
					when "type"
						s["type"] = @type_mapper.fetch(ele.attributes["name"])
						s["real_type"] = ele.attributes["name"]
					else
						raise ele.to_s
				end#case
			end#each_element
		end#each_element

		if e!=nil then
			s["access"] = e.elements["access"].text
		end#if
		if e.elements["description"]!=nil then
			s["description"] = e.elements["description"].text.strip.tr("\n","").squeeze(" ")
		end#if
		if e.elements["default"]!=nil then
			s["default"] = e.elements["default"].text
		end#if
		if e.elements["units"]!=nil then
			s["units"] = e.elements["units"].text
		end#if

		if s["type"] != nil then
			s["size"] = @type_sizer[s["real_type"]]
			s["length"] = @type_sizer[s["real_type"]] if s["length"] == nil
		end#if
		return s
	end#parseScalar

	def parseTypedefs(el_type, class_name = nil)
		name = class_name
		el_type.each_element() do
			|e|
			case e.attributes["basetype"]
				when "Enumeration"
					name = e.attributes["name"] if name == nil
					enum = Enum.new(name)
					e.each_element("namednumber") do
						|el|
						enum.add(el.attributes["name"], el.attributes["number"])
					end#each_element
					@datatree.addEnum(enum)
					return nil
				when "Integer32"
					return "Integer32"
				when "OctetString"
					return "OctetString"
				else
					p e.attributes["basetype"]
					raise "Unknown typedef tag #{e.name}"
			end#case
		end#each_element
	end#parseTypedefs

	def parseNodes(el_node)
		el_node.each_element() do
			|e|
			case e.name
				when "scalar"
					@datatree.addNode(parseScalar(e))
				when "node"
				when "table"
					t = parseTable(e)
					@datatree.addTable(t)
				else
					raise "Unknown node #{e.name}"
			end#case
		end#each_element
	end#parseNodes

	def parse()
		begin
		@doc.root().each_element() do
			|e|
			case e.name
				when "typedefs"
					parseTypedefs(e)
				when "nodes"
					parseNodes(e)
				else
			end#case
		end#each_element
		return @datatree
		rescue
			raise "Cannot parse #{@filename}:#{$!}"
		end
	end#parse
end#class SmiParser
