--------------------------------------------------------------------------------
-- Trombi - Copyright 2007-2008 Louis Paternault
-- 
-- This file is part of Trombi.
-- 
-- Trombi is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
-- 
-- Trombi is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
-- 
-- You should have received a copy of the GNU General Public License
-- along with Trombi.  If not, see <http://www.gnu.org/licenses/>.
--------------------------------------------------------------------------------

------------------------------------------------
--
-- Ce paquetage definit le type 'document', qui est une liste des elements qui seront presents dans le document final (sections, photos...), ainsi que des options de mise en page.
--
--
--------------------------------------------------

with Ada.strings.unbounded;
use Ada.strings.unbounded;

with listes, titre;
use titre;

package typedocument is

  -- Liste des elements presents dans le document final (titres et photos, mais aussi mise en forme...)
  type document is private;

  -- Renvoie un document vide
  function documentVide return document;

  -- Renvoie true ssi le document est vide
  function estVide(doc : document) return boolean;

  -- Ajoute une photo au document
  --   'nom' est le nom qui sera affiche sous la photo (la legende)
  --   'chemin' est le chemin d'acces au fichier
  --   'doc' est le document
  procedure ajoutePhoto(doc : in out document ; chemin, nom : string);

  -- Ajoute une section au document
  --   'nom' est le titre de la section
  --   'niveau' est le niveau de la section (section, subsection, paragraph...)
  procedure ajouteSection(doc : in out document ; nom : string ; niveau : positive);

  -- Ajoute une definition de taille des photos au document
  procedure ajouteTaille(doc : in out document ; hori : string ; vert : string);

  -- Ajoute une definition de nombre de photos par ligne au document
  procedure ajouteNombre(doc : in out document ; n : integer);

  -- Affiche la liste
  -- Utile uniquement pour debugger
  procedure affiche(doc : document ; n : integer := 3);

  -- Renvoie 'true' ssi le prochain element de la liste est le type demande
  function estTitre(doc : document) return boolean;
  function estPhoto(doc : document) return boolean;
  function estTaille(doc : document) return boolean;
  function estNombre(doc : document) return boolean;

  -- Renvoie les champs demandes.
  -- Precondition : ce champ existe. L'erreur "erreur inconnue" est levee s'il n'existe pas
  function photoLegende(doc : document) return string;
  function photoChemin(doc : document) return string;
  function titreTexte(doc : document) return string;
  function titreNiveau(doc : document) return niveauDeTitre;
  function tailleVert(doc : document) return string;
  function tailleHori(doc : document) return string;
  function nombreNombre(doc : document) return integer;

  -- Supprime le premier element
  procedure suivantdestructeur(doc : in out document);

private

  type genre is (
    photo,
    titre,
    taille,
    nombre,
    aucun
  );

  type element(leGenre : genre := aucun) is record
    case leGenre is
      when photo =>
        legende : unbounded_string;
        chemin : unbounded_string;
      when titre =>
        texte : unbounded_string;
        niveau : niveauDeTitre;
      when taille =>
        hori : unbounded_string;
        vert : unbounded_string;
      when nombre =>
        nombre : integer;
      when aucun =>
        null;
    end case;
  end record;

  package maListe is new listes(element);
  use maListe;

  -- Nous n'allons faire que des ajouts en queue.
  -- Autant qu'ils soient fait en temps O(1)
  type document is new dliste;

  --- Accesseurs sur les elements
  --- Levent l'exception "erreurImpossible" si le champs demande n'existe pas (si le champs 'legende' d'un titre est demande, par exemple)
  function leGenre(e : element) return genre;
  function photoLegende(e : element) return string;
  function photoChemin(e : element) return string;
  function titreTexte(e : element) return string;
  function titreNiveau(e : element) return niveauDeTitre;
  function tailleHori(e : element) return string;
  function tailleVert(e : element) return string;
  function nombreNombre(e : element) return integer;

end typedocument;
