----------------------------------------------------------------------
-- Trombi - Copyright 2007-2008 Louis Paternault
-- 
-- This file is part of Trombi.
-- 
-- Trombi is free software: you can redistribute it and/or modify
-- it under the terms of the GNU General Public License as published by
-- the Free Software Foundation, either version 3 of the License, or
-- (at your option) any later version.
-- 
-- Trombi is distributed in the hope that it will be useful,
-- but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
-- GNU General Public License for more details.
-- 
-- You should have received a copy of the GNU General Public License
-- along with Trombi.  If not, see <http://www.gnu.org/licenses/>.
----------------------------------------------------------------------

with erreurs, texte, trace;
use erreurs, texte, trace;

package body taille is


  ----------------------------------------
  -- Gere les commentaires de debuggage --
  profMax : integer := 1;
  procedure tracer(prof : positive ; chaine : string) is
  begin
    if prof <= profMax then
      tracerDebug(prof, chaine);
    end if;
  end tracer;
  ----------------------------------------

  procedure analyseTaille(chaine : string ; horizontal, vertical : out unbounded_string) is
  begin
    chaineAnalysee := to_unbounded_string(chaine);
    indiceCourant := 0;
    carSuivant;
    axiome(horizontal, vertical);
    if (courant /= finDeLigne) then
      raise erreurTaille;
    end if;
    tracer(1, "Taille lue : '" & to_string(horizontal) & "' x '" & to_string(vertical) & "'.");

  exception 
    when erreurTaille =>
      txtErreurTaille(to_string(chaineAnalysee));
      raise erreurArg;
    when erreurUnite =>
      txtErreurUnite(to_string(chaineAnalysee));
      raise erreurArg;
  end analyseTaille;

  -- Regles de la grammaire
  procedure axiome(largeur, hauteur : out unbounded_string) is
  begin
    tracer(1, "Entree dans axiome. Caractere lu : '" & caractereCourant & "'.");
    largeur := to_unbounded_string("");
    hauteur := to_unbounded_string("");
    case courant is 
      when minuscule =>
        auto;
      when chiffre | point =>
        largeur := to_unbounded_string(horizontal);
        if courant /= croix then
          raise erreurTaille;
        end if;
        croix;
        if not (courant = chiffre or courant =  point or courant =  finDeLigne) then
          raise erreurTaille;
        end if;
        hauteur := to_unbounded_string(verticalOptionnel);
      when croix =>
        croix;
        if not (courant =  chiffre or courant = point) then
          raise erreurTaille;
        end if;
        hauteur := to_unbounded_string(vertical);
      when others =>
        raise erreurTaille;
    end case;
  end axiome;

  procedure auto is
  begin
    tracer(1, "Entree dans auto. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when minuscule =>
        carSuivant;
        if not (courant = minuscule) then
          raise erreurTaille;
        end if;
        carSuivant;
        if not (courant = minuscule) then
          raise erreurTaille;
        end if;
        carSuivant;
        if not (courant = minuscule) then
          raise erreurTaille;
        end if;
        carSuivant;
        if (chaineAnalysee /= "auto") then
          raise erreurTaille;
        end if;
      when others =>
        raise erreurTaille;
    end case;
  end auto;

  procedure croix is
  begin
    tracer(1, "Entree dans croix. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when croix =>
        carSuivant;
      when others =>
        raise erreurTaille;
    end case;
  end croix;

  function verticalOptionnel return string is
  begin
    tracer(1, "Entree dans verticalOptionnel. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when finDeLigne => -- epsilon
        return "";
      when chiffre | point =>
        return vertical;
      when others =>
        raise erreurTaille;
    end case;
  end verticalOptionnel;
  
  function vertical return string is
  begin
    tracer(1, "Entree dans vertical. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre | point =>
        declare
          temp : string := nombre;
        begin
          if (courant /= minuscule) then
            raise erreurTaille;
          end if;
          return temp & unite;
        end;
      when others =>
        raise erreurTaille;
    end case;
  end vertical;
  
  function horizontal return string is
  begin
    tracer(1, "Entree dans horizontal. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre | point =>
        declare
          temp : string := nombre;
        begin
          if (courant /= minuscule) then
            raise erreurTaille;
          end if;
          return temp & unite;
        end;
      when others =>
        raise erreurTaille;
    end case;
  end horizontal;
  
  function unite return string is
    chaine : string(1..2);
  begin
    tracer(1, "Entree dans unite. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when minuscule =>
        chaine(1) := lettre;
        if courant /= minuscule then
          raise erreurTaille;
        end if;
        chaine(2) := lettre;

        if not est_unite(chaine) then
          tracer(1, chaine);
          raise erreurUnite;
        end if;
        return chaine;
      when others =>
        raise erreurTaille;
    end case;
  end unite;
  
  function lettre return character is
    carLu : character := caractereCourant;
  begin
    tracer(1, "Entree dans lettre. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when minuscule =>
        carSuivant;
        return carLu;
      when others =>
        raise erreurTaille;
    end case;
  end lettre;
  
  function nombre return string is
    temp : unbounded_string;
  begin
    tracer(1, "Entree dans nombre. caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre =>
        declare
          temp : string := entier;
        begin
          if not (courant = point or courant = minuscule) then
            raise erreurTaille;
          end if;
          return temp & suiteNombre;
        end;
      when point =>
        point;
        if not (courant = chiffre) then
          raise erreurTaille;
        end if;
        return "." & entier;
      when others =>
        raise erreurTaille;
    end case;
  end nombre;
  
  function suitenombre return string is
  begin
    tracer(1, "Entree dans suitenombre. caractereCourant lu : '" & caractereCourant & "'.");
    case courant is
      when point =>
        point;
        if not (courant = chiffre) then
          raise erreurTaille;
        end if;
        return "." & entier;
      when minuscule => --epsilon
        return "";
      when others =>
        raise erreurTaille;
    end case;
  end suitenombre;
  
  function entier return string is
  begin
    tracer(1, "Entree dans entier. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre =>
        declare
          temp : string := chiffre;
        begin
          if not (courant = chiffre or courant = point or courant = minuscule) then
            raise erreurTaille;
          end if;
          return temp & entierSuite;
        end;
      when others =>
        raise erreurTaille;
    end case;
  end entier;
  
  function entierSuite return string is
  begin
    tracer(1, "Entree dans entierSuite. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre =>
        declare
          temp : string := chiffre;
        begin
          if not (courant = chiffre or courant = point or courant = minuscule) then
            raise erreurTaille;
          end if;
          return temp & entierSuite;
        end;
      when point | minuscule => -- epsilon
        return "";
      when others =>
        raise erreurTaille;
    end case;
  end entierSuite;

  procedure point is
  begin
    tracer(1, "Entree dans point. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when point =>
        carSuivant;
      when others =>
        raise erreurTaille;
    end case;
  end point;

  function chiffre return string is
    temp : character;
  begin
    tracer(1, "Entree dans chiffre. Caractere lu : '" & caractereCourant & "'.");
    case courant is
      when chiffre =>
        temp := caractereCourant;
        carSuivant;
        return "" & temp; -- Conversion character -> string
      when others =>
        raise erreurTaille;
    end case;
  end chiffre;

  -- Passe au caractere suivant
  procedure carSuivant is
  begin
    indiceCourant := indiceCourant + 1;
    if (indiceCourant > to_string(chaineAnalysee)'length) then
      courant := finDeLigne;
    else
      caractereCourant := to_string(chaineAnalysee)(indiceCourant);
      if (caractereCourant in '0'..'9') then
        courant := chiffre;
      elsif (caractereCourant in 'a' .. 'z') then
        courant := minuscule;
      elsif (caractereCourant = '.') then
        courant := point;
      elsif (caractereCourant = 'X') then
        courant := croix;
      end if;
    end if;
  end carSuivant;

  function est_unite(chaine : string) return boolean is
  begin
    -- Consulter le fichier doc/grammaire/taille.txt pour savoir d'ou je sort cette liste d'unites
    if (
      chaine = "in" or
      chaine = "cm" or
      chaine = "mm" or
      chaine = "pt" or
      chaine = "sp" or
      chaine = "bp" or
      chaine = "pc" or
      chaine = "dd" or
      chaine = "cc"
    ) then
      return true;
    else
      return false;
    end if;
  end est_unite;

end taille;
