/* tpb - program to use the IBM ThinkPad(tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* includes {{{ */
#include <fcntl.h>
#include <sys/ioctl.h>
#include <linux/soundcard.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <wait.h>
#include "config.h"

#ifdef HAVE_LIBXOSD
#include <xosd.h>
#endif /* HAVE_LIBXOSD */

#include "cfg.h"
#include "tpb.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{*/
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: tpb.c,v 1.28 2002/11/04 11:53:59 mbr Exp $";
#endif
/* }}} */

/* global variables  {{{ */
extern config cfg;
/* }}} */

/* function prototypes {{{ */
int get_state(t_nvram_state *nvram_state);
int fork_app(char * cmd);
void set_volume_level(void);
int change_volume(int change);
int apmiser_running(void);
/* }}} */

int main(int argc, char **argv)/* {{{ */
{
	t_nvram_state nvram_state, last_nvram_state;
	int vol = 0;
	int display_state = 1;
	char *home;
	char *cfg_file;
	char *msg;
	char callbackcmd[CALLBACKCMDLENGTH];
	struct sigaction signal_action;
#ifdef HAVE_LIBXOSD
	xosd *osd_ptr = NULL;
#endif /* HAVE_LIBXOSD */

	/* ignore SIG_CHLD */
	signal_action.sa_handler = SIG_IGN;
#ifdef SA_NOCLDWAIT
	signal_action.sa_flags = SA_NOCLDWAIT;
#else
	signal_action.sa_flags = 0;
#endif
	sigemptyset(&signal_action.sa_mask);
	sigaction(SIGCHLD, &signal_action, NULL);

	/* initialize config */
	init_cfg();

	/* read global config file */
	parse_cfg_file(GLOBAL_CONFIG_FILE);

	/* read user config file */
	if((home = getenv("HOME")) != NULL) {
		cfg_file = (char *)malloc(strlen(home) + strlen("/") + strlen(PRIVATE_CONFIG_FILE) + 1);
		strcpy(cfg_file, home);
		strcat(cfg_file, "/");
		strcat(cfg_file, PRIVATE_CONFIG_FILE);
		parse_cfg_file(cfg_file);
		free(cfg_file);
	}

	/* parse options */
	parse_option(argc, argv);

	/* initialize osd {{{ */
#ifdef HAVE_LIBXOSD
	if(cfg.osd == STATE_ON) {
#ifdef HAVE_LIBXOSD0
		osd_ptr = xosd_init(cfg.osdfont, cfg.osdcolor, cfg.osdtimeout, cfg.osdpos, cfg.osdoffset, 0);
#elif HAVE_LIBXOSD1
		osd_ptr = xosd_init(cfg.osdfont, cfg.osdcolor, cfg.osdtimeout, cfg.osdpos, cfg.osdoffset, 0, 2);
#endif
		if(osd_ptr == NULL) {
			fprintf(stderr, "Unable to initialize xosd. Running without onsceen display\n");
		}
		else {
			xosd_set_align(osd_ptr, cfg.osdalign);
		}
	}
#endif /* HAVE_LIBXOSD */ /* }}} */

	/* to initialize struct */
	if(get_state(&nvram_state)) {
		_exit(1);
	}

	/* initialize volume */
	if(cfg.mixer == STATE_ON) {
		set_volume_level();
		vol = change_volume(0);
	}
	else {
		vol = nvram_state.volume_level * 100 / 14;
	}

	while(1) {
		/* save last state and get new one */
		memcpy(&last_nvram_state, &nvram_state, sizeof(t_nvram_state));
		if(get_state(&nvram_state)) {
			_exit(1);
		}

		/* determine the state of the Thinkpad button  {{{ */
		if(nvram_state.thinkpad_toggle != last_nvram_state.thinkpad_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts("ThinkPad button pressed");
			}
			if(cfg.tpbcmd != NULL) {
				if(fork_app(cfg.tpbcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s thinkpad pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of zoom  {{{ */
		if(nvram_state.zoom_toggle != last_nvram_state.zoom_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf("Zoom is %s\n", nvram_state.zoom_toggle ? "on" : "off");
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s zoom %s", cfg.callback, nvram_state.zoom_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.zoom_toggle ? "Zoom on" : "Zoom off");
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of the home button  {{{ */
		if(nvram_state.home_toggle != last_nvram_state.home_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts("Home button pressed");
			}
			if(cfg.homecmd != NULL) {
				if(fork_app(cfg.homecmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s home pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of the search button  {{{ */
		if(nvram_state.search_toggle != last_nvram_state.search_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts("Search button pressed");
			}
			if(cfg.searchcmd != NULL) {
				if(fork_app(cfg.searchcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s search pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of the mail button  {{{ */
		if(nvram_state.mail_toggle != last_nvram_state.mail_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {
			if(cfg.verbose == STATE_ON) {
				puts("Mail button pressed");
			}
			if(cfg.mailcmd != NULL) {
				if(fork_app(cfg.mailcmd)) {
					_exit(0);
				}
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s mail pressed", cfg.callback);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of ThinkLight {{{ */
		if(nvram_state.thinklight_toggle != last_nvram_state.thinklight_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf("ThinkLight is %s\n", nvram_state.thinklight_toggle ? "on" : "off");
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s thinklight %s", cfg.callback, nvram_state.thinklight_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.thinklight_toggle ? "ThinkLight on" : "ThinkLight off");
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of display  {{{ */
		if(nvram_state.display_toggle != last_nvram_state.display_toggle &&
			 nvram_state.hibernate_toggle == last_nvram_state.hibernate_toggle) {

			/* Some thinkpads have no hardware support to switch lcd/crt. They also
			 * don't reflect the current state in nvram_state.display_state. So, if
			 * nvram_state.display_toggle changes, but nvram_state.display_state does
			 * not change, simulate the display state.
			 */
			if(nvram_state.display_state == last_nvram_state.display_state) {
				display_state = display_state % 3 + 1;
			}
			else {
				display_state = nvram_state.display_state;
			}

			switch (display_state & 0x03) {
				case 0x1:
					msg = "LCD on, CRT off";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display lcd", cfg.callback);
					break;

				case 0x2:
					msg = "LCD off, CRT on";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display crt", cfg.callback);
					break;

				case 0x3:
					msg = "LCD on, CRT on";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s display both", cfg.callback);
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf("Display changed: %s\n", msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, "Display");
				xosd_display(osd_ptr, 1, XOSD_string, msg);
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of hv expansion  {{{ */
		if(nvram_state.expand_toggle != last_nvram_state.expand_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf("HV Expansion %s\n", nvram_state.expand_toggle & 0x01 ? "on" : "off");
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s expand %s", cfg.callback, nvram_state.expand_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.expand_toggle ? "HV Expansion on" : "HV Expansion off");
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine the state of the brightness buttons {{{ */
		if(nvram_state.brightness_level != last_nvram_state.brightness_level) {
			if(cfg.verbose == STATE_ON) {
				printf("Brightness changed: Level %d\n", nvram_state.brightness_level * 100 / 7);
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s brightness %d", cfg.callback, nvram_state.brightness_level * 100 / 7);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
		if(nvram_state.brightness_toggle != last_nvram_state.brightness_toggle) {
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, "Brightness");
				xosd_display(osd_ptr, 1, XOSD_percentage, nvram_state.brightness_level * 100 / 7);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the volume buttons {{{ */
		if(nvram_state.volume_level != last_nvram_state.volume_level) {
			if(cfg.mixer == STATE_ON) {
				vol = change_volume(MAXVOLUME / cfg.mixersteps * (nvram_state.volume_level - last_nvram_state.volume_level)); /* nvram_state.volume_level-last_nvram_state.volume_level gives the direction */
				set_volume_level();
				nvram_state.volume_level = last_nvram_state.volume_level; /* Do not recognize the change of volume_level in next iteration */
			}
			else {
				vol = nvram_state.volume_level * 100 / 14;
			}
			if(cfg.verbose == STATE_ON) {
				printf("Volume changed: Level %d\n", vol);
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s volume %d", cfg.callback, vol);
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
    /* show volume every time a volume button is pressed and not mute */
		if(nvram_state.volume_toggle != last_nvram_state.volume_toggle &&
			 !nvram_state.mute_toggle) {
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, "Volume");
				xosd_display(osd_ptr, 1, XOSD_percentage, vol);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the mute button {{{ */
		if(nvram_state.mute_toggle != last_nvram_state.mute_toggle) {
			if(cfg.verbose == STATE_ON) {
				printf("%s\n", nvram_state.mute_toggle ? "Mute" : "Unmute");
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s mute %s", cfg.callback, nvram_state.mute_toggle ? "on" : "off");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
		}
#ifdef HAVE_LIBXOSD
		if(nvram_state.mute_toggle != last_nvram_state.mute_toggle ||
			 (nvram_state.volume_toggle != last_nvram_state.volume_toggle && last_nvram_state.mute_toggle)) {
			if(cfg.mixer == STATE_ON) {
				if(nvram_state.mute_toggle) {
					change_volume(-vol); /* mute */
				}
				else {
					change_volume(vol); /* unmute */
				}
			}

			if(osd_ptr != NULL) {
				if(nvram_state.mute_toggle) {
					xosd_display(osd_ptr, 0, XOSD_string, "Mute");
					xosd_display(osd_ptr, 1, XOSD_percentage, 0);
				}
				else {
					xosd_display(osd_ptr, 0, XOSD_string, "Unmute");
					xosd_display(osd_ptr, 1, XOSD_percentage, vol);
				}
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of power {{{ */
		if(nvram_state.ac_state != last_nvram_state.ac_state) {
			if(cfg.verbose == STATE_ON) {
				printf("Power line changed: %s\n", nvram_state.ac_state ? "AC connected" : "AC disconnected");
			}
			if(cfg.callback != NULL) {
				snprintf(callbackcmd, sizeof(callbackcmd), "%s ac_power %s", cfg.callback, nvram_state.ac_state ? "connected" : "disconnected");
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				xosd_display(osd_ptr, 0, XOSD_string, nvram_state.ac_state ? "AC connected" : "AC disconnected");
				xosd_display(osd_ptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode AC {{{ */
		if(nvram_state.powermgt_ac != last_nvram_state.powermgt_ac) {
			switch(nvram_state.powermgt_ac) {
				case 0x4:
					msg = "PM AC high";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac high", cfg.callback);
					break;
					
				case 0x2:
					msg = "PM AC auto";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac auto", cfg.callback);
					break;
					
				case 0x1:
					msg = "PM AC manual";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_ac manual", cfg.callback);
					break;
					
				default:
					msg = "PM AC unknown";
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf("Power management mode AC changed: %s\n", msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				if (cfg.powermgt == STATE_ON ||  (cfg.powermgt == STATE_AUTO && !apmiser_running())) {
					xosd_display(osd_ptr, 0, XOSD_string, msg);
					xosd_display(osd_ptr, 1, XOSD_string, "");
				}
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode battery {{{ */
		if(nvram_state.powermgt_battery != last_nvram_state.powermgt_battery) {
			switch(nvram_state.powermgt_battery) {
				case 0x4:
					msg = "PM battery high";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery high", cfg.callback);
					break;

				case 0x2:
					msg = "PM battery auto";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery auto", cfg.callback);
					break;

				case 0x1:
					msg = "PM battery manual";
					snprintf(callbackcmd, sizeof(callbackcmd), "%s powermgt_battery manual", cfg.callback);
					break;

				default:
					msg = "unknown";
					break;
			}
			if(cfg.verbose == STATE_ON) {
				printf("Power management mode battery changed: %s\n", msg);
			}
			if(cfg.callback != NULL) {
				if(fork_app(callbackcmd)) {
					_exit(0);
				}
			}
#ifdef HAVE_LIBXOSD
			if(osd_ptr != NULL) {
				if (cfg.powermgt == STATE_ON ||  (cfg.powermgt == STATE_AUTO && !apmiser_running())) {
					xosd_display(osd_ptr, 0, XOSD_string, msg);
					xosd_display(osd_ptr, 1, XOSD_string, "");
				}
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* sleep for polltime */
		usleep(cfg.polltime);
	}

	_exit(0); /* never reached */
}/*}}}*/

int get_state(t_nvram_state *nvram_state)/*{{{*/
{
	int  fdsc;
	char buffer[114];

	/* open nvram for reading */
	/* must use open/close because seek is not supported by nvram */
	if((fdsc=open(cfg.nvram, O_RDONLY|O_NONBLOCK)) == -1) {
		fprintf(stderr, "Unable to open device %s: ", cfg.nvram);
		perror(NULL);
		return -1;
	}

	/* read nvram */
	if(read(fdsc, buffer, sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr, "Unable to read from device %s: ", cfg.nvram);
		perror(NULL);
		return -1;
	}

	/* close nvram device */
	if(close(fdsc) == -1) {
		fprintf(stderr, "Unable to close device %s: ", cfg.nvram);
		perror(NULL);
		return -1;
	}

	nvram_state->thinkpad_toggle   = ( buffer[0x57] & 0x08) >> 3;
	nvram_state->zoom_toggle       = (~buffer[0x57] & 0x20) >> 5;
	nvram_state->display_toggle    = ( buffer[0x57] & 0x40) >> 6;
	nvram_state->home_toggle       = ( buffer[0x56] & 0x01);
	nvram_state->search_toggle     = ( buffer[0x56] & 0x02) >> 1;
	nvram_state->mail_toggle       = ( buffer[0x56] & 0x04) >> 2;
	nvram_state->thinklight_toggle = ( buffer[0x58] & 0x10) >> 4;
	nvram_state->hibernate_toggle  = ( buffer[0x58] & 0x01);
	nvram_state->display_state     = ( buffer[0x59] & 0x03);
	nvram_state->expand_toggle     = ( buffer[0x59] & 0x10) >> 4;
	nvram_state->brightness_level  = ( buffer[0x5E] & 0x07);
	nvram_state->brightness_toggle = ( buffer[0x5E] & 0x20) >> 5;
	nvram_state->volume_level      = ( buffer[0x60] & 0x0f);
	nvram_state->volume_toggle     = ( buffer[0x60] & 0x80) >> 7;
	nvram_state->mute_toggle       = ( buffer[0x60] & 0x40) >> 6;
	nvram_state->ac_state          = ( buffer[0x57] & 0x10) >> 4;
	nvram_state->powermgt_ac       = ( buffer[0x39] & 0x07);
	nvram_state->powermgt_battery  = ( buffer[0x39] & 0x38) >> 3;

	return 0;
}/*}}}*/

int fork_app(char * cmd) { /* {{{ */
	//char *args[4];
	//args[0] = "sh";
	//args[1] = "-c";
	//args[2] = cmd;
	//args[3] = 0;

	switch(fork()) {
		case -1:
			return -1;
			break;

		case 0:
			{
				setsid(); /* children should not be killed if tpb ends */
				//execv("/bin/sh", args);
				system(cmd);
				_exit(0);
				return 127;
			}
			break;

		default:
			return 0;
			break;
	}
	return -1; /* never reached */
} /* }}} */

void set_volume_level(void) { /* {{{ */
	int  fdsc;
	char buffer;

	/* open nvram */
	if((fdsc = open("/dev/nvram", O_RDWR|O_NONBLOCK)) == -1) {
		fprintf(stderr, "Unable to open device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* jump to volume section */
	if(lseek(fdsc, 0x60, SEEK_SET) == -1 ) {
		fprintf(stderr, "Unable to seek device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* read nvram */
	if(read(fdsc, &buffer,sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr, "Unable to read from device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	buffer &= 0xf0;
	buffer |= 0x07;

	/* jump to volume section */
	if(lseek(fdsc, 0x60, SEEK_SET) == -1 ) {
		fprintf(stderr, "Unable to seek device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* write std value for volume */
	if(write(fdsc, &buffer, sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr, "Unable to write to device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	close(fdsc);

	return;

} /* }}} */

int change_volume(int change) { /* {{{ */
	int mixer;
	int volume;
	int left,right;

	/* open mixer */
	if((mixer = open(cfg.mixerdev, O_RDWR)) == -1) {
		fprintf(stderr, "Unable to open mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* read mixer volume */
	if(ioctl(mixer, SOUND_MIXER_READ_VOLUME, &volume) == -1) {
		fprintf(stderr, "Unable to read volume from mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* adjust volume */
	left = (volume & 0xff) + change;
	right = ((volume >> 8) & 0xff) + change;
	if(left < 0) {
		left = 0;
	}
	if(right < 0) {
		right = 0;
	}
	if(left > MAXVOLUME) {
		left = MAXVOLUME;
	}
	if(right > MAXVOLUME) {
		right = MAXVOLUME;
	}
	volume = left | (right << 8);

	/* write volume back to mixer */
	if(ioctl(mixer, SOUND_MIXER_WRITE_VOLUME, &volume) == -1) {
		fprintf(stderr, "Unable to write volume to mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* close mixer device */
	if(close(mixer) == -1 ) {
		fprintf(stderr, "Unable to close mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* calc volume percentage and return it */
	return ((left + right) / 2) * 100 / MAXVOLUME;

} /* }}} */

int apmiser_running(void) { /* {{{ */
	/* code inspired by comp.unix.programmer FAQ */
	char line[133];
	char *linep;
	char *token;
	char *cmd = NULL;
	FILE *fp;

	/* open command like a file */
	fp = popen("ps -e 2>/dev/null", "r");
	if (fp == NULL) {
		return 0;
	}

	/* get header */
	if (fgets(line, sizeof(line), fp) == NULL) {
		pclose(fp);
		return 0;
	}

	/* determine column of command name */
	linep = line;
	while(cmd == NULL)
	{
		if ((token = strtok(linep, " \t\n")) == NULL) {
			pclose(fp);
			return 0;
		}
		linep = NULL;

		if (strcmp("COMMAND", token) == 0 || strcmp("CMD", token) == 0) {
			cmd = token;
		}
	}

	/* check if any of the commands is apmiser */
	while(fgets(line, sizeof line, fp) != NULL) {
		if (strstr(strtok(cmd, " \t\n"), "apmiser") != NULL) {
			pclose(fp);
			return 1;
		}
	}

	pclose(fp);

	return 0;
} /* }}} */

/* vim:set sw=2:set ts=2: */
/* vim600:set fen:set fdm=marker:set fdl=0: */
