/* tpb - program to use the IBM ThinkPad(tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* includes {{{ */
#include <fcntl.h>
#include <linux/soundcard.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <wait.h>
#include "config.h"

#ifdef HAVE_LIBXOSD
#include <xosd.h>
#endif /* HAVE_LIBXOSD */

#include "cfg.h"
#include "tpb.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{*/
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: tpb.c,v 1.11.2.2 2002/09/25 12:07:52 mbr Exp $";
#endif
/* }}} */

extern config cfg;

int main(int argc, char **argv)/*{{{*/
{
#ifdef HAVE_LIBXOSD
	xosd *osdptr=NULL;
#endif /* HAVE_LIBXOSD */
	t_tpb tpbs,ltpbs;
	int vol;
	int mute;
	char *home;
	char *cfg_file;
	FILE *fdsc;

	/* Ignore SIGCHLD, so I don't have to wait for terminated childs (zombies) */
	signal(SIGCHLD,SIG_IGN);

	/* initialize config */
	init_cfg();

	/* read global config file */
	parse_cfg_file(GLOBAL_CONFIG_FILE);

	/* read user config file */
	if((home=getenv("HOME")) != NULL) {
		cfg_file = (char *)malloc(strlen(home)+strlen("/.tpbrc")+1);
		strcpy(cfg_file,home);
		strcat(cfg_file,"/.tpbrc");
		parse_cfg_file(cfg_file);
		free(cfg_file);
	}

	/* parse options */
	parse_option(argc, argv);

#ifdef HAVE_LIBXOSD
	/* initialize osd */
	if(cfg.osd) osdptr=xosd_init(cfg.osdfont, cfg.osdcolor, cfg.osdtimeout, cfg.osdpos, cfg.osdoffset, 0, 2);
#endif /* HAVE_LIBXOSD */

	/* if we use OSS mixer initialize the value of vl */
	if(cfg.mixer) set_vl();

	/* to initialize struct */
	if(get_state(&tpbs)) {
		_exit(1);
	}

	while(1) {
		/* save last state and get new one */
		memcpy(&ltpbs,&tpbs,sizeof(t_tpb));
		if(get_state(&tpbs)) {
			_exit(1);
		}

		/* determine the state of the Thinkpad button  {{{ */
		if((tpbs.tpb != ltpbs.tpb) && (tpbs.ht == ltpbs.ht)) {
			if(cfg.verbose) puts("ThinkPad button pressed");
			if(cfg.tpbcmd!=NULL) {
				if(fork_app(cfg.tpbcmd)) {
					_exit(0);
				}
			}
		} /* }}} */

		/* determine the state of ThinkLight {{{ */
		if(tpbs.kt != ltpbs.kt) {
			if(cfg.verbose) printf("ThinkLight is %s\n", tpbs.kt ? "on" : "off");
		} /* }}} */

		/* determine the state of display  {{{ */
		if(tpbs.ds != ltpbs.ds) {
			if(cfg.verbose) printf("Display changed: LCD %s, CRT %s\n", tpbs.ds & 0x01 ? "on" : "off", tpbs.ds & 0x02 ? "on" : "off");
		} /* }}} */

		/* determine the state of the brightness buttons {{{ */
		if(tpbs.bl != ltpbs.bl) {
			if(cfg.verbose) printf("Brightness changed: Level %d\n", tpbs.bl);
		}
#ifdef HAVE_LIBXOSD
		if(tpbs.bt != ltpbs.bt) {
			if(osdptr) {
				xosd_display(osdptr, 0, XOSD_string, "Brightness");
				xosd_display(osdptr, 1, XOSD_percentage, tpbs.bl*100/7);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the volume buttons {{{ */
		if(tpbs.vl != ltpbs.vl) {
			if(cfg.mixer) {
				vol=change_volume(MAXVOLUME/cfg.mixersteps*(tpbs.vl-ltpbs.vl)); /* tpbs.vl-ltpbs.vl gives the direction */
				set_vl();
				tpbs.vl=ltpbs.vl; /* Do not recognize the change of vl in next iteration */
			}
			else {
				vol=tpbs.vl*100/14;
			}
			if(cfg.verbose) printf("Volume changed: Level %d\n", vol);
		}
#ifdef HAVE_LIBXOSD
    /* show volume every time a volume button is pressed and not mute */
		if(tpbs.vt != ltpbs.vt && !tpbs.mt) {
			if(osdptr) {
				xosd_display(osdptr, 0, XOSD_string, "Volume");
				xosd_display(osdptr, 1, XOSD_percentage, vol);
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of the mute button {{{ */
		if(tpbs.mt != ltpbs.mt) {
			if(cfg.verbose) printf("%s\n", tpbs.mt ? "Mute" : "Unmute");
		}
#ifdef HAVE_LIBXOSD
		if(tpbs.mt != ltpbs.mt || (tpbs.vt != ltpbs.vt && ltpbs.mt)) {
			if(cfg.mixer) {
				if(tpbs.mt) {
					change_volume(-vol); /* mute */
				}
				else {
					change_volume(vol); /* unmute */
				}
			}

			if(osdptr) {
				if(tpbs.mt) {
					xosd_display(osdptr, 0, XOSD_string, "Mute");
					xosd_display(osdptr, 1, XOSD_percentage, 0);
				}
				else {
					xosd_display(osdptr, 0, XOSD_string, "Unmute");
					xosd_display(osdptr, 1, XOSD_percentage, vol);
				}
			}
		}
#endif /* HAVE_LIBXOSD */ /* }}} */

		/* determine the state of power {{{ */
		if(tpbs.ac != ltpbs.ac) {
			if(cfg.verbose) printf("Power line changed: %s\n", tpbs.ac ? "AC disconnected" : "AC connected");
#ifdef HAVE_LIBXOSD
			if(osdptr) {
					xosd_display(osdptr, 0, XOSD_string, tpbs.ac ? "AC disconnected" : "AC connected");
					xosd_display(osdptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode ac {{{ */
		if(tpbs.pma != ltpbs.pma) {
			char *msg;
			switch(tpbs.pma) {
				case 0x4: msg="PM ac high"; break;
				case 0x2: msg="PM ac auto"; break;
				case 0x1: msg="PM ac manual"; break;
				default: msg="PM ac unknown"; break;
			}
			if(cfg.verbose) printf("Power management mode ac changed: %s\n", msg);
#ifdef HAVE_LIBXOSD
			if(osdptr) {
					xosd_display(osdptr, 0, XOSD_string, msg);
					xosd_display(osdptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* determine power management mode battery {{{ */
		if(tpbs.pmb != ltpbs.pmb) {
			char *msg;
			switch(tpbs.pmb) {
				case 0x4: msg="PM battery high"; break;
				case 0x2: msg="PM battery auto"; break;
				case 0x1: msg="PM battery manual"; break;
				default: msg="unknown"; break;
			}
			if(cfg.verbose) printf("Power management mode battery changed: %s\n", msg);
#ifdef HAVE_LIBXOSD
			if(osdptr) {
					xosd_display(osdptr, 0, XOSD_string, msg);
					xosd_display(osdptr, 1, XOSD_string, "");
			}
#endif /* HAVE_LIBXOSD */
		} /* }}} */

		/* sleep for polltime */
		usleep(cfg.polltime);
	}

	_exit(0); /* never reached */
}/*}}}*/

int get_state(t_tpb *tpbs)/*{{{*/
{
	int  fdsc;
	char buffer[114];

	/* open nvram for reading */
	/* must use open/close because seek is not supported by nvram */
	if((fdsc=open(cfg.nvram, O_RDONLY|O_NONBLOCK)) == -1) {
		fprintf(stderr, "Unable to open device %s: ", cfg.nvram);
		perror(NULL);
		return(-1);
	}

	/* read nvram */
	if(read(fdsc,buffer,sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr,"Unable to read from device %s: " ,cfg.nvram);
		perror(NULL);
		return(-1);
	}

	/* close nvram device */
	if(close(fdsc) == -1) {
		fprintf(stderr, "Unable to close device %s: ", cfg.nvram);
		perror(NULL);
		return(-1);
	}

	tpbs->tpb=(buffer[0x57] & 0x08) >> 3;
	tpbs->kt=(buffer[0x58] & 0x10) >> 4;
	tpbs->ht=(buffer[0x58] & 0x01);
	tpbs->ds=(buffer[0x59] & 0x03);
	tpbs->bl=(buffer[0x5E] & 0x07);
	tpbs->bt=(buffer[0x5E] & 0x20) >> 5;
	tpbs->vl=(buffer[0x60] & 0x0f);
	tpbs->vt=(buffer[0x60] & 0x80) >> 7;
	tpbs->mt=(buffer[0x60] & 0x40) >> 6;
	tpbs->ac=(buffer[0x57] & 0x10) >> 4;
	tpbs->pma=(buffer[0x39] & 0x07);
	tpbs->pmb=(buffer[0x39] & 0x38) >> 3;

	return(0);
}/*}}}*/

int fork_app(char * cmd) { /* {{{ */
	//char *args[4];
	//args[0] = "sh";
	//args[1] = "-c";
	//args[2] = cmd;
	//args[3] = 0;

	switch(fork()) {
		case -1:
			return(-1);
			break;
		case 0:
			{
				setsid(); /* children should not be killed if tpb ends */
				//execv("/bin/sh", args);
				system(cmd);
				_exit(0);
				return(127);
			}
			break;
		default:
			return(0);
			break;
	}
	return(-1); /* never reached */
} /* }}} */

void set_vl(void) { /* {{{ */
	int  fdsc;
	char buffer;


	/* open nvram */
	if((fdsc=open("/dev/nvram", O_RDWR|O_NONBLOCK)) == -1) {
		fprintf(stderr, "Unable to open device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* jump to volume section */
	if(lseek(fdsc, 0x60,SEEK_SET) == -1 ) {
		fprintf(stderr,"Unable to seek device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* read nvram */
	if(read(fdsc,&buffer,sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr,"Unable to read from device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	buffer &= 0xf0;
	buffer |= 0x07;

	/* jump to volume section */
	if(lseek(fdsc, 0x60,SEEK_SET) == -1 ) {
		fprintf(stderr,"Unable to seek device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	/* write std value for volume */
	if(write(fdsc,&buffer,sizeof(buffer)) != sizeof(buffer)) {
		fprintf(stderr,"Unable to write to device %s: ", cfg.nvram);
		perror(NULL);
		_exit(1);
	}

	close(fdsc);

	return;

} /* }}} */

int change_volume(int change) { /* {{{ */
	int mixer;
	int volume;
	int left,right;

	/* open mixer */
	if((mixer = open(cfg.mixerdev, O_RDWR)) == -1) {
		fprintf(stderr, "Unable to open mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* read mixer volume */
	if(ioctl(mixer, SOUND_MIXER_READ_VOLUME, &volume) == -1) {
		fprintf(stderr, "Unable to read volume from mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* adjust volume */
	left = (volume & 0xff) + change;
	right = ((volume >> 8) & 0xff) + change;
	if(left < 0) left=0;
	if(right < 0) right=0;
	if(left > MAXVOLUME) left=MAXVOLUME;
	if(right > MAXVOLUME) right=MAXVOLUME;
	volume = left | (right << 8);


	/* write volume back to mixer */
	if(ioctl(mixer, SOUND_MIXER_WRITE_VOLUME, &volume) == -1) {
		fprintf(stderr, "Unable to write volume to mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* close mixer device */
	if(close(mixer) == -1 ) {
		fprintf(stderr, "Unable to close mixer device %s: ", cfg.mixerdev);
		perror(NULL);
		_exit(1);
	}

	/* calc volume percentage and return it */
	return(((left+right)/2)*100/MAXVOLUME);

} /* }}} */

/* vim:set sw=2:set ts=2: */
/* vim600:set fen:set fdm=marker:set fdl=0: */
