/* tpb - program to use the IBM ThinkPad (tm) special keys
 * Copyright (C) 2002 Markus Braun <markus.braun@krawel.de>
 *
 * This file is part of tpb.
 *
 * tpb is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * tpb is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with tpb; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* Meaning of bits in NVRAM of Thinkpad T21 {{{
 *
 * byte 0x57: xxxx xxxx
 *             |   |
 *             |   +-----> toggle if thinkpad button pressed
 *             +---------> toggle if video output is switched with Fn+F7 (LCD/CRT)
 *
 * byte 0x58: xxxx xxxx
 *               |
 *               +-------> keyboard light (0=off, 1=on)
 *
 * byte 0x59: xxxx xxxx
 *                   ||
 *                   ++--> state (1=LCD,2=CRT,3=both)
 *
 * byte 0x5E: xxxx xxxx
 *              |   |||
 *              |   +++--> brightness (0x0-0x7)
 *              +--------> toggle if LCD brightness is changed with Fn+Home and Fn+End
 *
 * byte 0x60: xxxx xxxx
 *            ||   ||||
 *            ||   ++++--> Volume level (0x0-0xE)
 *            ||
 *            |+---------> mute (0=off, 1=on)
 *            +----------> toggle if button (volume up, volume down, mute) pressed
 }}} */

/* includes {{{ */
#include <fcntl.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <wait.h>
#include "tpb.h"
/* }}} */

/* RCS version string for later identification using rcs ident {{{*/
#ifndef lint
static volatile const char *RCSid = "@(#) $Id: tpb.c,v 1.1 2002/06/23 11:07:46 mbr Exp $";
#endif
/* }}} */

int main(int argc, char **argv)/*{{{*/
{
	char verbose=0;
	int polltime=POLLTIME;
  char *tpbcmd=NULL;
	t_tpb tpbs,ltpbs;

	/* Ignore SIGCHLD, so I don't have to wait for terminated childs (zombies) */
	signal(SIGCHLD,SIG_IGN);

	/* parse options {{{ */
	while (1) {

		int option_index = 0;
		char *endptr;
		int c;
		static struct option long_options[] = {
			{"help",     0, NULL, 'h'},
			{"poll",     1, NULL, 'p'},
			{"thinkpad", 1, NULL, 't'},
			{"verbose",  0, NULL, 'v'},
			{NULL,       0, NULL,   0}
		};

		c = getopt_long (argc, argv, "hp:t:v", long_options, &option_index);
		if (c == -1)
			break;

		switch (c) {
			case 'h':
				print_usage(argv[0]);
				return(-1);

			case 'p':
				polltime=strtol(optarg, &endptr, 10);
				if(strlen(endptr)>0) {
					printf("Illegal polltime \"%s\"\n",optarg);
					return(-1);
				}
				break;

			case 't':
				tpbcmd=optarg;
				break;

			case 'v':
				verbose=1;
				break;

			default:
				print_usage(argv[0]);
				return(-1);
		}
	} /* }}}*/

	/* to initialize struct */
	if(get_state(&tpbs)) {
		return(-1);
	}

	while(1) {
		/* save last state and get new one */
		memcpy(&ltpbs,&tpbs,sizeof(t_tpb));
		if(get_state(&tpbs)) {
			return(-1);
		}

		/* determine the state of the Thinkpad button  {{{ */
		if(tpbs.tpb != ltpbs.tpb) {
			if (verbose) puts("ThinkPad button pressed");
			if (tpbcmd!=NULL) {
				switch(fork()) {
					case -1:
						return(-1);
						break;
					case 0:
						{
							char *args[4];
							args[0] = "sh";
							args[1] = "-c";
							args[2] = tpbcmd;
							args[3] = 0;
							setsid(); /* child should not be killed if tpb ends */
							execv("/bin/sh", args);
							return(127);
						}
						break;
					default:
						break;
				}
			}
		} /* }}} */

		/* determine the state of keyboard light  {{{ */
		if(tpbs.kl != ltpbs.kl) {
			if (verbose) printf("Keyboard light is %s\n", tpbs.kl ? "on" : "off");
		} /* }}} */

		/* determine the state of display  {{{ */
		if(tpbs.ds != ltpbs.ds) {
			if (verbose) printf("Display changed: LCD %s, CRT %s\n", tpbs.ds & 0x01 ? "on" : "off", tpbs.ds & 0x02 ? "on" : "off");
		} /* }}} */

		/* determine the state of the brightness buttons {{{ */
		if(tpbs.bl != ltpbs.bl) {
			if (verbose) printf("Brightness changed: Level %d\n", tpbs.bl);
		} /* }}} */

		/* determine the state of the volume buttons {{{ */
		if(tpbs.vl != ltpbs.vl) {
			if (verbose) printf("Volume changed: Level %d\n", tpbs.vl);
		} /* }}} */

		/* determine the state of the mute button {{{ */
		if(tpbs.ms != ltpbs.ms) {
			if (verbose) printf("Muting is %s\n", tpbs.ms ? "on" : "off");
		} /* }}} */

		/* sleep for polltime */
		usleep(polltime);
	}

	return(0); /* never reached */
}/*}}}*/

void print_usage(const char *prog)/*{{{*/
{
	printf("%s 0.1 (C) 2002 Markus Braun [%s]\n\n",prog,RCSid);
	printf("Usage: %s options\n\n",prog);
	printf("options:\n");
	printf("   -h | --help                display this help\n");
	printf("   -p DELAY | --poll DELAY    set delay between polls in microseconds [%d]\n",POLLTIME);
	printf("   -t CMD | --thinkpad CMD    string with command and options that is executed\n");
	printf("                              when ThinkPad button is pressed [none]\n");
	printf("   -v | --verbose             print more information about pressed keys\n");
	return;
}/*}}}*/

int get_state(t_tpb *tpbs)/*{{{*/
{
	int  fdsc;
	char buffer[114];

	/* open nvram for reading */
	/* must use open/close because seek is not supported by nvram */
	fdsc=open("/dev/nvram", O_RDONLY);
	if(fdsc==-1) {
		perror("Unable to open /dev/nvram");
		return(-1);
	}

	/* read nvram */
	if(read(fdsc,buffer,sizeof(buffer)) != sizeof(buffer)) {
		return(-1);
	}

	close(fdsc);

	tpbs->tpb=buffer[0x57] & 0x08;
	tpbs->kl=buffer[0x58] & 0x10;
	tpbs->ds=buffer[0x59] & 0x03;
	tpbs->bl=buffer[0x5E] & 0x07;
	tpbs->vl=buffer[0x60] & 0x0f;
	tpbs->ms=buffer[0x60] & 0x40;

	return(0);
}/*}}}*/
