# test_elliptical_cone.rb
#
# Copyright 2012-2013 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Tovero is free software; you can redistribute it and/or modify it
# under the terms of the GNU Lesser General Public License
# version 2.1 as published by the Free Software Foundation.
#
# Tovero is distributed in the hope that it will be useful, but
# WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
# Lesser General Public License for more details.  You should have
# received a copy of the GNU Lesser General Public License along with
# Tovero. If not, see <http://www.gnu.org/licenses/>.

# Ruby unit test (minitest) for the Elliptical_cone class

require 'minitest/autorun'
require 'test/unit'
require 'libtovero_support_rb_0'
require 'libtovero_math_rb_0'

include Libtovero_support_rb_0
include Libtovero_math_rb_0

class TestEllipticalCone < Test::Unit::TestCase
  def setup
    @Tolerances = GeometricTolerances.new
  end

  #
  # Solid validity checks
  #

  def test_default
    cone = EllipticalCone.new
    error = ErrorParam.new
    #
    assert cone.is_valid?(@Tolerances, error)
  end

  def test_zero_length_base_vector
    cone = EllipticalCone.new
    zero_length_vector = Vector.new
    cone.base_b = zero_length_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_degenerate_base_vectors
    cone = EllipticalCone.new
    small_vector = UnitVector::x * @Tolerances.distance
    cone.base_a = small_vector
    cone.base_b = small_vector
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_non_perpendicular_base_vectors
    cone = EllipticalCone.new
    cone.base_a = UnitVector::x * Distance.new(1.0, Distance::meter)
    cone.base_b = cone.base_a
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_zero_height
    cone = EllipticalCone.new
    cone.name = "cone"
    cone.height = Distance.new(0.0, Distance::meter)
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  def test_negative_ratio
    cone = EllipticalCone.new
    cone.name = "cone"
    cone.ratio = Unitless.new(-1.0)
    error = ErrorParam.new
    #
    refute cone.is_valid?(@Tolerances, error)
    assert (error.base.error_class == 'Math_error') and (error.base.code.equal? MathError::Validation)
  end

  #
  # Miscellaneous checks
  #

  def test_properties
    cone = EllipticalCone.new
    #
    assert_respond_to cone, :base
    assert_respond_to cone, :base=
    assert_respond_to cone, :base_a
    assert_respond_to cone, :base_a=
    assert_respond_to cone, :base_b
    assert_respond_to cone, :base_b=
    assert_respond_to cone, :height
    assert_respond_to cone, :height=
    assert_respond_to cone, :ratio
    assert_respond_to cone, :ratio=
  end

end
