// Units_mapper.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Units_mapper.hpp>
#include <tovero/math/geometry/Angle.hpp>
#include <tovero/math/geometry/Area.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <string>

using std::string;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Units_mapper::Units_mapper()
  : m_length_units("meter"),
    m_area_units("square_meter"),
    m_angle_units("radian")
{
  if (!length_map.size())
  {
    length_map["meter"] = Distance::meter;
    length_map["m"] = Distance::meter;
    length_map["centimeter"] = Distance(0.01 * Distance::meter);
    length_map["cm"] = Distance(0.01 * Distance::meter);
    length_map["millimeter"] = Distance::millimeter;
    length_map["mm"] = Distance::millimeter;
    length_map["foot"] = Distance::foot;
    length_map["ft"] = Distance::foot;
    length_map["inch"] = Distance::inch;
    length_map["in"] = Distance::inch;
  }
  if (!area_map.size())
  {
    area_map["square_meter"] = Area::square_meter;
    area_map["m^2"] = Area::square_meter;
    area_map["square_centimeter"] = (0.01 * Distance::meter) * (0.01 * Distance::meter);
    area_map["cm^2"] =  (0.01 * Distance::meter) * (0.01 * Distance::meter);
    area_map["square_millimeter"] = Area::square_millimeter;
    area_map["mm^2"] = Area::square_millimeter;
    area_map["square_foot"] = Area::square_foot;
    area_map["ft^2"] = Area::square_foot;
    area_map["square_inch"] = Area::square_inch;
    area_map["in^2"] = Area::square_inch;
  }
  if (!angle_map.size())
  {
    angle_map["radian"] = Angle::radian;
    angle_map["degree"] = Angle::degree;
  }
}

//
// Accessors/mutators
//

bool Units_mapper::set_length_units(const string& length_units)
{
  bool return_value = false;

  if (length_map.find(length_units) != length_map.end())
  {
    m_length_units = length_units;
    return_value = true;
  }

  return return_value;
}

bool Units_mapper::set_area_units(const string& area_units)
{
  bool return_value = false;

  if (area_map.find(area_units) != area_map.end())
  {
    m_area_units = area_units;
    return_value = true;
  }

  return return_value;
}

bool Units_mapper::set_angle_units(const string& angle_units)
{
  bool return_value = false;

  if (angle_map.find(angle_units) != angle_map.end())
  {
    m_angle_units = angle_units;
    return_value = true;
  }

  return return_value;
}

//
// Class data members
//

Units_mapper::Length_map_type Units_mapper::length_map;
Units_mapper::Area_map_type Units_mapper::area_map;
Units_mapper::Angle_map_type Units_mapper::angle_map;
