// Unit_vector.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/support/common.hpp>

using namespace Roan_trail::Tovero_math;

//
// Constructors
//

Unit_vector::Unit_vector(const Unitless& x,
                         const Unitless& y,
                         const Unitless& z)
{
  // force arguments to be normalized into a unit vector
  normalize(x,
            y,
            z,
            *this);
}

Unitless Unit_vector::normalize(const Unitless& x,
                                const Unitless& y,
                                const Unitless& z,
                                Unit_vector& return_u)
{
  Unit_vector u;
  Unitless length = sqrt(x * x + y * y + z * z);

  u.m_v[0] = x / length;
  u.m_v[1] = y / length;
  u.m_v[2] = z / length;

  return_u = u;

  return length;
}

//
// Accessors/mutators
//

void Unit_vector::set_at(size_t index, const Unitless& value)
{
  precondition((index >= 0) && (index < 3));

  m_v[index] = value;
  normalize(m_v[0],
            m_v[1],
            m_v[2],
            *this);
}

//
// Class constants
//

//
//   Predefined vectors
//

const Unit_vector Unit_vector::x = Unit_vector(Unitless::from_value(1.0),
                                               Unitless::from_value(0.0),
                                               Unitless::from_value(0.0));
const Unit_vector Unit_vector::y = Unit_vector(Unitless::from_value(0.0),
                                               Unitless::from_value(1.0),
                                               Unitless::from_value(0.0));
const Unit_vector Unit_vector::z = Unit_vector(Unitless::from_value(0.0),
                                               Unitless::from_value(0.0),
                                               Unitless::from_value(1.0));
const Unit_vector Unit_vector::zero = Unit_vector();
