// Solid_operation.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Solid_operation.hpp>
#include <tovero/math/geometry/Solid_combination_base.hpp>
#include <tovero/math/geometry/Solid_operand.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Solid_visitor.hpp>
#include <tovero/support/Reference_counting_base.hpp>
#include <tovero/support/common.hpp>
#include <string>

using std::string;
using namespace Roan_trail::Tovero_math;

//
// Constructor/copy
//

Solid_operation::Solid_operation(Solid_operand& left_operand,
                                 Solid_operand& right_operand,
                                 Solid_operator::Solid_operator_type op,
                                 const string& name)
  : Solid_combination_base(name),
    m_left_operand(&left_operand),
    m_right_operand(&right_operand),
    m_op(op)
{
  m_left_operand->reference();
  m_right_operand->reference();

  postcondition(mf_invariant(false));
}

Solid_operation::Solid_operation(const Solid_operation& o)
  : Solid_combination_base(o)
{
  set_left_operand(*o.m_left_operand);
  set_right_operand(*o.m_right_operand);
  m_op = o.m_op;

  postcondition(mf_invariant(false));
}

Solid_operation& Solid_operation::operator=(const Solid_operation& o)
{
  precondition(mf_invariant(false));

  if (this != &o)
  {
    Solid_combination_base::operator=(o);
    set_left_operand(*o.m_left_operand);
    set_right_operand(*o.m_right_operand);
    m_op = o.m_op;
  }

  postcondition(mf_invariant(false));
  return *this;
}

//
// Accessors/mutators
//

void Solid_operation::set_left_operand(Solid_operand& left_operand)
{
  precondition(mf_invariant());

  if (&left_operand != m_left_operand)
  {
    left_operand.reference();
    m_left_operand->unreference();
    m_left_operand = &left_operand;
  }

  postcondition(mf_invariant());
}

void Solid_operation::set_right_operand(Solid_operand& right_operand)
{
  precondition(mf_invariant());

  if (&right_operand != m_right_operand)
  {
    right_operand.reference();
    m_right_operand->unreference();
    m_right_operand = &right_operand;
  }

  postcondition(mf_invariant());
}

//
// Solid_visitor
//

Solid_visitor::Visit_result Solid_operation::accept(Solid_visitor& visitor) const
{
  precondition(mf_invariant());

  Solid_visitor::Visit_result result = visitor.visit_enter(*this);
  if (Solid_visitor::success != result)
  {
    goto exit_point;
  }

  result = m_left_operand->solid().accept(visitor);
  if (Solid_visitor::success != result)
  {
    goto exit_point;
  }

  result = m_right_operand->solid().accept(visitor);
  if (Solid_visitor::success != result)
  {
    goto exit_point;
  }

  result = visitor.visit_exit(*this);

 exit_point:
  return result;
}

//
// Protected member functions
//

//
//   Destructor
//

Solid_operation::~Solid_operation()
{
  precondition(mf_invariant(false));

  m_right_operand->unreference();
  m_left_operand->unreference();
}

//
//   Class invariant
//

bool Solid_operation::mf_invariant(bool check_base_class) const
{
  static_cast<void>(check_base_class); // avoid unused warning

  bool return_value = false;

  if (!m_left_operand
      || !m_right_operand)
  {
    goto exit_point;
  }

  return_value = (!check_base_class || Reference_counting_base::mf_invariant(check_base_class));
  goto exit_point;

 exit_point:
  return return_value;
}
