// Oblique_cylinder.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Oblique_cylinder.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Geometric_tolerances.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Oblique_cylinder::Oblique_cylinder(const string& name)
  : General_cone_base(name),
    m_base(Point::O),
    m_height(Unit_vector::z * Distance::meter),
    m_base_r(Unit_vector::x * Distance::meter)
{
}

Oblique_cylinder::Oblique_cylinder(const Point& base,
                                   const Vector& height,
                                   const Vector& base_r,
                                   const string& name)
  : General_cone_base(name),
    m_base(base),
    m_height(height),
    m_base_r(base_r)
{
}

//
// Predicates
//

bool Oblique_cylinder::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for non-zero height and base r vectors
  if (tolerances.is_distance_near_zero(m_height.length())
      || tolerances.is_distance_near_zero(m_base_r.length()))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height or base r vectors are non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for non-parallel height and base r vectors
  Unit_vector unit_h;
  m_height.normalize(unit_h);
  Unit_vector unit_r;
  m_base_r.normalize(unit_r);
  if (tolerances.is_cosine_near_zero(Unitless(1.0 - fabs(unit_h.dot(unit_r).value()))))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " + solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height and base r vectors are parallel";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

General_cone& Oblique_cylinder::generalize() const
{
  Unit_vector unit_h;
  m_height.normalize(unit_h);
  Unit_vector unit_r;
  const Distance base_r_length = m_base_r.normalize(unit_r);
  Vector base_b = unit_r.cross(unit_h) * base_r_length;

  General_cone* cone = new General_cone(m_base,
                                        m_height,
                                        m_base_r,
                                        base_b,
                                        m_base_r,
                                        base_b,
                                        name());

  return *cone;
}

Solid& Oblique_cylinder::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>(this));

  Unit_vector unit_h;
  m_height.normalize(unit_h);
  Unit_vector unit_r;
  const Distance base_r_length = m_base_r.normalize(unit_r);
  // check for height perpendicular to radius
  if (tolerances.is_cosine_near_zero(unit_r.dot(unit_h)))
  {
    // right circular cylinder
    Cylinder* cylinder = new Cylinder(m_base,
                                      m_height,
                                      base_r_length,
                                      name());
    return_value = cylinder;
  }

  return *return_value;
}
