// General_cone.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

// General conical (cylindrical) solid.  The perpendicular base a and
// base b vectors are added to the base center point to form the
// elliptical base.  The height vector is added to the base center
// point, and along with the top c and top d vectors, forms the
// elliptical top.  Base a and top c point in the same direction, but
// can have different magnitudes.  The same is true for base b and top
// d.

#ifndef TOVERO_MATH_GENERAL_CONE_HPP_
#define TOVERO_MATH_GENERAL_CONE_HPP_

#include <tovero/math/geometry/General_cone_base.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <string>

namespace Roan_trail
{
  namespace Tovero_math
  {
    class Geometric_tolerances;

    class General_cone : public General_cone_base
    {
    public:
      // constructor/copy
      General_cone(const std::string& name = "");
      General_cone(const Point& base,
                   const Vector& height,
                   const Vector& base_a,
                   const Vector& base_b,
                   const Vector& top_c,
                   const Vector& top_d,
                   const std::string& name = "");
      virtual Solid* clone_solid() const { return new General_cone(*this); }
      //   Note: let compiler generate default copy constructor and operator=
      // accessors/mutators
      const Point& base() const { return m_base; }
      void set_base(const Point& base) { m_base = base; }
      const Vector& height() const { return m_height; }
      void set_height(const Vector& height) { m_height = height; }
      const Vector& base_a() const { return m_base_a; }
      void set_base_a(const Vector& base_a) { m_base_a = base_a; }
      const Vector& base_b() const { return m_base_b; }
      void set_base_b(const Vector& base_b) { m_base_b = base_b; }
      const Vector& top_c() const { return m_top_c; }
      void set_top_c(const Vector& top_c) { m_top_c = top_c; }
      const Vector& top_d() const { return m_top_d; }
      void set_top_d(const Vector& top_d) { m_top_d = top_d; }
      // predicates
      virtual bool is_valid(const Geometric_tolerances& tolerances,
                            Roan_trail::Tovero_support::Error_param& return_error) const;
      // visitor
      virtual Solid_visitor::Visit_result accept(Solid_visitor& visitor) const { return visitor.visit(*this); }
      // other
      virtual std::string solid_class() const { return "General_cone"; }
      virtual General_cone& generalize() const { return *const_cast<General_cone*>(this); }
      virtual Solid& specialize(const Geometric_tolerances& tolerances) const;
    protected:
      // destructor
      virtual ~General_cone() {}
    private:
      // solid parameters
      Point m_base; // base center point
      Vector m_height;
      Vector m_base_a;
      Vector m_base_b;
      Vector m_top_c;
      Vector m_top_d;
    };
  }
}

#endif // TOVERO_MATH_GENERAL_CONE_HPP_
