// Cone.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#include <tovero/math/geometry/Cone.hpp>
#include <tovero/math/geometry/Cylinder.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/math/geometry/Point.hpp>
#include <tovero/math/geometry/Unit_vector.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <tovero/math/geometry/Vector.hpp>
#include <tovero/support/common.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Math_error.hpp>
#include <sstream>
#include <string>

using std::string;
using std::stringstream;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Math_error;
using namespace Roan_trail::Tovero_math;

//
// Constructor
//

Cone::Cone(const string& name)
  : General_cone_base(name),
    m_base(Point::O),
    m_height(Unit_vector::z * Distance::meter),
    m_base_radius(1.0 * Distance::meter),
    m_top_radius(1.0 * Distance::meter)
{
}

Cone::Cone(const Point& base,
           const Vector& height,
           const Distance& base_radius,
           const Distance& top_radius,
           const string& name)
  : General_cone_base(name),
    m_base(base),
    m_height(height),
    m_base_radius(base_radius),
    m_top_radius(top_radius)
{
}

//
// Predicates
//

bool Cone::is_valid(const Geometric_tolerances& tolerances, Error_param& return_error) const
{
  precondition(!return_error());

  bool return_value = false;

  start_error_block();

  // check for non-zero height vector
  if (tolerances.is_distance_near_zero(m_height.length()))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " << solid_class();
    diagnostic_stream << ((solid_name == "") ? "" : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": the height vector is zero";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  // check for positive radii
  if (tolerances.is_distance_near_zero(m_base_radius)
      || (m_base_radius < Distance(0.0 * Distance::meter))
      || tolerances.is_distance_near_zero(m_top_radius)
      || (m_top_radius < Distance(0.0 * Distance::meter)))
  {
    const string& solid_name = name();
    stringstream diagnostic_stream;
    diagnostic_stream << "Invalid " + solid_class();
    diagnostic_stream << ((solid_name == "") ? string("") : (string(" (") + solid_name + string(")")));
    diagnostic_stream << ": one or both of the radii are non-positive";
    on_error(true, new Math_error(error_location(),
                                  Math_error::validation,
                                  diagnostic_stream.str()));
  }

  return_value = true;
  goto exit_point;

  end_error_block();

  default_error_handler_and_cleanup(return_error,
                                    return_value,
                                    false);

 exit_point:
  postcondition(return_error.is_valid_at_return(return_value));
  return return_value;
}

//
// Other
//

General_cone& Cone::generalize() const
{
  Unit_vector unit_h;
  m_height.normalize(unit_h);
  Unit_vector unit_a = unit_h.perpendicular();
  Unit_vector unit_b = unit_a.cross(unit_h);
  Vector base_a = unit_a * m_base_radius;
  Vector base_b = unit_b * m_base_radius;
  Vector base_c = unit_a * m_top_radius;
  Vector base_d = unit_b * m_top_radius;

  General_cone* cone = new General_cone(m_base,
                                        m_height,
                                        base_a,
                                        base_b,
                                        base_c,
                                        base_d,
                                        name());
  return *cone;
}

Solid& Cone::specialize(const Geometric_tolerances& tolerances) const
{
  Solid* return_value = const_cast<Solid*>(static_cast<const Solid*>((this)));

  // check for equal base and top radii
  if (tolerances.is_distance_near_zero(m_top_radius - m_base_radius))
  {
    // right circular cylinder
    return_value = new Cylinder(m_base,
                                m_height,
                                m_base_radius,
                                name());
  }

  return *return_value;
}
