#!/usr/bin/ruby
#
# kinetophone.rb
#
# Copyright 2012-2013 Roan Trail, Inc.
#
# This file is part of Tovero.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
#   (1) Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
#   (2) Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in
#   the documentation and/or other materials provided with the
#   distribution.
#
#   (3) The name of the author may not be used to
#   endorse or promote products derived from this software without
#   specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
# IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
# DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
# INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
# (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
# SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
# STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
# IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
# POSSIBILITY OF SUCH DAMAGE.

#
# Command line argument processing first
#

mode = ARGV[0]

if mode == 'edit'
  raytrace = false
elsif mode == 'render' or mode == nil
  raytrace = true
else
  abort "error, if an argument is supplied, it must be edit or render"
end

# These load the extension libraries
require 'libtovero_support_rb_0'
require 'libtovero_math_rb_0'
require 'libtovero_graphics_rb_0'

# These are like "using namespace" in C++
include Libtovero_support_rb_0
include Libtovero_math_rb_0
include Libtovero_graphics_rb_0

m = Distance::meter
x_axis = UnitVector.new(Unitless.new(1.0), Unitless.new(0.0), Unitless.new(0.0))
torus_base = Point.new(0.0, 0.0, 5.0, m)

# Note: consistently setting the name of graphical entities makes
# it easier to work with the model in BRL-CAD, if necessary
base = Cylinder.new(Point.new,
                    Vector.new(0.0, 0.0, 2.0, m),
                    Distance.new(1.3, m),
                    "base.s")

sound_tor1 = Torus.new(torus_base,
                       x_axis,
                       Distance.new(3.0, m),
                       Distance.new(0.2, m),
                       "sound_tor1.s")

sound_tor2 = Torus.new(torus_base,
                       x_axis,
                       Distance.new(2.0, m),
                       Distance.new(0.2, m),
                       "sound_tor2.s")

sound_tor3 = Torus.new(torus_base,
                       x_axis,
                       Distance.new(1.0, m),
                       Distance.new(0.2, m),
                       "sound_tor3.s")

sound_cone = Cone.new(torus_base,
                      Vector.new(0.0, 0.0, 4.0, m),
                      Distance.new(0.1, m),
                      Distance.new(5.0, m),
                      "outer_cone.s")

sound_lines = SolidCombination.new;
sound_lines.name = "sound_lines.r"
sound_lines << sound_tor1
sound_lines << SolidMember.new(sound_cone, SolidOperator::Intersection_op)
sound_lines << sound_tor2
sound_lines << SolidMember.new(sound_cone, SolidOperator::Intersection_op)
sound_lines << sound_tor3
sound_lines << SolidMember.new(sound_cone, SolidOperator::Intersection_op)

outer_cone = Cone.new(Point.new(0.0, 0.0, 2.0, m),
                      Vector.new(0.0, 0.0, 2.0, m),
                      Distance.new(1.3, m),
                      Distance.new(2.3, m),
                      "outer_cone.s")

inner_cone = Cone.new(Point.new(0.0, 0.0, 2.1, m),
                      Vector.new(0.0, 0.0, 2.0, m),
                      Distance.new(1.2382, m),
                      Distance.new(2.2382, m),
                      "inner_cone.s")

shader = PhongShader::plastic
shader.shine = 5
color = Color.new(224, 212, 190, 0)
attributes = CombinationAttributes.new
attributes.shader = shader
attributes.color = color

speaker = Combination.new(attributes);
speaker.name = "speaker.r"
speaker << sound_lines
speaker << base
speaker << outer_cone
speaker << SolidMember.new(inner_cone, SolidOperator::Difference_op)

BC_DB = BCDatabase.new;
solid_list = BC_DB.top_solids;
# TODO: fix syntax (use << operator?)
solid_list << speaker

database_file = "kinetophone.g"

error = ErrorParam.new
success = BC_DB.write(database_file, true, error)
if (not success)
  puts("Could not write #{database_file}:")
  puts error.base.to_s
else
  if raytrace
    # display raytraced model
    #   options to "rt":
    #     -R [do not report overlaps]
    #     -l [lighting model number (7 = photon mapping)]
    #     -C [background color (R/G/B)]
    #     -w [width in pixels of rendering]
    #     -n [height in pixels (number of lines) of rendering]
    #     -M [read model/view info from stdin]
    #     <first positional arg> [database]
    #     <second positional arg> [entity in database to trace]
    #     (model/view info can be saved in mged with the "saveview" command)
    command = "rt -R -l7 -C86/59/39 -w1024 -n1024 -M #{database_file} speaker.r <<EOF\n\
            viewsize 20000;\n\
            orientation 0.0491 -0.564 -0.0743 0.821;\n\
            eye_pt -32300 110 14900 ;\n\
            start 0; clean; end;\n\
            EOF"
  else
    # [or] load the interactive editor
    command = "mged #{database_file}"
  end

  %x[#{command}]
end
