// Test_BC.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
//   (1) Redistributions of source code must retain the above copyright
//   notice, this list of conditions and the following disclaimer.
//
//   (2) Redistributions in binary form must reproduce the above copyright
//   notice, this list of conditions and the following disclaimer in
//   the documentation and/or other materials provided with the
//   distribution.
//
//   (3) The name of the author may not be used to
//   endorse or promote products derived from this software without
//   specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
// IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
// INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
// HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
// STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
// IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
// POSSIBILITY OF SUCH DAMAGE.

#include <tovero/graphics/brlcad/BC_database.hpp>
#include <tovero/graphics/base/Database.hpp>
#include <tovero/graphics/base/Color.hpp>
#include <tovero/graphics/base/Combination.hpp>
#include <tovero/graphics/base/Combination_attributes.hpp>
#include <tovero/graphics/base/Phong_shader.hpp>
#include <tovero/graphics/base/Toon_shader.hpp>
#include <tovero/support/error/Error.hpp>
#include <tovero/support/error/Graphics_error.hpp>
#include <tovero/math/geometry/Distance.hpp>
#include <tovero/math/geometry/Ellipsoid.hpp>
#include <tovero/math/geometry/Polyhedron.hpp>
#include <tovero/math/geometry/Solid.hpp>
#include <tovero/math/geometry/Solid_member.hpp>
#include <tovero/math/geometry/Solid_operator.hpp>
#include <tovero/math/geometry/Sphere.hpp>
#include <tovero/math/geometry/General_cone.hpp>
#include <tovero/support/Reference_counting_list.hpp>
#include <tovero/support/common.hpp>
#include <iostream>
#include <string>
#include <vector>
#include <cstddef>

using std::cout;
using std::endl;
using std::string;
using std::vector;
using Roan_trail::Tovero_support::Error;
using Roan_trail::Tovero_support::Error_param;
using Roan_trail::Tovero_support::Reference_counting_list;
using Roan_trail::Tovero_math::Distance;
using Roan_trail::Tovero_math::Ellipsoid;
using Roan_trail::Tovero_math::General_cone;
using Roan_trail::Tovero_math::Polyhedron;
using Roan_trail::Tovero_math::Point;
using Roan_trail::Tovero_math::Solid;
using Roan_trail::Tovero_math::Solid_member;
using Roan_trail::Tovero_math::Solid_operator;
using Roan_trail::Tovero_math::Sphere;

using namespace Roan_trail::Tovero_graphics;

int test1();
int test2();
int test_arb();

int main(int argc, char* argv[])
{
  int result = test1();
  // int result = test2();
  // int result = test_arb();

  return result;
}

int test1()
{
  int return_value = 1;

  BC_database BC_DB("test database");

  // create attributes
  Phong_shader* shader = &Phong_shader::plastic();
  Color* color = new Color(0,
                           0,
                           255,
                           0);
  Combination_attributes* attributes = new Combination_attributes;
  attributes->set_shader(shader);
  attributes->set_color(color);
  // create geometry
  Combination* combo = new Combination(attributes);
  combo->set_name("combo");
  Ellipsoid* ellipsoid = new Ellipsoid;
  ellipsoid->set_name("ellipsoid");
  combo->add_member(*(new Solid_member(*ellipsoid)));
  General_cone* cone = new General_cone;
  cone->set_base(Point(0.5 * Distance::meter,
                       0.0 * Distance::meter,
                       0.0 * Distance::meter));
  cone->set_name("cone");
  combo->add_member(*(new Solid_member(*cone, Solid_operator::difference_op)));

  Reference_counting_list<Solid>& out_solids = BC_DB.top_solids();
  out_solids.push_back(*combo);

  // write it out to a BRL-CAD binary database
  {
    cout << "Writing to test.g...";

    Error_param error;
    const bool wrote_DB = BC_DB.write("test.g",
                                      true,
                                      error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  // read the previously written BRL-CAD binary database back in
  {
    cout << "Reading BRL-CAD database back...";

    BC_database BC_DB_out;

    Error_param error;
    vector<string> objects;

    const bool did_read = BC_DB_out.read("test.g",
                                         objects,
                                         error);
    if (!did_read)
    {
      cout << endl << "Error reading test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << "Dump: " << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;

    // write it back out to a BRL-CAD binary database to complete the "round trip"
    cout << "Writing back to test_out.g...";
    const bool wrote_DB = BC_DB_out.write("test_out.g",
                                          true,
                                          error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test_out.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  return_value = 0;

 exit_point:
  return return_value;
}

int test2()
{
  int return_value = 1;

  BC_database BC_DB("test database");

  // create attributes
  Toon_shader* shader = new Toon_shader;
  Color* color = new Color(0,
                           0,
                           255,
                           0);
  Combination_attributes* attributes = new Combination_attributes;
  attributes->set_shader(shader);
  attributes->set_color(color);
  // create geometry
  Combination* combo = new Combination(attributes);
  combo->set_name("combo");
  Sphere* sphere = new Sphere;
  sphere->set_name("sphere");
  combo->add_member(*(new Solid_member(*sphere)));

  Reference_counting_list<Solid>& out_solids = BC_DB.top_solids();
  out_solids.push_back(*combo);

  // write it out to a BRL-CAD binary database
  {
    cout << "Writing to test.g...";

    Error_param error;
    const bool wrote_DB = BC_DB.write("test.g",
                                      true,
                                      error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  // read the previously written BRL-CAD binary database back in
  {
    cout << "Reading BRL-CAD database back...";

    BC_database BC_DB_out;

    Error_param error;
    vector<string> objects;

    const bool did_read = BC_DB_out.read("test.g",
                                         objects,
                                         error);
    if (!did_read)
    {
      cout << endl << "Error reading test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << "Dump: " << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;

    // write it back out to a BRL-CAD binary database to complete the "round trip"
    cout << "Writing back to test_out.g...";
    const bool wrote_DB = BC_DB_out.write("test_out.g",
                                          true,
                                          error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test_out.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  return_value = 0;

 exit_point:
  return return_value;
}

int test_arb()
{
  int return_value = 1;

  BC_database BC_DB("test database");

  // create attributes
  Phong_shader* shader = &Phong_shader::plastic();
  Color* color = new Color(0,
                           0,
                           255,
                           0);
  Combination_attributes* attributes = new Combination_attributes;
  attributes->set_shader(shader);
  attributes->set_color(color);
  // create geometry
  Combination* combo = new Combination(attributes);
  combo->set_name("combo");
  Polyhedron* polyhedron = new Polyhedron;
  polyhedron->set_name("arb8");
  combo->add_member(*(new Solid_member(*polyhedron)));

  Reference_counting_list<Solid>& out_solids = BC_DB.top_solids();

  // write it out to a BRL-CAD binary database
  {
    cout << "Writing to test.g...";

    out_solids.push_back(*combo);

    Error_param error;
    const bool wrote_DB = BC_DB.write("test.g",
                                      true,
                                      error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  // read the previously written BRL-CAD binary database back in
  {
    cout << "Reading BRL-CAD database back...";

    BC_database BC_DB_out;

    Error_param error;
    vector<string> objects;

    const bool did_read = BC_DB_out.read("test.g",
                                         objects,
                                         error);
    if (!did_read)
    {
      cout << endl << "Error reading test.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << "Dump: " << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;

    // write it back out to a BRL-CAD binary database to complete the "round trip"
    cout << "Writing back to test_out.g...";
    const bool wrote_DB = BC_DB_out.write("test_out.g",
                                          true,
                                          error);
    if (!wrote_DB)
    {
      cout << endl << "Error writing test_out.g" << endl;
      cout << "Diagnostic: " << error()->error_dictionary()[Error::diagnostic_error_key] << endl;
      cout << *error() << endl;
      goto exit_point;
    }

    cout << "done." << endl;
  }

  return_value = 0;

 exit_point:
  return return_value;
}
