// Node_mapper.hpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.

#ifndef TOVERO_GRAPHICS_NODE_MAPPER_HPP_
#define TOVERO_GRAPHICS_NODE_MAPPER_HPP_

#include <tovero/support/common.hpp>

#include <sstream>

#include <boost/bimap/bimap.hpp>
#include <boost/bimap/set_of.hpp>
// TODO: concept checks for template
#include <boost/concept/assert.hpp>
#include <boost/concept_check.hpp>
#include <map>
#include <string>

namespace Roan_trail
{
  namespace Tovero_graphics
  {
    template <class Node_type> class Node_mapper
    {
    public:
      // constructor/destructor/initialization
      Node_mapper();
      virtual ~Node_mapper();
      // typedefs
      typedef boost::bimaps::bimap<boost::bimaps::set_of<Node_type>,
                                   boost::bimaps::set_of<std::string> > Node_name_map_type;
      //
      void reset();
      const Node_name_map_type& nodes() const { return m_node_name_map; }
      // returns true if newly added, false if already exists
      bool add_node(Node_type node,
                    const std::string& default_node_name,
                    std::string& return_node_name);
      // look up a previously added node, returns whether or not found
      bool find_node_name(Node_type node, std::string& return_node_name) const;
      Node_type find_node_by_name(const std::string& node_name) const;
      // assign unique names to every node in the database
      static void name_database(Node_type database);
    private:
      Node_name_map_type m_node_name_map;
      std::map<std::string, int> m_prefix_counts;
      //
      void mf_make_unique_node_name(Node_type node,
                                    const std::string& node_name,
                                    const std::string& default_node_prefix,
                                    std::string& return_node_name);
      // prevent compiler from generating
      Node_mapper(const Node_mapper& mapper);
      Node_mapper& operator=(const Node_mapper& mapper);
    };

    //
    // Template implementation
    //

    //
    // Constructor/destructor/initialization
    //

    template <class Node_type> Node_mapper<Node_type>::Node_mapper()
    {
    }

    template <class Node_type> Node_mapper<Node_type>::~Node_mapper()
    {
    }

    //
    // Public member functions
    //

    template <class Node_type> void Node_mapper<Node_type>::reset()
    {
      m_node_name_map.clear();
      m_prefix_counts.clear();
    }

    template <class Node_type> bool Node_mapper<Node_type>::add_node(Node_type node,
                                                                     const std::string& default_node_name,
                                                                     std::string& return_name)
    {
      precondition(node);

      bool return_value = false;

      std::string node_name;

      typename Node_name_map_type::left_map::const_iterator node_p = m_node_name_map.left.find(node);
      if (node_p == m_node_name_map.left.end())
      {
        // first reference to this node
        // try the object's internal node name first
        node_name = node->name();
        mf_make_unique_node_name(node,
                                 node_name,
                                 default_node_name,
                                 node_name);
        m_node_name_map.insert(typename Node_name_map_type::value_type(node, node_name));
        return_value = true;
      }
      else
      {
        // node has already been added, return the name
        node_name = node_p->second;
        return_value = false;
      }

      return_name = node_name;
      goto exit_point;

    exit_point:
      postcondition(("" != return_name));
      return return_value;
    }

    template <class Node_type> bool Node_mapper<Node_type>::find_node_name(Node_type node,
                                                                           std::string& return_node_name) const
    {
      precondition(node);

      bool return_value = false;
      typename Node_name_map_type::left_map::const_iterator c = m_node_name_map.left.find(node);
      if (c == m_node_name_map.left.end())
      {
        // not found
        goto exit_point;
      }

      return_node_name = c->second;
      return_value = true;

    exit_point:
      return return_value;
    }

    template <class Node_type> Node_type
    Node_mapper<Node_type>::find_node_by_name(const std::string& node_name) const
    {
      precondition(("" != node_name));

      Node_type return_value = 0;
      typename Node_name_map_type::right_map::const_iterator c = m_node_name_map.right.find(node_name);
      if (c == m_node_name_map.right.end())
      {
        // not found
        goto exit_point;
      }

      return_value = c->second;

    exit_point:
      return return_value;
    }

    //
    // Private member functions
    //

    template <class Node_type> void
    Node_mapper<Node_type>::mf_make_unique_node_name(Node_type node,
                                                     const std::string& node_name,
                                                     const std::string& default_node_prefix,
                                                     std::string& return_node_name)
    {
      precondition(node);

      std::string working_node_prefix = node_name;
      const bool use_default = ("" == working_node_prefix);

      if (use_default)
      {
        if ("" == default_node_prefix)
        {
          // set working prefix to "Node";
          working_node_prefix = "Node";
        }
        else
        {
          // use supplied prefix
          working_node_prefix = default_node_prefix;
        }
      }

      std::stringstream s;
      while (true)
      {
        s.str("");
        int count = m_prefix_counts[working_node_prefix];
        assert((count < 10000000) && "error, count is unreasonable for unique node name");
        if (use_default)
        {
          // when using the default name always append a number
          // starting with 1
          if (0 == count)
          {
            count = 1;
          }
          s << working_node_prefix << "_" << count;
        }
        else
        {
          // when a non-blank node name is supplied,
          // only append a number if a duplicate name is found
          if (0 == count)
          {
            s << working_node_prefix;
          }
          else
          {
            s << working_node_prefix << "_" << count;
          }
        }
        m_prefix_counts[working_node_prefix] = count + 1;
        // see if the name is taken
        typename Node_name_map_type::right_map::const_iterator name_p = m_node_name_map.right.find(s.str());
        if (name_p == m_node_name_map.right.end())
        {
          // the name is unique, return it;
          return_node_name = s.str();
          break;
        }
      }
      postcondition(return_node_name != "");
    }
  }
}

#endif // TOVERO_GRAPHICS_NODE_MAPPER_HPP_
