// Cook_shader.cpp
//
// Copyright 2012-2013 Roan Trail, Inc.
//
// This file is part of Tovero.
//
// Tovero is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License
// version 2.1 as published by the Free Software Foundation.
//
// Tovero is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
// Lesser General Public License for more details.  You should have
// received a copy of the GNU Lesser General Public License along with
// Tovero. If not, see <http://www.gnu.org/licenses/>.
//
// Based on: ./src/liboptical/sh_cook.c from BRL-CAD (version 7.20.4)
// source:
//
//   Copyright (c) 1985-2011 United States Government as represented by
//   the U.S. Army Research Laboratory.
//
//   This library is free software; you can redistribute it and/or
//   modify it under the terms of the GNU Lesser General Public License
//   version 2.1 as published by the Free Software Foundation.
//
//   This library is distributed in the hope that it will be useful, but
//   WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
//   Lesser General Public License for more details.
//
//   You should have received a copy of the GNU Lesser General Public
//   License along with this file; see the file named COPYING for more
//   information.

#include <tovero/graphics/base/Cook_shader.hpp>
#include <tovero/math/geometry/Unitless.hpp>
#include <string>

using std::string;
using Roan_trail::Tovero_math::Unitless;
using namespace Roan_trail::Tovero_graphics;

//
// Constructors/copy
//

Cook_shader::Cook_shader(const Unitless& slope,
                         int shine,
                         const Unitless& specular,
                         const Unitless& diffuse,
                         const Unitless& transmitted,
                         const Unitless& reflected,
                         const Unitless& refractive_index,
                         const Unitless& extinction,
                         const string& name)
  : Shader(name),
    m_slope(slope),
    m_shine(shine),
    m_specular(specular),
    m_diffuse(diffuse),
    m_transmitted(transmitted),
    m_reflected(reflected),
    m_refractive_index(refractive_index),
    m_extinction(extinction)
{
}

Cook_shader::Cook_shader(const string& name)
  : Shader(name),
    m_slope(Unitless(0.2)),
    m_shine(10),
    m_specular(Unitless(0.7)),
    m_diffuse(Unitless(0.3)),
    m_transmitted(Unitless(0.0)),
    m_reflected(Unitless(0.0)),
    m_refractive_index(Unitless(1.0)), // air
    m_extinction(Unitless(0.0))
{
}

Cook_shader::Cook_shader(const Cook_shader& shader)
  : Shader(shader)
{
  m_slope = shader.m_slope;
  m_shine = shader.m_shine;
  m_specular = shader.m_specular;
  m_diffuse = shader.m_diffuse;
  m_transmitted = shader.m_transmitted;
  m_reflected = shader.m_reflected;
  m_refractive_index = shader.m_refractive_index;
  m_extinction = shader.m_extinction;
}

Cook_shader& Cook_shader::operator=(const Cook_shader& shader)
{
  if (this != &shader)
  {
    m_slope = shader.m_slope;
    m_shine = shader.m_shine;
    m_specular = shader.m_specular;
    m_diffuse = shader.m_diffuse;
    m_transmitted = shader.m_transmitted;
    m_reflected = shader.m_reflected;
    m_refractive_index = shader.m_refractive_index;
    m_extinction = shader.m_extinction;

    Shader::operator=(shader);
  }

  return *this;
}

//
// Shader presets
//

Cook_shader& Cook_shader::cook()
{
  return *(new Cook_shader);
}

Cook_shader& Cook_shader::mirror()
{

  return *(new Cook_shader(Unitless(0.2),   // slope
                           4,               // shine
                           Unitless(0.6),   // specular
                           Unitless(0.4),   // diffuse
                           Unitless(0.0),   // transmitted
                           Unitless(0.75),  // reflected
                           Unitless(1.65),  // refractive index
                           Unitless(0.0))); // extinction
}

Cook_shader& Cook_shader::glass()
{
  return *(new Cook_shader(Unitless(0.2),   // slope
                           4,               // shine
                           Unitless(0.7),   // specular
                           Unitless(0.3),   // diffuse
                           Unitless(0.8),   // transmitted
                           Unitless(0.1),   // reflected
                           Unitless(1.65),  // refractive index
                           Unitless(0.0))); // extinction
}
