# File usr/patch.n
# July 2005

# Copyright (C) 2005  D. R. Williamson

/*
   Words for obtaining and applying difference patches to files, using
   the Unix functions diff and patch.

------------------------------------------------------------------------

   This file serves as a model for using the infix parser.

   When this file was created, infix functions that loop (functions 
   for() and while()) had not been developed, so postfix DO ... LOOP 
   and BEGIN ... WHILE ... REPEAT are used.  

   As with most postfix words, they work under infix by appending an 
   argument list of the names of items that would normally be on the 
   postfix stack before the word runs.  So the postfix form,

      10 1 DO ... LOOP

   where DO consumes stack items 10 and 1, becomes for infix:

      DO(10,1); ... LOOP;

   This file runs with zero- and one-based indexing.

------------------------------------------------------------------------

   Difference and patch example usage:

      >> psource("patch.n");
      DIRS="src/";
      pnew="/tops_tar/tops-20051213/tops/" + DIRS;
      pold="/tops_tar/tops-20050630/tops/" + DIRS;

      T=diffs(pold,pnew);  # get all the patches (run eview(T) to see)

      (f)=patch(T,2,pold); # apply the 2nd group of patches in T
      patches(T,pold);     # apply all the patches in T to files in pold

      pnew_dir="/tops_tar/tops-20051213/tops/";
      pold_dir="/tops_tar/tops-20050630/tops/"; 
      DIRS="src/";
      DIFFS(DIRS, pold_dir, pnew_dir); # get all the differences
      PATCHES("patch.txt", pold_dir);  # apply all the differences

------------------------------------------------------------------------

   Latest run:
      Machine A is the source of patches.  

      Copy this file, patch.n, to usr/patch_prev.n, then edit patch.n
      eariler and later dates below to make the latest patch for 
      machine B.

      Making tar file on machine A:
         cd /opt; tartops; 
         DNLOAD=tops-`date "+%Y%m%d"`.tgz; gpg -b $DNLOAD; ll tops-*

      Patch tops-20060402_patch.tgz against tops-20061210.tgz

      1. On machine A, copy tops-20060402_patch.tgz into directory
      /tops_tar/tops-20060402/.
         > cp tops-20060402_patch.tgz /tops_tar/tops-20060402/

      Inside /tops_tar/tops-20060402, untar tops-20060402_patch.tgz
      and run file /opt/addons_20061210 to touch new files not in the 
      tar file:
         > cd /tops_tar/tops-20060402
         > cp /opt/addons_20061210 /tops_tar/tops-20060402/addons
         > remove all files; MAKE SURE WHERE YOU ARE: #/bin/rm -r tops
         > tar -zxvf tops-20060402_patch.tgz; addons

      2. On machine A, copy newest tar file, tops-20061210.tgz, into 
      directory /tops_tar/tops-20061210/ and untar it:
         /tops_tar > mkdir tops-20061210
         /tops_tar > cp /opt/tops-20061210.tgz tops-20061210   
         /tops_tar > cd tops-20061210
         /tops_tar/tops-20061210 > tar -zxvf tops-20061210.tgz

      3. On machine A copy and paste the following to create the
      patch file.  Messages "No such file or directory" from diff
      are for files that should be listed in addons.  Add them and
      redo the portion of step 1. within /tops_tar/tops-20060402,
      then run the following again:

      >> psource("patch.n");
      >> proj_old="/tops_tar/tops-20060402/tops/";
      >> proj_new="/tops_tar/tops-20061210/tops/";
      >> DIRS=pilen(\
            push(depth),\
               ".",\
               "admin/",\
               "apps/",\
               "doc/",\
               "src/",\
               "sys/",\
               "test/",\
               "test/drivers",\
               "timing/",\
               "tx/",\
               "usr/",\
               "usr/telnetd",\
            less(pull(depth()))\
         );
      >> DIFFS(DIRS, proj_old, proj_new);
      >> shell("mv patch.txt ~/patch_20060402-20061210.txt");

      4. Paste file addons into the top of the new patch file for use 
      on machine B and to the bottom of this file, with earlier ones,
      for future reference.

      On machine A, to check the patch, and on machine B to apply 
      the patch, run the following:

      5. Untar reference file, tops-20060402_patch.tgz, and run file 
      addons (for machine B, addons is included in the top of the
      patch file) to touch new file names:

      % cd /tops_tar/tmp; # and remove all files
      % cp /tops_tar/tops-20060402/tops-20060402_patch.tgz .
      % cp /tops_tar/tops-20060402/addons .
      % tar -zxvf tops-20060402_patch.tgz; addons

      6. Run PATCHES on the untarred reference file:
      % cd; tops
      >> psource("patch.n");
      >> proj_old="/tops_tar/tmp/tops/";
      >> PATCHES("patch_20060402-20061210.txt", proj_old);

      This is the output on machine B or when checking machine A.
      Patches in tops/test/drivers are new (patch does not preserve
      the x property, and chmod on machine B will need to be run to 
      make the scripts executable).

      PATCHES: patching files in /tops_tar/tmp/tops/.
      PATCHES: patching files in /tops_tar/tmp/tops/doc/
      PATCHES: patching files in /tops_tar/tmp/tops/src/
      PATCHES: patching files in /tops_tar/tmp/tops/sys/
      PATCHES: patching files in /tops_tar/tmp/tops/test/
      PATCHES: patching files in /tops_tar/tmp/tops/test/drivers
      PATCHES: patching files in /tops_tar/tmp/tops/usr/
      PATCHES: patching files in /tops_tar/tmp/tops/usr/telnetd
      >>

      7. Make a new reference tar file that future changes will be 
      patched against.  This is done on both machines A and B, immedi-
      ately after testing the patch on machine A and immediately after 
      applying the patch on machine B:
      >> chdir("/tops_tar/tmp"); // same directory as in step 5
      >> shell("tar -czvf tops-20061210_patch.tgz tops");
      >> shell("mv tops-20061210_patch.tgz /tops_tar");
      This is the file to be patched against in the future.

      8. On machine A, again in /tops_tar/tmp, cd into tops/ and run
      configure;make to verify that the patched files compile.

      9. From machine A, file ~/patch_20060402-20061210.txt can be
      emailed to machine B to bring it up to the latest date.

*/
//----------------------------------------------------------------------

indexbase(1); // set the base for array indices to 0 or 1

function (T) = diffs(old_dir, new_dir) {
/* List of differences between files in old_dir and new_dir.  Returned
   volume T contains the lines in the files of new_dir that are not in
   the files of old_dir.

   So they can be retrieved by patch(), each block of changed lines is 
   preceded by the line: Qhead filename: */
   { Qhead = "diffs_for_file: "; }
    
   T = tpurged(VOL);

   if(!rows((old_files=files(old_dir)))) return(T);
   if(!rows((new_files=files(new_dir)))) return(T);

   DO(rows(new_files),xbase);
      Q=new_files[I];

   // Get the differences.  Function diff1() runs Unix diff on a pair
   // of files:
      D = diff1(catpath(old_dir,Q), catpath(new_dir,Q));

   // Add any differences D, with a header, to the pile:
      if(chars(D)>0) T=[T; Qhead+Q; D];
   LOOP;
}

function DIFFS(dirs, proj_old, proj_new) {
/* For all the dirs in proj_old and proj_new, list the differences 
   between all the files and save to file patch.txt.

   So they can be retrieved by PATCH(), each block of changed lines is 
   preceded by the line: Qhead filename: */
   { Qhead = "patches_for_directory: "; }

   U=tpurged(VOL);

   DO(rows(dirs),xbase);
      dir=dirs[I];

      T=diffs(proj_old+dir, proj_new+dir); // running Unix diff

      if(rows(T)>0) U=[U; Qhead+dir; T];
   LOOP;

   dot("DIFFS: saving patches to file " + pwd + "patch.txt");
   nl();

   save(textput(U),"patch.txt"); // save all differences to file
   T=tpurged(VOL);
}

function (F)=files(dir) {
/* Obtain a list of relevant files in directory dir.  File types listed
   below are eliminated from the list before it is returned in F. */

   PWD=pwd();
   chdir(dir);
   
   (D,F)=rake((N=dirnames(dir)),file?(N));

   F=alphabetize(F,
      replace$(".." , " "), // no ..
      replace$(". " , " "), // no .

      qreplace(".o" , " "), // no .o files
      qreplace("CVS", " "), // no CVS
      qreplace("tops"," "), // no tops executable
      qreplace("core"," "), // no core files
      qreplace(".swp"," "), // no swap files
      qreplace(".sav"," "), // no save files

      noblanklines()
   );
   chdir(PWD);
}

function (f) = patch(T1, n, old_dir) {
/* For the difference file named in entry n of T1, patch the file in 
   old_dir to make it like new.

   Returned f is false if there is no nth entry in T1. */

   { debug = no; }

   T = [T1; diffs.Qhead];

/* L is a list of row indices in T where file names are located.  File
   names in T are given on rows with the unique pattern diffs.Qhead,
   starting in the leftmost column.  Restrict search in T for pattern
   diffs.Qhead to the leftmost chars: */
   L = grepr(T[*,xbase:chars(diffs.Qhead)], diffs.Qhead);
   if(qdx(n)>(rows(L)-1)) return(false);

// The nth file name comes from the second word of T(L(n)):
   (Fname,f) = (tpurged(STR), word(T[L[n]],ndx(2)));
   if(!f) return(false);
   if(debug) nl(dot(Fname));

// Patches in T for Fname are at indices L(n)+1 through L(n+1)-1:
   Tpatch = T[L[n]+1:L[n+1]-1];
   if(debug) nl(dot(Tpatch));

// Save patches on scratch file, for use in upcoming Unix patch command:
   save(Tpatch, scratch);
   Tpatch = tpurged(VOL);

// Run a shell command of the form: patch -e -i scratch Fname
   Fname = catpath(old_dir, Fname);
   COMMAND = "patch -e -i " + spaced(scratch) + Fname;
   shell(COMMAND);

   return(true);
}

function () = patches(T, pold) {
// Make all the patches to files in directory pold, using diffs from T:
   k = ndx(1);
   BEGIN();
   WHILE(patch(T,k,pold));
      k++;
   REPEAT;
}

function PATCHES(File, proj_old) {
/* Apply the patches in File to the changes in the dirs and files of
   proj_old. */

   { debug = yes; }

   U = [asciiload(File); DIFFS.Qhead];

/* M is a list of row indices in U where directory names are located.  
   Directory names in U are given on rows with the unique pattern 
   DIFFS.Qhead, starting in the leftmost column.  Restrict search in U 
   for pattern DIFFS.Qhead to the leftmost chars: */
   M = grepr(U[*,xbase:chars(DIFFS.Qhead)], DIFFS.Qhead);

// Apply patches to the files in the directories:
   k = ndx(1);
   BEGIN();
   // The kth directory name comes from the second word of U(M(k)):
      (dir,f) = (tpurged(STR), word(U[M[k]],ndx(2)));
      if(!f) return(nl(dot(" PATCHES: error reading directory name")));

      Dir = proj_old + dir;

      if(debug) (
         dot("PATCHES: patching files in "),
         dot(Dir),
         nl()
      );
   // Patches in U for Dir are at indices M(k)+1 through M(k+1)-1:
      Upatch = U[M[k]+1:M[k+1]-1];
      patches(Upatch,Dir);

   WHILE(qdx(k)<rows(M)-1);
      k++;
   REPEAT;
}
//----------------------------------------------------------------------
/*

Appendix. Keeping track of latest changes relative to a given tar file.

# File /opt/addons_20061210

# File addons for latest patch

# Making a patch of latest tops against tops-20060402_patch.tgz.

# To start out clean, run these two lines inside /tops_tar/tops-20060402

#    /tops_tar/tops-20060402 > /bin/rm -r tops
#    /tops_tar/tops-20060402 > tar -zxvf tops-20060402_patch.tgz; addons

# Any files not in file tops-20060402_patch.tgz must have a place holder
# added after it is untarred.  Then diff and patch will add them to
# the patches.  Below is a list of files that will have a place holder
# added by running this script.

# Files added after 20060402
# Touch the following names so diff and patch will provide these files:
touch tops/./Makefile.ibm64
touch tops/src/LINK64
touch tops/sys/uboot.n
touch tops/sys/uboot.nv
touch tops/test/cmplx_chk
touch tops/test/run_test.perl
touch tops/usr/patch_prev.n
touch tops/usr/root_pppoe
touch tops/usr/telnetd/make_lvtparse64

------------------------------------------------------------------------

# File /opt/addons_20060402

# File addons for latest patch

# Making a patch of latest tops against tops-20060329_patch.tgz.

# To start out clean, run these two lines inside /tops_tar/tops-20060329

#    /tops_tar/tops-20060329 > /bin/rm -r tops
#    /tops_tar/tops-20060329 > tar -zxvf tops-20060329_patch.tgz; addons

# Any files not in file tops-20060329_patch.tgz must have a place holder
# added after it is untarred.  Then diff and patch will add them to
# the patches.  Below is a list of files that will have a place holder
# added by running this script.

# Files added after 20060329
# Touch the following names so diff and patch will provide these files:
touch tops/test/bufdemo

------------------------------------------------------------------------

# File /opt/addons_20060329

# File addons for latest patch

# Making a patch of latest tops against tops-20060129_patch.tgz.

# To start out clean, run these two lines inside /tops_tar/tops-20060129

#    /tops_tar/tops-20060129 > /bin/rm -r tops
#    /tops_tar/tops-20060129 > tar -zxvf tops-20060129_patch.tgz; addons

# Any files not in file tops-20060129_patch.tgz must have a place holder
# added after it is untarred.  Then diff and patch will add them to
# the patches.  Below is a list of files that will have a place holder
# added by running this script.

# Files added after 20060129
# Touch the following names so diff and patch will provide these files:
touch tops/usr/eiv.n
touch tops/test/cms
touch tops/test/equiv_js
touch tops/test/purge_mem
touch tops/test/put4

------------------------------------------------------------------------

This shows a directory containing the project as of January 17, 2006.

Patches made against this directory on January 29, 2006 are in file
patch_20060117-20060129.txt.

[dale@clacker] /tops_tar/tops-20060117 > ll
total 1652
drwxr-xr-x    3 dale     comm         4096 Jan 29 18:55 ./
drwxr-xr-x    8 dale     comm         4096 Jan 29 19:08 ../
lrwxrwxrwx    1 dale     comm           20 Jan 29 18:45 addons -> /opt/addons_20060129*
drwxr-xr-x   11 dale     comm         4096 Jan 17 08:51 tops/
-rw-r--r--    1 dale     comm      1672997 Jan 17 09:01 tops-20060117_patch.tgz
[dale@clacker] /tops_tar/tops-20060117 > 

[dale@clacker] /tops_tar/tops-20060117 > more addons
# File /opt/addons_20060129
# File addons for latest patch

# Making a patch of latest tops against tops-20060117_patch.tgz.

# To start out clean, run these two lines inside /tops_tar/tops-20060117

#    /tops_tar/tops-20060117 > /bin/rm -r tops
#    /tops_tar/tops-20060117 > tar -zxvf tops-20060117_patch.tgz; addons

# Any files not in file tops-20060117_patch.tgz must have a place holder
# added after it is untarred.  Then diff and patch will add them to 
# the patches.  Below is a list of files that will have a place holder 
# added by running this script.

# Files added after 20060117
# Touch the following names so diff and patch will provide these files:
touch tops/test/drivers/dserv1
touch tops/test/drivers/tserv
[dale@clacker] /tops_tar/tops-20060117 > 

------------------------------------------------------------------------

This is a directory containing the project as of December 14, 2005:
[dale@clacker] /tops_tar/tops-20051214 > ll
total 1632
drwxr-xr-x    3 dale     comm         4096 Jan 17 07:47 ./
drwxr-xr-x    7 dale     comm         4096 Jan 17 08:08 ../
lrwxrwxrwx    1 dale     comm           20 Jan 17 07:38 addons -> /opt/addons_20060117*
drwxr-xr-x   11 dale     comm         4096 Jan 17 07:47 tops/
-rw-r--r--    1 dale     comm      1650709 Dec 14 21:38 tops-20051214.tgz
[dale@clacker] /tops_tar/tops-20051214 > 

File addons shows new files since December 14, 2005.  Running addons in
/tops_tar/tops-20051214 will touch these new names in the appropriate
directories, and DIFFS will add the entire files to the diffs file.

Here is addons for 20060117 shown in the directory listing above:

[dale@clacker] /tops_tar/tops-20051214 > more addons
# File /opt/addons_20060117
# File addons for latest patch

# Making a patch of latest tops against tops-20051214.tgz.

# To start out clean, run these two lines from /tops_tar/tops-20051214:

#    /tops_tar/tops-20051214 > /bin/rm -r tops
#    /tops_tar/tops-20051214 > tar -zxvf tops-20051214.tgz; addons

# Any files not in file tops-20051214.tgz must have a place holder
# added after it is untarred.  Then diff and patch will add them to 
# the patches.  Below is a list of files that will have a place holder 
# added by running this script.

# Files added after 20051214
# Touch the following names so diff and patch will provide these files:
touch tops/configure.pl
touch tops/doc/clientside_pitfalls.doc
touch tops/usr/xhost_start1.n
touch tops/usr/Xserver
[dale@clacker] /tops_tar/tops-20051214 > 

------------------------------------------------------------------------

Running DIFFS for the June 30, 2005 directory shown below, against the
latest directory, provides the files needed to bring the June 30, 2005 
version current on another machine.

This is a directory containing the project as of June 30, 2005:
[dale@clacker] /tops_tar/tops-20050630 > ll
total 1496
drwxr-xr-x 3 dale comm 4096 Dec 13 08:30 ./
drwxr-xr-x 6 dale comm 4096 Dec 13 07:51 ../
lrwxrwxrwx 1 dale comm  20  Dec 13 08:08 addons -> /opt/addons_20051213*
drwxr-xr-x11 dale comm 4096 Jun 30 09:13 tops/
-rw-r--r-- 1 dale comm 1514848 Jun 30 09:14 tops-20050630.tgz
[dale@clacker] /tops_tar/tops-20050630 > 

File addons shows new files since June 30, 2005.  Running addons in
/tops_tar/tops-20050630 will touch these new names in the appropriate
directories, and DIFFS will add the entire files to the diffs file.

Here is addons for December 13, 2005:
[dale@clacker] /tops_tar/tops-20050630 > more /opt/addons_20051213
# Making a patch of latest against tops-20050630.tgz.

# To start out clean, run these two lines from here 
# (here = /tops_tar/tops-20050630):

#    /tops_tar/tops-20050630 > /bin/rm -r tops
#    /tops_tar/tops-20050630 > tar -zxvf tops-20050630.tgz; addons

# Any files not in file tops-20050630.tgz must have a place holder
# added after tops-20050630.tgz is untarred.  Then diff and patch 
# will add them to the patches.  Below is a list of files that will
# have a place holder added by running this script.

# Touch the following names so diff and patch will provide these files:
mkdir tops/usr/telnetd
# 20051213
touch tops/doc/vtmap.txt
touch tops/usr/dserv1
touch tops/usr/Xclient.js
touch tops/usr/Xclient.html
touch tops/usr/xhost_start.n
touch tops/usr/telnetd/vtparse.c
touch tops/usr/telnetd/vtparse.h
touch tops/usr/telnetd/vtparse_table.c
touch tops/usr/telnetd/vtparse_table.h
touch tops/src/vt.c
touch tops/src/vt.h
touch tops/test/infix_equ
touch tops/test/iodemo1
# 20051007
touch tops/usr/telclient
touch tops/usr/telserver
# After 20050630 and before 20051007
touch tops/usr/patch.n
touch tops/usr/work.n
touch tops/sys/term.v
touch tops/usr/telnetd/MCONFIG
touch tops/usr/telnetd/README
touch tops/usr/telnetd/telnetd_wrap.c
touch tops/src/ssl.c
touch tops/src/ssl.h
touch tops/sys/internal_idx.n
touch tops/test/psparse 
touch tops/usr/client.pem 
touch tops/usr/dh1024.pem
touch tops/usr/patch.v
touch tops/usr/root.pem
touch tops/usr/server.pem
touch tops/doc/telnet.doc
[dale@clacker] /tops_tar/tops-20050630 > 
*/

