\ {{{1 GNU General Public License
{
Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}}
}

\  File nas.v  July 2002
{
   Copyright (c) 2002  D. R. Williamson
 
   Words for Nastran.

   NASTRAN is a registered trademark of the National Aeronautics and
   Space Administration.

   The words in this file are listed below.

   syspath "nas.v" cat asciiload  this " inline:" grepr reach dot

   inline: cwhere (qC n --- hT) \ files where card type C for n is found
   inline: DOF_SET (qF06 qSet --- hT) \ DOF table from F06 file
   inline: get_sorted (qFile --- hT) \ bulk data cards from .f06 file
   inline: get4sp (hFile qA -- hA_sp) \ huge matrix from op4 into sparse
   inline: gwhere (n --- hT) \ files where the GRID card for n is found
   inline: inCSTM (hCID hCSTM --- hC) \ CSTM definitions for CIDs
   inline: inGPL (hGRID hGPL --- hRows) \ rows in GPL where GRIDs are
   inline: inSet (hGRID hLDOF hMAP hUSET bit --- hRows) \ rows of bit
   inline: makeMPC (hE hG hC --- hT) \ make text matrix of MPC equations
   function (T) = make_tabled1(TID, P, t, sym1) P(t) into TABLED1 cards
   function (A, t) = read_tabled1(File1) read A(t) from TABLED1 cards

   Words for models from Nastran:
   inline: model (qFile qWord --- ) \ making model called Word
   inline: bgp (hModel --- hBGPDT) "bgpdt" db ; \ grid point data table
   inline: cstm (hModel --- hCSTM) "cstm" db ; \ coordinate system defs
   inline: db (hModel qDB --- hDB) \ data block DB for model
   inline: GEOM (hModel --- hT) \ geometry of Model
   inline: gpl (hModel --- hGPL) "gpl" db ; \ grid point id list
   inline: gridmap (hModel --- hMAP) "gridmap" db ; \ grid-dof map
   inline: lama (hModel --- hLAM) "lama" db ; \ modal frequencies
   inline: model? (hModel --- f) \ true flag if have library for Model
   inline: m66 (hModel --- hM66) "m66" db ; \ 6-by-6 mass matrix
   inline: _phi (hModel --- hPHI) "phi" db ; \ modal matrix
   inline: uset (hModel --- hUSET) "uset" db ; \ bits defining sets

   Words for Nastran Output2 files:
   inline: get2 (hFile qS --- hA) \ matrix named S from op2 file
   inline: get2c (hFile qS hCols --- hA) \ Cols of matrix S from op2
   inline: op2datablock (qS --- hT) \ data block S from Nastran op2 file
   inline: op2file (hFile --- f) \ initialize to read Nastran op2 file
   inline: op2map ( --- ) \ map op2 file for reading
   inline: op2mat (hMap hTrailer --- hA) \ mat A from op2 file
   inline: op2readtrailer ( --- hTlr qName) \ data block trailer, name
   inline: op2tlr (qS --- hA) \ trailer from Nastran data block S
   inline: toc2 (hFile --- ) \ table of contents of Nastran op2 file
   inline: BGPDT ( --- hA) \ location of grid points in basic system 0
   inline: CSTM ( --- hA) \ coordinate system definitions
   inline: GRIDmap ( --- hL) \ starting dof for each grid and spoint
   inline: EQEXIN ( --- hA) \ external/internal equivalence list
   inline: GID (hMAP hDOF --- hG) \ grid ID+dof list for given DOF list
   inline: GPL ( --- hA) \ grid point list in internal order
   inline: scanop2 (qFile --- ) \ scan a Nastran Output2 file
   inline: USET ( --- hA) \ bit sets for all degrees-of-freedom
   inline: USETbit (hMAP hUSET n --- hG) \ grid ID+dof list for set n


   Table trailer elements on Output2 files (BGPDT, CSTM, others):
      1 - number of grids plus spoints
      2 - number of coordinate systems
      3 - type of system (bit encoded)
      4 - precision of real-data record (1 SP, 2 DP)
      5 - length of real-data record
      6 - length of integer-data record

   Matrix trailer elements on Output2 files:
      1 - number of columns
      2 - number of rows
      3 - form of matrix (1 square, 2 rectangular, 6 symmetric, ...)
      4 - type of matrix (1 SPreal, 2 DPreal, 3 SPcmplx, 4 DPcmplx)
      5 - largest number of nonzero SP words among all cols
      6 - density times 10,000

   Matrix trailer element 3, form, can have the following values:
      1 - square
      2 - rectangular
      3 - diagonal
      4 - lower triangular factor
      5 - upper triangular factor
      6 - symmetric
      8 - identity
      9 - pseudoidentity
     10 - Cholesky factor
     11 - trapezoidal factor
     12 - sparse lower triangular factor (MSC(R) Version 67 only)
     13 - sparse lower triangular factor (MSC(R) Version 67.5, 68 only)
     15 - sparse unsymmetric triangular factors

   Matrix trailer element 4, type, can have the following values:
      1 - real single precision
      2 - real double precision
      3 - complex single precision
      4 - complex double precision
}

   CATMSG push no catmsg
   "sppartition" missing IF "sparse.v" source THEN

\-----------------------------------------------------------------------

\  Utilities for Nastran.

   inline: cwhere (qC n --- hT) \ files where card type C for n is found
{     Search files in a list of included Nastran bulk data files and
      find which file, or files, contain a card C for n.  

      Example: "GRID" 1095 cwhere

      For this word to run, a list of bulk data files has to be banked
      in the local library first.  Here is an example where names of 
      files are extracted from a Nastran .dat file that includes files 
      that provide bulk data:

       \ Load a file that contains file names:
            "mymodel.dat" asciiload (hT)
       \ Reach rows of T that have INCLUDE as the first 7 characters:
            (hT) its 1st 7 items catch "INCLUDE" grepe reach (hT1)
       \ Bank the list of include files into this word, cwhere:
            "INCLUDE" chars negate indent (hT2) "cwhere" "Files" bank
 
}     [ no is Files, no is Hash, yes is TRACE

        " cwhere: need list of files; use: (hT) 'cwhere' 'Files' bank"
        into message

        2000 into bins, defname into Name, no is %DATA
 
\       Here is the list of card types that will be processed:  
           "GRID CORD2R CORD1R CORD2C CORD1C CBAR CQUAD4 "
           "RBAR RBE2 RBE3 CONM1 CONM2 " pile
           words into Cards

{       A different Cards list can be banked here any time before this
        word is run and the hash is made (first run only).  Examples:
           
           Replacing the Cards list:
              "GRID PBAR MAT1" words "cwhere" "Cards" bank

           Adding to the present Cards list (yank and bank):
              "cwhere" "Cards" yank (hCards) "GENEL CQUAD8" words pile
              (hCard1s) "cwhere" "Cards" bank
}     ]
      Hash not
      IF Files rows any not
         IF message . nl return
         ELSE " cwhere: making hash for" . Files rows .i " files..." .
            %DATA hash?
            IF %DATA hash_close
            THEN "" "" bins Name "%DATA" localref hash_make

            Files rows 1st
            DO Files I quote filefound
               IF into File, File asciiload asciify chop push
                  TRACE IF 
                     nl " File: " . File .
                  THEN
                  peek (hT) 1st eight items catch
                  uppercase 1st word (f)
                  IF (hITEMS) into ITEMS

                     Cards rows 1st
                     DO ITEMS Cards I quote (type) grepe any?
                        IF peek (hT) swap reach (hDATA) dup push
                           TRACE IF 
                              nl sp sp " card: " .
                              Cards I quote .
                           THEN
                           2nd word (f)

                           IF \ Making keys, like 3002GRID
                              (hGIDs) right justify
                              Cards I quote strchop those rows pileof
                              park chop (hKey)

                              \ Making values, like CardImage+FileName
                              peek (hDATA) chop right justify
                              File those rows pileof two indent
                              park chop (hVal)

                              \ Adding to hash:
                              (hKey hVal) %DATA hash_add

                           ELSE three dump
                           THEN pull (hT) drop

                        THEN
                     LOOP

                  THEN
                  pull (hT) drop
               THEN
            LOOP
            nl yes is Hash

         THEN
      THEN
      (n) this type NUM = IF int$ THEN strchop 
      swap uppercase strchop cat
      %DATA swap (hHash hKey) hash_lookup pilen
   end

   inline: DOF_SET (qF06 qSet --- hT) \ DOF table from F06 file
{     Extract a DOF table for Set from an MSC(R) Nastran print file.

      The number of rows in text volume T equals the number of degrees-
      of-freedom.  Each row contains the row (DOF) number followed by
      the corresponding grid and local dof, in the form GRID-DOF.

      Incoming Set is the text that appears for the desired table.

      Examples.  These are the headings for the G set table:
         "strut.out" "G    DISPLACEMENT"     DOF_SET \ before version 70
         "strut.out" "G        DISPLACEMENT" DOF_SET \ 70 and above

      As shown above, a major enhancement in MSC(R) version 70 and 
      above has been to go from four to eight spaces in the titles of 
      the DOF tables.

}     [ 13200 (bytes) "1-PAGE" book
        no "FILE" book ] FILE filetrue IF FILE fclose THEN

      (qSet) "SET" book
      (qF06) this file? not
      IF (qF06) " DOF_SET: file not found: " . . nl return THEN
      (qF06) old ascii "FILE" file

    \ Offsets to all lines in FILE that contain the pattern SET:
      FILE SET fmap (hMap) any? not
      IF " DOF_SET: set pattern not found: " . SET . nl
         FILE fclose
         return
      THEN (hMap) 

    \ File offset to seek for first pattern:
      (hMap) this 1st pry "START" book \ at 1st pattern

    \ Bytes required to fetch all required pages:
      (hMap) these rows one >
      IF (hMap) delta partials (hP) this 2nd pry (bytes/page) swap 
         (hP) one endmost ontop (LAST) \ bytes to top of last page
         (bytes/page LAST) plus \ bytes to end of table
      ELSE (hMap) drop 1-PAGE
      THEN
      "FETCH" book

    \ Fetch the lines from FILE:
      FILE START fseek
      FILE FETCH fget (hT1) textget (hT)
      FILE fclose \ done with file

    { Lines wanted have the pattern 1= in the 1st word, as in:
         1=   1001-1   1001-2   1001-3   1001-4   1001-5   1001-6 ...
        11=   1002-5   1002-6   1003-1   1003-2   1003-3   1003-4 ...
        21=   1004-3   1004-4   1004-5   1004-6 
    } (hT) this 1st word (f) drop "1=" grepr (hT hRows) reach (hT)
      (hT) "=" chblank (hT) \ replace = with blank

    \ Read the row numbers in the first column of text:
      (hT) this 1st word (f) drop numerate (hR)
      (hR) zero pile (hR) \ append a zero row number

    \ When adjacent row numbers are not increasing, the table ends:
      (hR) delta 0> 1st those rows items swap rake drop
      1st swap 2nd pry nit items (hRows) \ want these rows from T
      (hT hRows) reach (hT)

    \ Take the 2nd through 10th columns of words:
      (hT) push 11 2nd
      DO peek I word (f) drop spaced LOOP pull drop
      ten parkn (hT) words (hT) \ a column (chain) of DOFs

    \ Put a list of integers on the left:
      (hT) these rows columnofints, spaced swap park (hT)

      "_" SET strchop 1st character "_DOF" cat cat naming
   end

   inline: get_sorted (qFile --- hT) \ bulk data cards from .f06 file
\     Get all sorted bulk data echoed in an MSC(R) Nastran output file.
\     The ENDDATA card is not returned in T.

      [ no is File ] File filetrue IF File close THEN

      no no blockofblanks dup into TITLE, into LABEL

      (qFile) strchop "Infile" book

      Infile file? not
      IF " get_sorted: file " Infile " not found" cat cat ersys return 
      THEN

      Infile -path -ext "Name" book
      Infile old ascii "File" open

    \ Bounds of the case control echo:
      File "C A S E    C O N T R O L    E C H O" fmap any? not
      IF File "C A S E    C O N T R O L   D E C K   E C H O" fmap 
         any? not
         IF " get_sorted: case control not found" ersys return THEN
      THEN (hOffset)

      (hOffset) 1st pry "L1" book
      File "BEGIN BULK" fmap 1st pry "L2" book

      File L1 fseek File L2 L1 - tic fget textget (hT)

    \ Get title, subtitle, and label from case control:
      (hT) dup "TITLE" tug noblanklines any?
      IF "=" tug -1 indent 1st 60 items catch noblanklines makes TITLE 
      THEN
      (hT) dup "LABEL" tug noblanklines any?
      IF "=" tug -1 indent 1st 60 items catch noblanklines makes LABEL 
      THEN

      (hT) "ECHO" tug noblanklines any?
      IF dup "EXCEPT" grepr any?
         IF drop notrailing ELSE "" THEN
      ELSE ""
      THEN "EXCEPT" book

    \ Bounds of the sorted bulk data echo:
      File "S O R T E D   B U L K   D A T A   E C H O" 
      fmap 1st pry "L1" book
      File "  ENDDATA" fmap one endmost ontop "L2" book \ last ENDDATA

      File L1 fseek File L2 L1 - tic fget textget (hT) 

      File purged "                       TOTAL COUNT=" fstr any?
      IF "=" tug -1 indent number not IF -1 ELSE one less THEN
      ELSE -1
      THEN "COUNT" book

      File close

    \ Remove unwanted lines, indent, and clip:
      (hT) " " those chars cats makes blankline
      (hT) ".." blankline qreplace
      "MSC/NASTRAN" blankline qreplace
      "MSC.NASTRAN" blankline qreplace
      "S O R T E D   B U L K   D A T A   E C H O" blankline qreplace

      TITLE rows any?
      IF (rows) 1st DO TITLE I quote blankline qreplace LOOP THEN
      LABEL rows any?
      IF (rows) 1st DO LABEL I quote blankline qreplace LOOP THEN

      -30 indent, 1st 80 items catch
      noblanklines (hT)

    \ Check counts:
      (hT) these rows "nc" book
      nc COUNT <>
      IF " get_sorted error: cards found =" . nc .i
         ", file total count =" . COUNT .i nl

         EXCEPT any?
         IF "   Discrepancy may be due to cards not echoed to "
            Infile cat . nl
            "   See exceptions on card " . (EXCEPT) . nl
         THEN

      THEN

      "_" Name ".bdf" cat cat naming
   end

   inline: get4sp (hFile qA -- hA_sp) \ huge matrix from op4 into sparse
{
      Example:

      Reading sparse matrix KGG took 472 seconds for 16M bufsize, and
      247 seconds for 64M bufsize:

         -rw-r--r--   1  dale  dale   302643048 Aug 24 14:38 widget.op4

         [tops@clacker] ready > op4 toc4
          MGG  20044 by 20044  form 6  type 2
          MGGD  20044 by 1  form 6  type 2
          KGG  20044 by 20044  form 6  type 2
          KGGD  20044 by 1  form 6  type 2

         [tops@clacker] ready > time push op4 'KGG' get4sp
                                time pull - beep .i
          472
          stack elements:
                0 sparse: _KGG  20044 by 20044
          [1] ok!
         [tops@clacker] ready >
}
      [ 64E6 "bufsize" book ]

      "A" book, "File" book
      File dup rewind _toc4 lowercase (hT)
      (hT) dup " " A lowercase " " cat cat grepr any?

      IF (hT) 1st pry quote numerate
         this 1st pry "rows" book, 2nd pry "Cols" book

         xbase push 0based \ use 0-based for easier indexing within DO

         bufsize eight / integer (terms) \ terms in buffer at once
         rows / integer "cols" book      \ full cols per buffer read
         Cols cols /mod (rem quot)       \ quot sets of reads, then rem

         (quot) dup push any
         IF peek 1st
            DO File dup rewind A I cols * cols items get4c sparse LOOP
            peek spparkn (hQuot)
         ELSE rows zero null sparse
         THEN (hQuot)

         File dup rewind A pull cols * four roll
         (rem) items get4c sparse (hRem)

         (hQuot hRem) two spparkn

         pull indexbase
         "_" A cat naming

      ELSE (hT) drop " get4sp: matrix " A " not found" cat cat ersys
      THEN
   end

   inline: gwhere (n --- hT) \ files where the GRID card for n is found
{     Search files in a list of included Nastran bulk data files and
      find which file, or files, contain a GRID card for n.

      Warning: assumes bulk data is in 8-character fields.  See word
      cwhere for a more general word that works on free-field data.

      For this word to run, a list of bulk data files has to be banked
      in the local library first.  Here is an example where names of 
      files are extracted from a Nastran .dat file that includes files 
      that provide bulk data:

       \ Load a file that contains file names:
            "mymodel.dat" asciiload (hT)
       \ Reach rows of T that have INCLUDE as the first 7 characters:
            (hT) its 1st 7 items catch "INCLUDE" grepe reach (hT1)
       \ Bank the list of include files into this word, gwhere:
            "INCLUDE" chars negate indent (hT2) "gwhere" "Files" bank
}
      [ no is Files, no is Hash
        " gwhere: need list of files; use: (hT) 'gwhere' 'Files' bank"
        into message
        2000 into bins, defname into Name, no is %DATA
      ]
      Hash not
      IF Files rows any not
         IF message . nl return
         ELSE " gwhere: making hash for" . Files rows .i " files..." .
            %DATA hash?
            IF %DATA hash_close
            THEN "" "" bins Name "%DATA" localref hash_make
            Files rows 1st
            DO Files I quote filefound
               IF this asciiload, these 1st 4 items catch
                  "GRID" grepe any? \ reach grid IDs from field 2:
                  IF reach 9 ndx 8 items catch (hGIDs)
                     swap those rows pileof (hNameFile) %DATA hash_add
                  ELSE 2drop
                  THEN
               THEN
            LOOP
            nl yes is Hash
         THEN
      THEN
      (n) int$ %DATA swap (hHash hKey) hash_lookup pilen
   end

   inline: inCSTM (hCID hCSTM --- hC) \ CSTM definitions for CIDs
{     Returned C has one column for each coordinate system id in CID.

      The 14 rows of C contain the following:
           1: coordinate system id
           2: type: 1=rectangular, 2=cylindrical, 3=spherical
         3-5: vector from origin of frame b to origin of frame p,
              expressed in frame b
        6-14: Cbp, the direction cosine matrix from frame p to frame b 

      Frame b is the basic frame, system 0, to which all frames chain.

      Notes:
         If id=0 (system 0), its column receives type 1, null origin,
            and identity direction cosine matrix
         If id=-1 (an spoint), its column receives null in rows 2-14

}     [ defname is name
        list: 1, 0 0 0, 1 0 0, 0 1 0, 0 0 1 ; makes basic
        13 1 null makes spoint
        14 0 null makes PURGED
      ]

      this 1st reach bend, 1st those rows items park (hXY) "XY" book
      "Clist" book
      (hCID) hand this 0> "R" book
      (hCID) R rake "CID" book, "CID0" book

      CID any? \ doing ids greater than 0:
      IF XY swap (hXY hCID) find (hF)
         these totals ontop those rows <>
         IF (hF) CID swap rake drop
            name sp . ": reference system ids not found in CSTM:" .  nl
            mformat swap "%7.0f" mformatset .m mformatset
            nl "" ersys return
         ELSE (hF) drop
         THEN
         Clist XY CID look catch
      ELSE PURGED
      THEN
      (hCID)

      CID0 any? \ doing ids of 0 and -1; these are not in CSTM:
      IF (hCID0) this rake (hC0 hC1)

         swap (hC0) any?
         IF (hC0) bend basic those cols clone pile ELSE PURGED THEN
         (hC0)

         swap (hC1) any?
         IF (hC1) bend spoint those cols clone pile ELSE PURGED THEN
         (hC1)

         (hC hC1) CID0 mesh

      ELSE PURGED
      THEN 
      (hCID0)

      (hCID0 hCID) swap R mesh (hC)
   end
   
   inline: inGPL (hGRID hGPL --- hRows) \ rows in GPL where GRIDs are
\     List of rows in GPL where items in list GRID are located.
      [ defname is name ] no is x
      (hGPL) 1st those rows items (hY) park (hXY) "XYtable" book

      (hGRID) hand "x" book

      XYtable x find (hF) these totals ontop those rows <>
      IF (hF) x swap rake drop
         name sp . ": grids not found in GPL set:" . nl 
         mformat swap "%7.0f" mformatset .m mformatset
         nl "" ersys return
      THEN
 
      XYtable x look
   end

   inline: inSet (hGRID hLDOF hMAP hUSET bit --- hRows) \ rows of bit
\     List of rows where GRID(LDOF) points are located in the set
\     defined by bit.
      [ list: 1 0 ; "apart" book, defname is name ]
      (hMAP hUSET bit) USETbit (hA)
      (hA) apart claw (hLDOF1 hGRID1) 10 *f plus (hX)
      1st those rows items (hY) park (hXY) "XYtable" book

      (hGRID hLDOF) hand "LDOF" book hand "GRID" book
      LDOF GRID 10 *f plus "x" book

      XYtable x find (hF) these totals ontop those rows <>
      IF (hF) GRID that rake drop
         LDOF rot rake drop park
         name sp . ": these grid dof pairs not found in set:" . nl
         mformat swap "%7.0f" mformatset .m mformatset
         nl "" ersys return 
      ELSE (hF) drop
      THEN
 
      XYtable x look
   end

   inline: makeMPC (hE hG hC --- hT) \ make text matrix of MPC equations
{     hE(N) = list of grid*10 + dof for the dependent of equation n
      hG(N,K) = matrix of grid*10 + dof for independent of each equation
      hC(N,K) = matrix of coefficients: C(i,j) is coefficient for G(i,j)
         in equation E(i)

      All matrices have the same number of rows; G and C have the same
      number of columns.

      Multipoint constraint equation n has the form

         SF*E(n) = sum[SF*C(n,k)*G(n,k)], k=1,cols in G and C

      where

          E(n) is displacement of Grid(Dof) defined by E(n),
          X(G(n,k)) is displacement of Grid(Dof) defined by G(n,k),
          SF is a scale factor provided by this word.

      Cards created are in the form Nastran requires:

         -SF*E(n) + sum[SF*C(n,k)*G(n,k)] = 0

      To run the example below:
      [tops@steelee] ready > "nas.v" "MPC Example" msource

         MPC Example
            99 "makeMPC" "SET" bank
            1 "makeMPC" "SF" bank
            "%8.1f" "makeMPC" "FORM" bank 3 1
            DO list: 100 101 102 ; 10 *f I +d (hE, dependent) \

               list: 200 201 202 ; (independent 1)  \
               list: 300 301 302 ; (independent 2)  \
               list: 400 401 402 ; (independent 3)  \
               list: 500 501 502 ; (independent 4)  \
               4 parkn 10 *f I +d (hG, independent) \

               list: 2 2 2 ; list: 3 3 3 ; \
               list: 4 4 4 ; list: 5 5 5 ; \
               4 parkn (hC) \
               (hE hG hC) makeMPC \
            LOOP 3 pilen dot

            halt
}     [ 
      \ Use word bank to change SF, FORM, and SET from outside (see
      \ the example above):
        1000 is SF      \ to squeeze more digits
        "%8.2f" is FORM \ must be compatible with SF; 8 chars max
        1 "SET" book    \ MPC set number

{       Example of SF and FORM:
           For SF=1000 and FORM="%8.2f" values are scaled by 1000 and
           have 2 fractional digits.  Then a number that uses all 8
           characters in the 8 character field is something like
              -9999.12

           This word gives the first (dependent) entry in an MPC equa-
           tion a coefficient of -1.0.  So when SF=1000, the dependent 
           entry has a coefficient equal to -1000.
}
        "%8.0f" dup cat "2INTS" book
        (10*G+DOF --- G DOF) "10 /mod swap" "GDOF" inlinex
      ]
      2INTS FORM cat "FORMAT" book
      1st push, 1 indexbase
      SF *factor into C, (hG) into G, (hE) into E
      C cols makes cmax
      no no blockofblanks (hT)
      E rows 1st
      DO no is count
       \ First card:
         "MPC" 8 +trailing, SET int$ 8 +trailing cat (hT1)

         list: E I pry GDOF, SF negate ; bend
         FORMAT format (hT2)

         list: G I 1st fetch GDOF, C I 1st fetch ; bend
         FORMAT format (hT3) cat cat (hT1)

       \ Additional cards:
         (hT hT1) pile, one count bump, count cmax <
         IF C cols 2nd
            DO list: G J I fetch GDOF, C J I fetch ; bend
               FORMAT format
               one count bump, count cmax <
               IF list: G J I 1+ fetch GDOF, C J I 1+ fetch ; bend
                  FORMAT format cat
                  one count bump
               THEN
               16 indent pile, 2
            +LOOP
         THEN
      LOOP 80 +trailing "_MPC" naming, pull indexbase
   end

   inline: makeMPC2 (hE hG hC --- hT) \ text matrix of MPC equations
{     Same as makeMPC() except double field cards are written.

      hE(N) = list of grid*10 + dof for dependent of equation n
      hG(N,K) = matrix of grid*10 + dof for independent of each equation
      hC(N,K) = matrix of coefficients: C(i,j) is coefficient for G(i,j)
         in equation E(i)

      All matrices have the same number of rows; G and C have the same
      number of columns.

      Multipoint constraint equation n has the form

         SF*E(n) = sum[SF*C(n,k)*G(n,k)], k=1,cols in G and C

      where

          E(n) is displacement of Grid(Dof) defined by E(n),
          X(G(n,k)) is displacement of Grid(Dof) defined by G(n,k),
          SF is a scale factor provided by this word.

      Cards created are in the form Nastran requires:

         -SF*E(n) + sum[SF*C(n,k)*G(n,k)] = 0

      To run the example below:
      [tops@steelee] ready > "nas.v" "MPC2 Example" msource

         MPC2 Example
            99 "makeMPC2" "SET" bank
            3 1
            DO list: 100 101 102 ; 10 *f I +d (hE, dependent) \

               list: 200 201 202 ; (independent 1)  \
               list: 300 301 302 ; (independent 2)  \
               list: 400 401 402 ; (independent 3)  \
               list: 500 501 502 ; (independent 4)  \
               4 parkn 10 *f I +d (hG, independent) \

               list: 2 2 2 ; list: 3 3 3 ; \
               list: 4 4 4 ; list: 5 5 5 ; \
               4 parkn (hC) \
               (hE hG hC) makeMPC2 \
            LOOP 3 pilen dot

            halt
}     [ 
      \ Use word bank to change SF, FORM, and SET from outside (see
      \ the example above):
        1000 is SF       \ to squeeze more digits
        "%0.10f" is FORM \ must be compatible with SF; 16 chars max
        1 "SET" book     \ MPC set number

{       Example of SF and FORM:
           For SF=1000 and FORM="%0.10f" values are scaled by 1000 and
           have 10 fractional digits.  Then a number that uses all 16
           characters in a double field is something like
              -9999.0123456789

           This word gives the first (dependent) entry in an MPC equa-
           tion a coefficient of -1.0.  So when SF=1000, the dependent 
           entry has a coefficient equal to -1000.
}
        1 "NC" book \ first continuation number
      \ Note: NC is not reset, so multiple calls to this word in the 
      \ same run produces continuations with no duplicates.

      \ This macro makes a continuation string containing NC:
        '"*MPC" NC ints "%04X" format + "CONT" book 1 NC bump'
        "cont" macro

        (10*G+DOF --- DOF G) "10 /mod " "GDOF" macro
      ]
      1st push, 1 indexbase
    \ Store the matrices from the stack:
      (hC) SF * into C, (hG) into G, (hE) into E

      C cols makes cmax
      VOL tpurged (hT)
      E rows 1st
      DO no is count
       \ First card (makes two double field cards):
         "MPC*" 8 blpad, SET intstr 16 blpad + (qS1)
         E I pry GDOF (G) intstr 16 blpad swap
         (DOF) intstr 16 blpad + (qS2) +
         SF negate FORM format left justify (qS3) +
         72 blpad cont CONT + (hC1) \ first half of first card

         CONT 8 blpad
         G I 1st fetch GDOF (G) intstr 16 blpad swap
         (DOF) intstr 16 blpad + + (qS2)
         C I 1st fetch FORM format left justify (qS3) +
         72 blpad cont CONT + (hC2) \ second half of first card
         (hC1 hC2) pile (hT1)

       \ Additional cards (continuation cards):
         (hT hT1) pile, one count bump, count cmax <
         IF C cols 2nd
            DO CONT 24 blpad
               G J I fetch GDOF (G) intstr 16 blpad swap
               (DOF) intstr 16 blpad + + (qS2)
               C J I fetch FORM format left justify (qS3) +
               72 blpad cont 
               CONT + (hC1) pile \ first half of continuation card
      
               one count bump, count cmax <

               IF CONT 8 blpad
                  G J I 1+ fetch GDOF (G) intstr 16 blpad swap
                  (DOF) intstr 16 blpad + + (qS2)
                  C J I 1+ fetch FORM format left justify (qS3) +
                  72 blpad cont 
                  CONT + (hC2) pile \ second half of continuation card
                  one count bump
               ELSE CONT 72 blpad cont
                  CONT + (hC2) pile \ second half of continuation card
               THEN 2
            +LOOP
         THEN
      LOOP 80 blpad "_MPC" naming, pull indexbase
   end

{" This word is written in infix.  Gather its text and run eval.

   function (T) = make_tabled1(TID, P, t, sym1) {
   /* Format P(t) into TABLED1 cards for Nastran. 

      Example:
         if(missing("sine")) source("mmath.v");
         if(missing("make_tabled1")) source("nas.v");

         T = make_tabled1(99, sine(1,100,0.0,0.005,24), "TDX");

         nl(dot(T));
   */
      {
      // Initial formats for the 8 character fields:
         X_FORMAT = "%8.4f";
         Y_FORMAT = "%8.5f";
      }
      sym = catch(sym1+"0000", [1:3]); // 3-char continuation symbol

   /* Store data row-wise in four columns (partial last row is lost): */
      Rows = 4*integer(rows(P)/4); // truncating; remainder is lost
      Y = foldr(P[1:Rows], 4)';
      X = foldr(t[1:Rows], 4)';

   /* Four pairs of (X,Y) data per line: */
      FORMAT = cats(X_FORMAT + Y_FORMAT, 4);

      HEAD = "TABLED1 " + intstr(TID);
      DATA = [format(catch([X,Y], [1,5,2,6,3,7,4,8]), FORMAT);" ENDT"];
      SEQU = nose(format(ints([1:rows(DATA)]), "%04X"), "+" + sym);

   /* Assemble the pieces, using brackets to enforce precedence: */
      T = [[HEAD ; [SEQU , DATA]] , [SEQU ; " "]];

      X = Y = purged;
   }
"} eval \ parse infix into an inline and add to library

{" This word is written in infix.  Gather its text and run eval.

   function (A, t) = read_tabled1(File1) {
   /* From File1, read A(t) from Nastran TABLED1 cards.

      Assumes 8 character fields, 8 numbers per line (time, value,
      time, value, ...).

      Assumes the last card holds ENDT, and it is skipped. 

      Example:
         source("nas.v"); source("mmath.v");

      // Make some tabled1 cards and save to file:
         T = make_tabled1(99, sine(1,100,0.,0.005,24), "TDX");
         save(T, "TD.txt");

      // Read the tabled1 cards:
         (A, t) = read_tabled1("TD.txt");

      // Display the tabled1 cards and the matrices read from them:
         nl(dot(T)); 
         nl(dot([t, A]));
         delete("TD.txt");
   */
      fieldwidth(8);
      File = remtabf(File1); // tabs into equivalent spaces

   /* Get cards 2nd through next-to-last, 2nd through 9th fields: */
      T = asciiload(File)[<< 2nd over rows 2 - items >>, ndx(9:72)];
  

      A = chain(matread([spaced(crop(T, field(2))),
                         spaced(crop(T, field(4))),
                         spaced(crop(T, field(6))),
                         spaced(crop(T, field(8)))], 4)');
                         
      t = chain(matread([spaced(crop(T, field(1))),
                         spaced(crop(T, field(3))),
                         spaced(crop(T, field(5))),
                         spaced(crop(T, field(7)))], 4)');

      T = purged;
}
"} eval \ parse infix into an inline and add to library

\-----------------------------------------------------------------------

   inline: model (qFile qWord --- ) \ making model called Word
{     This word makes a model word called Word, using data from File.
      File is an output2 file from Nastran, containing required data
      blocks.

      Each model word contains data blocks defining its characteristics
      taken from File.

      Later, saying the model word name puts its handle on the stack to
      be used for extracting data blocks from its local library.  See
      "Utilities for model words" in this file for simple words that
      fetch particular data blocks.

      Example: fetching basic grid definition for model called stator,
      and coordinate system definitions for model called rotor:

         stator bgp (bgpdt.stator)
         rotor cstm (cstm.rotor)
}
      [ {" This is the text for Word to be created:
           [ defname "Word" book

\            Inlines like this are used for big data blocks booked to
\            files instead of memory (using inline fbank--see below):

                "phi" this "." Word cat cat swap -inlinex \ phi.Word

           ] Word (qModel) \ Word puts qmodel on stk, the model's handle

        "} (hT) "make_model" book

        no no blockofblanks "Models" book
        no "OP2" book

\       This inline, called fbank, fbooks a matrix with name Mat.Word:
        "(qWord qMat) strchop '.' rot cat cat fbook" "fbank" inlinex
      ]
      true one STR stkok and, two STR stkok and not
      IF "model" stknot return THEN

      that filefound not
      IF " model: file not found: " other cat ersys return THEN

      "File" book, strchop "Word" book drop

    \ Adding Word to list Models, kept here in the local library:
      Models this Word grepe reach rows any not
      IF Models, Word pile onto Models THEN

    \ Making the model word:
      " Making model " . Word . nl
      CATMSG (f) no catmsg, make_model Word inlinex, (f) catmsg

      OP2 filetrue IF OP2 fclose THEN fortclose \ closing existing

    \ Opening the output2 file:
      File old binary "OP2" file \ open the file
      OP2 op2file not            \ map the file 

      IF fortclose OP2 fclose
         nl " model: not output2 file: " File cat ersys return 
      THEN

    \ Storing data blocks in local library of the model word:
      nl
      " Data blocks:" .
      " gpl"    dup . GPL    Word rot bank
      " bgpdt"  dup . BGPDT  Word rot bank
      " uset"   dup . USET   Word rot bank
      " gridmap" dup . GRIDmap Word rot bank
      " cstm"   dup . CSTM   Word rot bank

      " m66"  dup . OP2 "M66"   get2 Word rot bank
      " lama" dup . OP2 "LAMAI" get2 Word rot bank
      " phi"  dup . OP2 "PHIU1" get2 Word rot fbank \ to file, not mem

      nl fortclose OP2 fclose
   end

\  Utilities for model words:

   inline: bgp (hModel --- hBGPDT) "bgpdt" db ; \ grid point data table

   inline: cstm (hModel --- hCSTM) "cstm" db ; \ coordinate system defs

   inline: db (hModel qDB --- hDB) \ data block DB for model
      2dup extract rev "." rot cat cat naming ;

   inline: GEOM (hModel --- hT) \ geometry of Model
\     All grid locations and their motion reference systems in a table,
\     ready for eview.

      no STR stkok not IF "GEOM" stknot return THEN
      this model? not
      IF " GEOM: " swap " is not a model" cat cat . nl return THEN
      push
      peek gpl "%8.0f" format chop right justify " " tail \ grid IDs
      peek bgp 2nd 3 items catch " %8.3f " 3 cats format  \ location
      peek bgp 1st catch peek cstm inCSTM bend            \ coord sys
      "%8.0f %3.0f " "%8.3f " 12 cats cat format
      three parkn "_GEOM." pull cat naming
   end

   inline: gpl (hModel --- hGPL) "gpl" db ; \ grid point id list

   inline: gridmap (hModel --- hMAP) "gridmap" db ; \ grid-dof map 

   inline: lama (hModel --- hLAM) "lama" db ; \ modal frequencies

   inline: model? (hModel --- f) \ true flag if have library for Model
      "gpl" localref exists?
   end

   inline: m66 (hModel --- hM66) "m66" db ; \ 6-by-6 mass matrix

   inline: _phi (hModel --- hPHI) "phi" db ; \ modal matrix

   inline: uset (hModel --- hUSET) "uset" db ; \ bits defining sets

\-----------------------------------------------------------------------

\  Words for reading Nastran Output2 files. 

   inline: get2 (hFile qS --- hA) \ matrix named S from op2 file
\     Works for tables too.
      that filetrue not IF that notafile return THEN
      that op2file
      IF lop (qS) op2datablock (hA)
      ELSE " get2: not a valid Nastran output2 file" . nl
      THEN
   end

   inline: get2c (hFile qS hCols --- hA) \ Cols of matrix S from op2
      other filetrue not IF other notafile return THEN
      other op2file
      IF (hCols) hand "op2mat" "Cols" bank
         lop (qS) op2datablock (hA)
      ELSE " get2c: not a valid Nastran output2 file" . nl
      THEN
   end

   inline: op2datablock (qS --- hT) \ data block S from Nastran op2 file
\     Reads table or matrix S from Nastran op2 file.
\     Ignores character case of name S. 

      [ 8 is +type \ Fortran records to jump to reach data block type
      ]
      "op2map" "MapOP2" extract rows any not
      IF (qS) drop 
         " op2datablock: run op2file to map file" ersys purged return 
      THEN
 
      (qS) any? not
      IF "op2map" "Names" extract (hNames)
            no "op2map" "MapOP2" extract pile (hRecs)
            one those rows nit items one pile (hRows)
            park (hXY) fortrec? (rec) look ontop (row)
         (hNames row) ndx quote (qS)
      THEN

      (qS) "op2map" "Names" extract 
      lowercase that lowercase grepe any?
      IF ontop "nrec" book ELSE (qS) drop purged return THEN (qS)

      "op2map" "Tlr" extract nrec reach "trailer" book
      "op2map" "MapOP2" extract nrec pry

      +type plus "rec" book \ this is the 1-based rec where type is
      "fortfile" "Map" extract (hMap) rec ndx reach

      2nd catch ontop 0= \ data block type, 0 if table, 1 if mat
      IF \ " reading table " . dup (qS) . nl

         rec two plus fortseek
         depth push
         BEGIN
            fortrec \ read a record of the table
            fortrec fortendian import4 ontop 0< \ until negative key
         UNTIL
         depth pull less pilen

      ELSE \ " reading matrix " . dup (qS) . nl

       \ These phrases were written interactively, and are most
       \ easily understood by stepping through them interactively.

         "fortfile" "Map" extract (hMap)

       \ Pinpoint the rows in Map from fortfile that are for this 
       \ matrix:

         (hMap) this 2nd catch nullr    \ op2 data blocks end with 0 0
         1st rec items those rows teeth \ rake out unwanted top part
         rake lop 1st those rows items  \ reach these rows for mat

         swap rake swap 1st pry qdx nit \ 0 0 ending index this mat
         1st swap items reach           \ reach these rows for mat
         rec +d (hRows)                 \ add starting offset to each

       \ Fetch rows from Map for this matrix; there are at least 
       \ two rows per matrix column:
         (hMap hRows) reach (hMap)

       \ Park Fortran record numbers on the left:
         rec tic those rows items swap park (hMap1)

         trailer (hTlr)

         (hMap1 hTlr) op2mat (hA) \ reading the matrix cols

      THEN

      (qS) "_" rot cat naming
   end

   inline: op2file (hFile --- f) \ initialize to read Nastran op2 file
{     Returns flag f false if File is not a valid Nastran op2 file.

      Sets up the library in word fortfile if it currently is not
      servicing a file, or if the handle for the file it is servicing
      is no longer valid.

      See word op2map for the tables that are available after this
      word runs.
}
      [ "(hF1 hF2 --- f) less abs totals ontop 0<>" "differ?" inlinex ]

      "hNew" book
      hNew filetrue not IF false return THEN

      "fortfile" "File" extract "hOld" book

      hOld filetrue \ fortfile is still open
      IF hOld dup rewind
         hNew dup rewind differ?     \ error if not the same handles
         IF hNew fortfile drop false \ get error msg from word fortfile
            return
         THEN
      ELSE hNew fortfile
      THEN

      "op2map" "Fhandle" extract "hOld" book

      hOld filetrue \ Fhandle is still open
      IF hOld dup rewind
         hNew dup rewind differ? \ remap if not same handles
      ELSE true
      THEN (f)
      IF op2map \ Fortran file map and op2 map
         "op2map" "MapOP2" extract rows any
      ELSE true
      THEN (f)
   end

   inline: op2map ( --- ) \ map op2 file for reading
{     Here are phrases for obtaining useful tables from this word's 
      library:

         "op2map" "Names" extract  \ list of op2 data block names
         "op2map" "Tlr" extract    \ list of op2 data block trailers
         "op2map" "MapOP2" extract \ map of records to op2 data blocks

      This word is run by op2file.  Once op2file has run, the tables
      listed above are available.

      This word uses the Fortran file map from word fortfile, and makes
      and stores MapOP2.  MapOP2 is a list of Fortran record numbers 
      that precede each matrix and table on the op2 file.
 
      Description of Map from word fortfile:

      Each row of Map from word fortfile (which uses word fortmap), and
      used here to map the op2 file, pertains to a record in a Fortran 
      file.

      Column 1 of Map contains record size (bytes), column 2 contains
      the value of the first four bytes of the record interpreted as
      an integer, and column 3 contains the number of bytes offset to
      the record (a 0-based index).

      In a Nastran .op2 file, sequential records in Map that are of
      this form

         col 1  col 2   col 3
           4      -1   7453820 (header row minus 2, pattern 4 -1)
           4       7   7453832 (header row minus 1, pattern 4 7)
          28     103   7453844 (header row, pattern 28)

      precede each matrix or table.  Records of 28 generally correspond
      to the header record of a matrix or table.  If the item is a ma-
      trix, the header record contains the Nastran matrix trailer.
}
      [ purged "MapOP2" book, file.sizeof 1 null is Fhandle ]

      "fortfile" "File" extract "Fhandle" book

      "fortfile" "Map" extract (hMap)

      dup 1st catch "Map1" book
      2nd catch "Map2" book

      one Map1 rows items (hRecNo, 1-based list)

      Map1 28 those dims fill = (f1) \ rows with 28 in col 1
 
      Map1 4 those dims fill =
      Map2 7 those dims fill = and \ rows with 4 and 7
      one lag (f2) \ lagged 1 to align with the rows of 28

      Map1 4 those dims fill =
      Map2 -1 those dims fill = and \ rows with 4 and -1
      two lag (f3) \ lagged 2 to align with the rows of 28

      (f1 f2 f3) and and (hR) \ header rows contain true

      purged "Map1" book
      purged "Map2" book

      (hRecNo hR) rake lop dup "MapOP2" book

      (hMapOP2) push
      no no blockofblanks \ hNames
      no six null \ hTrailers (6 columns)

      peek rows 1st
      DO peek (hMapOP2) I pry (k) fortseek

         op2readtrailer (hTlr qName)

         push (hTrailers hTlr) bend pile
         swap (hNames qName) pull pile swap

      LOOP pull drop
      (hNames hTrailers) "Tlr" book, "Names" book
   end

   inline: op2mat (hMap hTrailer --- hA) \ mat A from op2 file
{     Incoming Map is a portion of the map from fortfile, with a list
      of Fortran record numbers tacked on the left.  The third column 
      of Map holds the 'keys' for an op2 file.  A negative key ends 
      the set of rows in Map that pertain to a matrix column.

      The four columns of Map contain:
         1 - Fortran record number, 1-based, suitable for fortseek
         2 - record size (bytes)
         3 - value of the first four bytes of the record, interpreted
             as an integer; in this case, the integer gives the start-
             ing row in the column where the data to be read will go
         4 - the number of file bytes offset to the record (0-based
             index for fseek)
}
      [
      \ A calling word can bank a list of desired columns into Cols
      \ and it will be used the very next time.  Word get2c uses this
      \ feature.

      \ If Cols is purged, all columns are returned:
           purged is Cols \ list of matrix column numbers to return
      ]
      (hTlr) this
      1st pry "nCols" book this
      2nd pry "nRows" book this
      3rd pry "Form" book 
      4th pry "Type" book

    \ Pull out just the relevant records (rows) from Map:
      (hMap) this 2nd catch four -d 0<>, that 3rd catch 0< or (hR)
      (hMap hR) rake lop (hMap) "Map" book

      Cols rows any
      IF Cols nCols ndx those dims fill > totals ontop 0<>
         IF " op2mat: one or more requested columns are out of range"
            ersys purged return
         THEN

         Map \ getting records for the listed Cols:

         1st Map rows items, Map 3rd catch (M1 M2) 2dup

         (M1 M2) 0< dup sling1 plus (hR) \ 1 at beginning of each col
         (M1 hR) rake lop Cols reach "r1" book \ start rows within Map

         (M1 M2) 0< rake lop
         Cols reach "r2" book \ ending rows within Map

         list: Cols rows 1st DO r1 I pry, r2 I pry thru LOOP ; (hR)
         (hMap hR) reach (hMap) "Map" book

         Map 3rd catch 0< totals abs ontop "nCols" book

         purged is Cols \ purging Cols for next time

      THEN

    \ The number of rows in Map column 3 that contain a negative key
    \ must equal the number of matrix columns, nCols:

      Map its 3rd catch 0< rake lop rows, nCols <>

      IF " op2mat: matrix columns and number of records do not agree"
         ersys purged return
      THEN

    \ Running speedy C word op2mat1 (it replaces slower high level 
    \ loop of op2mat now relegated to the Appendix):

      "op2map" "Fhandle" extract (hFile)

      Map nRows nCols Type fortendian op2mat1

      purged is Map \ purge Map; it can be big

    \ Keeping the Fortran record in synch with file pointer, since
    \ word op2mat1 bypasses the fortfile utilities:

      "op2map" "MapOP2" extract (hFortRec) dup
      "fortfile" "Map" extract 3rd catch (hOffsets)

      "fortfile" "File" extract fpos (hOffsets pos) bsearch drop
      (hFortRec nrec) bsearch drop 

      (hFortRec nrow) pry (nrec) fortseek
   end

   inline: op2readtrailer ( --- hTlr qName) \ data block trailer, name
{     Used by word op2map to read trailer record and data block name.  
      Assumes word op2map has positioned to the correct record.

      Items in rows of Nastran matrix trailer:
         1 - number of columns 
         2 - number of rows
         3 - form of matrix (1 square, 2 rectangular, 6 symmetric, ...)
         4 - type of matrix (1 SPreal, 2 DPreal, 3 SPcmplx, 4 DPcmplx)
         5 - largest number of nonzero SP words among all cols
         6 - density times 10,000
}
      fortrec fortendian import4
      2nd six items reach (hTlr)

      four fortstep \ move ahead four Fortran records, to Name

      fortrec (qName) 1st eight items catch strchop (qName)
   end

   inline: op2tlr (qS --- hA) \ trailer from Nastran data block S
\     Trailer for data block S from Nastran op2 file.
      (qS) "op2map" "Names" extract
      lowercase swap (qS) lowercase grepe any?
      IF ontop (row) "op2map" "Tlr" extract swap reach bend
         "_tlr" naming
      ELSE purged
      THEN
   end

   inline: scanop2 (qFile --- ) \ scan a Nastran Output2 file
\     Same as script usr/scanop2.
      "FILE" book
      FILE file? not IF "file not found: " . FILE . nl halt THEN

      FILE old binary "BIN" file # open file, make handle called BIN
      BIN op2file drop
      out cr0 # map the OP2 file, clear comment from screen
      fortclose

      "Name Rec Titles: Cols Rows Form Type Nzwd %Dens*100" # col titles

      BIN "toc2" >stk asciify    # run toc2 with output table to stk
      "Mapping...done" these chars spaces replace$

      left justify pile neat     # pile titles, neat columns
      "Trailer titles apply to matrices, not tables" # footnote

      pile dot nl fortclose BIN fclose
   end

   inline: toc2 (hFile --- ) \ table of contents of Nastran op2 file
{     Showing the contents of a Nastran op2 file, such as:

      [tops@steelee] ready > op2 toc2
       GPL     5 trailer:   2777       0       0       0       0       0
       USET   29 trailer:      0    7262   32768     515       0       0
       BGPDT  48 trailer:   2777       0    7262       2    1880 3110003
       M66    72 trailer:      6       6       6       2      12   10000
       LAMA  116 trailer:      1    2064       2       1    2064   10000
       PHIX  135 trailer:   2064    1230       2       2    1998    8081

      The number following the data block name is its Fortran record
      number, a 1-based index.

      The last table of contents written remains in this library, and
      it is called toc2.  

      To retrieve it, use: "toc2" "toc2" extract (hT)
                       or: "toc2" again extract (hT)
}
      [ 0 0 blockofblanks "toc2" book ]

      this filetrue not IF notafile return THEN
  
      this fpos push dup rewind
      (hFile) dup op2file (f)
  
      IF "op2map" "Names" extract 
         "op2map" "MapOP2" extract "%10.0f" format
         chop right justify " " nose park
         (hNames) one indent " trailer:" tail

         "op2map" "Tlr" extract (hTlr)
         "%8.0f" those cols cats format park
         dup "toc2" book, cr dot nl

      ELSE " toc2: not a valid Nastran output2 file" . nl
      THEN
      (hFile) pull fseek
   end

\-----------------------------------------------------------------------

\  Words for specific Nastran tables on Output2 files.

   inline: BGPDT ( --- hA) \ location of grid points in basic system 0
{     From Nastran BGPDT table, returned A has four columns: CID, X, Y,
      Z.

      Each row of A corresponds to a grid or an spoint, and the sequence
      of rows of A matches the internal order of grids and spoints.

      CID, 0 or greater, is the reference coordinate system for grid 
      motion.  CID=-1 (no coordinate system) for spoints.
}
      [ four makes Rows \ CID, X, Y, Z
        list: 0 1 1 1 ; makes R0
        list: 0 0 0 1 1 1 ; makes dpR0
      ]
      "BGPDT" op2datablock any? not IF "BGPDTS" op2datablock THEN any?
      IF (hT) dup fortendian import4 (hI)
      
         "op2datablock" "trailer" extract 4th pry 0=

         IF \ single precision table:
            (hT hI) these rows Rows /mod lop R0 swap repeat "R" book
            (hI) R rake drop (hI) swap
            (hT) fortendian import4f (hF) R rake lop (hF)
            (hI hF) R (hRake)

         ELSE \ double precision table:
            (hI) these rows 12 spikes rake lop (hI) swap
            (hT) fortendian import8 (hF)
            (hF) these rows dpR0 rows /mod lop dpR0 swap repeat (hR)
            (hF hR) rake lop (hI hF)
            R0 them rows repeat (hRake)

         THEN (hI hF hRake) tier Rows foldr bend

         "_BGPDT" naming

      ELSE purged
      THEN
   end

   inline: CSTM ( --- hA) \ coordinate system definitions
{     Entire Nastran CSTM table into A, each column pertaining to 
      a coordinate system.

      Returned A has 14 rows:
         Row 1: coordinate system ID
         Row 2: coordinate system type (1=rec, 2=cyl, 3=sph)
         Row 3-5: [t1 t2 t3] (origin, Ob)
         Row 6-14: [r11 r21 r31 r12 r22 r32 r13 r23 r33] (dir cos, Cba)

      Origin, Ob, and direction cosine matrix, Cba, are used as follows
      to express frame A vector, xa, in frame B:

         xb = Ob + Cba*xa

      where Ob is a vector from the origin of B to the origin of A,
      expressed in B (returned in rows 3-5), and Cba is the direction
      cosine matrix that expresses frame A vector, xa, in frame B (re-
      turned in rows 6-14: a 3-by-3 matrix stored by columns).

      For all matrices, frame B is the rectangular reference frame of 
      a Nastran model, called the basic system or system 0.
}

      [ list: 0 0 ; 12 ones pile makes R0, R0 rows makes Rows 

\       From Nastran, 3-by-3 direction cosine matrices are stored by 
\       rows; these vectors reach terms in a sequence such that the
\       terms are stored by columns in returned A:

         \ Case of single precision CSTM:
              list: 1 2 3 4 5 ; \ ID, type, origin
            \ Rows to reach for storing dir cos by columns:
              list: 1 4 7 2 5 8 3 6 9 ; 5 +d \ dir cos
              pile makes ReachSP \ for single precision data

         \ Case of double precision CSTM:
              list: 1 2 3 ; \ origin
            \ Rows to reach for storing dir cos by columns:
              list: 1 4 7 2 5 8 3 6 9 ; 3 +d \ dir cos
              pile makes ReachDP \ for double precision data
      ]

      "CSTM" op2datablock any? not IF "CSTMS" op2datablock THEN any?
      IF (hT) this fortendian import4 (hI)

         "op2datablock" "trailer" extract 4th pry 0=

         IF (hT hI) \ single precision matrices:

            these rows R0 rows /mod lop R0 swap repeat "R" book
            (hI) R rake drop (hI) swap
         
            (hT) fortendian import4f (hF) R rake lop (hF)
            (hI hF) R tier Rows foldr ReachSP ndx reach

         ELSE (hT hI) lop \ double precision matrices:

            (hI) four foldr bend \ cols: ID, type, IntRec, RealRec
            yes 4th sorton (hT)  \ sort rows by vals in RealRec column
            1st two items (ID, type) catch bend (hID)

          \ Reading RealRecs (all in one Fortran record), 12 items
          \ per RealRec:
            fortrec? (rec) four plus fortseek \ jumping 4 records ahead
            fortrec fortendian import8 12 foldr ReachDP ndx reach (hC)
            (hID hC) pile 
         THEN (hA)

       \ Putting columns in ascending sort by ID:
         (hA) this 1st reach bend, 1st those rows items park
         (hB) yes sort 2nd catch (hA hR) catch (hA)

         "_CSTM" naming

      ELSE purged
      THEN
   end

   inline: GRIDmap ( --- hL) \ starting dof for each grid and spoint
{     Rows in list L correspond to internal sequence; the length of L
      equals the total number of grids plus spoints.

      This word requires BGPDT and GPL from Nastran op2 file.

         1: starting dof (one-based)
         2: external number of grid or spoint
         3: number of dof (6 if grid, 1 if spoint)

      Using CID=-1 from BGPDT to distinguish spoints from grids.
}
      BGPDT 1st catch (CID) 0< (hF) push
      list: one \ start one-based Nastran DOF indexing
         peek (hF) rows 2nd
         DO peek (hF) I 1- pry 0= IF 6 ELSE 1 THEN that plus
         LOOP pull drop
      end 

      (hDOFstart)       \ 1. Starting DOF
      GPL               \ 2. External ID
      over delta -1 lag \ 3. Number of DOF

      three parkn "_DOF" naming
   end

   inline: EQEXIN ( --- hA) \ external/internal equivalence list
\     Returns Nastran EQEXIN table.
      "EQEXIN" op2datablock any? not IF "EQEXINS" op2datablock THEN any?
      IF (hT) fortendian import4 "_EQEXIN" naming
      ELSE purged
      THEN
   end

   inline: GID (hMAP hDOF --- hG) \ grid ID+dof list for given DOF list
{     Incoming values in vector DOF are 1-based g-set degrees-of-
      freedom.

      This word requires MAP created by GRIDmap.

      For each row of incoming DOF, column 1 of returned G holds the
      corresponding grid or spoint number, and column 2 holds the local
      degree-of-freedom (1-6 for grid, 0 for spoint).

      Warning: does not check if an item in incoming DOF list exceeds 
      the maximum degree-of-freedom.
}
      (hMAP hDOF) hand swap this rev
      1st catch 1st those rows items park over look (hStart)
      rot swap (hGRIDmap hStart) reach
      this 2nd catch (hG)
      rot rot 1st catch less one +d (hD)
      park "_GID" naming
   end

   inline: GPL ( --- hA) \ grid point list in internal order
\     Returns Nastran GPL table.
      "GPL" op2datablock any? not IF "GPLS" op2datablock THEN any?
      IF (hT) fortendian import4 "_GPL" naming
      ELSE purged
      THEN
   end

   inline: USET ( --- hA) \ bit sets for all degrees-of-freedom
{     Returns Nastran USET table.

      Returned A is a one-column matrix with number of rows equal to
      the degrees-of-freedom in the model.

      Ones in the leftmost 32 bits of row k of A define the sets in
      which degree-of-freedom k appears; Nastran is restricted to 32
      sets, so the remaining 32 bits in row k are null.

      See Nastran Programmer's Manual, Vol. I, p. 2.3-100 for defini-
      tions of bit positions for the various 32 sets.  Here are some
      often-used values:
            Bit Position     Set
                1 - 6      U1 - U6
                  10          A
                  11          B
                  24          L
                  25          A
                  26          F
                  27          N
                  28          G
                  29          R
                  30          O
                  31          S
                  32          M
}
      "USET" op2datablock any?
      IF "uset" book
         uset \ expanding with nulls from 32- to 64-bit elements

         list: 0 0 0 0 1 1 1 1 ; those chars four slash repeat
         1st those rows items bob rake trash

         NULLch them chars two star cats hand (uset8)
         dup push (uset hRows uset8) cram pull (uset8) vol2mat (hA)

         "_USET" naming, purged "uset" book
      ELSE purged
      THEN
   end

   inline: USETbit (hMAP hUSET n --- hG) \ grid ID+dof list for set n
{     Return G for grids and spoints that occupy set n; n is a bit 
      number from 1 to 32.  

      The set occupancy for each degree-of-freedom (6 for each grid 
      and 1 for each spoint) is determined by the bit pattern in the 
      vector returned by word USET.
 
      This word requires MAP from word GRIDmap and USET from word USET.

      Column 1 of returned G holds a grid or spoint number, and column
      2 holds the local degree-of-freedom (1-6 for grid, 0 for spoint).
}
      (hMAP hUSET n) dup (n) push ndx bit 
      one \ start one-based Nastran DOF indexing
      those rows items swap rake lop (hMAP hDOF) GID
      (hG) "_USETbit_" pull (n) suffix naming
   end

   pull catmsg private halt

\-----------------------------------------------------------------------

;  Appendix

   Lines to test word DOF_SET:

     junk junk junk
     junk junk junk
         G        DISPLACEMENT

      1= 101-1 101-2 101-3 101-4 101-5 101-6 102-1 102-2 102-3 102-4 =10
     11= 102-5 102-6 103-1 103-2 103-3 103-4 103-5 103-6 104-1 104-2 =20

         G        DISPLACEMENT

     21= 104-3 104-4 104-5 104-6 105-1 105-2 105-3 105-4 105-5 105-6 =20
     31= 106-0 107-0 
    junk junk junk
    junk junk junk

\-----------------------------------------------------------------------

  _inline: op2mat (hMap hTrailer hCols --- hA) \ mat A from op2 file
{     Incoming Map is a portion of the map from fortfile, with a list
      of Fortran record numbers tacked on the left.  The third column
      of Map holds the 'keys' for an op2 file.  A negative key ends
      the set of rows in Map that pertain to a matrix column.

      The four columns of Map contain:
         1 - Fortran record number, 1-based, suitable for fortseek
         2 - record size (bytes)
         3 - value of the fires four bytes of the record, interpreted
             as an integer; in this case, the integer gives the start-
             ing row in the column where the data to be read will go
         4 - the number of file bytes offset to the record (0-based
             index for fseek)

      The six rows of Trailer contain:
         1 - number of columns
         2 - number of rows
         3 - form of matrix (1 square, 2 rectangular, 6 symmetric, ...)
         4 - type of matrix (1 SPreal, 2 DPreal, 3 SPcmplx, 4 DPcmplx)
         5 - largest number of nonzero SP words among all cols
         6 - density times 10,000
}
      (hCols) "Cols" book

      (hTlr) this
      1st pry "nCols" book this
      2nd pry "nRows" book this
      3rd pry "Form" book
      4th pry (type) dup
      (type) 2 mod 0= "DP" book \ DP=true if double precision
      (type) 2 > "CP" book      \ CP=true if complex

      "Map" book

    \ The number of rows in Map column 3 that contain a negative key
    \ must equal the number of matrix columns, nCols:

      Map its 3rd catch 0< rake lop rows, nCols <>

      IF " op2mat: matrix columns and number of records do not agree"
         ersys purged return
      THEN

    \ Pull out just the relevant records (rows) from Map:
      Map this 2nd catch four -d 0<>, Map 3rd catch 0< or (hR)
      (hMap hR) rake lop (hMap) "Map" book

    \ Reading matrix columns.  Each column record has a 4-byte int
    \ starting row (1-based), followed by floating point bytes.
    \ If the record has 4 or fewer bytes, the column is null.

      [ 4 1 null "R0" book ]

      DP IF "import8" ELSE "import4f" THEN ptr (n) "import" book

      "op2map" "Fhandle" extract "File" book

time push
      nRows one null (hC0) \ first column
      Map rows 1st
      DO Map I reach, (hR) this 3rd pry its 0>
         IF (hR n) "r0" book
            File that 4th pry fseek
            File swap 2nd pry fget (hT)
            (hT) R0 those chars four less ones pile (hR) claw lop (hT)
            (hT) any?
            IF (hC hT) fortendian import exe (hC)
               (hC0 hC) r0 ndx those rows items them ram
            THEN
         ELSE (hR r0) 2drop nRows one null (hC0) \ starting next column
         THEN
      LOOP drop \ getting rid of unneeded next column
      nCols parkn
time pull less " ET: . .i nl
   end

\-----------------------------------------------------------------------

\  Early, experimental words for reading Nastran op2 files.

  _inline: get2 (hFile qS --- hA) \ THIS IS FOR LEARNING ONLY
\     Reading data block named S from Nastran Output2 file.

\     If file is of different endian than the machine, reset ENDIAN
\     in this library with a phrase such as (see man endian):

\        PDP_ENDIAN "get2" "ENDIAN" bank

      [ endian is ENDIAN ] \ the machine's endian is default

      (qS) into NAME
      purged is TRAILER
      no is ERR
      no is END

      (hFile) ENDIAN fortfile

      fortrewind
      BEGIN
         fortint (k) its 0=
         IF drop true "END" book
         ELSE (k) its 3 =
            IF (k) drop, op2labelskip
            ELSE (k) its 1 = \ skip remainder if k=1
               IF (k) drop, op2recordskip drop
               ELSE (k) 2 <>
                  IF yes "ERR" book
                  ELSE op2readtrailer (hTlr qName) NAME strcmp 0<>
                     IF (hTlr) drop \ not NAME
                     ELSE \ found NAME
                        (hTlr) "TRAILER" book
                        op2key (k f)
                        IF dup this (k) 2 > swap (k) 4 <> and
                           IF (k) two less "LENHD" book
                              zero "ITYPE" book
                              true (f) \ data is in header record
                           ELSE (k) drop
                              fortskip \ skipping 2nd instance of NAME
                              fortint (k) -3 =
                              IF fortint (k) 1 =
                                 IF fortint "ITYPE" book
                                    fortbackspace fortbackspace
                                 THEN
                              ELSE yes "ERR" book
                              THEN
                              false (f) \ data is not in header record
                           THEN (f) "INHEAD" book
                        ELSE (k) drop yes "ERR" book
                        THEN
                     THEN
                  THEN
                  \ <<< traffic here for each matrix on file
               THEN
               \ <<<< tons of traffic here, skipping recs to find NAME
            THEN
         THEN
         TRAILER rows any \ found Name if TRAILER not purged
         ERR or
         END or
      UNTIL
   end

  _inline: op2file (hFile --- f) \ initialize to read Nastran op2 file
{     Returns flag f false if File is not a valid Nastran op2 file.

      Sets up the library in word fortfile if it currently is not
      servicing a file, or if the handle for the file it is servicing
      is different from the handle of incoming File.

      See word op2map for the tables that are available after this
      word runs.
}
      this filetrue not IF drop false return THEN

      "fortfile" "File" extract (hFile) filetrue not
      IF (hFile) fortfile op2map \ Fortran file map and op2 map

      ELSE (hFile) file.handle pry (hF1) \ fortfile .handle
         "op2map" "Fhandle" extract (hFile2) dup push
         (hFile2) file.handle pry (hF2)  \ op2map .handle
         (hF1 hF2) <>                    \ different .handles?
         pull (hFile2) filetrue not or   \ or invalid handle?
         IF op2map THEN                  \ new map for op2
      THEN
      "op2map" "MapOP2" extract rows any
   end

  _inline: op2key ( --- key f)
      [ " op2key: bad key = " is msg ]

      fortint (k) dup 1 <>
      IF msg over int$ cat ersys fail return THEN drop

      fortint (k) dup 0<>
      IF msg over int$ cat ersys fail return THEN drop

      fortint (k) dup 2 <
      IF msg over int$ cat ersys fail return THEN true
   end

  _inline: op2labelskip ( --- ) \ skip output2 file label
\     Skipping label of 7 records.
      "fortfile" "Rec" extract seven plus fortseek
   end

  _inline: op2readtrailer ( -- hTlr qName) \ data block trailer and name
{     Items in trailer:
         1 number of columns
         2 number of rows
         3 form of matrix (square, 2 rectang, 6 symmetric, ...)
         4 type of matrix (1 SPreal, 2 DPreal, 3 SPcmplx, 4 DPcmplx)
         5 largest number of nonzero SP words among all columns
         6 density*10,000
}
      [ no is echo ]
      fortrec (qName) 1st quote strchop (qName)

      fortint (k) -1 <>
      IF yes "get2" "ERR" bank, purged swap return THEN

      fortint (k) 7 <>
      IF yes "get2" "ERR" bank, purged swap return THEN

      fortrec fortendian import4
      2nd six items reach (hTlr)
      swap (hTlr qName)

      echo
      IF " Datablock " over cat dot
         " trailer: " dot over bend
         "%6.0f" three pick rows cats format dot nl
      THEN

      fortint (k) -2 <>
      IF yes "get2" "ERR" bank, purged swap return THEN
   end

  _inline: op2recordskip ( --- k) \ skip rest of record after key=1
      zero is k
\     Testing for end of data block
      fortint (k1) 0<>
      fortint (k2) this "k2" book 0<>
      (fk1 fk2) or
      IF k2 0< not \ not end of data block
         IF BEGIN fortrec drop fortint (k) 0> not UNTIL THEN
      ELSE one is k \ end of data block
      THEN k
   end

  _inline: toc2 (hFile --- )
      yes "op2readtrailer" "echo" bank
      "" get2
      no "op2readtrailer" "echo" bank
   end

   private halt

