/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2014  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef X11
/* xterm.c  October 1999

   Copyright (c) 1999-2014   D. R. Williamson

X11 references:

   1. Scheifler, R. W., J. Gettys, et al, "X Window System," third
      edition, Digital Press, 1992.

   2. Nye, A., "Xlib Programming Manual for Version 11, Volume One,"
      third edition, O'Reilly & Associates, 1992.

   3. Redhat 5.2 Linux: /usr/doc/HOWTO/XFree86-HOWTO

   4. Linux and Unix:
         xterm -help
         man xterm
         man XXX for X11 command XXX, i.e., man XrmMergeDatabases

   5. Some X11 elements adapted from: 
         Gnuplot, gplt_x11.c, distributed with Redhat 5.2 Linux
         Copyright 1986-1993, 1998   Thomas Williams, Colin Kelley

         Information about one window/plot, from Gnuplot:
            typedef struct plot_struct {
                    Window window;
                    Pixmap pixmap;
                    unsigned int posn_flags;
                    int x,y;
                    unsigned int width, height; / window size /
                    unsigned int px, py; / pointsize /
                    int ncommands, max_commands;
                    char **commands;
            } plot_struct;
*/
#include <stdio.h>
#include <stdlib.h>

#include "main.h"
#ifndef DEPSTK
   #include "stk.h"
#endif

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#ifndef MAINLIB
   #include "lib.h"
#endif
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "term.h"
#include "tex.h"
#include "tag.h"

/*---------------------------------------------------------------------

   The following is from term.h, and is shown as a comment here only 
   for reference.  

   These enums match the elements in the structs that are defined in 
   xterm.v.

   Uppercase enum items correspond to similarly ordered elements in
   structures of xterm.v:
enum _ecb {BP,BR,CE,CM,EE,KP,KR,ME,VE,ecbsize};
enum _gcb {GCC,GAM,GAV,gcbsize};
enum _gav {FUNC,PLMASK,FORE,BACK,LINEWID,LINESTY,CAPSTY,JOINSTY,
           FILSTY,FILRUL,ARCMOD,PIXTILE,PIXSTIP,TSX,TSY,FONT,SUBMOD,
           EXPOSE,CLIPX,CLIPY,CLIPMASK,DASHOFF,DASHES,GAVsize};
enum _pcb {PIX,PW,PH,XHOT,YHOUT,pcbsize};
enum _wav {BACKPMAP,BACKPIX,BORDPMAP,BORDPIX,BGRAV,WGRAV,BACKSTOR,
           BACKGPLANE,BACKGPIX,SAVEU,EVENTM,PROPNOM,OVERRIDE,
           COLORM,CURSOR,WAVsize};
enum _wcb {WIN,USR,W,H,X,Y,FLAGS,WAM,WAV,ECB,TYP,wcbsize};

----------------------------------------------------------------------*/

/* Added line styles. 

   See line drawing examples that can be run at the ready prompt in
   XDrawLine1(), this file.  Search for "Drawing examples." 

   Wed Sep  8 12:48:55 PDT 2010.  Add LineRectangle.
   Tue May 21 09:22:20 PDT 2013.  Add LineStep. 

   Macros for added styles are defined in xterm.v to be consistent with
   the following enum called added_line_styles: */
   enum added_line_styles {
      LineRectangle=LineDoubleDash+1,
      LineStep=LineDoubleDash+2,
   };

/* Notes:

   See /usr/include/X11/X.h for standard line styles. 

   Added line styles are hidden from the X11 system in GCcreate() and 
   GCline(), and are referenced in functions such as line(), linec(), 
   linet() and XDrawLine1(). */

/*--------------------------------------------------------------------*/

/* For point-to-point drawing, this loop appears in linec(), linet() 
   and linet1():*/
   #define LINEDIAG \
      for(k=0;k<lines;k++) { \
         if(bounds1(*X,*Y,*(X+1),*(Y+1),Ymin,Ymax,&X2,&Y2)) { \
            if(bounds1(*(X+1),*(Y+1),*X,*Y,Ymin,Ymax,&X1,&Y1)) { \
               if(!(X1==X2 && Y1==Y2)) { /* not a single point: */ \
                  segs++; \
                  *x1=(short)X1; \
                  *y1=(short)Y1; \
                  *x2=(short)X2; \
                  *y2=(short)Y2; \
                  x1++; y1++; x2++; y2++; \
               } \
            } \
         } \
         X--; \
         Y--; \
      }

/*--------------------------------------------------------------------*/

/* For LineStep drawing, this loop appears in linet() and linet1(): */
   #define LINESTEP \
   /* LineStep makes one diagonal line into two: \
\
         a horizontal line from (x1,y1) to (x2,y1) \
         a vertical line from (x2,y1) to (x2,y2). \
\
      This line style corresponds to waiting from x1 to x2 to receive \
      information that y has changed from y1 to y2; point-to-point \
      drawing does not model this. */ \
\
      for(k=0;k<lines;k++) { \
         if(*X>Xmin && *(X+1)<Xmax) { \
         /* Horizontal line: */ \
            Y1=*Y; /* horizontal line is at height Y1 */ \
            if(Y1<Ymax && Y1>Ymin) { \
               segs++; \
               *x1=(short)*X; \
               *x2=(short)MIN(*(X+1),Xmax); \
               *y1=*y2=(short)Y1; \
               x1++; y1++; x2++; y2++; \
            } \
         /* Vertical line: */ \
         /* Fri Feb 14 13:36:12 PST 2014.  Corrected to draw a \
            vertical line when both Y1 and Y2 are outside the \
            zoomed region: */ \
            if(*Y>*(Y+1)) { Y2=*Y; Y1=*(Y+1); } \
            else { Y1=*Y; Y2=*(Y+1); } \
            Y1=MIN(MAX(Y1,Ymin),Ymax); \
            Y2=MAX(MIN(Y2,Ymax),Ymin); \
            if(Y1>=Ymin && Y2<=Ymax) { \
               segs++; \
               *x1=*x2=(short)*(X+1); \
               *y1=(short)Y1; \
               *y2=(short)Y2; \
            } \
            x1++; y1++; x2++; y2++; \
         } \
         X--; \
         Y--; \
      }

/*--------------------------------------------------------------------*/

/* wincreate() */
struct wcb *winlast=NULL;
struct wcb *wintable=NULL;

int bitmapf() /* bitmapf (hP qFile --- ) */
/* Read bitmap file and store elements into pixmap control block. */
{
   Pixmap pix;
   double *hP;
   unsigned int height,width;
   int ret,xhot,yhot;

   if(!Dpy) {
      stkerr(" bitmapf: ",TERMNOT);
      return 0;
   }
   if(tos->typ!=STR) {
      stkerr(" bitmapf: ",STRNOT);
      return 0;
   }
   strchop();

   ret=XReadBitmapFile(Dpy,Root,tos->tex,&width,&height,&pix,
      &xhot,&yhot);

   drop();
   if(tos->typ!=MAT) {
      stkerr(" bitmapf: ",MATNOT);
      XFreePixmap(Dpy,pix);
      return 0;
   }
   if(tos->row!=pcbsize) {
      gprintf(" bitmapf: require hP vector of %d rows",pcbsize); nc();
      stkerr("","");
      XFreePixmap(Dpy,pix);
      return 0;
   }
   dup1s();
   named(); /* name of hP onto stack */
   if(catfetch(tos->tex)==NULL) {
      stkerr(" bitmapf: ",PIXCATNOT);
      drop();
      XFreePixmap(Dpy,pix);
      return 0;
   }
   drop(); /* name qS off stack */
   if(TRACE) {
      gprintf(" bitmap address: %lX",pix); nc();
      gprintf(" bitmap width, height: %d-by%d",width,height); nc();
   }
   hP=tos->mat;
/* WORK IN PROGRESS; THIS IS NOT FINISHED */
   return 1;
}

int bounds(double x1, double y1, double x2, double y2, \
   double xMin, double xMax, double yMin, double yMax, \
   double *X, double *Y)
/* Clip point (x2,y2) of line (x1,y1)-(x2,y2) to keep within bounds
   of rectangle with opposing corners (xMin,yMin), (xMax,yMax). 

   Returns 1 if point is inside rectangle;
   Returns 0 if line has been clipped. */
{
   *X=x2;
   *Y=y2;

   if(outside(Y,yMin,yMax)) 
      if(interpolate(x1,y1,x2,y2,X,Y)) return 0;

   if(outside(X,xMin,xMax)) 
      if(interpolate(y1,x1,y2,x2,Y,X)) return 0;

   return 1;
}

int bounds1(double x1, double y1, double x2, double y2, \
   double yMin, double yMax, double *X, double *Y)

/* Clip point (x2,y2) of line (x1,y1)-(x2,y2) to keep within bounds
   of rectangle with opposing corners (xMin,yMin), (xMax,yMax).

   Used by linet() where only Y is subject to clipping.

   Returns 1 if point is inside rectangle;
   Returns 0 if line has been clipped. */
{
   *X=x2;
   *Y=y2;
   if(outside(Y,yMin,yMax))
      if(interpolate(x1,y1,x2,y2,X,Y)) return 0;

   return 1;
}

int colorpix() /* colorpix (qColor --- pix) */
{
   XColor pixscreen,pixexact;
   int ret;

   if(tos->typ!=STR) {
      stkerr(" color: ",STRNOT);
      return 0;
   }
   strchop();
   ret=XLookupColor(Dpy,DefaultColormap(Dpy,Scr),tos->tex,&pixexact,
      &pixscreen);
   if(!ret) {
      stkerr(" colorpix: ",COLORNOT);
      return 0;
   }
   XAllocNamedColor(Dpy,DefaultColormap(Dpy,Scr),tos->tex,&pixscreen,
      &pixexact);

   return(drop() && pushuint((unsigned long)pixscreen.pixel));
}

int defButtonPressedEvent() /* (hE --- ) */
{
/* From tests: button (BPbu): 1 left, 2 both (or middle), 3 right */

   gprintf(" defButtonPressedEvent"); nc();
   return 1;
}

int defButtonReleasedEvent() /* (hE --- ) */
{
/* From tests: button (BRbu): 1 left, 2 both or middle), 3 right */
   
   gprintf(" defButtonReleasedEvent"); nc();
   return 1;
}

int defClientMessageEvent() /* (hE --- ) */
{
   gprintf(" defClientMessageEvent"); nc();

/*
               if(event.xclient.message_type == WM_PROTOCOLS &&
                  event.xclient.format == 32 &&
                  event.xclient.data.l[0] == WM_DELETE_WINDOW) {
printf(" delete plot.  WM_PROTOCOLS,WM_DELETE_WINDOW: %X,%X\n\r",WM_PROTOCOLS,WM_DELETE_WINDOW);
                  plot_struct *plot = find_plot(event.xclient.window);
                  if (plot) delete_plot(plot);
               }
*/

   return 1;
}

int defConfigureEvent() /* (hE --- ) */
{
   gprintf(" defConfigureEvent"); nc();
   return 1;
}

int defExposeEvent() /* (hE --- ) */
{
   gprintf(" defExposeEvent"); nc();
   return 1;
}

int defKeyPressedEvent() /* (hE --- ) */
{
   gprintf(" defKeyPressedEvent"); nc();
   return 1;
}

int defKeyReleasedEvent() /* (hE --- ) */
{
   gprintf(" defKeyReleasedEvent"); nc();
   return 1;
}

int defMotionEvent() /* (hE --- ) */
{
   gprintf(" defMotionEvent"); nc();
   return 1;
}

int defVisibilityEvent() /* (hE --- ) */
{
/* From tests:
      state (VEst): 
           0 when fully visible,
           1 when partially covered 
           2 when fully covered */

   gprintf(" defVisibilityEvent"); nc();
   return 1;
}
   
int fontload() /* fontload (qS --- 0 | XFont fid) */
/* For given font name, S, loads font and returns XFontStruct pointer,
   XFont, and fontid, fid. */
{
   XFontStruct *font;
   Font fid;

   if(tos->typ!=STR) {
      stkerr(" fontload: ",STRNOT);
      return 0;
   }
   strchop();
   font=XLoadQueryFont(Dpy,tos->tex);
   if(font==NULL) {
      return(drop() && pushint(0));
   }
   fid=font->fid;

   return(
      drop() &&
      pushuint((unsigned long)font) &&
      pushuint((unsigned long)fid) 
   );
}

int GCclip() /* GCclip (hL hGCB --- ) */
/* Fri Jul 25 06:35:34 PDT 2014. 

   Vector GCB is a graphics control block (gcb) as defined in xterm.v,
   and it contains the graphic context, gc, being set here.

   Set the clip region in GCB(gc).  From L, clip rectangle x, y, width
   and height are computed and a corresponding region in graphics con-
   text (GCB(gc)) is stored.

   When GCB(gc) is used later, (x,y) is added to the origin of the dis-
   play, (clip_x_origin, clip_y_origin), to establish the origin of the
   clip region.

   Usage of X11 functions to set a region for clipping is shown in Ref-
   erence 2, Example 3-15 (p. 78).

   Reference 2, Figure 5-9 (p. 134) shows the clip region and its rela-
   tionship to clip_x_origin, and clip_y_origin.

   Regarding (clip_x_origin,clip_y_origin):

      Unless set in gav attributes when GCB is created (by GCcreate()), 
      the clip origin is at (clip_x_origin,clip_y_origin)=(0,0).  

      Alternatively, function XSetClipOrigin(Dpy,gc,x,y) can be used to 
      set (clip_x_origin,clip_y_origin)=(x,y).  See Reference 1, p 187.

      If (clip_x_origin,clip_y_origin)=(0,0), then L, in effect, de-
      fines the origin of the clip region when Xrectangle elements
      rec.x and rec.y are set to xmin and ymin, respectively, in the 
      expressions below.

   Warning: since GCB disappears from the stack after this function
   runs, changes made to GCB(gc) will be lost unless GCB is an item in
   the catalog, or duplicated elsewhere on the stack, or duplicated on
   the local stack (see push). */
{
   GC gc;
   Region reg;
   XRectangle rec;

   double *GCB,*L;
   short xmax,xmin,ymax,ymin;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" GCclip: ",MATNOT2);
      return 0;
   }
   if(tos->row!=gcbsize) {
      gprintf(" GCclip: require GCB vector of %d rows",gcbsize);
      nc();
      stkerr("","");
      return 0;
   }
   GCB=tos->mat;
   gc=(GC)patlong(*(GCB+GCC)); /* GCC element holds gc */

   L=(tos-1)->mat;
   xmin=(short)MIN(*L,*(L+1));
   xmax=(short)MAX(*L,*(L+1));
   ymin=(short)MIN(*(L+2),*(L+3));
   ymax=(short)MAX(*(L+2),*(L+3));

   rec.x=xmin;
   rec.width=xmax-xmin;
   rec.y=ymin;
   rec.height=ymax-ymin;

   reg=XCreateRegion();
   XUnionRectWithRegion(&rec,reg,reg);
   XSetRegion(Dpy,gc,reg);
   XDestroyRegion(reg);

   return(drop2());
}

int GCcreate() /* GCcreate (hW hGC --- ) */
/* Create a graphics context.  Vector hGC is a graphics context block
   (gcb) structure, as defined in xterm.v.  

   Vector hW is a window control block (wcb) as defined in xterm.v.

   While wcb hW is provided here to make GC, this GC can be used for 
   drawing with any other hW of the same depth.

   Warning: since GC disappears from the stack after this function runs,
   before calling this function GC must be an item in the catalog, or
   duplicated elsewhere on the stack, or duplicated on the local stack
   (see push).

   Tue May 21 09:22:20 PDT 2013.  For linestyle=LineStep, set line 
   style in gcv.line_style to LineSolid.  LineStep will later be used
   in linet().

   Wed Sep  8 12:48:55 PDT 2010.  If linestyle==LineRectangle, set line
   style in gcv.line_style to LineSolid.  Line style LineRectangle will
   later be used to draw lines where the lines are the diagonals of 
   rectangles; see linet(). */
{
   double *hA,*hGC,*hW;
   XGCValues gcv;
   unsigned long GCmask;
   GC GContext;
   Window win;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" GCcreate: ",MATNOT2);
      return 0;
   }
   hW=(tos-1)->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */
   if(!winfind(win,wintable)) {
      stkerr(" GCcreate: ",WINNOT);
      return 0;
   }
   if(tos->row!=gcbsize) {
      gprintf(" GCcreate: require hGC vector of %d rows",gcbsize); 
      nc();
      stkerr("","");
      return 0;
   } 
   hGC=tos->mat;
   if(*(hGC+GCC)>0) {
      dup1s();
      GCfree();
   }
/* Graphics context attributes mask (GAM) and vector (GAV): */
   if(*(hGC+GAM)!=-INF) {
   /* Bit pattern of GAM mask element is already unsigned long, so
      just copy it into GCmask: */
      memcpy(&GCmask,(unsigned long *)(hGC+GAM),sizeof(long));

      if(*(hGC+GAV)!=-INF) {
         pushd(*(hGC+GAV));
         exe(); /* putting GC attributes vector on tos */
         if(tos->row!=GAVsize) {
            gprintf(" GCcreate: require attributes vector of %d rows",
               GAVsize); 
            nc();
            stkerr("","");
            return 0;
         }
         hA=tos->mat; /* graphics context attributes vector */
         gcv.function=(int)patint(*(hA+FUNC));
         gcv.plane_mask=patlong(*(hA+PLMASK));
         gcv.foreground=patlong(*(hA+FORE));
         gcv.background=patlong(*(hA+BACK));
         gcv.line_width=patint(*(hA+LINEWID));

      /* Special handling for added line styles (also in GCline()): */
         if(*(hA+LINESTY)==LineRectangle || *(hA+LINESTY)==LineStep)
            gcv.line_style=LineSolid;
         else gcv.line_style=patint(*(hA+LINESTY));

         gcv.cap_style=patint(*(hA+CAPSTY));
         gcv.join_style=patint(*(hA+JOINSTY));
         gcv.fill_style=patint(*(hA+FILSTY));
         gcv.fill_rule=patint(*(hA+FILRUL));
         gcv.arc_mode=patint(*(hA+ARCMOD));
         gcv.tile=(Pixmap)patlong(*(hA+PIXTILE));
         gcv.stipple=(Pixmap)patlong(*(hA+PIXSTIP));
         gcv.ts_x_origin=patint(*(hA+TSX));
         gcv.ts_y_origin=patint(*(hA+TSY));
         gcv.font=(Font)patlong(*(hA+FONT));
         gcv.subwindow_mode=patint(*(hA+SUBMOD));
         gcv.graphics_exposures=(Bool)patint(*(hA+EXPOSE));
         gcv.clip_x_origin=patint(*(hA+CLIPX));
         gcv.clip_y_origin=patint(*(hA+CLIPY));
         gcv.clip_mask=(Pixmap)patlong(*(hA+CLIPMASK));
         gcv.dash_offset=patint(*(hA+DASHOFF));
         gcv.dashes=(char)patint(*(hA+DASHES));
         drop();
      }
      else {
         stkerr(" GCcreate: ",GCATTNOT);
         return 0;
      }
   }
   else {
      GCmask=GCFunction | GCBackground | GCForeground;
      gcv.function=GXcopy;
      gcv.background=XBlackPixel(Dpy,Scr);
      gcv.foreground=XWhitePixel(Dpy,Scr);
   }
/* Creating GC from GCmask and properties in gcv: */
   GContext=XCreateGC(Dpy,win,GCmask,&gcv);
   *(hGC+GCC)=(long)GContext;

   return(drop2());
}

int GCfore() /* GCfore (pix hGC --- ) */
/* Set the foreground color in GC. */
{
   GC GCon;
   double *hGC,*hGAV;
   long pix;

   pix=(long)(tos-1)->real;

   hGC=tos->mat;
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* putting GC attributes vector, gav, on tos */

   hGAV=tos->mat;
   *(hGAV+FORE)=(double)pix;

   if(*(hGC+GCC)!=-INF) {
      GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

      XSetForeground(Dpy,GCon,pix);
   }
   return(drop() && drop2());
}

int GCfree() /* GCfree (hGC --- ) */
/* Free a graphics context. */
{
   double *hGC;
   GC GContext;

   if(tos->typ!=MAT) {
      stkerr(" GCfree: ",MATNOT);
      return 0;
   }
   hGC=tos->mat;
   if(*(hGC+GCC)<=0) return(drop());
   
   GContext=(GC)patlong(*(hGC+GCC));
   XFreeGC(Dpy,GContext);
   *(hGC+GCC)=-INF;

   return(drop());
}

int GCline() /* GCline (linestyle hGC --- ) */
/* Set line style in GC. 

   Tue May 21 09:22:20 PDT 2013.  For linestyle=LineStep, set line style
   in gcv.line_style to LineSolid.  LineStep will later be used in 
   linet().

   Wed Sep  8 12:48:55 PDT 2010.  If linestyle==LineRectangle, set line
   style in GAV.GCC to LineSolid.  Line style LineRectangle will later 
   be used to draw lines where the lines are the diagonals of rectan-
   gles; see linet(). */
{
   GC GCon;
   double *hGC,*hGAV;
   int line_width,line_style,cap_style,join_style;

   hGC=tos->mat;

   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* putting GC attributes vector, gav, on tos */

   hGAV=tos->mat;
   *(hGAV+LINESTY)=(tos-2)->real;

   if(*(hGC+GCC)!=-INF) {

      line_width=patint(*(hGAV+LINEWID));

   /* Special handling for added line styles (also in GCcreate()): */
      if(*(hGAV+LINESTY)==LineRectangle || *(hGAV+LINESTY)==LineStep)
         line_style=LineSolid;
      else line_style=patint(*(hGAV+LINESTY));

      cap_style=patint(*(hGAV+CAPSTY));
      join_style=patint(*(hGAV+JOINSTY));

      GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

      XSetLineAttributes(Dpy,GCon,line_width,line_style,
         cap_style,join_style);
   }
   return(drop() && drop2());
}

int interpolate(double x1, double y1, double x2, double y2, \
   double *X2, double *Y2)
/* Interpolate X2 to go with Y2 on the line defined by end points
   (x1,y1) and (x2,y2). */
{
   if(y2==y1) return 1;
   *X2=x2 + (x2-x1)*(*Y2-y2)/(y2-y1);
   return 0;
}

int line() /* line (hW hGC hY hX --- ) */
/* Draw line segments from X(i),Y(i) to X(i+1),Y(i+1) in window W
   with graphics context GC.  Does not clip line segments. 

   See line drawing examples in XDrawLine1(), this file. */
{
   register Window win;
   register GC GCon;
   register int k=0;
   register double *X,*Y;
   double *hW,*hGC;
   int lines,style;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || \
      (tos-2)->typ!=MAT || (tos-3)->typ!=MAT) {
      stkerr(" line: ",STKNOT);
      return 0;
   }
   hW=(tos-3)->mat;
   win=(Window)patlong(*(hW+WIN)); /* WIN element of hW holds Window */

   hGC=(tos-2)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* GCC element of hGC holds GC */

/* Fetch line style from GC.GAV.LINESTY: */
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* putting GC attributes vector, gav, on tos */
   style=(int)patint(*(tos->mat+LINESTY));
   drop();

   lines=(tos->row)-1;

/* Drawing lines from right to left: */
   Y=((tos-1)->mat)+lines-1;
   X=(tos->mat)+lines-1;

   switch(style) {

      default:
      case LineRectangle:
         for(;k<lines;k++) {
            XDrawLine1(Dpy,win,GCon,style,(short)*(X+1),(short)*(Y+1),
               (short)*X,(short)*Y);
            X--;
            Y--;
         }
      break;

      case LineStep:
         for(;k<lines;k++) {
            XDrawLine1(Dpy,win,GCon,style,(short)*(X+1),(short)*(Y+1),
               (short)*(X+1),(short)*Y);
            XDrawLine1(Dpy,win,GCon,style,(short)*(X+1),(short)*Y,
               (short)*X,(short)*Y);
            X--;
            Y--;
         }
      break;
   }
   return(drop2() && drop2());
}

int line_seg() /* line_seg (hW hGC hC hS hY1 hY2 hX1 hX2 hL --- ) */
/* Tue Jul 22 06:30:00 PDT 2014

   Draw line segments from (X1(k),Y1(k)) to (X2(k),Y1(k)) where X* and
   Y* are vectors of length nr.  Line segments are clipped to the rec-
   tangle defined by L.  

   Vector S, also of length nr, contains states of Y1, i.e., S(k) is
   the state of Y1(k), k=1:nr.  

   Values of S(k), when one-based, are numbered 1, 2, ... s.  Vector
   C of s rows contains a pixel color value for each of the s values
   of state S to use when segments of state j, j=1:s, are drawn.

   Line segments are drawn in order of state S, i.e., segments of state
   1 are drawn first and segments of state s are drawn last.  Line seg-
   ments may overlap and a later state hide an earlier one.  If this is
   unacceptable, it is necessary to preprocess the data before calling
   this function.  Function cposn() in wapp.c is an example of pre-
   processing in a specific application.

   Notes on using XDrawSegments() to draw line segments:  

   This shows setting line color and calling XDrawSegments() to draw a 
   number of segments equal to segs:
      XSetForeground(Dpy,GCon,(long)*C);
      XDrawSegments(Dpy,Win,GCon,(XSegment*)u,segs);

   X11 release 5 or later is assumed (not checked), where the call to 
   XDrawSegments() uses multiple requests to draw all segments when
   segs exceeds the machine's XMaxRequestSize limit (see Reference 2, 
   p. 152).

   At the ready prompt, use word dprops to see the machine's value of
   XMaxRequestSize.  The number of line segments is limited to:
      (XMaxRequestSize-3)/2 
   (see Reference 2, p. 153). 

   Here's how to draw with XDrawLine() instead of XDrawSegments():
      XSetForeground(Dpy,GCon,(long)*C);
      x1=u;
      y1=x1+1;
      x2=y1+1;
      y2=x2+1;
      for(k=0;k<segs;k++) {
         XDrawLine(Dpy,Win,GCon,*x2,*y2,*x1,*y1);
         x1+=4; x2+=4; y1+=4; y2+=4;
      } */ 
{
   register Window Win;
   register GC GCon;
   register double *C,*S,*X1,*X2,*Y1,*Y2;
   register int k;
   register short *x1,*y1,*x2,*y2;

   int nclr,nr,segs=0;
   short *u=NULL;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT || (tos-7)->typ!=MAT) {
      stkerr(" line_seg: ",STKNOT);
      return 0;
   }
/* Fri Jul 25 14:45:10 PDT 2014.  If L is not purged, run GCclip(): */
   if(tos->row) {
      pushstr("7 pick"); xmain(0); /* hL hGCB */
      if(!GCclip()) {
         gprintf(" line_seg: error setting clip region");
         nc();
         stkerr("","");
         return 0;
      }
   }
   else drop(); /* drop purged L */

   C=(tos-5)->mat; nclr=(tos-5)->row;
   S=(tos-4)->mat; nr=(tos-4)->row;
   Y1=(tos-3)->mat;
   Y2=(tos-2)->mat;
   X1=(tos-1)->mat;
   X2=tos->mat;

   if(tos->row!=nr || (tos-1)->row!=nr || (tos-2)->row!=nr || 
      (tos-3)->row!=nr) {
      gprintf(\
         " line_seg: S, Y1, Y2, X1, X2 number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2(); drop2();
      return 0;
   }
   u=malloc(1+4*nr*sizeof(short));
   if(u==NULL) {
      stkerr(" line_seg: ",MEMNOT);
      drop2(); drop2(); drop2(); drop2();
      return 0;
   }
   x1=u;
   y1=x1+1;
   x2=y1+1;
   y2=x2+1;
   for(k=0;k<nr;k++) {
      segs++;
      *x1=(short)*X1;
      *x2=(short)*X2;
      *y1=(short)*Y1;
      *y2=(short)*Y2;
      x1+=4; x2+=4; y1+=4; y2+=4;
      X1++; Y1++; X2++; Y2++;
   }
   if(segs) {
      Win=(Window)patlong(*((tos-7)->mat+WIN)); /* WCB(WIN) holds Win */
      GCon=(GC)patlong(*((tos-6)->mat+GCC)); /* GC(GCC) holds GCon */

      XSetForeground(Dpy,GCon,(long)*C);
      XDrawSegments(Dpy,Win,GCon,(XSegment*)u,segs);
   }
   mallfree((void *)&u);
   return(drop2() && drop2() && drop2() && drop2());
}

int lineb() /* lineb (hW hGC hR hf hY hX hL --- ) */
/* Mon Jul 14 03:13:14 PDT 2014

   In window W with graphics context GC and vectors X and Y, draw hori-
   zontal line segments (belts) from X(i),Y(i) to X(i+1),Y(i) clipped
   to rectangle limits given in L.  

   Incoming f is a matrix of truth values (zero and nonzero) with num-
   ber of rows that matches the rows of X and Y.  Column k of f makes
   curve k by using values of X(i),Y(i) only where in f(i,k) is true.

   Incoming vector R has number of rows equal to (or greater than) the
   number of columns in f, and R(k) contains the color pixel value for
   curve k (i.e., the curve where f(*,k) is true).

   Curves from the columns of f are drawn in order from last column to 
   first.  Thus the first column of f is drawn last and is on top of
   all others.

   Values in incoming X are in ascending order, as in a list of times.

   Line style in GC is ignored, and lines are drawn in the form of
   style LineStep except that only the horizontal portion of the step
   is drawn and the vertical portion is not. */
{
   register Window Win;
   register GC GCon;
   register int j,k;
   register double *f=0,*R=0,*X=0,*Y=0;
   double *hW,*hGC,Y1,Xmin,Xmax,Ymin,Ymax;
   int cols,lines,r1,r2,rows,segs=0;
   short *u1=NULL,*u2=NULL,*v=NULL,*x1,*x2,*y;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT) {
      stkerr(" lineb: ",STKNOT);
      return 0;
   }
   hW=(tos-6)->mat;
   Win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

/* Fetch GC: */
   hGC=(tos-5)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   rows=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,rows,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,rows,&r2);
   
   u1=malloc(1+rows*sizeof(short));
   u2=malloc(1+rows*sizeof(short));
   v=malloc(1+rows*sizeof(short));

   if(u1==NULL || u2==NULL || v==NULL) {
      stkerr(" lineb: ",MEMNOT);
      drop2(); drop2(); drop2();
      return 0;
   }
   if((tos-1)->row!=rows || (tos-2)->row!=rows) {
      gprintf(" lineb: X, Y, and f number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   cols=(tos-2)->col;

   R=(tos-3)->mat;
/* Sun May 18 11:27:57 PDT 2014.  Too many colors is ok. */
   if(cols>(tos-3)->row) {
      gprintf(" lineb: number of R colors is less than columns of f");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */
   for(j=cols-1;j>-1;j--) { /* draw from last column of f to first */

   /* Drawing line segments from right to left: */
      X=(tos->mat)+r2-1; /* using X and X+1 each step */
      Y=(tos-1)->mat+r2-1;
      f=((tos-2)->mat+j*rows)+r2-1;
      segs=0;
      x1=u1;
      x2=u2;
      y=v;

   /* LineStep makes one diagonal line into two: a vertical one from 
      (x2,y1) to (x2,y2) and a horizontal one from (x1,y1) to (x2,y1), 
      where x2>x1.  Here only the horizontal line at y=y1 is drawn: */
      for(k=0;k<lines;k++) {
         if(!(*f==0 && *(f+1)==0)) {
            Y1=*Y;
            if(*f && (Y1<Ymax && Y1>Ymin)) {
            /* Horizontal line: */
               segs++;
               *x1=(short)*X;
               *x2=(short)MIN(*(X+1),Xmax);
               *y=(short)Y1;
               x1++; x2++; y++;
            }
         }
         X--;
         Y--;
         f--;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         x2=u2;
         y=v;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,Win, GCon,*x2,*y,*x1,*y);
            x1++;
            x2++;
            y++;
         }
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&u2);
   mallfree((void *)&v);

   return(drop2() && drop2() && drop2());
}

int lineb1() /* lineb1 (hW hGC hR hf hY hX hL --- ) */
/* Mon Jul 14 18:59:00 PDT 2014

   In window W with graphics context GC, draw horizontal line segments
   (belts) from X(i), Y(i,j) to X(i+1),Y(i,j) clipped to rectangle 
   limits given in L.  

   Incoming f is a matrix of truth values (zero and nonzero) with size
   that matches the rows and columns of Y.  Column k of f makes curve k
   by using values of X(i),Y(i,k) only where in f(i,k) is true.

   Incoming vector R has number of rows equal to (or greater than) the
   number of columns in Y, and R(k) contains the color pixel value for
   curve Y(*,k).

   Curves in the columns of Y are drawn in order from last column to 
   first.  Thus the first column of Y is drawn last and is on top of
   all others.

   Values in incoming X are in ascending order, as in a list of times.

   Line style in GC is ignored, and lines are drawn in the form of
   style LineStep except that only the horizontal portion of the step
   is drawn and the vertical portion is not.  Use function linef() to 
   get both portions of the step. */
{
   register Window Win;
   register GC GCon;
   register int j,k;
   register double *f=0,*R=0,*X=0,*Y=0;
   double *hW,*hGC,Y1,Xmin,Xmax,Ymin,Ymax;
   int cols,lines,r1,r2,rows,segs=0;
   short *u1=NULL,*u2=NULL,*v=NULL,*x1,*x2,*y;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT) {
      stkerr(" lineb1: ",STKNOT);
      return 0;
   }
   hW=(tos-6)->mat;
   Win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

/* Fetch GC: */
   hGC=(tos-5)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   rows=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,rows,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,rows,&r2);
   
   u1=malloc(1+rows*sizeof(short));
   u2=malloc(1+rows*sizeof(short));
   v=malloc(1+rows*sizeof(short));

   if(u1==NULL || u2==NULL || v==NULL) {
      stkerr(" lineb1: ",MEMNOT);
      drop2(); drop2(); drop2();
      return 0;
   }
   if((tos-1)->row!=rows || (tos-2)->row!=rows) {
      gprintf(" lineb1: X, Y, and f number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   cols=(tos-1)->col;
   if((tos-2)->col!=cols) {
      gprintf(" lineb1: Y and f number of columns must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   R=(tos-3)->mat;
/* Sun May 18 11:27:57 PDT 2014.  Too many colors is ok. */
   if(cols>(tos-3)->row) {
      gprintf(" lineb1: number of R colors is less than columns of Y");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */
   for(j=cols-1;j>-1;j--) { /* draw Y from last column to first */

   /* Drawing line segments from right to left: */
      X=(tos->mat)+r2-1; /* using X and X+1 each step */
      Y=((tos-1)->mat+j*rows)+r2-1;
      f=((tos-2)->mat+j*rows)+r2-1;
      segs=0;
      x1=u1;
      x2=u2;
      y=v;

   /* LineStep makes one diagonal line into two: a vertical one from
      (x2,y1) to (x2,y2) and a horizontal one from (x1,y1) to (x2,y1),
      where x2>x1.  Here only the horizontal line at y=y1 is drawn: */
      for(k=0;k<lines;k++) {
         if(!(*f==0 && *(f+1)==0)) {
            Y1=*Y;
            if(*f && (Y1<Ymax && Y1>Ymin)) {
            /* Horizontal line: */
               segs++;
               *x1=(short)*X;
               *x2=(short)MIN(*(X+1),Xmax);
               *y=(short)Y1;
               x1++; x2++; y++;
            }
         }
         X--;
         Y--;
         f--;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         x2=u2;
         y=v;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,Win, GCon,*x2,*y,*x1,*y);
            x1++;
            x2++;
            y++;
         }
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&u2);
   mallfree((void *)&v);

   return(drop2() && drop2() && drop2());
}

int lineb2() /* lineb2 (hW hGC hR hf hY hX hL --- ) */
/* Mon Jul 14 10:50:19 PDT 2014

   In window W with graphics context GC and vectors X and Y, draw a
   box with diagonal corners given by X(i),Y(i)+1 and X(i+1),Y(i)-1, 
   clipped to rectangle limits given in L.  

   Incoming f is a matrix of truth values (zero and nonzero) with num-
   ber of rows that matches the rows of X and Y.  Column k of f makes
   curve k by using values of X(i),Y(i) only where in f(i,k) is true.

   Incoming vector R has number of rows equal to (or greater than) the
   number of columns in f, and R(k) contains the color pixel value for
   curve k (i.e., the curve where f(*,k) is true).

   Curves from the columns of f are drawn in order from last column to 
   first.  Thus the first column of f is drawn last and is on top of
   all others.

   Values in incoming X are in ascending order, as in a list of times.

   Line style in GC is ignored, and lines are drawn in the form of
   style LineStep except that only the horizontal portion of the step
   is drawn and the vertical portion is not. */
{
   register Window Win;
   register GC GCon;
   register int d=5,j,k;
   register double *f=0,*R=0,*X=0,*Y=0;
   double *hW,*hGC,Y1,Xmin,Xmax,Ymin,Ymax;
   int cols,lines,r1,r2,rows,segs=0;
   short *u1=NULL,*u2=NULL,*v=NULL,*x1,*x2,*y;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT) {
      stkerr(" lineb2: ",STKNOT);
      return 0;
   }
   hW=(tos-6)->mat;
   Win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

/* Fetch GC: */
   hGC=(tos-5)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   rows=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,rows,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,rows,&r2);
   
   u1=malloc(1+rows*sizeof(short));
   u2=malloc(1+rows*sizeof(short));
   v=malloc(1+rows*sizeof(short));

   if(u1==NULL || u2==NULL || v==NULL) {
      stkerr(" lineb2: ",MEMNOT);
      drop2(); drop2(); drop2();
      return 0;
   }
   if((tos-1)->row!=rows || (tos-2)->row!=rows) {
      gprintf(" lineb2: X, Y, and f number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   cols=(tos-2)->col;

   R=(tos-3)->mat;
/* Sun May 18 11:27:57 PDT 2014.  Too many colors is ok. */
   if(cols>(tos-3)->row) {
      gprintf(" lineb2: number of R colors is less than columns of f");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */
   for(j=cols-1;j>-1;j--) { /* draw from last column of f to first */

   /* Drawing line segments from right to left: */
      X=(tos->mat)+r2-1; /* using X and X+1 each step */
      Y=(tos-1)->mat+r2-1;
      f=((tos-2)->mat+j*rows)+r2-1;
      segs=0;
      x1=u1;
      x2=u2;
      y=v;

      for(k=0;k<lines;k++) {
         if(!(*f==0 && *(f+1)==0)) {
            Y1=*Y;
            if(*f && (Y1<Ymax && Y1>Ymin)) {
            /* Horizontal line: */
               segs++;
               *x1=(short)*X;
               *x2=(short)MIN(*(X+1),Xmax);
               *y=(short)Y1;
               x1++; x2++; y++;
            }
         }
         X--;
         Y--;
         f--;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         x2=u2;
         y=v;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,Win, GCon,*x2,*y+d,*x1,*y+d);
            XDrawLine(Dpy,Win, GCon,*x1,*y+d,*x1,*y-d);
            XDrawLine(Dpy,Win, GCon,*x1,*y-d,*x2,*y-d);
            XDrawLine(Dpy,Win, GCon,*x2,*y-d,*x2,*y+d);
            x1++;
            x2++;
            y++;
         }
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&u2);
   mallfree((void *)&v);

   return(drop2() && drop2() && drop2());
}

int linec() /* linec (hW hGC hY hX hL --- ) */
/* Draw line segments from X(i),Y(i) to X(i+1),Y(i+1) in window W with
   graphics context GC.  Line segments are clipped to rectangle limits
   given in L.

   See line drawing examples in XDrawLine1(), this file.

   Wed Jul 21 07:03:47 PDT 2010.  XDrawLine() into a separate loop. */
{
   register Window win;
   register GC GCon;
   register int k=0;
   register double *X=0,*Y=0;
   double *hW,*hGC,X1,X2,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   int len,lines,segs=0,style,typ;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT) {
      stkerr(" linec: ",STKNOT);
      return 0;
   }
   hW=(tos-4)->mat;
   win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */
   typ=(Window)patlong(*(hW+TYP));

   hGC=(tos-3)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Fetch line style from GC.GAV.LINESTY: */
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* GC attributes vector, gav, on tos */
   style=(int)patint(*(tos->mat+LINESTY));
   drop();

/* Tue May 21 13:49:18 PDT 2013.  Style LineStep works in line(), but
   clipping requires lines of style LineStep to use linet() which is
   run when wcb.typ is equal to 1 (see word pExpose in sys/plot.v): */
   if(style==LineStep && typ!=1) {
      gprintf(" linec: cannot clip LineStep style; use linet()");
      nc();
      drop(); drop2(); drop2();
      return 0;
   }
/* Wed May 22 20:42:41 PDT 2013.  Style LineRectangle works in line(),
   but clipping requires lines of LineRectangle to use linet() which is
   run when wcb.typ is equal to 1 (see word pExpose in sys/plot.v): */
   if(style==LineRectangle && typ!=1) {
      gprintf(" linec: cannot clip LineRectangle style; use linet()");
      nc();
      drop(); drop2(); drop2();
      return 0;
   }
/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

   len=tos->row;
   u1=malloc(1+len*sizeof(short));
   v1=malloc(1+len*sizeof(short));
   u2=malloc(1+len*sizeof(short));
   v2=malloc(1+len*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL) {
      stkerr(" linec: ",MEMNOT);
      drop2(); drop2();
      return 0;
   }
   x1=u1;
   y1=v1;
   x2=u2;
   y2=v2;

/* Drawing lines from right to left: */
   lines=tos->row-1;
   X=(tos->mat)+lines-1;
   Y=((tos-1)->mat)+lines-1;
   
   LINEDIAG

   if(segs) {

      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;

      for(k=0;k<segs;k++) {
         XDrawLine( Dpy,win,GCon,*x2,*y2,*x1,*y1);
         x1++;
         x2++;
         y1++;
         y2++;
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);

   return(drop2() && drop2());
}

int linef() /* linef (hW hGC hR hf hY hX hL --- ) */
/* Fri Nov 22 08:36:12 PST 2013

   In window W with graphics context GC, draw line segments from X(i),
   Y(i,j) to X(i+1),Y(i+1,j) clipped to rectangle limits given in L.  

   Incoming vector R has number of rows equal to (or greater than) the
   number of columns in Y, and R(j) contains the color pixel value for
   curve Y(*,j).

   Curves in the columns of Y are drawn in order from last column to 
   first.  Thus the first column of Y is drawn last and is on top of
   all others.

   Values in incoming X are in ascending order, as in a list of times.

   Therefore, clipping in X can be done ahead of time to select just 
   those rows of X that are in the X-clip region.  Binary search is 
   used on X, so it must be in ascending order. 

   Incoming f is a matrix that matches the rows and columns of Y.  

   This function works the same as linet1() if all terms in f are non-
   zero.

   For default line style, if f(m,k)=0 and/or f(n,k)=0 for line segment 
   [y(m,k),y(n,k)], the line segment will not be drawn.

   Sat Nov 30 17:45:38 PST 2013.  For line style LineStep, which draws
   two lines (one vertical and one horizontal), part of the step will 
   be drawn if either f(m,k)!=0 or f(n,k)!=0. */
{
   register Window win;
   register GC GCon;
   register int j,k;
   register double *f=0,*R=0,*X=0,*Y=0;
   double *hW,*hGC,X1,X2,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   int cols,len,lines,r1,r2,rows,segs=0,style;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT) {
      stkerr(" linef: ",STKNOT);
      return 0;
   }
   hW=(tos-6)->mat;
   win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

/* Fetch line style from GC.GAV.LINESTY: */
   hGC=(tos-5)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* GC attributes vector, gav, on tos */
   style=(int)patint(*(tos->mat+LINESTY));
   drop();

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   rows=len=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,len,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,len,&r2);
   
   if(style==LineStep) len=len+len; /* makes two lines for each one */
   u1=malloc(1+len*sizeof(short));
   v1=malloc(1+len*sizeof(short));
   u2=malloc(1+len*sizeof(short));
   v2=malloc(1+len*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL) {
      stkerr(" linef: ",MEMNOT);
      drop2(); drop2(); drop2();
      return 0;
   }
   if((tos-1)->row!=rows || (tos-2)->row!=rows) {
      gprintf(" linef: X, Y, and f number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   cols=(tos-1)->col;
   if((tos-2)->col!=cols) {
      gprintf(" linef: Y and f number of columns must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   R=(tos-3)->mat;
/* Sun May 18 11:27:57 PDT 2014.  Too many colors is ok. */
   if(cols>(tos-3)->row) {
      gprintf(" linef: number of R colors is less than columns of Y");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */
   for(j=cols-1;j>-1;j--) { /* draw Y from last column to first */

   /* Drawing line segments from right to left: */
      X=(tos->mat)+r2-1; /* using X and X+1 each step */
      Y=((tos-1)->mat+j*rows)+r2-1;
      f=((tos-2)->mat+j*rows)+r2-1;
      segs=0;
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;

      switch(style) {

         case LineStep:
      /* LineStep makes one diagonal line into two: a vertical one
         from (x2,y1) to (x2,y2) and a horizontal one from (x1,y1)
         to (x2,y1), where x2>x1: */
         for(k=0;k<lines;k++) {
            if(!(*f==0 && *(f+1)==0)) {
               Y1=*Y;
               Y2=*(Y+1);
               if(*(f+1) && ((*(X+1)<Xmax) /* Y1 or Y2 not outside: */
                  && (!outside(&Y1,Ymin,Ymax) \
                  ||  !outside(&Y2,Ymin,Ymax)))) {
               /* Vertical line: */
                  segs++;
                  *x1=*x2=(short)*(X+1);
                  *y1=(short)MAX(Y1,Ymin);
                  *y2=(short)MIN(Y2,Ymax);
                  x1++; y1++; x2++; y2++;
               }
               Y1=*Y;
               if(*f && (Y1<Ymax && Y1>Ymin)) {
               /* Horizontal line: */
                  segs++;
                  *x1=(short)*X;
                  *x2=(short)MIN(*(X+1),Xmax);
                  *y1=*y2=(short)Y1;
                  x1++; y1++; x2++; y2++;
               }
            }
            X--;
            Y--;
            f--;
         }
         break;

         default:
         for(k=0;k<lines;k++) {

         /* Mon Apr  7 22:03:42 PDT 2014.
            This could be improved by drawing the diagonal that applies
            to the vertical and horizontal steps made by LineStep, i.e.,
            from (x1,y1) to (x2,y2).  Now, if *f is an isolated point,
            nothing is drawn because *(f+1)=0. */

            if(*f!=0 && *(f+1)!=0) {
               if(bounds1(*X,*Y,*(X+1),*(Y+1),Ymin,Ymax,&X2,&Y2)) {
                  if(bounds1(*(X+1),*(Y+1),*X,*Y,Ymin,Ymax,&X1,&Y1)) {
                     if(!(X1==X2 && Y1==Y2)) { /* not a single point: */
                        segs++;
                        *x1=(short)X1;
                        *x2=(short)X2;
                        *y1=(short)Y1;
                        *y2=(short)Y2;
                        x1++;
                        y1++;
                        x2++;
                        y2++;
                     }
                  }
               }
            }
            X--;
            Y--;
            f--;
         }
         break;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         y1=v1;
         x2=u2;
         y2=v2;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,win,GCon,*x2,*y2,*x1,*y1);
            x1++;
            x2++;
            y1++;
            y2++;
         }
      }
   }
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);

   return(drop2() && drop2() && drop2());
}

int lineg() /* lineg (hW hGC hR hN hP hY hX hL --- ) */
/* Thu Jun 26 12:58:17 PDT 2014

   In window W with graphics context GC, draw vertical lines (grills)
   from X(k),P(k,j) to X(k),Y(k) clipped to rectangle limits given in
   L.

   The dimensions of N match those of P, and vertical lines are drawn 
   from P(k,j) only if flag N(k,j) is not zero. 

   Column vector R has number of rows equal to (or greater than) the 
   number of columns in P, and R(j) contains the color pixel value for
   the vertical lines made using column j of P and N.

   Lines are drawn in order from last column of P to first, so lines
   made with the first column of P are on top.

   Values in X are in ascending order. 

   Fri Jun 27 09:38:38 PDT 2014.  Testing.  See word line_tests() in 
   file usr/uboot.v.  Paste the following four lines at the ready prompt
   to run line_tests() on line drawing functions linet1(), linef() and
   lineg():
      syspath "../usr/uboot.v" +                   \
      "#def line_tests" "#end line_tests" msource1 \
      syspath "../usr/uboot.v" +                   \
      "#def Test lines" "#end Test lines" msource1 \ */
{
   register Window Win;
   register GC GCon;
   register int j,k;
   register double *N=0,*P=0,*R=0,*X=0,*Y=0;
   double *hW,*hGC,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   int cols,lines,r1,r2,rows,segs=0;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT || (tos-7)->typ!=MAT) {
      stkerr(" lineg: ",STKNOT);
      return 0;
   }
   hW=(tos-7)->mat;
   Win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

   hGC=(tos-6)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   rows=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,rows,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,rows,&r2);

   u1=malloc(1+rows*sizeof(short));
   v1=malloc(1+rows*sizeof(short));
   u2=malloc(1+rows*sizeof(short));
   v2=malloc(1+rows*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL) {
      stkerr(" lineg: ",MEMNOT);
      drop2(); drop2(); drop2(); drop();
      return 0;
   }
   if((tos-1)->row!=rows || (tos-2)->row!=rows || (tos-3)->row!=rows) {
      gprintf(" lineg: X, Y, P and N number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2(); drop();
      return 0;
   }
   cols=(tos-2)->col;
   if((tos-3)->col!=cols) {
      gprintf(" lineg: P and N number of columns must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2(); drop();
      return 0;
   }
   R=(tos-4)->mat;
   if(cols>(tos-4)->row) { /* too many colors is ok */
      gprintf(" lineg: number of R colors is less than columns of P");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2(); drop();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */

/* Moving from right to left: */
   for(j=cols-1;j>-1;j--) { /* from last column of P to first */
      X=tos->mat+r2-1;
      Y=(tos-1)->mat+r2-1;
      P=(tos-2)->mat+j*rows+r2-1;
      N=(tos-3)->mat+j*rows+r2-1;
      segs=0;
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;

   /* Draw vertical lines from (X,Y) to (X,P) when N is not zero: */
      for(k=0;k<lines;k++) {
         if(*N && *X<=Xmax) {
            Y1=*Y;
            Y2=*P;
            outside(&Y1,Ymin,Ymax);
            outside(&Y2,Ymin,Ymax);
            segs++;
            *x1=*x2=(short)*X;
            *y1=(short)Y1;
            *y2=(short)Y2;
            x1++; y1++; x2++; y2++;
         }
         X--;
         Y--;
         P--;
         N--;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         y1=v1;
         x2=u2;
         y2=v2;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,Win, GCon,*x2,*y2,*x1,*y1);
            x1++;
            x2++;
            y1++;
            y2++;
         }
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);

   return(drop2() && drop2() && drop2() && drop());
}

int linem() /* linem (hW hGC hR hV hY hX hL hD --- ) */
/* Fri May 24 11:02:14 PDT 2013
   Mon Aug 12 14:46:36 PDT 2013.  Revised to use color vector R and
   GC of one column; and to allow negative values in D.

   Make a mat of horizontal lines.

   Vectors X and Y define a set of points (X[k],Y[k]) in a plane.

   In window W with graphics context GC, draw horizontal line segments
   from (X[i],Y[i]) to (X[i]+DELTA,Y[i]) clipped to rectangle limits 
   given in L.  Data for computing X offset DELTA is given in incoming
   vector D.

   R(k) contains the color pixel value for the mat line pegged to Y,
   and D(k) contains its length.  

   To peg a mat line to Y, vector V, the same length as Y, holds an in-
   dex to a row in color vector R (and length vector D) for every point
   in Y; when values in V are converted from user index base to zero-
   based for this function, any that are less than zero are ignored and
   no line is drawn.  

   Note that the maximum zero-based index in V is limited to the number
   of rows in R (and D) minus one.  This is checked before drawing.

   Line style for all lines is in GC.  

   Values in incoming X are in ascending order, as in a list of times.
   Step in X, used for computing DELTA, is assumed to be given by the
   first two X values.

   Since X values are ascending, clipping in X can be done ahead of
   time to select just those rows of X that are in the X-clip region.  
   Binary search is used on X because it is in ascending order.

   Row k of vector D contains the number of X rows that define the 
   length of its mat, e.g., D[k]=DELTA[k]/dX, where dX is the step in
   X.  Using the first step in X, DELTA[k] for D[k] is computed below
   as DELTA[k]=D[k]*(X[1]-X[0]). 

   D can be negative for a mat with a Y peg on the right.  But see the
   warning below.

   Warning: Mats are always drawn from left to right.  This means that
   on zoom in, a mat with negative D will not display if the curve of
   its right peg is not in the zoomed view.  With an appropriate lag on
   the peg, it may be possible to convert mats with negative D into 
   mats with positive D to fit the left-to-right operation. */
{
   Window Win;
   GC GCon;
   register double *X=0,*Y=0,*V=0;
   double *D,DELTA,*R,X1,X2,Y1,Xmin,Xmax,Ymin,Ymax;
   int k,len,lines,nC,r1,r2,segs=0,Z=0;
   short *w=NULL,*u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2,*z;

   cop(); /* make a copy so D values can be changed */

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT ||
      (tos-6)->typ!=MAT || (tos-7)->typ!=MAT) {
      stkerr(" linem: ",STKNOT);
      return 0;
   }
   Win=(Window)patlong(*((tos-7)->mat+WIN));
   GCon=(GC)patlong(*((tos-6)->mat+GCC)); 

   R=(tos-5)->mat;
   nC=(tos-5)->row; /* number of colors */

   D=tos->mat;
   if(nC!=tos->row) {
      gprintf(" linem: %d rows of D do not match %d rows of R",\
         tos->row,nC);
      nc();
      stkerr("","");
      return 0;
   }
/* Loading the clip rectangle limits from L: */
   X=(tos-1)->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   lop(); /* L off stack */

/* Defining horizontal axis X: */
   X=(tos-1)->mat;
   len=(tos-1)->row;

/* Find rows of X that enclose the X-clip region, but always start from
   the first row because X1 may be left of X-clip: */
   r1=0; /* mat line X1 may be left of X-clip, where it is pinned */
   bsearchd(Xmax,X,len,&r2);

/* For each row of D, scale D into its mat length DELTA using the
   initial step in X: */
   X1=(*(X+1)-*X);
   for(k=0;k<nC;k++) *(D+k)=*(D+k)*X1;

   u1=malloc(1+len*sizeof(short));
   v1=malloc(1+len*sizeof(short));
   u2=malloc(1+len*sizeof(short));
   v2=malloc(1+len*sizeof(short));
   w=malloc(1+len*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL || w==NULL) {
      stkerr(" linem: ",MEMNOT);
      return 0;
   }
   x1=u1;
   y1=v1;
   x2=u2;
   y2=v2;
   memset(w,0,len*sizeof(short));
   z=w;

   lines=r2-r1+1; /* lines equals number of points */
   X=((tos-1)->mat)+r1;
   Y=((tos-2)->mat)+r1;
   V=((tos-3)->mat)+r1;

   for(k=0;k<lines;k++) {
      if((*(V+k)-XBASE)>(nC-1)) {
         gprintf(" linem: V index %d is out of range for %d colors",\
            (int)(*(V+k)-XBASE),nC);
         nc();
         stkerr("","");
         return 0;
      }
   }
/* Drawing lines from left to right is done so mat lines later in X 
   will be drawn on top of earlier ones: */
   for(k=0;k<lines;k++) {
      Y1=*Y;
      Z=(int)(*V-XBASE);
      if(Z>-1 && Y1<Ymax && Y1>Ymin) {
         DELTA=*(D+Z);
         if(DELTA>0) {
            X1=MAX(*X,Xmin);
            X2=MIN(*X+DELTA,Xmax);
         }
         else {
            X1=MAX(*X+DELTA,Xmin);
            X2=MIN(*X,Xmax);
         }
         if(X1<X2) {
         /* Horizontal line segment at Y1 (=Y2) from X1 to X2: */
            segs++;
            *x1=(short)X1;
            *x2=(short)X2;
            *y1=*y2=(short)Y1;
            *z=(short)Z;
            x1++; y1++; x2++; y2++; z++;
         }
      }
      X++;
      Y++;
      V++;
   }
   if(segs) {
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;
      z=w;
      for(k=0;k<segs;k++) {
         XSetForeground(Dpy,GCon,(long)*(R+*z));
         XDrawLine(Dpy,Win,GCon,*x2,*y2,*x1,*y1);
         x1++; y1++; x2++; y2++; z++;
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);
   mallfree((void *)&w);

   return(drop() && drop2() && drop2() && drop2());
}

int linem1() /* linem1 (hW hGC hP hR hV hY hX hL --- ) */
/* Tue Jan 21 12:52:48 PST 2014

   Make a mat of horizontal lines.

   Column matrices X and Y define a set of points in a plane XY.

   In window W with graphics context GC, draw horizontal line segments
   from (X[i],Y[i]) to (X[i]+D,Y[i]) clipped to rectangle limits given
   in L.  

   Lengths for horizontal offset D are given in incoming vector V, and
   are assumed to be zero or greater.  Only lengths that fit ranges
   defined in range matrix R will be drawn.

   For line length D=V(m) that fits within range R(k,2)-R(k,1),
      R(k,1) <= D < R(k,2),
   a horizontal line of length D will be drawn from origin Y(m) with 
   color pixel value given by P(k).
 
   Line style for all lines is in GC.  

   Values in incoming X are in ascending order, as in a list of times.
   Since X values are ascending, clipping in X can be done ahead of
   time to select just those rows of X that are in the X-clip region.  
   Binary search is used on X because it is in ascending order. */
{
   Window Win;
   GC GCon;
   register double *X=0,*Y=0,*V=0;
   double D,*P,*R,R1,R2,X1,X2,Y1,Xmin,Xmax,Ymin,Ymax;
   int j=0,k,len,lines,ncv,r1,r2,segs=0;
   long *c0=NULL,*cv,CV;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;
   short w1,h1,h2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT ||
      (tos-6)->typ!=MAT || (tos-7)->typ!=MAT) {
      stkerr(" linem1: ",STKNOT);
      return 0;
   }
   Win=(Window)patlong(*((tos-7)->mat+WIN));
   GCon=(GC)patlong(*((tos-6)->mat+GCC)); 

   P=(tos-5)->mat;
   ncv=(tos-5)->row; /* number of color values */

   R=(tos-4)->mat;
   if(ncv!=(tos-4)->row) {
      gprintf(" linem1: %d rows of R do not match %d rows of P",\
         (tos-4)->row,ncv);
      nc();
      stkerr("","");
      return 0;
   }
/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Defining horizontal axis X: */
   X=tos->mat;
   len=tos->row;

/* Find rows of X that enclose the X-clip region, but always start from
   the first row because X1 may be left of X-clip: */
   r1=0; /* mat line X1 may be left of X-clip, where it is pinned */
   bsearchd(Xmax,X,len,&r2);

   u1=malloc(1+ncv*len*sizeof(short));
   v1=malloc(1+ncv*len*sizeof(short));
   u2=malloc(1+ncv*len*sizeof(short));
   v2=malloc(1+ncv*len*sizeof(short));
   c0=malloc(1+ncv*len*sizeof(long));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL || c0==NULL) {
      stkerr(" linem1: ",MEMNOT);
      return 0;
   }
   x1=u1;
   y1=v1;
   x2=u2;
   y2=v2;
   cv=c0;
   lines=r2-r1+1; /* lines equals number of points */

   for(;j<ncv;j++) {
      R1=*R;
      R2=*(R+ncv);
      CV=(long)*P;
      X=tos->mat+r1;
      Y=((tos-1)->mat)+r1;
      V=((tos-2)->mat)+r1;

   /* Drawing lines from left to right is done so mat lines later in X 
      will be drawn on top of earlier ones: */
      for(k=0;k<lines;k++) {
         Y1=*Y;
         D=ABS(*V);
         if((R1<=D && D<R2) && Y1<Ymax && Y1>Ymin) {
            X1=MAX(*X,Xmin);
            X2=MIN(*X+D,Xmax);
            if(X1<X2) {
            /* Horizontal line segment at Y1 (=Y2) from X1 to X2: */
               segs++;
               *x1=(short)X1;
               *x2=(short)X2;
               *y1=*y2=(short)Y1;
               *cv=CV;
               x1++; y1++; x2++; y2++; cv++;
            }
         }
         X++;
         Y++;
         V++;
      }
      R++;
      P++;
   }
   if(segs) {
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;
      cv=c0;
      CV=*cv;
      XSetForeground(Dpy,GCon,CV);
      for(k=0;k<segs;k++) {
         if(CV!=*cv) {
            CV=*cv;
            XSetForeground(Dpy,GCon,CV);
         }
         XDrawLine(Dpy,Win,GCon,*x2,*y2,*x1,*y1);

      /* Put a knob at the x2 tip, 1 pixel up, 1 pixel down and 1 pixel
         wide.  The horizontal line is 1 pixel wide, so for the verti-
         cal need 2 pixels in one direction (y2+2) and 1 pixel in the 
         other (y2-1): */  
         h1=*y2-1; h2=*y2+2; w1=*x2-1;
         XDrawLine(Dpy,Win,GCon,*x2,h1,*x2,h2); /* at x2 */
         XDrawLine(Dpy,Win,GCon,w1,h1,w1,h2);   /* at x2-1 */

         x1++; y1++; x2++; y2++; cv++;
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);
   mallfree((void *)&c0);

   return(drop() && drop2() && drop2() && drop2());
}

int linet() /* linet (hW hGC hY hX hL --- ) */
/* Tue Jul 20 10:20:15 PDT 2010

   In window W with graphics context GC, draw line segments from X(i),
   Y(i) to X(i+1),Y(i+1) clipped to rectangle limits given in L.  

   Values in incoming X are in ascending order, as in a list of times.

   Therefore, clipping in X can be done ahead of time to select just 
   those rows of X that are in the X-clip region.  Binary search is 
   used on X, so it must be in ascending order.

   See line drawing examples in XDrawLine1(), this file. */
{
   register Window win;
   register GC GCon;
   register int k=0;
   register double *X=0,*Y=0;
   double *hW,*hGC,X1,X2,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   double X2prev;
   int len,lines,r1,r2,segs=0,style;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT) {
      stkerr(" linet: ",STKNOT);
      return 0;
   }
   hW=(tos-4)->mat;
   win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */
   
   hGC=(tos-3)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Fetch line style from GC.GAV.LINESTY: */
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* GC attributes vector, gav, on tos */
   style=(int)patint(*(tos->mat+LINESTY));
   drop();

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   len=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,len,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,len,&r2);
   
   if(style==LineStep) len=len+len; /* makes two lines for each one */
   u1=malloc(1+len*sizeof(short));
   v1=malloc(1+len*sizeof(short));
   u2=malloc(1+len*sizeof(short));
   v2=malloc(1+len*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL) {
      stkerr(" linet: ",MEMNOT);
      drop2(); drop2();
      return 0;
   }
   x1=u1;
   y1=v1;
   x2=u2;
   y2=v2;

/* Drawing lines from right to left: */
   lines=r2-r1; /* lines equals number of steps */
   X=(tos->mat)+r2-1; /* using X and X+1 each step */
   Y=((tos-1)->mat)+r2-1;

   switch(style) {

      default:
         LINEDIAG 
      break;
 
      case LineStep:
         LINESTEP 
      break;

   /* Sun Aug 11 13:46:30 PDT 2013.  Line style LineRectangle no longer
      works correctly when there is clipping.  For filling rectangles,
      use new function rgnt() instead. */
    
      case LineRectangle:

      X2prev=*(X+1);
      for(k=0;k<lines;k++) {
         if(bounds1(*X,*Y,*(X+1),*(Y+1),Ymin,Ymax,&X2,&Y2)) {
            if(bounds1(*(X+1),*(Y+1),*X,*Y,Ymin,Ymax,&X1,&Y1)) {
               if(!(X1==X2 && Y1==Y2)) { /* not a single point: */
                  segs++;
                  *y1=(short)Y1;
                  *x1=(short)X1;
                  *y2=(short)Y2;
               /* The following corrects for clipping in X: */
                  *x2=(short)MIN(X2,X2prev);

                  /**x2=(short)X2;*/

               x1++; y1++; x2++; y2++;
               }
            }
         }
         X2prev=*(X+1);
         X--;
         Y--;
      }
      break;
   }
   if(segs) {
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;

      for(k=0;k<segs;k++) {
         XDrawLine1(Dpy,win,GCon,style,*x2,*y2,*x1,*y1);
         x1++;
         x2++;
         y1++;
         y2++;
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);

   return(drop2() && drop2());
}

int linet1() /* linet1 (hW hGC hR hY hX hL --- ) */
/* Mon Jul 15 12:24:36 PDT 2013

   In window W with graphics context GC, draw line segments from X(i),
   Y(i,j) to X(i+1),Y(i+1,j) clipped to rectangle limits given in L.  

   Incoming vector R has number of rows equal to the number of columns
   in Y, and R(j) contains the color pixel value for curve Y(*,j).

   Curves in the columns of Y are drawn in order from last column to 
   first.  Thus the first column of Y is drawn last and is on top of
   all others.

   Values in incoming X are in ascending order, as in a list of times.

   Therefore, clipping in X can be done ahead of time to select just 
   those rows of X that are in the X-clip region.  Binary search is 
   used on X, so it must be in ascending order. */
{
   register Window win;
   register GC GCon;
   register int j,k;
   register double *R=0,*X=0,*Y=0;
   double *hW,*hGC,X1,X2,Y1,Y2,Xmin,Xmax,Ymin,Ymax;
   int cols,len,lines,r1,r2,rows,segs=0,style;
   short *u1=NULL,*v1=NULL,*u2=NULL,*v2=NULL,*x1,*y1,*x2,*y2;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT) {
      stkerr(" linet1: ",STKNOT);
      return 0;
   }
   hW=(tos-5)->mat;
   win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */

/* Fetch line style from GC.GAV.LINESTY: */
   hGC=(tos-4)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */
   pushd(*(hGC+GAV));
   if(!exe()) return 0; /* GC attributes vector, gav, on tos */
   style=(int)patint(*(tos->mat+LINESTY));
   drop();

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   len=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,len,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,len,&r2);
   
   if(style==LineStep) len=len+len; /* makes two lines for each one */
   u1=malloc(1+len*sizeof(short));
   v1=malloc(1+len*sizeof(short));
   u2=malloc(1+len*sizeof(short));
   v2=malloc(1+len*sizeof(short));

   if(u1==NULL || v1==NULL || u2==NULL || v2==NULL) {
      stkerr(" linet1: ",MEMNOT);
      drop2(); drop2();
      return 0;
   }
   cols=(tos-1)->col;
   rows=(tos-1)->row;

   R=(tos-2)->mat; 
/* Sat May  3 11:16:48 PDT 2014.  Too many colors is ok. */
   if(cols>(tos-2)->row) {
      gprintf(" linet1: number of R colors is less than columns of Y");
      nc();
      stkerr("","");
      drop2(); drop2(); drop();
      return 0;
   }
   lines=r2-r1; /* lines equals number of steps */

   for(j=cols-1;j>-1;j--) { /* draw Y from last column to first */
   /* Drawing line segments from right to left: */
      X=(tos->mat)+r2-1; /* using X and X+1 each step */
      Y=((tos-1)->mat+j*rows)+r2-1;
      segs=0;
      x1=u1;
      y1=v1;
      x2=u2;
      y2=v2;

      switch(style) {

         default:
            LINEDIAG 
         break;

         case LineStep:
            LINESTEP 
         break;
      }
      if(segs) {
         XSetForeground(Dpy,GCon,(long)*(R+j));
         x1=u1;
         y1=v1;
         x2=u2;
         y2=v2;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,win,GCon,*x2,*y2,*x1,*y1);
            x1++;
            x2++;
            y1++;
            y2++;
         }
      }
   }
   mallfree((void *)&u1);
   mallfree((void *)&v1);
   mallfree((void *)&u2);
   mallfree((void *)&v2);

   return(drop2() && drop2() && drop());
}

int linetX() /* linetX (hW hGC hR hY hX hL --- ) */
/* Sun Jul 13 10:40:32 PDT 2014

   Fri Jul 25 05:42:31 PDT 2014.  Clipping with this program's func-
   tions removed and X11 clipping installed using function GCclip();
   clipping still uses incoming L.

   Wed Jul 23 08:38:05 PDT 2014.  Revised to use XDrawSegments() in-
   stead of XDrawLine().  See notes in line_seg() related to function
   XDrawSegments().

   In window W with graphics context GC, draw line segments from X(i),
   Y(i,j) to X(i+1),Y(i+1,j), clipped the rectangle defined by L.

   Incoming vector R has number of rows equal to the number of columns
   in Y, and R(j) contains the color pixel value for curve Y(*,j).

   Line style in GC is ignored, and lines are drawn in the form of
   style LineStep except that only the horizontal portion of the step
   is drawn and the vertical portion is not. */
{
   register Window win;
   register GC gc;
   register double *R=0,*X=0,*Y=0;
   register int j=0,k;
   register short *x1,*x2,*y1,*y2;

   int cols,nr,rows,segs;
   short *u=NULL;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT) {
      stkerr(" linetX: ",STKNOT);
      return 0;
   }
/* Fri Jul 25 08:17:42 PDT 2014.  If L is not purged, run GCclip(): */
   if(tos->row) {
      pushstr("4 pick"); xmain(0); /* hL hGCB */
      if(!GCclip()) {
         gprintf(" linetX: error setting clip region");
         nc();
         stkerr("","");
         return 0;
      }
   }
   else drop(); /* drop purged L */

   if(tos->row!=(tos-1)->row) {
      gprintf(\
         " linetX: Y and X number of rows must be equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop();
      return 0;
   }
   rows=tos->row;
   cols=(tos-1)->col;

   if(cols>(tos-2)->row) {
      gprintf(" linetX: number of R colors is less than columns of Y");
      nc();
      stkerr("","");
      drop2(); drop2(); drop();
      return 0;
   }
   u=malloc(1+4*rows*sizeof(short));
   if(u==NULL) {
      stkerr(" linetX: ",MEMNOT);
      drop2(); drop2(); drop();
      return 0;
   }
   win=(Window)patlong(*((tos-4)->mat+WIN)); /* WCB(WIN) holds win */
   gc=(GC)patlong(*((tos-3)->mat+GCC)); /* GC(GCC) holds gc */

   R=(tos-2)->mat; 
   nr=rows-1;

   for(;j<cols;j++) {
      X=(tos->mat);
      Y=(tos-1)->mat+j*nr;
      segs=0;
      x1=u;
      y1=x1+1;
      x2=y1+1;
      y2=x2+1;

      for(k=0;k<nr;k++) {
         segs++;
         *x1=(short)*X;
         *x2=(short)*(X+1);
         *y1=*y2=(short)*Y;
         x1+=4; x2+=4; y1+=4; y2+=4;
         X++; Y++;
      }
      if(segs) {
         XSetForeground(Dpy,gc,(long)*(R+j));
         XDrawSegments(Dpy,win,gc,(XSegment*)u,segs);
      }
   /* This is how to do the drawing with XDrawLine():

      if(segs) {
         XSetForeground(Dpy,gc,(long)*(R+j));
         x1=u;
         y1=x1+1;
         x2=y1+1;
         y2=x2+1;
         for(k=0;k<segs;k++) {
            XDrawLine(Dpy,win,gc,*x2,*y2,*x1,*y1);
            x1+=4; x2+=4; y1+=4; y2+=4;
         }
      } */
   }
   mallfree((void *)&u);
   return(drop2() && drop2() && drop());
}

int pminmax() /* pminmax (hA --- min max) */
/* Fetching smallest and largest terms in matrix, and ignoring UNDEF. */
{
   register double *A,max=-INF,min=INF;
   int k=0,len,rows;

   if(tos->typ!=MAT) {
      stkerr(" pminmax: ",MATNOT);
      return 0;
   }
   A=tos->mat;
   rows=tos->row;
   len=rows*tos->col;

   for(k=0;k<len;k++) {
      if((ABS(*A)) != (ABS(UNDEF))) {
         if(*A<min) min=*A;
         if(*A>max) max=*A;
      }
      A++;
   }
   return(
      drop() &&
      push(NUM,NULL,NOTAG,min,NULL,NULL,0,0,NULL) &&
      push(NUM,NULL,NOTAG,max,NULL,NULL,0,0,NULL)
   );
}

int rgnt() /* rgnt (hW hGC hY1 hY2 hX hL --- ) */
/* Fri Aug  2 04:43:25 PDT 2013

   In window W with graphics context GC, fill the region between lines
   Y1(X) and Y2(X), clipped to rectangle limits given in L, with the
   color given in GC.

   Values in incoming X are in ascending order, as in a list of times.

   Therefore, clipping in X can be done ahead of time to select just
   those rows of X that are in the X-clip region.  Binary search is
   used on X, so it must be in ascending order. */
{
   register Window Win;
   register GC GCon;
   register int k;
   register double *X,*Y1,*Y2;
   double *hW,*hGC,Xmin,Xmax,Ymin,Ymax;
   int n=0,np,r1,r2;
   short *x0=NULL,*y0=NULL,*w0=NULL,*h0=NULL,*x,*y,y2,*w,*h;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT) {
      stkerr(" rgnt: ",STKNOT);
      return 0;
   }
   hW=(tos-5)->mat;
   Win=(Window)patlong(*(hW+WIN)); /* hW(WIN) holds Window */
  
   hGC=(tos-4)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

/* Loading the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   np=tos->row;
   X=tos->mat;
   bsearchd(Xmin,X,np,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,np,&r2);
   np=r2-r1+1;
   if(np<2) {
   /* Sat Sep 21 11:32:47 PDT 2013.  Not an error.  Display message if
      wtrace is on, then silently return. */
      if(WTRACE) {
         gprintf(" rgnt: clipped X does not define two or more points");
         nc();
      }
      drop2(); drop2(); drop();
      return 1;
   }
   x=x0=malloc(1+np*sizeof(short));
   y=y0=malloc(1+np*sizeof(short));
   w=w0=malloc(1+np*sizeof(short));
   h=h0=malloc(1+np*sizeof(short));
   if(x0==NULL || y0==NULL || w0==NULL || h0==NULL) {
      stkerr(" rgnt: ",MEMNOT);
      drop2(); drop2(); drop();
      return 0;
   }
/* Store clipped points for rectangles: */
   X=(tos->mat)+r1;
   rot(); cop(); Y1=(tos->mat)+r1;
   rot(); cop(); Y2=(tos->mat)+r1;

   if(*Y1>Ymax) *Y1=Ymax; else if(*Y1<Ymin) *Y1=Ymin;
   if(*Y2>Ymax) *Y2=Ymax; else if(*Y2<Ymin) *Y2=Ymin;

   y2=(short)MIN(MAX(*Y1,*Y2),Ymax);
/* Wed Dec 11 03:41:00 PST 2013.  If *h>0, add 1 pixel for the
   height of a line: */
   if((*h=y2-(*y=(short)MAX(MIN(*Y1,*Y2),Ymin)))) *h+=1;
   *w=1+(short)*(X+1)-(*x=(short)*X);
   n++; x++; w++; y++; h++;
   X++; Y1++; Y2++;

   for(k=1;k<np-1;k++) {
      if(*Y1>Ymax) *Y1=Ymax; else if(*Y1<Ymin) *Y1=Ymin;
      if(*Y2>Ymax) *Y2=Ymax; else if(*Y2<Ymin) *Y2=Ymin;

      y2=(short)MIN(MAX(*Y1,*Y2),Ymax);
   /* Wed Dec 11 03:41:00 PST 2013.  If *h>0, add 1 pixel for the
      height of a line: */
      if((*h=y2-(*y=(short)MAX(MIN(*Y1,*Y2),Ymin)))) *h+=1;
      *w=1+(short)*(X+1)-(*x=(short)*X);

      if(*y!=*(y-1) || *h!=*(h-1)) { n++; x++; w++; y++; h++; }
      else { *(w-1)+=*w-1; } /* keep n the same and just bump w */

      X++; Y1++; Y2++;
   }
/* Draw rectangles: */
   x=x0;
   y=y0;
   w=w0;
   h=h0;
   for(k=0;k<n;k++) {

   /* Wed Dec 11 03:41:00 PST 2013.  Skip *h of zero: */
      if(*h) XFillRectangle(Dpy,Win,GCon,*x,*y,*w,*h);
      x++; w++; y++; h++;
   }
   mallfree((void *)&x0);
   mallfree((void *)&y0);
   mallfree((void *)&w0);
   mallfree((void *)&h0);

   return(drop2() && drop2() && drop());
}

int rgnt1() /* rgnt1 (hW hGC hR hY1 hY2 hX hL --- ) */
/* Thu Oct  3 12:01:50 PDT 2013

   Column k of Y1 and column k of Y2 define a pair of lines that bound
   a region.

   In window W with graphics context GC, fill the region between the
   kth pair of lines Y1(X,k) and Y2(X,k), clipped to rectangle limits 
   given in L, with the color given in R(k) used in graphics context GC.

   Values in incoming X are in ascending order, as in a list of times.

   Therefore, clipping in X can be done ahead of time to select just
   those rows of X that are in the X-clip region.  Binary search is
   used on X, so it must be in ascending order. 

   Rows of Y1, Y2 and X must be equal.  This is not checked. */
{
   register Window Win;
   register GC GCon;
   register int j=0,k;
   register double *X,*Y1,*Y2;
   double *X0,*Y01,*Y02;
   double *W,*hGC,*R,Xmin,Xmax,Ymin,Ymax;
   int cols,n,nclr,np,r1,r2,rows;
   short *x0=NULL,*y0=NULL,*w0=NULL,*h0=NULL,*x,*y,y2,*w,*h;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT || (tos-4)->typ!=MAT || (tos-5)->typ!=MAT || \
      (tos-6)->typ!=MAT) {
      stkerr(" rgnt1: ",STKNOT);
      return 0;
   }
   W=(tos-6)->mat;
   Win=(Window)patlong(*(W+WIN)); /* W(WIN) holds Window */
  
   hGC=(tos-5)->mat;
   GCon=(GC)patlong(*(hGC+GCC)); /* hGC(GCC) holds GC */

   R=(tos-4)->mat; /* colors */
   nclr=(tos-4)->row;

/* Load the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=MIN(*(X+2),*(X+3));
   Ymax=MAX(*(X+2),*(X+3));
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   np=tos->row;
   X=tos->mat;
   rows=tos->row;
   bsearchd(Xmin,X,np,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,np,&r2);
   np=r2-r1+1;

   if(np<2) {
   /* Display message if wtrace is on, then silently return. */
      if(WTRACE) {
         gprintf(\
            " rgnt1: clipped X does not define two or more points");
         nc();
      }
      drop2(); drop2(); drop2();
      return 1;
   }
   x=x0=malloc(1+np*sizeof(short));
   y=y0=malloc(1+np*sizeof(short));
   w=w0=malloc(1+np*sizeof(short));
   h=h0=malloc(1+np*sizeof(short));
   if(x0==NULL || y0==NULL || w0==NULL || h0==NULL) {
      stkerr(" rgnt1: ",MEMNOT);
      drop2(); drop2(); drop2();
      return 0;
   }
   X0=(tos->mat)+r1;
   rot(); cop(); Y01=(tos->mat)+r1;
   cols=tos->col;
   rot(); cop(); Y02=(tos->mat)+r1;
   if(tos->col!=cols) {
      gprintf(" rgnt1: Y1 and Y2 columns are not equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
/* Sat May  3 11:16:48 PDT 2014.  Too many colors is ok. */
   if(cols>nclr) { 
      gprintf(" rgnt1: number of R colors is less than columns of Y");
      nc();
      stkerr("","");
      drop2(); drop2(); drop2();
      return 0;
   }
   for(;j<cols;j++) {
      X=X0;
      Y1=Y01+j*rows;
      Y2=Y02+j*rows;
      x=x0;
      y=y0;
      w=w0;
      h=h0;
      n=0;
      
   /* Store clipped points for rectangles: */
      if(*Y1>Ymax) *Y1=Ymax; else if(*Y1<Ymin) *Y1=Ymin;
      if(*Y2>Ymax) *Y2=Ymax; else if(*Y2<Ymin) *Y2=Ymin;

      y2=(short)MIN(MAX(*Y1,*Y2),Ymax);
   /* Wed Dec 11 03:41:00 PST 2013.  If *h>0, add 1 pixel for the
      height of a line: */
      if((*h=y2-(*y=(short)MAX(MIN(*Y1,*Y2),Ymin)))) *h+=1;
      *w=1+(short)*(X+1)-(*x=(short)*X);
      n++; x++; w++; y++; h++;
      X++; Y1++; Y2++;

      for(k=1;k<np-1;k++) {
         if(*Y1>Ymax) *Y1=Ymax; else if(*Y1<Ymin) *Y1=Ymin;
         if(*Y2>Ymax) *Y2=Ymax; else if(*Y2<Ymin) *Y2=Ymin;

         y2=(short)MIN(MAX(*Y1,*Y2),Ymax);

      /* Wed Dec 11 03:41:00 PST 2013.  If *h>0, add 1 pixel for the
         height of a line: */
         if((*h=y2-(*y=(short)MAX(MIN(*Y1,*Y2),Ymin)))) *h+=1;
         *w=1+(short)*(X+1)-(*x=(short)*X);

         if(*y!=*(y-1) || *h!=*(h-1)) { n++; x++; w++; y++; h++; }
         else { *(w-1)+=*w-1; } /* keep n the same and just bump w */

         X++; Y1++; Y2++;
      }
   /* Draw rectangles: */
      x=x0;
      y=y0;
      w=w0;
      h=h0;
      XSetForeground(Dpy,GCon,(long)*(R+j));
      for(k=0;k<n;k++) {
      /* Wed Dec 11 03:41:00 PST 2013.  Skip *h of zero: */
         if(*h) XFillRectangle(Dpy,Win,GCon,*x,*y,*w,*h); 
         x++; w++; y++; h++; 
      }
   }
   mallfree((void *)&x0);
   mallfree((void *)&y0);
   mallfree((void *)&w0);
   mallfree((void *)&h0);

   return(drop2() && drop2() && drop2());
}

int wdemo() /* wdemo ( --- ) */
{
   Window win1;
   XEvent event;
   XSetWindowAttributes attributes;
   Cursor cursor_shape;
   XFontStruct *fontinfo;
   GC gr_context1,gr_context2;
   XGCValues gr_values;

   attributes.background_pixel=XWhitePixel(Dpy,Scr);
   attributes.background_pixel=25; /* red */
   attributes.border_pixel=XBlackPixel(Dpy,Scr);
   attributes.border_pixel=26; /* green */
 
   win1=XCreateWindow(Dpy,Root,0,0,
        XDisplayWidth(Dpy,Scr)-400,
        XDisplayHeight(Dpy,Scr)-200,
        30,Dep,InputOutput,Vis,
        CWBackPixel | CWBorderPixel,&attributes);
   XSelectInput(Dpy,win1,ExposureMask | KeyPressMask);
   
   gr_values.function=GXcopy;
   gr_values.plane_mask=AllPlanes;
   gr_values.foreground=BlackPixel(Dpy,Scr);
   gr_values.background=WhitePixel(Dpy,Scr);

   gr_context1=XCreateGC(Dpy,win1,
               GCFunction | GCPlaneMask | GCForeground | GCBackground,
               &gr_values);

   gr_values.function=GXxor;
   gr_values.foreground=WhitePixel(Dpy,Scr);
   gr_values.background=BlackPixel(Dpy,Scr);
   gr_context2=XCreateGC(Dpy,win1,
               GCFunction | GCPlaneMask | GCForeground | GCBackground,
               &gr_values);

   fontinfo=XLoadQueryFont(Dpy,"6x10");
   
   cursor_shape=XCreateFontCursor(Dpy,XC_heart);
   XDefineCursor(Dpy,win1,cursor_shape);

   XSetFont(Dpy,gr_context1,fontinfo->fid);
   XSetFont(Dpy,gr_context2,fontinfo->fid);

   XMapWindow(Dpy,win1);

   while(1) {
      XNextEvent(Dpy,&event);

      switch(event.type) {
      case Expose:
         XClearWindow(Dpy,win1);
         XDrawString(Dpy,win1,gr_context1,50,50,"Hello",5);
         XDrawImageString(Dpy,win1,gr_context2,20,20,"Hello",5);

         XFillRectangle(Dpy,win1,gr_context1,150,150,111,111);
         XFillRectangle(Dpy,win1,gr_context2,200,180,111,111);
         break;

      case KeyPress:
         return 1;
      }
   }
}

int wdemo1() /* wdemo ( --- ) */
/* Run a standalone X window demo */
{
   Window window;
   XSetWindowAttributes attributes;
   XGCValues gr_values;
   GC gr_context;
   XFontStruct *fontinfo;
   XEvent event;
   XColor color,dummy;
/*
   unsigned long width=350,height=200;
*/
   unsigned long width=50,height=50;

/* Testing pixmap: */
   int retpix,xhot,yhot;
   unsigned int hei,wid;
   Pixmap pix;

/* Testing XImage: */
   XImage *x;
   Drawable d;

   attributes.background_pixel=XWhitePixel(Dpy,Scr);

   window=XCreateWindow(Dpy,Root,
      200,200,width,height,5,Dep,InputOutput,Vis,
      CWBackPixel,&attributes);

   XSelectInput(Dpy,window,ExposureMask | KeyPressMask
       | ButtonPressMask);
   fontinfo=XLoadQueryFont(Dpy,"6x10");

   XAllocNamedColor(Dpy,DefaultColormap(Dpy,Scr),
      "black",&color,&dummy);

   gr_values.font=fontinfo->fid;
printf(" gr_values.font: %X\n\r",(unsigned int)gr_values.font);
   gr_values.foreground=color.pixel;
   gr_values.background=XWhitePixel(Dpy,Scr);
   gr_context=XCreateGC(Dpy,window,(GCForeground | GCBackground | GCFont),&gr_values);
   XFlush(Dpy);
   XMapWindow(Dpy,window);
   XFlush(Dpy);

/* reading a pixmap: */
   retpix=XReadBitmapFile(Dpy,Root,
      "/usr/include/X11/bitmaps/woman",&wid,&hei,&pix,&xhot,&yhot);
   printf(" retpix,wid,hei,xhot,yhot: %d,%d,%d,%d,%d\n\r",
      retpix,wid,hei,xhot,yhot);
   printf(" window,pix,: %X,%X\n\r",(unsigned int)window, \
      (unsigned int)pix);

/* get an image: */
/*
This bombs without displaying window first:
   ximage=XGetImage(Dpy,window,0,0,width,height,AllPlanes,XYPixmap);
   printf(" ximage.width,ximage.height: %d,%d\n\r",ximage->width,ximage->height);

But getting pixmap is ok:
   x=XGetImage(Dpy,pix,0,0,75,75,AllPlanes,XYPixmap);
   gprintf("    width, height: %d-by-%d",x->width,x->height); nc();
*/

   while(1) {
      XNextEvent(Dpy,&event);

      switch(event.type) {

      case ButtonPress:

/* XYPixmap uses 1 bit per pixel, ZPixmap uses 8 bits per pixel, and
   therefore has about 8 times as many bytes_per_line to hold them.

   Origin x, plus width-x must not be greater than width; probably
   true for y and height too.
 */

      d=window;
      x=XGetImage(Dpy,d,50,0,width-50,height,AllPlanes,XYPixmap);

   d=pix;
   x=XGetImage(Dpy,d,0,0,75,75,AllPlanes,XYPixmap);

      /* this code will go into word XImage: */
   gprintf(" Properties of drawable %X",d); nc();
   gprintf("    width, height: %d-by-%d",x->width,x->height); nc();
   gprintf("    xoffset: %d",x->xoffset); nc();
   gprintf("    format: %d",x->format); nc();
   gprintf("    (formats: %d=XYBitmap, %d=XYPixmap, %d=ZPixmap)", \
      XYBitmap,XYPixmap,ZPixmap); nc();
   gprintf("    data: %X",x->data); nc();
   gprintf("    byte_order: %d",x->byte_order); nc();
   gprintf("    bitmap_bit_order: %d",x->bitmap_bit_order); nc();
   gprintf("    (byte and bit order: %d=LSBFirst, %d=MSBFirst)", \
      LSBFirst,MSBFirst); nc();
   gprintf("    bitmap_pad: %d",x->bitmap_pad); nc();
   gprintf("    (bitmap pad: 8, 16, 32 either XY or ZFormat)"); nc();
   gprintf("    depth: %d",x->depth); nc();
   gprintf("    bytes_per_line: %d",x->bytes_per_line); nc();
   gprintf("    bits_per_pixel(Z): %d",x->bits_per_pixel); nc();
   gprintf("    Bits in Z arrangement:"); nc();
   gprintf("       red_mask(Z): %X",x->red_mask); nc();
   gprintf("       green_mask(Z): %X",x->green_mask); nc();
   gprintf("       blue_mask(Z): %X",x->blue_mask); nc();
   gprintf("    f.create_image: %X",x->f.create_image); nc();
   gprintf("    f.destroy_image: %X",x->f.destroy_image); nc();
   gprintf("    f.get_pixel: %X",x->f.get_pixel); nc();
   gprintf("    f.put_pixel: %X",x->f.put_pixel); nc();
   gprintf("    f.sub_image: %X",x->f.sub_image); nc();
   gprintf("    f.add_pixel: %X",x->f.add_pixel); nc();

   XDestroyImage(x);

   break;

      case Expose:
/*
         XCopyPlane(Dpy,pix,window,gr_context,0,0,wid,hei,50,50,1);
*/
         XCopyPlane(Dpy,pix,window,gr_context,0,0,wid,hei,0,0,1);
      break;

/* XCopyPlane copies up to wid-by-hei bits of pix, and no more; it
   does not stretch to fit larger rectangle. */


/*
      case Expose:
         XDrawLine(Dpy,window,gr_context,0,0,100,100);
         XDrawString(Dpy,window,gr_context,100,100,"hello",5);
      break;
*/
      case KeyPress:
/*
         XCloseDisplay(Dpy);
*/
      return(pushuint((unsigned long)window));
      }
   }
}

int winadd(struct wcb *new, struct wcb **table, struct wcb **end)
/* Adding a new element to linked list of windows being managed.

   Making a doubly linked list in sorted order, after:
      Schildt, H., "C: The Complete Reference,"
      Osborne McGraw-Hill, 1995

   Fancier than required, but useful as a model for general
   linked lists. */
{
   register struct wcb *old,*p;

   if(*end==NULL) { /* very 1st element in list */
      new->nex=NULL;
      new->pre=NULL;
      *end=new;
      *table=new;
      return 1;
   }
   p=*table;
   old=NULL;
   while(p) {
      if(strcmp(p->cat->nam,new->cat->nam)<0) {
         old=p;
         p=p->nex;
      }
      else { /* error if already have nam */
         if(strcmp(p->cat->nam,new->cat->nam)==0) {
            stkerr(" winadd: ",WINDUPE);
            return 0;
         }
         if(p->pre) { /* inserting in front of p */
            p->pre->nex=new;
            new->nex=p;
            new->pre=p->pre;
            p->pre=new;
            return 1;
         }
         new->nex=p; /* inserting in front of 1st element */
         new->pre=NULL;
         p->pre=new;
         *table=new;
         return 1;
      }
   }
   old->nex=new; /* putting at end */
   new->nex=NULL;
   new->pre=old;
   *end=new;
   return 1;
}

int winclear() /* winclear (hW x y w h) */
/* Clear a rectangle w-by-h at location x,y */
{
   Window win;
   struct wcb *wincb;
   int exposure=0,x,y;
   int h,w;

   if(!popint((int *)&h) || !popint((int *)&w) ||
      !popint(&y) || !popint(&x)) return 0;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XClearArea(Dpy,win,x,y,w,h,(Bool)exposure);
      return(drop());
   }
   stkerr(" winclear: ",WINUNINIT);
   return 0;
}

int wincopy() /* wincopy (hWfrom hWto hGC --- ) */
/* Copy window Wfrom to window Wto. */
{
   Window Wfrom,Wto;
   GC GCon;
   double *hWto;

   GCon=(GC)patlong(*((tos->mat)+GCC)); /* GCC element holds GC */
   drop();

   hWto=tos->mat;
   Wto=(Window)*(hWto+WIN);

   Wfrom=(Window)*(((tos-1)->mat)+WIN);
   
   XCopyArea(Dpy,Wfrom,Wto,GCon,0,0,
      (unsigned int)*(hWto+W),(unsigned int)*(hWto+H),0,0);

   return(drop2());
}

int wincreate() /* wincreate (hW qTitle --- ) */
/* Create a window.  Vector hW is a window control block (wcb) struc-
   ture, as defined in xterm.v, that is booked in the catalog.
   If the window in hW exists, it is destroyed first. */
{
   double *hW,*hA;
   char *S,*Title;
   unsigned long CWmask;
   struct wcb *p;
   XSetWindowAttributes watt;

   if(tos->typ!=STR) {
      stkerr(" wincreate: ",STRNOT);
      return 0;
   }
   strchop();

   if((tos-1)->typ!=MAT) {
      stkerr(" wincreate: ",MATNOT);
      return 0;
   }
   if((tos-1)->row!=wcbsize) {
      gprintf(" wincreate: require hW vector of %d rows",wcbsize); 
      nc();
      stkerr("","");
      return 0;
   } 
   if((Title=(char *)memgetn(tos->tex,tos->col))==NULL) {
      return 0;
   }
   if(!*Title) {
      mallfree((void *)&Title);
   }
   drop(); /* qTitle off stack */

   dup1s(); /* hW */
   named(); /* S is name of hW: */
   if((S=(char *)memgetn(tos->tex,tos->col))==NULL) {
      drop();
      mallfree((void *)&Title);
      return 0;
   }
   drop(); /* name qS off stack */

   if(winfindname(S,wintable)) { 
      if(!( /* destroying existing window first: */
         dup1s() &&
         winfree()
      )) {
         stkerr(" wincreate: ",WINDESNOT);
         mallfree((void *)&Title);
         mallfree((void *)&S);
         return 0;
      }
   }
   if((p=malloc(sizeof(struct wcb)))==NULL) { /* item for linked wcb */
      stkerr(" wincreate: ",MEMNOT);
      mallfree((void *)&S);
      return 0;
   }
   p->cat=(catitem *)catfetch(S); /* hW must be in catalog */
   mallfree((void *)&S);
   if(p->cat==NULL) {
      stkerr(" wincreate: ",WINCATNOT);
      mallfree((void *)&Title);
      mallfree((void *)&p);
      return 0;
   }
   hW=tos->mat; /* structure of wcb (xterm.v) and enum _wcb (term.h) */
   p->vis=0; /* window has not been mapped */

   if(*(hW+W)==-INF) *(hW+W)=gW;
   if(*(hW+H)==-INF) *(hW+H)=gH;
   if(*(hW+X)==-INF) *(hW+X)=gX;
   if(*(hW+Y)==-INF) *(hW+Y)=gY;
   if(*(hW+FLAGS)==-INF) *(hW+FLAGS)=gFlags;

/* Window attributes mask (WAM) and attributes vector (WAV): */
   if(*(hW+WAM)!=-INF) {
   /* Bit pattern of WAM mask element is already unsigned long, so 
      just copy it into CWmask: */
      memcpy(&CWmask,(unsigned long *)(hW+WAM),sizeof(long));
      if(*(hW+WAV)!=-INF) {
         pushd(*(hW+WAV));
         exe(); /* putting attributes vector on tos */
         if(tos->row!=WAVsize) {
            gprintf(" wincreate: require attributes vector of %d rows",
               WAVsize);
            nc();
            stkerr("","");
            mallfree((void *)&Title);
            mallfree((void *)&p);
            return 0;
         }
         hA=tos->mat; /* window attributes vector */
         watt.background_pixmap=(Pixmap)patlong(*(hA+BACKPMAP));
         watt.background_pixel=patlong(*(hA+BACKPIX));
         watt.border_pixmap=(Pixmap)patlong(*(hA+BORDPMAP));
         watt.border_pixel=patlong(*(hA+BORDPIX));
         watt.bit_gravity=patint(*(hA+BGRAV));
         watt.win_gravity=patint(*(hA+WGRAV));
         watt.backing_store=patint(*(hA+BACKSTOR));
         watt.backing_planes=patlong(*(hA+BACKGPLANE));
         watt.backing_pixel=patlong(*(hA+BACKGPIX));
         watt.save_under=(Bool)patint(*(hA+SAVEU));
         watt.event_mask=(long)patint(*(hA+EVENTM));
         watt.do_not_propagate_mask=(long)patint(*(hA+PROPNOM));
         watt.override_redirect=(Bool)patint(*(hA+OVERRIDE));
         watt.colormap=(Colormap)patlong(*(hA+COLORM));
         watt.cursor=(Cursor)patlong(*(hA+CURSOR));
         drop();
      }
      else {
         stkerr(" wincreate: ",WINATTNOT);
         mallfree((void *)&Title);
         mallfree((void *)&p);
         return 0;
      }
   }
   else {
      CWmask=CWBackPixel;
      watt.background_pixel=XBlackPixel(Dpy,Scr);
   }
/* Creating window from properties in wcb, CWmask, and watt: */
   p->win=wcreate(Title,(unsigned int)*(hW+FLAGS),(int)*(hW+X),
      (int)*(hW+Y),(unsigned int)*(hW+W),(unsigned int)*(hW+H),
      CWmask,&watt);

   *(hW+WIN)=(long)p->win; /* Window ptr into win element of hW */

   drop(); /* hW off the stack */
   mallfree((void *)&Title);

   if(winadd(p,&wintable,&winlast)) return 1;
   else XDestroyWindow(Dpy,(Window)p->win);
   mallfree((void *)&p);
   return 0;
}

int windisp() /* windows ( --- ) */
/* Displaying window properties. */
{
   if(wintable) winlist(wintable);
   else {
      gprintf(WINSNOT);
      nc();
   }
   return 1;
}

struct wcb *winfind(Window window, struct wcb *table)
/* Finding window in linked list of windows. */
{
   register struct wcb *p;

   if(!window) return(struct wcb *)NULL;

   p=table;
   while(p) {
      if(p->win==window) break;
      p=p->nex;
   }
   return p;
}

struct wcb *winfindname(char *nam, struct wcb *table)
/* Finding named window in linked list of windows. */
{
   register struct wcb *p;
   char *nam1;

   p=table;
   nam1=tagged(nam,"CODE__"); 
   while(p) {
      if(strcmp(p->cat->nam,nam1)==0) break;
      p=p->nex;
   }
   if(p) return p;

   p=table;
   nam1=tagged(nam,"DATA__"); 
   while(p) { 
      if(strcmp(p->cat->nam,nam1)==0) break;
      p=p->nex;
   }
   return p;
}

int winfree() /* winfree (hW --- ) */
/* Destroy window and remove it from wintable linked list. */
{
   Window win;
   struct wcb *p;
   double *hW;

   if(tos->typ!=MAT) {
      stkerr(" winfree: ",MATNOT);
      return 0;
   }
   hW=tos->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */

   p=winfind(win,wintable);
   if(!p) {
      stkerr(" winfree: ",WINNOT);
      return 0;
   }
   if(p->win) { /* p->win is the same as win=(Window)*(hW+WIN) */
      XDestroyWindow(Dpy,p->win);
      *(hW+WIN)=-INF;
      p->vis=0;
   }
   winrem(p,&wintable,&winlast);
   return(drop());
}

void winlist(struct wcb *windows)
/* Displaying items in linked list of windows. */
{
   register struct wcb *p;

   p=windows;
   gprintf(" Windows:"); nc();

   while(p) {
      pushq2(p->cat->nam,strlen(p->cat->nam));
      notag();
      gprintf("  %s: Window=%lX",tos->tex,p->win);
      drop(); nc();
      p=p->nex;
   }
}

int winraise() /* winraise (hW --- ) */
{
   Window win;
   struct wcb *wincb;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XRaiseWindow(Dpy,win);
      return(drop());
   }
   stkerr(" winraise: ",WINUNINIT);
   return 0;
}

void winrem(struct wcb *p, struct wcb **table, struct wcb **end)
/* Removing an item from a linked list. */
{
   if(*table==p) {
      *table=p->nex;
      if(*table) (*table)->pre=NULL;
      else *end=NULL;
   }
   else {
      p->pre->nex=p->nex;
      if(p!=*end) p->nex->pre=p->pre;
      else *end=p->pre;
   }
   mallfree((void *)&p);
}

int winresize() /* winresize (hW w h --- ) */
{ 
   Window win;
   struct wcb *wincb;
   int h,w;

   if(!popint((int *)&h) || !popint((int *)&w)) return 0;

   win=(Window)*((tos->mat)+WIN); /* WIN element holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(wincb) {
      XResizeWindow(Dpy,win,w,h);
      return(drop());
   }
   stkerr(" winresize: ",WINUNINIT);
   return 0;
}

int winshow() /* winshow (hW --- ) */
{
   Window win;
   struct wcb *wincb;
   double *hW;

   if(tos->typ!=MAT) {
      stkerr(" winshow: ",MATNOT);
      return 0;
   }
   hW=tos->mat;
   win=(Window)*(hW+WIN); /* WIN element of hW holds Window */
   wincb=winfind(win,wintable); /* find win in linked list */

   if(!wincb) {
      stkerr(" winshow: ",WINNOT);
      return 0;
   }
   if(wincb->vis==1) return(drop()); /* window already visible */

   if(win) {
      XMapWindow(Dpy,win);
      wincb->vis=1; /* visible */
      return(drop());
   }
   stkerr(" winshow: ",WINUNINIT);
   return 0;
}

int xclip() /* xclip (hY hX hL --- hY1 hX1 hL1) */
/* Thu Oct  3 18:17:45 PDT 2013

   This function makes similar clipping expressions in linem(), linet()
   and rgnt() available for high level use.  

   For x clip values in L, reduce X and Y to just the rows that are
   enclosed.

   Assumes values in X are in ascending order with no duplicates. 

   For Y, the clip min and max values in returned L1 match those in L.

   A timing case below shows this function is two or more times faster
   than equivalent high level code.

   Test case.

      Paste at the ready prompt the following line:
         syspath "../src/xterm.c" + "XCLIP_TEST" msource

      This shows running the test case:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Thu Oct  3 19:57:06 PDT 2013
      [tops@kaffia] ready > syspath "../src/xterm.c" + 
                            "XCLIP_TEST" msource

       Case 1: no clipping:

                    L
       Row 1:        0
       Row 2:        8
       Row 3:      101
       Row 4:      209

                   Y(1)     Y(2)       X
       Row 1:      101      201        0
       Row 2:      102      202        1
       Row 3:      103      203        2
       Row 4:      104      204        3
       Row 5:      105      205        4
       Row 6:      106      206        5
       Row 7:      107      207        6
       Row 8:      108      208        7
       Row 9:      109      209        8

       Case 2: clip to rows 4, 5 and 6:

                    L
       Row 1:        3
       Row 2:        5
       Row 3:      101
       Row 4:      209

                   Y(1)     Y(2)       X
       Row 1:      104      204        3
       Row 2:      105      205        4
       Row 3:      106      206        5
      [tops@kaffia] ready > 

      XCLIP_TEST

      1 9 uniform "X" book
      101 9 items 201 9 items park "Y" book
      list: X pminmax Y pminmax ; "L" book
      " Case 1: no clipping:" nl . nl
      Y X L xclip (hY1 hX1 hL1) .m nl park nl .m nl

      " Case 2: clip to rows 4, 5 and 6:" nl . nl
      list: X 4 pry X 6 pry Y pminmax ; "L" book
      Y X L xclip (hY1 hX1 hL1) .m nl park nl .m

      halt 

   Timing case.

      Code in high level word tgraphDraw.  The phrases inside the IF 
      THEN branch are high level equivalents to this function, xclip().

      This code runs both the high level code and function xclip()
      to show the timing of each.  Results are shown following:

      ...

      L rows 0>
      IF
         X "Xsav" book Y "Ysav" book L "Lsav" book

         time push

       \ Find rows of X that enclose the X-clip region (these phrases
       \ mimic the expressions in rgnt1() and other functions in file
       \ xterm.c):
         L 1st pry L 2nd pry 2dup max "Xmax" book min "Xmin" book
         X Xmin bsearch drop (r1)
         (r1) X over pry Xmin < IF (r1) 1+ THEN (r1)
         X Xmax bsearch drop (r2)
         (r1 r2) over - 1+ (r1 n) items (hR) \ X and Y rows to keep
         (hR) X over reach "X" book
         (hR) Y swap reach "Y" book
         Xmin L 1st poke Xmax L 2nd poke
      ELSE
         X 1st pry X dup rows ndx pry (x1 x2) Y pminmax (y1 y2)
         (x1 x2 y1 y2) 4 listn (hL) "L" book
      THEN
      time pull - 1E6 * "  ET high level:" nl . .i nl

      time push
      Ysav Xsav Lsav xclip "Lx" book "Xx" book "Yx" book
      time pull - 1E6 * " ET native code:" . .i nl

      Y Yx - null?
      X Xx - null? and
      IF " high level and native code results agree" . nl
      ELSE " high level and native code results do not agree" . nl
      THEN

      ...

      Results.  Times in microseconds are shown for several zooms in
      and zooms out.  The native code version is about two or more
      times faster.

      Initial graph:
        ET high level: 1119738
       ET native code: 386896
       high level and native code results agree

      Zoom in:
      EUZ13 13092 09:58 CDT Tue Sep 3, 2013
      EUZ13 13606 06:45 CDT Wed Sep 25, 2013
        ET high level: 178425
       ET native code: 85546
       high level and native code results agree

      Zoom in:
      EUZ13 13216 21:40 CDT Thu Sep 12, 2013
      EUZ13 13594 01:44 CDT Tue Sep 24, 2013
        ET high level: 74906
       ET native code: 39711
       high level and native code results agree

      Zoom in:
      EUZ13 13312 02:44 CDT Wed Sep 18, 2013
      EUZ13 13586 14:37 CDT Thu Sep 19, 2013
        ET high level: 20344
       ET native code: 10565
       high level and native code results agree

      Zoom out:
        ET high level: 285610
       ET native code: 40143
       high level and native code results agree

      Zoom out:
        ET high level: 519031
       ET native code: 80788
       high level and native code results agree

      Zoom out:
        ET high level: 950598
       ET native code: 527569
       high level and native code results agree */
{
   register int j=0;
   int cols,np,r1,r2,rows;
   register double *X,*Y,*Y0,*Y1,*X1;
   double *L1,Xmin,Xmax,Ymin,Ymax;

/* Load the clip rectangle limits from L: */
   X=tos->mat;
   Xmin=MIN(*X,*(X+1));
   Xmax=MAX(*X,*(X+1));
   Ymin=*(X+2);
   Ymax=*(X+3);
   drop(); /* L off stack */

/* Find rows of X that enclose the X-clip region: */
   np=tos->row;
   X=tos->mat;
   rows=tos->row;
   bsearchd(Xmin,X,np,&r1); /* takes nearest X below Xmin */
   if(*(X+r1)<Xmin) r1+=1;
   bsearchd(Xmax,X,np,&r2);
   np=r2-r1+1;

/* Return just rows r1:r2 in Y1 and X1: */
   swap(); /* Y to tos */
   cols=tos->col;
   if(tos->row!=rows) {
      gprintf(" xclip: X and Y rows are not equal");
      nc();
      stkerr("","");
      drop2(); drop2(); drop();
      return 0;
   }
/* Populate clipped matrix Y1: */
   Y0=tos->mat+r1;
   if(!matstk(np,cols,"_Y1")) return 0;
   Y1=tos->mat;
   for(;j<cols;j++) {
      Y=Y0+j*rows;
      Y1=(tos->mat)+j*np;
      memcpy(Y1,Y,np*sizeof(double));
   }
   lop(); swap(); /* Y off stack, X to tos */

/* Populate clipped matrix X1: */
   X=tos->mat+r1;
   if(!matstk(np,1,"_X1")) return 0;
   X1=tos->mat;
   memcpy(X1,X,np*sizeof(double));
   lop(); /* X off stack */

/* Return updated clip vector: */
   if(!matstk(4,1,"_L1")) return 0;
   L1=tos->mat;
   *L1=Xmin;
   *(L1+1)=Xmax;
   *(L1+2)=Ymin;
   *(L1+3)=Ymax;

   return 1;
} 

void XDrawLine1(Display *Dpy, Window Win, GC gc, int style, \
   short x1, short y1, short x2, short y2)
/* Wed Sep  8 16:25:59 PDT 2010

   Driver for drawing a line. 

   Defined X11 line styles are:
      int line_style: LineSolid=0, LineOnOffDash=1, LineDoubleDash=2
   and they are handled below by XDrawLine().  

   Wed Sep  8 16:25:59 PDT 2010.  Add style LineRectangle where a rec-
   tangle is defined by the diagonal of the given line.  No line is 
   drawn, but the rectangle is filled with the color of the given line
   (see Examples 3 and 4 below).  

   Tue May 21 09:22:20 PDT 2013.  Add style LineStep (see Examples 1 
   and 2 below).

   Drawing examples.

   1. Line style LineStep.  Tue May 21 10:19:31 PDT 2013.  

      Paste at the ready prompt the two lines following:
         syspath "../src/xterm.c" + 
         "#def Drawing Example 1" "#end Drawing Example 1" msource1
\
      At a glance the curves look the same, but using the left mouse
      buttom to zoom in and view a point where the curves cross, the
      green curve is seen to be style LineStep.

      How to zoom:

         To zoom in: Left click two points to define the diagonal 
         corners of an imaginary rectangle, then right click.

         Continuing this action (left, left, right) will zoom in 
         closer.

         Right clicking will zoom back out through each zoom in.

      #def Drawing Example 1

       \ Load plot words:
         "pExpose" missing IF "plot.v" source THEN
         "sine" missing IF "mmath.v" source THEN

       \ For zooming with LineStep, Window control block type, wcb.typ,
       \ must be 1 to invoke linet() (see linet() in this file and word
       \ pExpose in file sys/plot.v).  

       \ This sets wcb.typ to 1 in the WCB we'll be using from plot.v:
         1 plotWCB wcb.typ poke \ graph will use linet()

       \ Define line colors and styles:
         list:
            "DodgerBlue"  "LineSolid"     \ line 1
            "ForestGreen" "LineStep"      \ line 2
         end (hL) colorset (hLines hColors) graphset

       \ Define lines: a sine and a sine lagged by 150 degrees:
         1 2 2pi * 000 0.001 1000 sine (hS ht) drop \ line 1, sine
         1 2 2pi * 150 0.001 1000 sine (hL ht)      \ line 2, lagged

       \ Draw lines:
         (hS hL ht) rev park (ht hP) swap (hP ht) plot pause

       \ Close and reset to default lines and colors:
         plotclose

      #end Drawing Example 1

   2. Line style LineStep.  Wed May 22 10:02:51 PDT 2013.

      Used to debug zooming with LineStep.

      Paste at the ready prompt the two lines following:
         syspath "../src/xterm.c" + 
         "#def Drawing Example 2" "#end Drawing Example 2" msource1
\
      To zoom in, see above "How to zoom."

      #def Drawing Example 2

         "pExpose" missing IF "plot.v" source THEN
         "sine" missing IF "mmath.v" source THEN
         1 plotWCB wcb.typ poke \ graph will use linet()

       \ Define line colors and styles:
         "LineStep" "LS" book
         "LineSolid" "LD" book
         list:
            "blue"  LS \ line 1
            "green" LS \ line 2
            "red"   LS \ line 3
            "blue"  LD \ line 4
            "green" LD \ line 5
            "red"   LD \ line 6
         end (hL) colorset (hLines hColors) graphset

       \ Define lines:
         4 10 uniform dup 2 + dup 2 + dup 3 + dup 2 + dup 2 +     
         6 parkn "V" book
         1st V rows items "t" book

         V t plot pause plotclose

      #end Drawing Example 2

   3. Line style LineRectangle.  Thu Sep  9 18:21:39 PDT 2010.  

      Paste at the ready prompt the two lines following:
         syspath "../src/xterm.c" + 
         "#def Drawing Example 3" "#end Drawing Example 3" msource1
\
      To zoom in, see above "How to zoom."

      #def Drawing Example 3

       \ Load plot words:
         "weave" missing IF "plot.v" source THEN

       \ Zooming with LineRectangle is done by linet().  Window control
       \ block type, wcb.typ, must be 1 to invoke linet() (see linet()
       \ in this file and word pExpose in file sys/plot.v).

       \ This sets wcb.typ to 1 in the WCB we'll be using:
         1 plotWCB wcb.typ poke \ graph will use linet() 

       \ Define line colors and styles:
         list: 
            "blue"  "LineSolid"     \ line 1
            "green" "LineSolid"     \ line 2
            "red"   "LineRectangle" \ line 3
            "red"   "LineRectangle" \ line 4
         end (hL) colorset (hLines hColors) graphset

       \ Define lines:
         list: 1 50 thru ; dup reversed pile "Y1" book \ line 1       
         Y1 10 - "Y2" book                             \ line 2
         Y1 Y2 weave "Y3" book "Y4" book               \ lines 3 and 4

       \ Make an ascending order X axis (required for linet()):
         1 Y1 rows items "X" book

       \ Make plot:
         Y1 Y2 Y3 Y4 4 parkn X (hY hX) plot pause plotclose

      #end Drawing Example 3

   4. Line styles LineStep and LineRectangle.  
      Wed May 22 20:58:53 PDT 2013.

      Paste at the ready prompt the two lines following:
         syspath "../src/xterm.c" + 
         "#def Drawing Example 4" "#end Drawing Example 4" msource1
\
      To zoom in, see above "How to zoom."

      #def Drawing Example 4

         "pExpose" missing IF "plot.v" source THEN
         "sine" missing IF "mmath.v" source THEN

       \ Zooming for line styles LineStep and LineRectangle is only
       \ handled by linet(), and wcb.typ must equal 1 to use linet()
       \ (see pExpose() in sys/plot.v).

       \ This sets wcb.typ to 1 in the WCB we'll be using from plot.v:
         1 plotWCB wcb.typ poke \ graph will use linet()

       \ Define line colors and styles:
         list:
            "DodgerBlue"  "LineStep"             \ line 1 stepped
            "ForestGreen" "LineSolid"            \ line 2
            "ForestGreen" "LineStep"             \ line 2 stepped
            "#586949" (broccoli) "LineRectangle" \ line 2 fill rectangle
         end (hL) colorset (hLines hColors) graphset

         1 5 0 0.05 26 sine (hS1 ht) "t" book "S1" book \ line 1 data
         S1 1 lag "S2" book                             \ line 2 data

         S1 S2 S2 S2 4 parkn t plot pause plotclose

      #end Drawing Example 4
*/ {
   short h,w,x,y;

   if(style==LineRectangle) {
      x=MIN(x1,x2);
      y=MIN(y1,y2);
      w=ABS(x2-x1);
      h=ABS(y2-y1);
      XFillRectangle(Dpy,Win,gc,x,y,w,h);
      return;
   }
   XDrawLine(Dpy,Win,gc,x1,y1,x2,y2);
   return;
}

void XEventHandler(int event) /* (hE --- ) */
/* Handling an event that occurred in a window; hE on stack contains
   event info.
   Prototype file term.h contains the enumerations that define the 
   elements of E for various events (prototype for XEventHandler() 
   is also in term.h).
   The first element of E is always the window of the event. */
{
   Window win;
   XWindowAttributes watt;
   struct wcb *wincb;
   double *hW,*hH=NULL;
   #define NoHandlers -1
   int handler;
   
   win=(Window)*(tos->mat); /* from within, so no stack check first */

   if(!(wincb=winfind(win,wintable))) {
      drop();
      return;
   }
   if(WTRACE) {
      gprintf(" XEventHandler.  Event, window, wcb: %d, %s, %lX", \
         event,wincb->cat->nam,(unsigned int)win); nc();
   }
   oncat=wincb->cat; /* need this globally */
   (*(unsigned long (*)())oncat->exe)(); /* puts hW on stack */

   hW=tos->mat;
   if(*(hW+ECB)<0) { /* ptrs for exe() are always negative; see ptr() */
      handler=event;
      pushd(*(hW+ECB)); /* ptr to stack */
      exe(); /* puts hH on stack, list of event handlers */ 
      hH=tos->mat; 
      /* (hE hW hH) current stack elements */
   }
   else {
      handler=NoHandlers; 
   /* (hE hW) current stack elements */
   }
   switch(handler) { 

      case ButtonPress:
         pushd(*(hH+BP)); /* ptr */
      break;

      case ButtonRelease:
         pushd(*(hH+BR)); /* ptr */
      break;

      case ClientMessage:
         pushd(*(hH+CM)); /* ptr */
      break;

      case ConfigureNotify:
         pushd(*(hH+CE)); /* ptr */
      break;

      case Expose:

      /* Keep window size up to date for scaling */
         XGetWindowAttributes(Dpy,win,&watt);
         hW=(tos-1)->mat;
         *(hW+W)=(double)watt.width;
         *(hW+H)=(double)watt.height;

         pushd(*(hH+EE)); /* ptr */
      break;

      case KeyPress:
         pushd(*(hH+KP)); /* ptr */
      break;

      case KeyRelease:
         pushd(*(hH+KR)); /* ptr */
      break;

      case MotionNotify:
         pushd(*(hH+ME)); /* ptr */
      break;

      case VisibilityNotify: 
         pushd(*(hH+VE)); /* ptr */
      break;

      default:
         drop2(); drop(); /* dropping (hE hW hH) stack elements */
      return;

      case NoHandlers:
         drop2(); /* dropping (hE hW) stack elements */
      return;
   }
/* hE hW hH ptr */ lop(); 

/* hE hW ptr */ if(tos->real) exe(); /* run the handler */ 

                else { /* clear stk; no handler if ptr is zero: */
                   drop2();
                   drop();
                }
   return;
   #undef NoHandlers
}

int Ximageprops() /* XImage (d --- ) */
/* Display XImage structure for Drawable d. */
{
   unsigned long d;

   if(!popuint(&d)) return 0;

   gprintf(" Properties of drawable %lX",d); nc();
/*
   gprintf("    width, height: %d-by-%d",d.width,d.height); nc();
*/
return 1;

}

int XRegionFree() /* winregionfree (hR --- ) */
/* Destroy the region created by XRegionSet. */
{
   Region region;
   unsigned long R;

   if(!popuint(&R)) return 0;
   
   region=(Region)R;
   XDestroyRegion(region);
   return 1;
}

int XRegionSet() /* winregion (hEv hGC --- hR) */
/* Defines region for rectangle returned in expose event data, eventEE.
   Using region to redraw window has not been successful: clipped 
   regions are left undrawn. */
{
   Region region;
   XRectangle rect;
   double *Ev;
   GC GCon;

   region=XCreateRegion();
   GCon=(GC)patlong(*((tos->mat)+GCC)); /* GCC element holds GC */
   
   Ev=(tos-1)->mat;
   rect.x=(short)*(Ev+EEx);
   rect.y=(short)*(Ev+EEy);
   rect.width=(unsigned short)*(Ev+EEw);
   rect.height=(unsigned short)*(Ev+EEh);

   XUnionRectWithRegion(&rect,region,region);
   XSetRegion(Dpy,GCon,region);

   return(
      drop2() &&
      pushuint((unsigned long)region)
   );
}

/*
Creating wdraw, draw in window:

GC gc=(GC)0;

int wdraw()
{
Notes from display(plot), Xmain.c:

Using struct plot_struct *plot;

double xscale, yscale, pointsize;

    * set scaling factor between internal driver & window geometry *
   xscale=plot->width/4096.0;  yscale=plot->height/4096.0

   scale units of picture to the window
   px=(int)xscale*pointsize;
   py=(int)yscale*pointsize;

   create new GC (graphics context) and new pixmap:
   if(gc) XFreeGC(Dpy,gc);

   gc=XCreateGC(Dpy,plot->pixmap,0,(XGCValues *)0);
   XSetFont(Dpy,gc,font->fid);

   * set pixmap background *
   XSetForeground(Dpy,gc,colors[0]);
   XFillRectangle(Dpy,plot->pixmap,gc,0,0,plot->width,plot->height);
   XSetBackground(Dpy,gc,colors[0]);


In these, Raise and Clear should be set in wcreate:
   * top the window but don't put keyboard or mouse focus into it. *
   if(win->Raise) XMapRaised(Dpy,plot->window);
   * momentarily clear the window first if requested *
   if(win->Clear) {
      XClearWindow(Dpy,plot->window);
      XFlush(Dpy);
   }
*/
#endif
