/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2013  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* stk.h  March 1999

Copyright (c) 1999  D. R. Williamson

   Note:
      Where this file appears as #include "stk.h", place it highest, 
      above other program header files, but below #include "main.h"
      if it is present. 
*/

#define _depth (int)(tos-stack)
int _depthMIN; /* minimum stack depth when xx clears */
#define _ldepth (int)(tosloc-stklocal)
#define _fdepth (int)(tosfun-stkfunction)

#define DEPSTK 131072 /* items max in stack */
#define DEPLOCSTK 32  /* items max in local stack */
#define DEPFUNSTK 128 /* items max in function stack */
#define ERR_MSG_SIZE 160 /* size of snprintf-built messages to stkerr() */

/* Update these stack types in word.p if this enumeration changes: */
enum stktyp {DEF=1,NUM,MAT,STR,VOL,PTR}; 

typedef struct {

   int typ; /* type (one of enum stktyp listed above) */
   char *tok; /* name given by function that pushed item to stack */
   long long tag; /* 64 bit int tag; see tag.h, tag.c for possible settings */
   double real; /* real value if typ NUM */
   double imag; /* value of imaginary if typ NUM and tag is complex */
   double *mat; /* pointer to nums if typ MAT array (stored by cols) */
   char *tex; /* ptr to text if typ STR or typ VOL (stored by rows) */
   int row; /* rows in typ MAT or VOL or PTR (=1 for typ STR) */
   int col; /* cols per row in typ MAT or chars per row in STR or VOL */
   unsigned long (**ptr)(); /* typ PTR vec of pointers (len=row) */

/* The counts defined next, carried locally but visible to the system
   when the item is on top of the stack or when it has been pulled up
   from the catalog, determine its life span.  When cnt and cntc both
   become zero, memory used by the item is freed (functions drop() and 
   catfree()), eliminating the need for a global memory manager to track
   memory allocations.  When the stkitem is no longer needed, it simply
   dies on its last use: */

   int *cnt; /* pointer to count of instances on stack */
   int *cntc; /* pointer count of instances in catalog */

/* nam is the name of the catalog item that pushed this stk item, and 
   is in nam for info only when stack item contents are displayed.
   Stack items do not themselves use catalog names for anything since
   all are born on the stack and most die on it without ever seeing the
   catalog. */

   char *nam; /* from catalog item; nam is info only in stack display
   and is the name given when it was booked into the catalog, as shown
   below for word my-two-bits booked into the catalog followed by word
   props to display the stkitem elements in this struct (the tag ",@*"
   appended to my-two-bits locates it in the main library--see discus-
   sion in design.doc and in lib.h): 

                Tops 0.5
      Mon Aug 30 22:13:50 PDT 1999
      [tops@gutter] ready > 0.25 "my-two-bits" book
   
      [tops@gutter] ready > my-two-bits props
       NUM: tok (null), val 0.25, mat 0, tex 0
            row 0, col 0, ptr 0, cnt 1, cntc 1, nam my-two-bits,@*

      [tops@gutter] ready > 

   Update showing changed catalog tag on nam, and added stkitem
   elements (tag, real, imag):

               Tops 2.4.3
      Wed Apr 20 07:26:15 PDT 2005
      [tops@clacker] ready > 0.25 "my-two-bits" book

      [tops@clacker] ready > my-two-bits props
       NUM: tok (null), tag 0, real 0.25, imag 0, mat 0, tex 0
            row 0, col 0, ptr 0, cnt 1, cntc 1, nam my-two-bits,1s:*

      [tops@clacker] ready > 
*/

} stkitem; 

/* Global stkitems */
stkitem stack[1+DEPSTK]; /* this defines the stack for the program */ 
stkitem stklocal[1+DEPLOCSTK]; /* this is the very small local stack
                                  for words push, peek, and pull */
stkitem stkfunction[1+DEPFUNSTK]; /* function stack used by parser */

/* The five variables below are used when pushing a catalog item back 
   on the stack.  They allow push() to give the stack item the values 
   of tag, imag, cnt, cntc, and nam that it has in the catalog.  

   For a new stack item, function push() sets these to default values
   like 0 and NULL.  Using these variables temporarily overrides that 
   behavior when a catalog item is pushed to the stack so its catalog
   values are obtained.  Values are set back to the initial values 
   (set in stkinit()) as soon as push() runs; see exe.c for examples
   of use. */
long long CTAG; /* tag of stkitem in catalog, initialized to NOTAG */
double CIMAG; /* imag NUM of stkitem in catalog, initialized to 0 */
int *CNT;  /* initialized to NULL */
int *CNTC; /* initialized to NULL */
char *NAM; /* initialized to NULL */

/* Note: pointers in struct stkitem (*tok, *mat, *tex and *ptr) must
   be to memory that can be freed when the stack item is dropped from
   the stack.  Segmentation fault will occur if a pointer not from
   malloc(), say a pointer to a string or array built into the program,
   is assigned to one of these pointers and it ends up on the stack. 

   Some memory allocation functions are available to make this job 
   easier, such as (see mem.c):

      int matstk(int rows, int cols, char *name) // ( --- hA) //
      int strstk(int chars, char *name) // ( --- qS) //
      int volstk(int rows, int chars, char *name) // ( --- hT) //

      double *memget(int rows, int cols) -- array for 8-byte numbers
      char *memgetc(int rows, int chars) -- array for characters
      char *memgetn(char *name, int len) -- copies chars to mallocked

   and many uses of them can be found in functions for native words
   that use the stack, including those in files mat.c and tex.c.

   For pushing items to the stack, stk.c provides these functions:

      int pushd(double d) 
      int pushdx(double d) 
      int pushint(int i) 
      int pushq2(char *quote, int len)
      int pushmat(char *name, double *mat, int rows, int cols)
      int pushuint(unsigned int u)

   and these for popping items from the stack:

      int popbool(int *n);
      int popd(double *x);
      int popdx(double d) 
      int popint(int *n);
      int popuint(unsigned int *u);
*/

/* Messages for stack operations (other messages are in ctrl.h): */
#define ARRAYNOT "expect matrix or volume on stack"
#define BITOUT "bit index error: expect 0-63 (0based) or 1-64 (1based)"
#define BORDUNK "unknown byte order"
#define CANNOTPOP "cannot pop empty stack"
#define COLSNOT "column sizes are not equal"
#define COLSNOTV "column size does not match partition vector"
#define ROWSNOTV "row size does not match partition vector"
#define COMPLEXNOT "expect complex matrix on stack"
#define CROPNOT "expect starting and ending indices on stack"
#define EMPTYSTK "empty stack"
#define FEEDNOT "end of source input"
#define FLAGNOT "expect flag, true or false, on stack"
#define FILENOT "expect file name on the stack"
#define FUNSTKFUL "cannot push onto full function stack"
#define FUNSTKEMT "function stack is empty"
#define HEXNOT "hex character out of range"
#define KEYONLY "only when interactive"
#define INSUFITM "insufficient number of items on stack"
#define LISTBAD "stack element for list is not a number or a string"
#define LISTNOT "cannot create list"
#define LISTNEED "expect number or string on top of stack"
#define LOCSTKEMT "local stack is empty"
#define LOCSTKFUL "cannot push onto full local stack"
#define LOCSTKNOT "push/pull ops not equal (-stkbal stops this check)"
#define MATCHNOT "two stack item types do not match"
#define MATNOT "expect matrix on stack"
#define REALNOT "expect real-number item(s) on stack"
#define SPARSENOT "expect sparse matrix on stack"
#define MATNOT2 "expect two matrices on stack"
#define MATNOT3 "expect three matrices on stack"
#define MATORVOLNOT "expect matrix or volume on stack"
#define MATPURG "matrix is purged"
#define MATSNOT "matrix on top of stack; expect top two items to match"
#define MATSNOTC "matrices are not compatible"
#define MEMNOT "memory allocation failure"
#define MEMRESIZE "memory resize failure"
#define NEEDOFF "expect offset on stack"
#define NEEDTHREE "expect three on stack"
#define NEEDTWO "expect two items on stack"
#define NOTSUPT "stack item type not supported"
#define NUMNOT "expect number on stack"
#define NUMUNOT "expect unsigned number on stack"
#define NUMSNOT "number on top of stack; expect top two items to match"
#define NUMS2NOT "expect two numbers on stack"
#define NUMORMATNOT "expect number or matrix on stack"
#define NUMORMATNOT2 "expect two numbers or two matrices on stack"
#define NUMORMATORSTRNOT "expect number, matrix or string on stack"
#define NUMORMATORSTRNOT2 "expect number, matrix or string pairs"
#define NUMORSTRNOT "expect string or number on stack"
#define OFFBEYOND "offset is beyond stack"
#define ONTOFULL "cannot push onto full stack"
#define OUTCHAR "seeking character that is out of bounds"
#define OUTCOL "seeking column that is out of bounds"
#define OUTMAT "term is out of bounds"
#define OUTROW "seeking row that is out of bounds"
#define PTRNOT "expect vector of pointers on stack"
#define RETNOT "return stack not empty"
#define ROWSNOT "row sizes are not equal"
#define SBITERR "error finding sign bit"
#define SEEDBAD "specify number above 0 and not greater than 2147483646"
#define SOURCENOT "source file not found"
#define SQUNOT "expect square matrix"
#define STKNOT "stack items not as expected"
#define STKNOTC "stack items are not compatible" 
#define STRNOT "expect string on stack"
#define STRNOTNUM "cannot convert string into number"
#define STRORVOLNOT "expect string or volume on stack"
#define STRSNOT "string on top of stack; expect top two items to match"
#define STRSNOT2 "expect pair of strings on stack"
#define STRNOT2 "expect pair of strings on stack"
#define VOLNOT "expect volume on stack"
#define VOLSTRMATNOT "expect volume, string or matrix on stack"
#define VOLSNOT "volume on top of stack; expect top two to match"
#define VOLSNOTC "volumes are not compatible"

int anyq(); /* any? (hA --- f) */
int clear_stack(); /* xx (... --- ) */
int clear_temp_stack(); /* xl (... --- ...) */
int clrfs(); /* clrfs (... --- ...) */
int cmplxdbl(); /* real-imag (hC --- hAr hAi) */
int cmplxmatch(); /* cmplxmatch (hA hB --- hA hB) */
int cmpximag(); /* Im (hC --- hAi) */
int cmpxreal(); /* Re (hC --- hAr) */
int cols(); /* rows (hA --- r) */
int dblcmplx(); /* CMPX (hAr hAi --- hC) */
int depth(); /* depth ( --- n) */
int depthMIN(); /* depthMIN ( --- n) */
int depthMIN_set(); /* depthMIN_set (n --- ) */
int depthSTK(); /* depthSTK ( --- n) */
int dims(); /* dims (hA --- r c) */
int drop(); /* drop (u v --- u) */
int drop2(); /* 2drop (x y z --- x) */
int dup1s(); /* dup (x --- x x) */
int dup2s(); /* 2dup (x y --- x y x y) */
int dup3s(); /* 3dup (x y z --- x y z x y z) */
int fsdepth(); /* fsdepth ( --- n) */
int is_complex_word(); /* is_complex (hC --- f) */
int is_scalar(stkitem *x);
#define loclin(row,width) row*width /* to 1st char at row in text */
#define locvec(col,rows) col*rows /* to 1st row at col in mat */
int lop(); /* lop (x y --- y) */
int lpeek(); /* peek (x --- x y) */
int lpick(); /* peel (x 2 --- x u) */
int lpull(); /* pull (x --- x y) */
int lpush(); /* push (x y --- x) */
int named(); /* named (hA --- qS) */
int naming(); /* naming (hA qS --- hB) */
int other(); /* other ( u v w --- u v w u) */
int over(); /* over (u v --- u v u) */
stkitem *peek();
char *peekq(); 
int pick(); /* pick (u v w 2 --- u v w u) */
int pickfs(); /* pickfs (x 2 --- x u) */
int plop(); /* plop (u v --- u u v) */
stkitem *pop();
int popbool(int *n); 
int popd(double *x);
int popdc(double *x);
int popdx(double *xr, double *xi);
int popint(int *n); 
int popint1(int *n);
unsigned long **popptr(long *len, char **tag);
int popuint(unsigned long *u);
int props(); /* props (hA --- ) */
int pullfs(); /* pullfs (x --- x y) */
int push(int typ, char *tok, int tag, double real, double *mat, 
         char *text, int rows, int cols, unsigned long (**ptr)());
int pushd(double d);
int pushdx(double dr, double di);
int pushfs(); /* pushfs (x y --- x) */
int pushint(int i);
int pushmat(char *name, double *mat, int rows, int cols);
int pushptr(unsigned long (**ptr)(), int len, char *tag);
int pushptr1(unsigned long (**ptr)());
int pushq(char *quote, int len);
int pushq1(char *quote);
int pushq2(char *quote, int len);  
int pushstr(char *quote);  
int pushq3(); /* "..." ( --- qS) */
int pushtex(char *name, char *text, int lins, int width); 
int pushtex1(char *token);
int pushtex3(); /* {"..."} ( --- hT) */
int pushuint(unsigned int u);
int rev(); /* rev (u v w --- w u v) */
int revn(); /* revn (a b c d e 4 --- a e d c b) */
int roll(); /* roll (u v w 2 --- v w u) */
int rot(); /* rot (u v w --- v w u) */
int rows(); /* rows (hA --- r) */
int sizeof_item(); /* sizeof (hA --- nBytes) */
int spcols(); /* spcols (hA --- r) */
int spdims(); /* spdims (hA --- r c) */
int sprows(); /* sprows (hA --- r) */
void stkinit();
int stkdepth(); /* returns int equal to stack depth */ 
void stkerr(char *s1, char *s2);
int stkerrabs; /* int = count bumped by stkerr, always increasing */
int stkerrcount; /* int = count bumped by stkerr, may be reset */
int stkset(char *message);
int swap(); /* swap (u v --- v u) */
int tdepth(); /* tdepth ( --- n) */
void tocmplx(int rows, int cols, double *Ar, double *Ai, double *C);
void todblx(int rows, int cols, double *C, double *Ar, double *Ai);
int tok(); /* tok (hA --- qS) */
void tokfree(); /* (hA --- hA) */
int tos1(); /* tos (... --- ...) */
stkitem *tos;
stkitem *tosloc;
stkitem *tosfun;

/*----------------------------------------------------------------------

   Stack operator return flags:

   Stack functions when successful return 1, and 0 otherwise, except
   for the following:

      peek() returns the pointer to the top of stack, tos.

      pop() returns the pointer to the former top of stack item that
      was popped.  Its arrays will have been freed if its age counts
      (cnt and cntc) dropped to 0.

      popptr(&num,&tag) returns the pointer to an array of pointers,
      with the number of pointers and local library tag they reference
      returned as arguments.  The function that builds an array of
      pointers in the first place, inline1(), also books it into the
      catalog, ensuring that it is never freed when it is popped from
      the stack.

   Demonstrating automatic memory management:

      In the following, a 500-by-500 identity matrix is created and
      saved in the catalog as A, then its properties are displayed.
      The pointer to the matrix, containing 2 million bytes, is element
      mat in the struct stkitem that is shown by word props.

      Pointer mat can be seen to toggle between 4029A008 and 40483008
      as the first matrix A is created then replaced by a second ma-
      trix A.

      For the operating system to reuse memory pointers like this, it
      must mean that they are being freed as each old A falls out of
      the catalog and off of the stack, when its catalog count, cntc,
      and stack count, cnt, both equal zero.

               Tops 0.5.1
      Sat Sep 11 12:57:30 PDT 1999
      [tops@gutter] ready > 500 identity is A, A props
       MAT: tok _identity, val 0.000000, mat 4029A008, tex 0
            row 500, col 500, ptr 0, cnt 1, cntc 1, nam A,@*

      [tops@gutter] ready > 500 identity is A, A props
       MAT: tok _identity, val 0.000000, mat 40483008, tex 0
            row 500, col 500, ptr 0, cnt 1, cntc 1, nam A,@*

      [tops@gutter] ready > 500 identity is A, A props
       MAT: tok _identity, val 0.000000, mat 4029A008, tex 0
            row 500, col 500, ptr 0, cnt 1, cntc 1, nam A,@*

      [tops@gutter] ready > 500 identity is A, A props
       MAT: tok _identity, val 0.000000, mat 40483008, tex 0
            row 500, col 500, ptr 0, cnt 1, cntc 1, nam A,@*

      [tops@gutter] ready > bye
              Good-bye
      Sat Sep 11 12:58:04 PDT 1999
      [user@gutter] /home/dale/proj/sage/sou >

--------------------------------------------------------------------- */
/* end of file stk.h */
