/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2013  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* stk.c  March 1999

Copyright (c) 1999  D. R. Williamson
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "inpo.h"
#include "lib.h"
#include "mem.h"
#include "sparse.h"
#include "op4.h"
#include "tag.h"
#include "tex.h"

int anyq() /* any? (hA --- false or hA true) */
/* If item on the stack has any rows or characters, return it and
   true; otherwise, drop it and return false.

   In the case of a number, if it is >0 return it and true; other-
   wise, drop it and return false. */
{
   SparseMatrix m;

   switch(tos->typ) {

      case NUM:
         if(tos->real>0) return(pushint(xTRUE));
         else return(
            drop() &&
            pushint(xFALSE)
         );
      case VOL:
         if(is_sparse(tos)) {
            m=sparse_overlay(tos);
            if(!m.H[ROWS]>0 || !m.H[COLS]>0)
               return(
                  drop() &&
                  pushint(xFALSE)
               );
            else return(pushint(xTRUE));
        }
      case MAT:
      case PTR:
      case STR:
         if(!tos->row>0 || !tos->col>0)
            return(
               drop() &&
               pushint(xFALSE)
            );
         else return(pushint(xTRUE));

      default:
         return(
            drop() &&
            pushint(xFALSE)
         );
   }
}

int clear_stack() /* xx (... --- ) */
{
   while(_depth>_depthMIN) drop();
   return 1;
}

int clear_temp_stack() /* xl (... --- ...) */
{
   while(_ldepth) {
      lpull();
      drop();
   }
   return 1;
}

int clrfs() /* clrfs (... --- ...) */
{
   while(_fdepth) {
      pullfs();
      drop();
   }
   return 1;
}

int cmplxdbl() /* real-imag (hC --- hAr hAi) */
/* Complex matrix C is split into separate matrices real and imaginary.

   The even rows in C hold the real terms, and odd rows hold the
   imaginary terms (for 0-based indexing). 

   No test for complex tag is made, so any MAT in complex form can be 
   converted (of course, it ought to have an even number of rows).  

   Ar and Ai have half the number of rows of C. */
{
   int cols,rows;
   double Ai,Ar;

   if(tos->typ==NUM) {
      Ar=tos->real;
      Ai=tos->imag;
      return(
         drop() &&
         pushd(Ar) &&
         pushd(Ai) 
      );
   }
   if(is_sparse(tos)) {
      return spreal_imag();
   }
   if(tos->typ!=MAT) {
      stkerr(" real-imag: ",MATNOT);
      return 0;
   }
   if(!matstk((rows=(tos->row)/2),(cols=tos->col),"_Ar")) return 0;
   if(!matstk(rows,cols,"_Ai")) return 0;

   todblx(rows,cols,(tos-2)->mat,(tos-1)->mat,tos->mat);

   return(rot() && drop());
}

int cmplxmatch() /* cmplxmatch (hA hB --- hA hB) */
/* If A or B is complex, make the other one match the complex type by 
   adding a null imaginary part. */
{
   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" cmplxmatch: ",MATNOT2);
      return 0;
   }
   if(is_complex(tos) || is_complex(tos-1)) {
      if(!is_complex(tos)) {
         dup1s();
         dims();
         null();
         dblcmplx();
      }
      else {
         if(!is_complex(tos-1)) {
            swap();
            dup1s();
            dims();
            null();
            dblcmplx();
            swap();
         }
      }
   }
   return 1;
}

int cmpximag() /* Im (hC --- hAi) */
/* Extract the imaginary part of complex C.

   The even rows in C hold the real terms, and odd rows hold the
   imaginary terms (for 0-based indexing). 

   No test for complex tag is made, so any MAT in complex form can be 
   converted (of course, it ought to have an even number of rows).  

   Ai has half the number of rows of C. */
{
   register int k=0;
   int cols,rows;
   double ai;
   double *C;
   double *Ai;

   if(tos->typ==NUM) {
      ai=tos->imag;
      return(
         drop() &&
         pushd(ai) 
      );
   }
   if(is_sparse(tos)) {
      return(spreal_imag() && lop());
   }
   if(tos->typ!=MAT) {
      stkerr(" cmpimag: ",MATNOT);
      return 0;
   }
   C=tos->mat;

   if(!matstk((rows=(tos->row)/2),(cols=tos->col),"_Ai")) return 0;
   Ai=tos->mat;

   for(;k<rows*cols;k++) {
      C++;

      *Ai=*C;
      Ai++;
      C++;
   }
   return(lop());
}

int cmpxreal() /* Re (hC --- hAr) */
/* Extract the real part of complex C.

   The even rows in C hold the real terms, and odd rows hold the
   imaginary terms (for 0-based indexing). 

   No test for complex tag is made, so any MAT in complex form can be 
   converted (of course, it ought to have an even number of rows).  

   Ar has half the number of rows of C. */
{
   register int k=0;
   int cols,rows;
   double ar,*Ar,*C;

   if(tos->typ==NUM) {
      ar=tos->real;
      return(
         drop() &&
         pushd(ar) 
      );
   }
   if(is_sparse(tos)) {
      return(spreal_imag() && drop());
   }
   if(tos->typ!=MAT) {
      stkerr(" cmpxreal: ",MATNOT);
      return 0;
   }
   C=tos->mat;

   if(!matstk((rows=(tos->row)/2),(cols=tos->col),"_Ar")) return 0;
   Ar=tos->mat;

   for(;k<rows*cols;k++) {
      *Ar=*C;
      Ar++;
      C++;
      C++;
   }
   return(lop());
}
int cols() /* cols (hA --- r) */
{
   if(peek()!=NULL) {
      if (!is_sparse(tos)) { /* not sparse */
          pushint(tos->col);
          return(lop());
      } else {
         return(spcols());
      }
   }
   stkerr(" ",CANNOTPOP);
   return 0;
}

int dblcmplx() /* complex (hAr hAi --- hC) */
/* Complex real and imaginary parts in separate matrices are moved into
   into a single matrix with twice as many rows. 

   Terms of Ar go into the even rows of C (0, 2, 4, ...), and terms 
   of Ai go into the odd rows (1, 3, 5, ...) of C. 

   If both Ar and Ai are NUMs, then C is a NUM with real=Ar and imag=Ai.
*/
{
   int cols,rows;

   if(stkdepth()<2) {
      stkerr(" complex: ",NEEDTWO);
      return 0;
   }
   if(tos->typ==NUM && (tos-1)->typ==NUM) {
      pushint(0);
      tos->real=(tos-2)->real;
      tos->imag=(tos-1)->real;
      set_complex(tos);
      return(lop() && lop());
   }
/* Turn NUMS into 1-by-1 MATS: */
   if(tos->typ==NUM) typnum2mat();
   if((tos-1)->typ==NUM) {
      swap();
      typnum2mat();
      swap();
   }
   if((tos->typ!=MAT     || (tos-1)->typ!=MAT) &&      /* not dense matrices */
      (!is_sparse(tos) || (!is_sparse(tos-1))) ) { /* not sparse matrices */
      stkerr(" complex: ",NUMORMATNOT2);
      return 0;
   }
   if (tos->typ==MAT) { /* dense inputs */
      if((rows=tos->row)!=(tos-1)->row ||
         (cols=tos->col)!=(tos-1)->col) {
         stkerr(" complex: ",MATSNOTC);
         return 0;
      }
      if(!matstk(2*rows,cols,"_C")) return 0;
      set_complex(tos);
      tocmplx(rows,cols,(tos-2)->mat,(tos-1)->mat,tos->mat);

      return(lop() && lop());
   } 
   else {            /* sparse inputs */
      set_complex(tos);
      return spadd_cx();
   }
}

int depth() /* depth ( --- n) */
/* Pushes depth of stack to the stack, returns 1 if no error.

   Note: stkdepth() returns the actual depth; this function, depth(),
   does not.  This function simply puts depth on the stack and always
   returns 1 (true) if successful. */
{
   return(pushint(_depth));
}

int depthMIN() /* depthMIN ( --- n) */
{
   return(pushint(_depthMIN));
}

int depthMIN_set() /* depthMIN_set (n --- ) */
{
   return(popint(&_depthMIN));
}

int depthSTK() /* depthSTK ( --- n) */
/* Pushes the maximum depth of stack to the stack. */
{
   return(pushint(DEPSTK));
}

int dims() /* dims (hA --- r c) */
{
   if(peek()!=NULL) {
      return(
         dup1s() &&
         rows() &&
         swap() &&
         cols() 
      );
   }
   stkerr(" ",CANNOTPOP);
   return 0;
}

int drop() /* drop (u v --- u) */
/* Drops topmost item from stack, and frees its memory if no other 
   copies of it are on the stack or in the catalog.

   This function, and function catfree(), control the freeing of all
   mallocked memory (that's mallocked as in "frolicked in the park"). */
{
   if(tos==stack) {
      stkerr(" ",CANNOTPOP);
      return 0;
   }
   tokfree(); /* applies to items not in catalog */

   (*(tos->cnt))--; /* decrementing the stk item's stack count */

/* Freeing stack item if its stack and catalog counts are both zero: */
   if(*(tos->cnt)==0 && *(tos->cntc)==0) {
      mallfree((void *)&tos->tok);
      mallfree((void *)&tos->mat);
      mallfree((void *)&tos->tex);
      mallfree((void *)&tos->ptr);
      mallfree((void *)&tos->cnt);
      mallfree((void *)&tos->cntc);
   /* Never free tos->nam: it is a catalog name, and names are
      never deleted from the catalog. */
   }
   tos--;
   return 1;
}

int drop2() /* 2drop (x y z --- x) */
{
   if(tos<stack+2) {
      stkerr(" 2drop: ",NEEDTWO);
      return 0;
   }
   return(drop() && drop());
}

int dup1s() /* dup (x --- x x) */
{
   register int delta;

   if((delta=_depth)<1) {
      stkerr(" dup: ",EMPTYSTK);
      return 0;
   }
   if(delta==DEPSTK) {
      stkerr(" dup: ",ONTOFULL);
      return 0;
   }
   (*(tos->cnt))++;

   *(tos+1)=*tos;
   tos++;
   return 1;
}

int dup2s() /* 2dup (x y --- x y x y) */
{
   register int delta;

   if((delta=_depth)<2) {
      stkerr(" 2dup: ",NEEDTWO);
      return 0;
   }
   if(delta==DEPSTK-1) {
      stkerr(" 2dup: ",ONTOFULL);
      return 0;
   }
/* this is the same as "over over" */
   *(tos+1)=*(tos-1);
   tos++;
   (*(tos->cnt))++;

   *(tos+1)=*(tos-1);
   tos++;
   (*(tos->cnt))++;

   return 1;
}

int dup3s() /* 3dup (x y z --- x y z x y z) */
{
   int delta;

   if((delta=_depth)<3) {
      stkerr(" 3dup: ",NEEDTHREE);
      return 0;
   }
   if(delta==DEPSTK-2) {
      stkerr(" 3dup: ",ONTOFULL);
      return 0;
   }

   *(tos+1)=*(tos-2);
   tos++;
   (*(tos->cnt))++;

   *(tos+1)=*(tos-2);
   tos++;
   (*(tos->cnt))++;

   *(tos+1)=*(tos-2);
   tos++;
   (*(tos->cnt))++;

   return 1;
}

int fsdepth() /* fsdepth ( --- n) */
/* Pushes depth of function stack to the stack, returns 1 if no 
   error. */
{
   return(pushint(_fdepth));
}

int is_complex_word() /* is_complex (hC --- f) */
/* Flag f true if C is complex. */
{
   return(
      pushint(xTRUE*is_complex(tos)) && lop()
   );
}

int is_scalar(stkitem *x)
/* Returns 1 if x is NUM or 1-by-1 MAT, real or complex. */
{
   if(x->typ==NUM) return 1;
   if(x->typ!=MAT) return 0;
   if(is_complex(x)) return(x->row==2 && x->col==1);
   else return(x->row==1 && x->col==1);
}

int lop() /* lop (x y --- y) */
{
   stkitem temp;

   if(_depth<2) {
      stkerr(" lop: ",NEEDTWO);
      return 0;
   }
   else {
      temp=*tos;
      *tos=*(tos-1);
      *(tos-1)=temp;
      return drop();
   }
}

int lpeek() /* peek (x --- x y) */
/* Copying top of local stack to stack. */
{
   if(tosloc==stklocal) {
      stkerr(" peek: ",LOCSTKEMT);
      return 0;
   }
   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++;
   memcpy(tos,tosloc,sizeof(stkitem));
   (*(tos->cnt))++; /* another stack instance */

   return 1;
}

int lpick() /* peel (x 2 --- x u) */
/* Copying nth item of the local stack to the program top of stack. */
{
   register int offset;

   if(tos->typ!=NUM) {
      stkerr(" peel: ",NEEDOFF);
      return 0;
   }
   offset=(int)pop()->real;

   if(offset>=_ldepth || offset<0) {
      stkerr(" peel: ",OFFBEYOND);
      return 0;
   }
   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++;
   memcpy(tos,tosloc-offset,sizeof(stkitem));
   (*(tos->cnt))++; /* another stack instance */

   return 1;
}

int lpull() /* pull (x --- x y) */
/* Moving top of local stack to stack. */
{
   if(tosloc==stklocal) {
      stkerr(" pull: ",LOCSTKEMT);
      return 0;
   }
   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++;
   memcpy(tos,tosloc,sizeof(stkitem));
   tosloc--;

   return 1;
}

int lpush() /* push (x y --- x) */
/* Moves top of stack item to top of local stack.  The item's stack
   count, cnt, is unchanged.  */
{
   if(tosloc==(stklocal+DEPLOCSTK)) {
      stkerr(" push: ",LOCSTKFUL);
      return 0;
   }
   if(tos==stack) {
      stkerr(" ",CANNOTPOP);
      return 0;
   }
   tosloc++; /* adding to local stack */
   memcpy(tosloc,tos,sizeof(stkitem));
   tos--; /* removing from stack, as drop() does */

   return 1;
}

int named() /* named (hA --- qS) */
/* Providing catalog name or token name of stack item. */
{
   if(!stkdepth()) {
      stkerr(" named: ",EMPTYSTK);
      return 0;
   }
   if(*(tos->cntc) && tos->nam) {
      return( /* item is in catalog--give stk.nam name: */
         pushq2(tos->nam,strlen(tos->nam)) && lop()
      );
   }
   else {
      if(tos->tok)
         return( /* item is just on stack--give stk.token name: */
            pushq2(tos->tok,strlen(tos->tok)) && lop()
         );
      else
         return(
            pushq2("",0) && lop()
         );
   }
}

int naming() /* naming (hA qS --- hB) */
/* Changing to S the token name element of A, tos.tok. */
{
   char *name;

   if(stkdepth()<2) {
      stkerr(" naming: ",STKNOT);
      return 0;
   }
   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" naming: ",STRORVOLNOT);
      return 0;
   }
   strchop();
/* Stack items belonging to catalog items can't be renamed.  Rather 
   than giving an error, just copy A (word cop()) and name the copy. */
   if(*(tos-1)->cntc) {
      swap(); 
      cop(); 
      swap();
   }
   if((name=(char *)memgetn(tos->tex,tos->col))==NULL) return 0;
   drop(); /* S is dropped */

   tokfree();     /* free the existing tok name */
   tos->tok=name; /* store the new one */
   return 1;
}

int other() /* other ( u v w --- u v w u) */
/* Copying 3rd stack item to top of stack. */
{
   if(_depth<3) {
      stkerr(" other: ",NEEDTHREE);
      return 0;
   }
   *(tos+1)=*(tos-2);
   tos++;

   (*(tos->cnt))++;

   return 1;
}

int over() /* over (u v --- u v u) */
/* Copying second on stack to top of stack. */
{
   register int delta;

   if((delta=_depth)<2) {
      stkerr(" over: ",NEEDTWO);
      return 0;
   }
   if(delta==DEPSTK) {
      stkerr(" over: ",ONTOFULL);
      return 0;
   }
   *(tos+1)=*(tos-1);
   tos++;

   (*(tos->cnt))++;

   return 1;
}

stkitem *peek()
/* Returning pointer to top of stack.  Same as tos. */
{
   if(tos!=stack) return tos;
   return NULL;
}

char *peekq()
/* Returning pointer to top of stack if it contains STR. */
{
   register stkitem *stk;

   if((stk=peek())==NULL || stk->typ!=STR) {
      stkerr(" peekq: ",STRNOT);
      return NULL;
   }
   return(stk->tex);
}

int pick() /* pick (u v w 2 --- u v w u) */
/* Copying nth stack item to top of stack. */
{
   stkitem *temp;
   register int offset;

   if((temp=peek())==NULL) {
      stkerr(" pick: ",NEEDOFF);
      return 0;
   }
   else {
      if(temp->typ!=NUM) {
         stkerr(" pick: ",NEEDOFF);
         return 0;
      }
   }
   offset=(int)pop()->real;
   if(offset>=_depth || offset<0) {
      stkerr(" pick: ",OFFBEYOND);
      return 0;
   }
   *(tos+1)=*(tos-offset);
   tos++;

   (*(tos->cnt))++;

   return 1;
}

int pickfs() /* pickfs (x 2 --- x u) */
/* Copying nth item of function stack to the program top of stack. */
{
   register int offset;

   if(tos->typ!=NUM) {
      stkerr(" pickfs: ",NEEDOFF);
      return 0;
   }
   offset=(int)pop()->real;

   if(offset>=_fdepth || offset<0) {
      stkerr(" pickfs: ",OFFBEYOND);
      return 0;
   }
   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++;
   memcpy(tos,tosfun-offset,sizeof(stkitem));
   (*(tos->cnt))++; /* another stack instance */

   return 1;
}

int plop() /* plop (u v --- u u v) */
/* Sat Jul 13 11:58:07 PDT 2013.  Duplicating second stack item. */
{
   register int delta;

   if((delta=_depth)<2) {
      stkerr(" plop: ",NEEDTWO);
      return 0;
   }
   if(delta==DEPSTK) {
      stkerr(" plop: ",ONTOFULL);
      return 0;
   }
   return(over() && swap());
}

stkitem *pop()
/* Popping top of stack, returning its pointer. */
{
   if(drop()) return (tos+1);
   return NULL;
}

int popbool(int *n)
/* Popping top of stack if it is xTRUE or xFALSE and returning its
   int value in arg n. */
{
   register stkitem *stk;

   if((stk=(peek()))==NULL || stk->typ!=NUM || \
      !(stk->real==xFALSE || stk->real==xTRUE)) {
      gprintf(" popbool: expect %d (true) or %d (false) on stack", \
         xTRUE,xFALSE); nc();
         stkerr("","");
         return 0;
   }
   *n=(int)pop()->real;
   return 1;
}

int popd(double *x)
/* Popping top of stack if it is NUM and returning its fp value in
   arg x. */
{
   if(tos->typ!=NUM) {
      stkerr(" popd: ",NUMNOT);
      return 0;
   }
   *x=pop()->real; /* 8-byte floating point */
   return 1;
}

int popdc(double *x)
/* Popping top of stack if it is MAT and returning its first two fp 
   values in two elements of *x, to be used as a double-complex number. 
*/
{
   if(tos->typ!=MAT) {
      stkerr(" popdc: ",MATNOT);
      return 0;
   }
   *x=*tos->mat;
   *(x+1)=*(tos->mat+1);
   return drop();
}

int popdx(double *xr, double *xi)
/* Popping top of stack if it is NUM and returning its real and imag
   fp values in *xr and *xi, respectively. */
{
   if(tos->typ!=NUM) {
      stkerr(" popdx: ",NUMNOT);
      return 0;
   }
   *xr=tos->real;
   *xi=tos->imag;
   return drop();
}

int popint(int *n)
/* Popping top of stack if it is NUM and returning its int value in
   arg n. */
{
   if(tos->typ!=NUM) {
      stkerr(" popint: ",NUMNOT);
      return 0;
   }
   *n=(int)pop()->real;
   return 1;
}

int popint1(int *n)
/* Popping top of stack if it is NUM or MAT.  Returning int value of
   NUM or int value of the first element of MAT. 

   Used by ifbranch() when interpreting IF ... ELSE ... THEN control
   structure. */
{
   if(tos->typ==NUM) {
      *n=(int)pop()->real;
      return 1;
   }
   if(tos->typ==MAT) {
      *n=(int)*tos->mat; /* taking the first element of real matrix */
      return(drop());
   }
   stkerr(" popint1: ",NUMNOT);
   return 0;
}

unsigned long **popptr(long *len, char **tag)
/* Returns pointer to a column vector of pointers; the number of
   pointers and a pointer to the local library tag they reference
   are returned in arg len and arg tag. */
{
   if(tos->typ!=PTR) {
      stkerr(" popptr: ",PTRNOT);
      return 0;
   }
   *len=tos->row; /* the length of this vector of pointers */
   *tag=tos->tok; /* tag was put into tok by pushptr() */

   return(unsigned long **)pop()->ptr;
}

int popuint(unsigned long *u)
/* Popping top of stack if it is NUM and returning its unsigned int
   value in arg u. */
{
   long long U;

   if(tos->typ!=NUM) {
      stkerr(" popuint: ",NUMUNOT);
      return 0;
   }
   U=(long long)pop()->real;
   *u=(unsigned long)U;
   return 1;
}

int props() /* props (hA --- ) */
/* Showing struct stkitem properties of the item on stack. */
{
   char stktyp[4];

/* These are all the elements in struct stkitem (see stk.h): */
   char *tok;
   int tag;
   double real=0;
   double imag=0;
   double *mat;
   char *tex;
   int row;
   int col;
   unsigned long (**ptr)();
   int *cnt;
   int *cntc;
   char *nam;

   if(_depth<1) {
      stkerr(" props: ",EMPTYSTK);
      return 0;
   }
   tok=tos->tok;
   if(!tok) tok="(null)";
   tag=tos->tag;
   real=tos->real;
   imag=tos->imag;
   mat=tos->mat;
   tex=tos->tex;
   row=tos->row;
   col=tos->col;
   ptr=tos->ptr;
   cnt=tos->cnt;
   cntc=tos->cntc;
   nam=tos->nam;

   if(!nam || !*cntc) nam="(null)";

   switch(tos->typ) {

   case NUM:
      strcpy(stktyp,"NUM");
   break;

   case MAT:
      strcpy(stktyp,"MAT");
   break;

   case STR:
      strcpy(stktyp,"STR");
   break;

   case VOL:
      strcpy(stktyp,"VOL");
   break;

   case PTR:
      strcpy(stktyp,"PTR");
   break;

   default:
      stkerr(" props: ",TYPNOT);
      drop();
      return 0;
   break;
   }
   if(real==0 && imag==0) {
      gprintf(" %s: tok %s, tag %d, real 0, imag 0, mat %lX, tex %lX", \
         stktyp,tok,tag,mat,tex);
   }
   else {
      gprintf(
         " %s: tok %s, tag %d, real %G, imag %G, mat %lX, tex %lX", \
         stktyp,tok,tag,real,imag,mat,tex);
   }
   nc();
   gprintf("      row %d, col %d, ptr %lX, cnt %d, cntc %d, nam %s",
      row,col,ptr,*cnt,*cntc,nam);
   nc();

   return(drop());
}

int pullfs() /* pullfs (x --- x y) */
/* Moving top of function stack to stack. */
{
   if(tosfun==stkfunction) {
      stkerr(" pullfs: ",FUNSTKEMT);
      return 0;
   }
   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++;
   memcpy(tos,tosfun,sizeof(stkitem));
   tosfun--;

   return 1;
}

int push(int typ, char *tok, int tag, double real, double *mat, \
   char *tex, int row, int col, unsigned long (**ptr)())
/* Making a stack item. */
{
   char *nam1,*tok1;

/* For this enumeration: enum stktyp {DEF=1,NUM,MAT,STR,VOL,PTR},
   typname below gives name: */
   char typname[28]={'U','N','K','\0',
                     'D','E','F','\0',
                     'N','U','M','\0',
                     'M','A','T','\0',
                     'S','T','R','\0',
                     'V','O','L','\0',
                     'P','T','R','\0'};

   if(tos==(stack+DEPSTK)) {
      stkerr(" ",ONTOFULL);
      return 0;
   }
   tos++; /* the very first push goes to *(stack+1), not *(stack+0) */

   tos->typ=typ;
   tos->tok=tok;
   tos->tag=tag;
   tos->real=real;
   tos->imag=0;
   tos->mat=mat;
   tos->tex=tex;
   tos->row=row;
   tos->col=col;
   tos->ptr=ptr;
   tos->nam=NAM;

   if(!CNT) {
      if((tos->cnt=(int *)malloc(sizeof(int)))==NULL) {
         stkerr(" push: ",MEMNOT);
         return 0;
      }
      *(tos->cnt)=0;
   }
   else tos->cnt=CNT;

   if(!CNTC) {
      if((tos->cntc=(int *)malloc(sizeof(int)))==NULL) {
         stkerr(" push: ",MEMNOT);
         return 0;
      }
      *(tos->cntc)=0;
   }
   else { 
      tos->tag=CTAG; /* pass along the catalog item's tag */
      tos->imag=CIMAG; /* pass along the imag NUM part of cat item */

      tos->cntc=CNTC;

      if(TRACE) {
         if(!tok) tok1="(null)"; 
         else tok1=tok;

         if(!NAM) nam1="(null)";
         else nam1=NAM;

         if(chout) nc();

         gprintf(
       " catalog item to stk: tok %s, typ %s, cnt %d, cntc %d, nam %s",
         tok1,typname+loclin(tos->typ,4),1+*tos->cnt,*tos->cntc,nam1);
         nc();
      }
   }
   (*(tos->cnt))++;
   return 1;
}

int pushd(double d) {
   return(push(NUM,NULL,NOTAG,d,NULL,NULL,0,0,NULL));
}

int pushdx(double dr, double di) {
   if((push(NUM,NULL,NOTAG,dr,NULL,NULL,0,0,NULL))) {
      tos->imag=di;
      set_complex(tos);
      return 1;
   }
   return 0;
}

int pushfs() /* pushfs (x y --- x) */
/* Moves top of stack item to top of function stack.  The item's stack
   count, cnt, is unchanged.  */
{
   if(tosfun==(stkfunction+DEPFUNSTK)) {
      stkerr(" pushfs: ",FUNSTKFUL);
      return 0;
   }
   if(tos==stack) {
      stkerr(" ",CANNOTPOP);
      return 0;
   }
   tosfun++; /* adding to function stack */
   memcpy(tosfun,tos,sizeof(stkitem));
   tos--; /* removing from stack, as drop() does */

   return 1;
}

int pushint(int i) {
   return(push(NUM,NULL,NOTAG,(double)i,NULL,NULL,0,0,NULL));
}

int pushmat(char *name, double *mat, int rows, int cols) {
   return(push(MAT,name,NOTAG,0,mat,NULL,rows,cols,NULL));
}

int pushptr(unsigned long (**ptr)(), int len, char *tok) {
/* Pushes an array of len pointers to the stack, with name tok.
   Warning: ptr and tok must be mallocked memory. */
   return(push(PTR,tok,NOTAG,0,NULL,NULL,len,1,ptr));
}

int pushptr1(unsigned long (**ptr)()) {
/* Pushes single pointer to the stack with cntc=1, so it appears the 
   item is in the catalog so that ptr is not freed when popped from 
   the stack.
   ONLY FOR INTERNAL USE DURING ASSEMBLY OF AN INLINE (exe.c). */

   static int cnt=0,cntc=1;
   int ret=0;
   char *tok="jmp addr";

   CNT=&cnt;
   CNTC=&cntc;

   ret=push(PTR,tok,NOTAG,0,NULL,NULL,1,1,ptr);

   CNT=NULL;
   CNTC=NULL;

   return(ret);
}

int pushq(char *quote, int len) {
/* Pushes string to stack.
   Warning: quote must be mallocked memory; see pushq2. */
   return(push(STR,NULL,NOTAG,0,NULL,quote,1,len,NULL));
}

int pushq1(char *token)
/* Pushes a line of text in quotes to the stack without the quotes. */
{  int len;
   memmove(token,token+1,(len=strlen(token)-2));
   *(token+len)='\0';
   return(pushq(token,len)); /* send mallocked token, no offset. */
}

int pushq2(char *quote, int len)
/* Moves string to mallocked memory and then pushes it to the stack. */
{  char *str;
   if((str=(char *)memgetn(quote,len))!=NULL) {
      return(push(STR,NULL,NOTAG,0,NULL,str,1,len,NULL));
   } return 0;
}

int pushstr(char *quote)
/* Same as pushq2, except doesn't need a string length */
{  char *str;
   int   len;
   len = strlen(quote);
   if ((str = (char *) memgetn(quote, len)) != NULL) {
      return(push(STR, NULL, NOTAG, 0, NULL, str, 1, len, NULL));
   } return 0;
}

int pushq3() /* "..." or '...' ( --- qS) */
/* Pushes to stack the string in mallocked tokenq from tokenget(). */
{
   return(pushq1(tokenq));
}

int pushtex(char *name, char *text, int lines, int width) {
   return(push(VOL,name,NOTAG,0,NULL,text,lines,width,NULL));
}

int pushtex1(char *token)
/* Pushes a body of text enclosed by {" ... "} to the stack.
   Caution: pointer token will be invalid after this function--it is
   freed in textget (thus incoming token must be a mallocked pointer).
*/
{  int len;

   memmove(token,token+2,(len=strlen(token)-4));
   *(token+len)='\0'; /* terminator expected by textget */

   return(
      pushq(token,len+1) &&
      textget(1)
   );
}

int pushtex3() /* {"..."} ( --- hT) */
/* Pushes to stack the text in mallocked tokenq from tokenget(). */
{
   return(pushtex1(tokenq));
}

int pushuint(unsigned int u) {
   return(push(NUM,NULL,NOTAG,(double)u,NULL,NULL,0,0,NULL));
}

int rev() /* rev (u v w --- w u v) */
{
   stkitem temp;

   if(_depth<3) {
      stkerr(" rev: ",NEEDTHREE);
      return 0;
   }
   temp=*tos;
   *(tos)=*(tos-1);
   *(tos-1)=*(tos-2);
   *(tos-2)=temp;
   return 1;
}

int revn() /* revn (a b c d e 4 --- a e d c b) */
/* Reversing the order of the topmost n stack items. */
{
   int n;
   stkitem *A;
   register int k=0;

   if(!popint(&n)) return 0;

   if(_depth<n) {
      gprintf(" revn: require %d items on the stack",n);
      nc();
      stkerr("","");
      return 0;
   }
   if((A=malloc(1+n*sizeof(stkitem)))==NULL) {
      stkerr(" revn: ",MEMNOT);
      return 0;
   }
   for(;k<n;k++) *(A+k)=*(tos-k);
   for(k=0;k<n;k++) *(tos-k)=*(A+n-1-k);

   mallfree((void *)&A);
   return 1;
}

int roll() /* roll (u v w 2 --- v w u) */
{
   stkitem *temp;
   register int offset;

   if((temp=peek())==NULL) {
      stkerr(" roll: ",NEEDOFF);
      return 0;
   }
   else {
      if(temp->typ!=NUM) {
         stkerr(" roll: ",NEEDOFF);
         return 0;
      }
   }
   offset=(int)pop()->real;
   if(offset>=_depth || offset<0) {
      stkerr(" roll: ",OFFBEYOND);
      return 0;
   }
   *temp=*(tos-offset);
   memcpy((tos-offset),(tos-offset+1),offset*sizeof(stkitem));
   *tos=*temp;
   return 1;
}

int rot() /* rot (u v w --- v w u) */
{
   stkitem temp;

   if(_depth<3) {
      stkerr(" rot: ",NEEDTHREE);
      return 0;
   }
   temp=*(tos-2);
   *(tos-2)=*(tos-1);
   *(tos-1)=*(tos);
   *tos=temp;
   return 1;
}

int rows() /* rows (hA --- r) */
{
   if(peek()!=NULL) {
      if (!is_sparse(tos)) { /* not sparse */
          if(is_complex(tos)) pushint((tos->row)/2);
          else pushint(tos->row);
          return(lop());
      } else {
          return(sprows());
      }
   }
   stkerr(" ",CANNOTPOP);
   return 0;
}

int sizeof_item() /* sizeof (hA --- nBytes) */
/* Allocated bytes for stack item A.  Bytes do not include overhead
   of 32 to 40 bytes per stack item. */
{
   if(stkdepth()<1) {
      stkerr(" sizeof: ",STKNOT);
      return 0;
   }
   switch(tos->typ) {

      case MAT:
         pushd((tos->row)*(tos->col)*sizeof(double));
         return(lop());

      case NUM:
         pushd(sizeof(double));
         return(lop());

      case PTR:
         pushd((tos->row)*sizeof(unsigned long));
         return(lop());

      case STR:
      case VOL:
         pushd((tos->row)*(tos->col));
         return(lop());
   }
   stkerr(" sizeof: ",NOTSUPT);
   return 0;
}

int spcols() /* spcols (hA --- r) */
/* Columns of sparse matrix on stack. */
{
   return(
      spdims() && lop()
   );
}

int spdims() /* spdims (hA --- r c) */
/* Rows and columns of sparse matrix on stack. */
{
    int cols,rows;
    SparseMatrix       m;

    if (!is_sparse(tos)) {
        stkerr(" spdims: ",SPARSENOT);
        return 0;
    }
    m     = sparse_overlay(tos);

/*  Getting values before drop() below: */
    cols=m.H[COLS]; 
    rows=m.H[ROWS];

    return(
      drop() &&
      pushint(rows) &&
      pushint(cols)
    );
}

int sprows() /* sprows (hA --- r) */
/* Rows of sparse matrix on stack. */
{
   return(
      spdims() && drop()
   );
}

int stkdepth() /* returns integer equal to stack depth */
{
   return(_depth);
}

void stkerr(char *s1, char *s2)
{
   char s[129]={0};
   if(strlen(s1)) {
      strncat(s,s1,128); 
      strncat(s,s2,128); 
      gprintf("%s",s); nc();
   }
   stkerrcount++;
   stkerrabs++;
   _depthMIN=0;

   reperr();
   return;
}

void stkinit() {

   if(TRACE) {
      gprintf(" initializing stk"); nc();
   }
   stkerrcount=0;
   stkerrabs=0;

   tos=stack;

/* Sat Jul 13 15:28:08 PDT 2013.  Initial allocations for ints cnt and
   cntc at the beginning of the stack: */
   if((tos->cnt=(int *)malloc(sizeof(int)))==NULL) {
      stkerr(" stkinit: ",MEMNOT);
      return;
   }
   *(tos->cnt)=0;

   if((tos->cntc=(int *)malloc(sizeof(int)))==NULL) {
      stkerr(" stkinit: ",MEMNOT);
      return;
   }
   *(tos->cntc)=0;

/* Sat Jul 13 15:28:08 PDT 2013.  Without the initial allocations using
   malloc() that are added above, the program may segfault.  Malloc() 
   operations like the ones added above already properly continue in 
   push() every time a new item is pushed to the stack.  But the ini-
   tial unallocated items have always been here at the beginning of the
   stack.  

   After 13 years of continuous use, it took clearing the stack with 
   function xx() while testing new function plop() to make the error
   appear. 

   Fixing this old stkinit() bug made things all right for a while 
   until running new plop() caused another problem, and a bug was 
   found in plop().

   With the bug in plop() fixed, the program works again.  

   But in fact, now that plop() works properly the changes made here
   to fix the old program bug could be removed and the program would 
   work fine, perhaps for another decade.

   In other words, if plop() had been written without a bug in the
   first place, it and the program would have worked and the old bug 
   here would not have been uncovered.

   If there can be bugs that are hidden (maybe dormant is a better word
   for a bug) until new features are added that also have bugs, when is
   a program ever free of bugs? 

   When you see someone claiming a program is fully debugged, you are
   looking at a fool or a liar.  Developers of software for pilotless
   aircraft and cars without drivers certainly have their work cut out
   for them.

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Sat Jul 13 12:48:03 PDT 2013
      [tops@kaffia] ready > 3

       stack elements:
             0 number: 3
       [1] ok!
      [tops@kaffia] ready > 1 2 plop

       stack elements:
             0 number: 2
             1 number: 1
             2 number: 1
             3 number: 3
       [4] ok!
      [tops@kaffia] ready > xx

      [tops@kaffia] ready > 1 2 plop
      Segmentation fault (core dumped)
      [dale@kaffia] /opt/tops/tops/src > 

   Excerpts from debugger gdb show failure in push() (see #3 below) be-
   cause the allocations had not been not done here in stkinit() when
   it is run at program start up:

      [dale@kaffia] /opt/tops/tops/src > gdb tops core.4083 
      GNU gdb Red Hat Linux (5.1.90CVS-5)
      Copyright 2002 Free Software Foundation, Inc.
      GDB is free software, covered by the GNU General Public License,\
         and you are welcome to change it and/or distribute copies of \
         it under certain conditions.
      Type "show copying" to see the conditions.
      There is absolutely no warranty for GDB.  Type "show warranty" \
         for details.
      This GDB was configured as "i386-redhat-linux"...
      Core was generated by `tops'.
      Program terminated with signal 11, Segmentation fault.

      ...

      (gdb) where
      #0  0x4207a938 in chunk_alloc () from /lib/i686/libc.so.6
      #1  0x4207cc8a in malloc_check () from /lib/i686/libc.so.6
      #2  0x42079fbd in malloc () from /lib/i686/libc.so.6
      #3  0x080cf653 in push (typ=4, tok=0x0, tag=0, real=0, mat=0x0, \
         tex=0x8d4a158 "_noblanklines", row=1, col=13, ptr=0x0) at \
         stk.c:1049
      #4  0x080d1309 in pushq2 (quote=0x8c195a0 "_noblanklines", \
         len=13) at stk.c:1174
      #5  0x080e110e in strchop () at tex.c:2384
      #6  0x080d0921 in naming () at stk.c:644
      #7  0x080dfca8 in noblanklines () at tex.c:1613
      #8  0x08091da5 in run () at exe.c:2057
      #9  0x080901b3 in exeinline () at exe.c:758
      #10 0x08092df3 in catexe (word=0x8b91ce0 "console") at exe.c:168
      #11 0x080a7d6d in perform () at main.c:81
      #12 0x080a8022 in source () at main.c:186
      #13 0x080a891b in main (argc=1, argv=0xbffff7e4) at main.c:42
      #14 0x42017499 in __libc_start_main () from /lib/i686/libc.so.6
      (gdb) q
      [dale@kaffia] /opt/tops/tops/src > tops
*/
   tosloc=stklocal;
   tosfun=stkfunction;
   _depthMIN=0;

   CNT=NULL;
   CNTC=NULL;
   CTAG=0;
   CIMAG=0;
   NAM=NULL;

   return;
}

int stkset(char *message)
/* Returns typ if both stack items are the same, or else displays
   message and returns 0.  Not used much any more. */
{
   int typ;
   if(stkdepth()<2) { stkerr(message,NEEDTWO); return 0; }
   if((typ=(tos->typ))==((tos-1)->typ)) return typ;
   if(typ==MAT) { stkerr(message,MATSNOT); return 0; }
   if(typ==NUM) { stkerr(message,NUMSNOT); return 0; }
   if(typ==STR) { stkerr(message,STRSNOT); return 0; }
   if(typ==VOL) { stkerr(message,VOLSNOT); return 0; }
   stkerr(message,STKNOT); return 0;
}

int swap() /* swap (u v --- v u) */
{
   stkitem temp;

   if(_depth<2) {
      stkerr(" swap: ",NEEDTWO);
      return 0;
   }
   else {
      temp=*tos;
      *tos=*(tos-1);
      *(tos-1)=temp;
      return 1;
   }
}

int tdepth() /* tdepth ( --- n) */
/* Pushes depth of local stack to the stack, returns 1 if no error. */
{
   return(pushint(_ldepth));
}

void tocmplx(int rows, int cols, double *Ar, double *Ai, double *C)
/* Separate real and imaginary matrices combined into one complex.

   C has twice as many rows as Ar and Ai.

   The even elements in C hold the real terms, the odd elements hold
   the imaginary terms (0-based indexing). */
{
   register int k=0;

   for(;k<rows*cols;k++) {

      *C=*Ar;
      Ar++;
      C++;

      *C=*Ai;
      Ai++;
      C++;
   }
}

void todblx(int rows, int cols, double *C, double *Ar, double *Ai)
/* Complex matrix split into separate matrices real and imaginary.
  
   The rows in Ar and Ai equal half the rows in C. */
{
   register int k=0;

   for(;k<rows*cols;k++) {

      *Ar=*C;
      Ar++;
      C++;

      *Ai=*C;
      Ai++;
      C++;
   }
}

int tok() /* tok (hA --- qS) */
/* Stack item tok name to stack. */
{
   char *tok;

   if(_depth<1) {
      stkerr(" tok: ",EMPTYSTK);
      return 0;
   }
   tok=tos->tok;

   if(tok) pushq2(tok,strlen(tok));
   else pushq2("",0); /* an empty string to stack */

   return(lop());
}

void tokfree() /* (hA --- hA) */
/* Freeing memory of tos.tok if not anywhere else. */
{
   int k,len,nomatch=1;

   if(*(tos->cntc)) return; /* can't free if item is in catalog */

   if(tos->tok) {
/*    tos.cnt gives the total instances of the tos item on the program
      stack and on the local stack and on the function stack. */

      if(*tos->cnt>1) { /* looking for other instances of A's tos.tok */
         len=_depth;
         k=1;
         while(k<len && nomatch) { /* Look on the program stack */
            nomatch=(!(tos->tok==(tos-k)->tok));
            k++;
         }
         if(nomatch) { /* Look on the local (temp) stack */
            k=0;
            len=_ldepth;
            while(k<len && nomatch) {
               nomatch=(!(tos->tok==(tosloc-k)->tok));
               k++;
            }
         }
         if(nomatch) { /* Look on the function stack */
            k=0;
            len=_fdepth;
            while(k<len && nomatch) {
               nomatch=(!(tos->tok==(tosfun-k)->tok));
               k++;
            }
         }
      /* If no other instances, can free tos.tok */
         if(nomatch) {
            mallfree((void *)&tos->tok);
            tos->tok=0;
         }
      }
      else {
         mallfree((void *)&tos->tok);
         tos->tok=0;
      }
   }
}

int tos1() /* tos (... --- ...) */
/* This word simply allows an infix phrase like A=tos() to work. */
{
   if(!stkdepth()) {
      stkerr(" tos: ",EMPTYSTK);
      return 0;
   }
   return 1;
}
