/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of spsolve.c: Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */
#if defined(UMFPACK)
/* headers {{{1 */
#include <stdio.h>
#include <stdlib.h>        /* free    */
#include <string.h>        /* strncmp */
#include <math.h>          /* fabs    */
#include <umfpack.h>

#include "stk.h"
#include "main.h"
#include "exe.h"           /* xmain */
#include "sparse.h"
#include "tag.h"
#include "spsolve.h"

#include "ctrl.h"
#include "inpo.h"
#include "mem.h"
/* headers 1}}} */

/* Mods to UMFPack source code {{{1
UMFPACKv4.3/UMFPACK/Source/umfpack_load_numeric.c 
    cp GLOBAL Int UMFPACK_save_numeric -> GLOBAL Int UMFPACK_load_numeric_mem

UMFPACKv4.3/UMFPACK/Source/umfpack_save_numeric.c
    cp GLOBAL Int UMFPACK_load_numeric -> GLOBAL Int UMFPACK_save_numeric_mem
    added  GLOBAL Int UMFPACK_numeric_bytes
    added  #define WRITEMEM

UMFPACKv4.3/UMFPACK/Source/umf_version.h
    added (needs to go in four places, see DINT, DLONG, ZINT, ZLONG)
    #define UMFPACK_save_numeric_mem umfpack_di_save_numeric_mem
    #define UMFPACK_load_numeric_mem umfpack_di_load_numeric_mem
    #define UMFPACK_numeric_bytes    umfpack_di_numeric_bytes
1}}} */

/* words */
/* int  splu          splu    ( hA_sp --- hA_sp hLU_sp ) {{{1 */
int  splu()      
/* 
 * man entry:  splu {{{2
 * (hA_sp ---  hA_sp hLU_sp) Factor the sparse matrix [A] into [L] and [U] triangular factors using UMFPACK.  [L] and [U] are stored in the same matrix, [LU].  Use spfbs to do forward/backward substitution with [LU] to solve a system of equations.  spdiag can be used to extract the diagonal of [U] (useful for computing a determinant or the Sturm count).  [A] is kept on the stack because spfbs will need it.
  10 10 .2 sprand  10 speye +  # [A]  (add [I] to guarantee full rank)
  splu
  10 2 random                  # [B]
  spfbs
 * category: math::matrix::sparse
 * related: sparse, spadd, spmult, solve, spfbs, speig_near_shift, spdiag
 * 2}}}
 */ 
{
int DEBUG = 0;

    SparseMatrix A;
    double *null = (double *) NULL ;
    void   *Symbolic,  *Numeric ;
    int     r, c, s, a_ptr, end_row, A_is_indexed,
           *A_rowind, *A_colptr;

    double  Control [UMFPACK_CONTROL];  /* used only when DEBUG != 0 */
/*
char   *nname = "num.bin";
char   *sname = "sym.bin";
*/

if (DEBUG) gprintf("top of splu\n");
    if (!is_sparse(tos)) {
        stkerr(" splu: [A] ", SPARSENOT); 
        return 0;
    }

    A = sparse_overlay(tos);
    A_is_indexed = is_indexed(tos);

    if (A.H[ROWS] != A.H[COLS]) {
        stkerr(" splu: [A] is not square ", ROWSNOT); 
        return 0;
    }

    if ((A_rowind = (int *)   malloc( A.H[n_NONZ] * sizeof(int)))   == NULL) {
        stkerr(" splu:  A_rowind ", MEMNOT);
        return 0;
    }
    if ((A_colptr = (int *)   malloc((A.H[COLS]+1)* sizeof(int)))   == NULL) {
        stkerr(" splu:  A_colptr ", MEMNOT);
        free(A_rowind);
        return 0;
    }

    /* Copy A's indexing. */
    a_ptr = 0;
    for (c = 0; c < A.H[COLS]; c++) {
        A_colptr[c] = a_ptr;
        for (s = A.S_start[c]; s < A.S_start[c+1]; s++) {
            end_row = A.S[s].start_row + A.S[s].len - 1;
            for (r = A.S[s].start_row; r <= end_row; r++) {
                A_rowind[a_ptr++] = r;
            }
        }
    }
    A_colptr[c] = a_ptr;

    (void) umfpack_di_symbolic (A.H[ROWS], A.H[COLS], A_colptr, A_rowind, A.N, 
                                &Symbolic, null, null) ;
    (void) umfpack_di_numeric  (A_colptr, A_rowind, A.N, Symbolic, 
                                &Numeric,  null, null);
    free(A_rowind);
    free(A_colptr);

if (DEBUG) {
Control [UMFPACK_PRL] = 6 ;
printf ("\nNumeric factorization of A: ") ;
(void) umfpack_di_report_numeric (Numeric, Control);
}
    /*
    error = umfpack_di_save_numeric( Numeric,  nname);
    if (error) {
        snprintf(T, size_T, "num save to %s error %d ", nname, error);
        stkerr(" splu:  ", T);
        return 0;
    }
    */

    /*
    error = umfpack_di_save_symbolic(Symbolic, sname);
    if (error) {
        snprintf(T, size_T, " sym save to %s error %d ", sname2, error);
        stkerr(" splu:  ", T);
        return 0;
    }
    */

    umfpack_di_free_symbolic (&Symbolic);

    return umfpack_numeric_to_tos(Numeric, A);
} /* 1}}} */
/* int  spfbs         spfbs   ( hA hLU_sp hB --- hA hLU_sp hX ) {{{1 */
int  spfbs()      
/* 
 * man entry:  spfbs {{{2
 * (hA hLU_sp hB --- hA hLU_sp hX ) Given a right hand side matrix [B] and the LU decomposition of a sparse matrix [A], performs forward- and backward substitution on the triangular factors in [LU] to return the solution [X] to [A][X] = [B].  [B], [A], and [LU] remain on the stack to simplify getting multiple solutions for the same input matrix (as of v4.3, UMFPACK only handles single column right hand sides).
  10 10 .2 sprand  10 speye + is A A # add [I] to guarantee full rank
  splu                     ( A LU ) 
  10 1 random is B B       ( A LU B )        
  spfbs                    ( A LU X )
  lop                      ( A X )
  star B - abs maxfetch    ( Ax-B )
Internal indexing:  If both [A] and [B] are internally indexed then they do not need to be size compatible--if necessary [B] is reordered and/or expanded so that its row indices align with [A]'s column indices.  If [A] is indexed, its column indices are copied to [X]'s row indices.  If [B] is indexed, its column indices are copied to [X]'s column indices.  An error is reported if any of [B]'s row indices are not in [A]'s column indices.
 * category: math::matrix::sparse
 * related: sparse, spadd, spmult, solve, spfbs
 * 2}}}
 */ 
{
int DEBUG = 0;
    SparseMatrix A;
    char    *name = "_X";
    int     i, j, r, c, s, error, a_ptr, end_row, B_nrows, B_ncols,
            found_index, A_indexed, B_indexed,
            B_cmplx = 0, indices_align = 1, B_NPT = 1,
           *A_rowind, *A_colptr, 
           *Br_idx = 0, *Bc_idx = 0, *Xr_idx = 0, *Xc_idx = 0;
    double *null = (double *) NULL ;
    double *B, *X, *B_compatible;
    void   *Numeric ;
    double Control [UMFPACK_CONTROL];
#define size_T 160
    char    T[size_T+1];
/*
char   *nname = "num.bin";
*/

if (DEBUG) gprintf("top of spfbs\n");

    if (is_sparse(tos)) {
        stkerr(" spfbs: [B] ", MATNOT); 
        return 0;
    }
    if (!is_factor_lu(tos-1) || !is_umfpack(tos-1)) {
        stkerr(" spfbs: [LU] ", "not an UMFPACK factor"); 
        return 0;
    }
    if (!is_sparse(tos-2)) {
        stkerr(" spfbs: [A] ", SPARSENOT); 
        return 0;
    }
    A = sparse_overlay(tos-2);
    if (A.H[ROWS] != A.H[COLS]) {
        stkerr(" spfbs: [A] is not square ", ROWSNOT); 
        return 0;
    }
    A_indexed = is_indexed(tos-2);

    if (is_complex(tos))
        B_cmplx = 1;
    B_nrows   = tos->row;
    B_ncols   = tos->col;
    if (B_cmplx) {
        B_nrows /= 2;
        B_NPT    = 2;
    }
    B_indexed = is_indexed(tos);
    if (B_indexed) {
        Br_idx = MAT_ROW_IDX(tos);
        Bc_idx = MAT_COL_IDX(tos);
    }
    B = tos->mat;
    B_compatible = B; /* if B isn't compatible, create B_compatible later */

    if (A_indexed && B_indexed) { /* {{{2 */
        /* The hardest case; have to see if indices are compatible.
         * If they aren't have to make a compatible B vector which
         * means a lot of index traversal.
         */
        if (B_nrows != A.H[COLS]) { /* different sizes--can't possibly align */
            indices_align = 0;
        } else {
            for (r = 0; r < B_nrows; r++) {
                if (Br_idx[r] != A.col_idx[r]) {
                    indices_align = 0;  /* dang! extra effort needed */
                    break;
                }
            }
        }

        if (!indices_align) {  
            /* Bummer, have to create a right hand side vector which is 
             * compatible with [A].  This is a lot of work, O(N^2), because
             * have to make an index map showing where [B]'s indices line
             * up with [A]'s.
             */
            if ((B_compatible = (double *) 
                                malloc(B_nrows*B_ncols*B_NPT * sizeof(double))) 
                    == NULL) {
                stkerr(" spfbs:  B_compatible ", MEMNOT);
                return 0;
            }
            for (i = 0; i < B_nrows*B_ncols*B_NPT; i++) B_compatible[i] = 0.0;
            for (r = 0; r < B_nrows; r++) {
                found_index = 0;
                for (j = 0; j < A.H[COLS]; j++) {
                    if (Br_idx[r] == A.col_idx[j]) {
                        /* copy B's terms for this row to the working RHS */
                        for (c = 0; c < B_ncols; c++) {
if (DEBUG) {
gprintf("B_compatible[%2d] = B[%2d] ( = % 12.5e)\n",
j + c*B_nrows*B_NPT, r + c*B_nrows*B_NPT, B[r + c*B_nrows*B_NPT]);
}
                            B_compatible[j + c*B_nrows*B_NPT] = 
                                       B[r + c*B_nrows*B_NPT];
                            if (B_cmplx) {
                                B_compatible[j + c*B_nrows*B_NPT + 1] = 
                                           B[r + c*B_nrows*B_NPT + 1];
                            }
                        }
                        found_index = 1;
                        break;
                    }
                }
                if (!found_index) {
                    snprintf(T, size_T, 
                        "internal row index B[%d]=%d does not appear in [A]", 
                        r+XBASE, Br_idx[r]);
                    stkerr(" spfbs: ", T);
                    return 0;
                }
            }
        }
    } /* 2}}} */

    /* create compressed column index vectors to pass to UMFPACK {{{2 */
    if ((A_rowind = (int *)   malloc( A.H[n_NONZ] * sizeof(int)))   == NULL) {
        stkerr(" spfbs:  A_rowind ", MEMNOT);
        return 0;
    }
    if ((A_colptr = (int *)   malloc((A.H[COLS]+1)* sizeof(int)))   == NULL) {
        stkerr(" spfbs:  A_colptr ", MEMNOT);
        free(A_rowind);
        return 0;
    }
    a_ptr = 0;
    for (c = 0; c < A.H[COLS]; c++) {
        A_colptr[c] = a_ptr;
        for (s = A.S_start[c]; s < A.S_start[c+1]; s++) {
            end_row = A.S[s].start_row + A.S[s].len - 1;
            for (r = A.S[s].start_row; r <= end_row; r++) {
                A_rowind[a_ptr++] = r;
            }
        }
    }
    A_colptr[c] = a_ptr;
    /* 2}}} */

    /* bring LU to tos for UMFPACK conversion */
    swap();   /* ( hA hLU hB ---  hA hB hLU ) */

    /*
    error = umfpack_di_load_numeric( &Numeric,  nname);
    if (error) {
        snprintf(T, size_T, "umfpack_di_load_numeric load error %d ", error);
        stkerr(" spfbs:  ", T);
        return 0;
    }
    */

    if (!is_factor_lu(tos) || !is_umfpack(tos)) {
        stkerr(" spfbs:  ", "[LU] is not a UMFPACK factor");
        return 0;
    }
    
    error = umfpack_numeric_from_tos(&Numeric);
    if (error < 0) {
        snprintf(T, size_T, " umfpack_numeric_from_tos error %d ", error);
        stkerr(" spfbs:  ", T);
        return 0;
    }

if (DEBUG) {
Control [UMFPACK_PRL] = 6 ;
printf ("\nNumeric in fbs: ") ;
(void) umfpack_di_report_numeric (Numeric, Control);
}

    if (A_indexed || B_indexed) {
        if (!matstk_idx(B_nrows, B_ncols, name)) return 0;
    } else {
        /* don't bother adding internal indices to X */
        if (!matstk(    B_nrows, B_ncols, name)) return 0;
    }
    X = tos->mat;   /* ( hA hB hLU hX ) */

    if (A_indexed || B_indexed) {
        Xc_idx = MAT_COL_IDX(tos);
        Xr_idx = MAT_ROW_IDX(tos);
        set_indexed(   tos);
        set_index_used(tos);
        if        (A_indexed) {
            memcpy(Xr_idx, A.col_idx, A.H[COLS] * sizeof(int) );
            /* if [B] isn't indexed but [A] is, [X]'s column indices are 1..n */
            for (c=0; c<B_ncols; c++) Xc_idx[c] = c + XBASE;
        }

        /* [X] gets [B]'s column indexing regardless of [A]'s index status */
        if (B_indexed) {
            memcpy(Xc_idx, Bc_idx    , B_ncols * sizeof(int) );
        }
    }

    /* Until UMFPACK can do FBS with right hand sides having more than
     * one column, have to loop over the columns one at a time
     */
    for (c = 0; c < B_ncols; c++) {
        (void) umfpack_di_solve(UMFPACK_A, A_colptr, A_rowind, A.N, 
                               &X[c*B_nrows], 
                               &B_compatible[c*B_nrows], 
                                Numeric, null, null);
    }
    umfpack_di_free_numeric (&Numeric) ;

    rot(); drop();   /* ( hA hLU hX ) */

    free(A_rowind);
    free(A_colptr);
    if (!indices_align) free(B_compatible);

    return 1;

}
/* 1}}} */

int  umfpack_numeric_to_tos   (void        *NumericHandle, /* {{{1 */
                               SparseMatrix A)
{
  /* Take a UMFPACK numeric factor and the input matrix it was
   * computed from and pack the factor into a tops VOL on tos.
   */ 
int DEBUG = 0;
    char *name  = "_LU_num";
    int   nBytes_Numeric, error;
#define size_T 160
    char    T[size_T+1];
double Control [UMFPACK_CONTROL];

    /*
    NumericType *Numeric ;
    Numeric = (NumericType *) NumericHandle ;
    */
    void *Numeric ;
    Numeric = (void *) NumericHandle ;

if (DEBUG) {
umfpack_di_defaults (Control) ;
Control [UMFPACK_PRL] = 6 ;
}
    nBytes_Numeric  = umfpack_di_numeric_bytes( Numeric );

    if (!volstk(1, nBytes_Numeric, name)) return 0;
    error = umfpack_di_save_numeric_mem( Numeric,  tos->tex);
    /*
     * A bit of a kludge:  haven't figured out how to determine
     * the number rows or cols from only the Numeric entry, so am
     * storing this integer value in the space meant for the real
     * value of a scalar.  This is needed for the spdiag fuction
     * if given a sparse matrix factor.
     */
    tos->real = (double) A.H[COLS];
if (DEBUG) gprintf("umfpack_numeric_to_tos error=%d\n", error);
    if (error) {
        snprintf(T, size_T, " error %d ", error);
        stkerr(" umfpack_numeric_to_tos:  ", T);
        return 0;
    }

    set_umfpack(  tos);
    set_factor_lu(tos);

    return 1;
}
/* 1}}} */
int  umfpack_numeric_from_tos (void ** NumericHandle ) /* {{{1 */
{
  /* Take the item at top of stack and create from it a UMFPACK
   * Numeric data variable.
   * Based on UMFPACK's umfpack_load_numeric.c 
   */
int DEBUG = 0;
    /*
    NumericType *Numeric ;
    */
    void *Numeric ;
#define size_T 160
    char  T[size_T+1];
    int   error;

    if (!is_umfpack(tos) || !is_factor_lu(tos) || is_symbolic(tos)) {
        return -1;
    }

    *NumericHandle = (void *) NULL ;

    error = umfpack_di_load_numeric_mem(&Numeric,  tos->tex);
if (DEBUG) gprintf("umfpack_numeric_from_tos error=%d\n", error);
    if (error) {
        snprintf(T, size_T, " error %d ", error);
        stkerr(" umfpack_numeric_from_tos:  ", T);
        return 0;
    }

    *NumericHandle = (void *) Numeric ;

    return 1;
}
/* 1}}} */
#endif
