/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
}}}1 */

/* 
   sparse.h
   Albert Danial  (pre-cvs: Oct 2 2001; Feb 23 2002)
 */

/* The items of this enum comprise the sparse matrix header. */
enum { ROWS              = 0,

       COLS                 ,

       n_STR                ,

       n_NONZ               ,   /* number of non-zero *terms*; a complex *
                                 * value is a single term                */
       COMPLX               ,   /* 0 => real          1 => complex       */

       DATA_SIZE                /* in bytes; total length of .data[]        */
                                /* This should always stay as the last term */
                                /* of the header since the constant         */
                                /* SPARSE_HDR_SIZE is defined from it.      */
};

/* The sparse magic numbers came from:
           perl -e 'for (1..4) {printf "%d\n", rand(1000000000)}'
 */
#define BPI 4   /* bytes per int    word ( = 8 on IA64) */
#define BPD 8   /* bytes per double word                */
#define SPARSE_MAGIC_SIZE           4
#define SPARSE_MAGIC_0       97038593
#define SPARSE_MAGIC_1      338847044
#define SPARSE_MAGIC_2      905434576
#define SPARSE_MAGIC_3      450194058
#define SPARSE_HDR_SIZE     DATA_SIZE + 1
#define SPARSE_START        BPI*SPARSE_MAGIC_SIZE
#define SPARSE_ZERO_THRESH    1.0e-16
#define NUM_PER_TERM(a)       ((a) == (1)  ? (2)  : (1))
#ifdef   BLAS
#ifdef FORT_UDSC
void   daxpy_();
void   dcopy_();
void   dscal_();
double ddot_ ();
#else
void   daxpy();
void   dcopy();
void   dscal();
double ddot();
#endif
#endif

typedef struct {
    int     len       ; /* Number of terms in the string (a complex       */
                        /* number counts as a single term).               */
    int     start_row ; /* Zero based, so first row has start_row = 0.    */
    int     N_idx     ; /* Index into N[] to first numeric term for this  */
                        /* string.                                        */
} str_t;

typedef struct {
    int     *magic    ;
    int     *H        ;         /* size = SPARSE_HDR_SIZE  */
    int     *S_start  ;         /* size = # columns + 1    */
    int     *N_start  ;         /* size = # columns + 1    */
    str_t   *S        ;         /* size = n_strings_total  */
    int     *row_idx  ;         /* size = 0 | # rows       */
    int     *col_idx  ;         /* size = 0 | # columns    */
    double  *N        ;         /* size = n_nonzeros_total */

    char    *data     ;         /* main data block         */

} SparseMatrix;

typedef struct {
    int      row      ;         /* row index    (0-based) */
    int      col      ;         /* column index (0-based) */
    double   Re       ;         /* real part              */
    double   Im       ;         /* imaginary part (used only for complex vals)*/
} ijv;

typedef struct {
    int      nRows    ;
    int      nCols    ;
    int      cmplx    ;         /* 0=Real    1=Complex      */
    int      nTerms   ;         /* length of data (d) array */
    int      indexed  ;         /* 0=no, 1=yes, internally indexed */
    int     *row_idx  ;         /* size = 0 | # rows        */
    int     *col_idx  ;         /* size = 0 | # columns     */
    ijv     *d        ;         /* pointer to array of i,j,vRe,vIm terms */
} IJVMatrix;

SparseMatrix sparse_overlay(stkitem *X);
int  sparse();      /* sparse      (hA --- hA_sp) */
int  dense();       /* dense       (hA_sp --- hA) */
int  spbend();      /* spbend      (hA_sp --- hA_sp') */
int  sprand();      /* sprand      (r c rho --- hA_sp) */
int  sperror();     /* sperror     (hA_sp --- hA_sp e ) */
int  spdump();      /* spdump      (hA_sp --- hA_sp) */
int  speye();       /* speye       (n --- hI_sp) */
int  spdiag();      /* spdiag      (hA_sp --- hD) */
int  spy();         /* spdiag      (hA_sp --- ) */
int  spnull();      /* spnull      (r c --- hN_sp) */
int  spadd();       /* spadd       (hA_sp hB_sp --- hC_sp) */
int  spadd_rr();
int  spadd_rc();
int  spadd_cc();
int  spadd_cx();
int  spmult();      /* spmult      (hA_sp hB --- hC) */
int  spmult_sd_rr();
int  spmult_sd_rc();
int  spmult_sd_cr();
int  spmult_sd_cc();
int  spmult_sym_low_sd_rr();
int  spmult_ss_rr();
int  spmult_ss_rc();
int  spmult_ss_cr();
int  spmult_ss_cc();
int  spscale();     /* spscale     (hA_sp f --- hAf_sp) */
int  spones();      /* spones      (hA_sp --- hO_sp) */
int  spreal_imag(); /* spreal_imag (hC_sp --- hRe_sp hIm_sp) */
int  spparkn();     /* spparkn     (hA1_sp ... hAn_sp n --- hA_sp) */
int  spconvert();   /* spconvert   (hIJV --- hA_sp)  */
int  sp2ijv();      /* sp2ijv      (hA_sp --- hIJV)  */
int  sppilen();     /* sppilen     (hA1_sp ... hAn_sp n --- hA_sp) */
int  spcatch();     /* spcatch     (hA_sp c | hA_sp hCols --- hB_sp) */
int  spreach();     /* spcatch     (hA_sp c | hA_sp hRows --- hB_sp) */
int  spcomb();      /* spcomb      (hA_sp c01 --- hA0_sp hA1_sp) */
int  sprake();      /* sprake      (hA_sp r01 --- hA0_sp hA1_sp) */
int  spmesh();      /* spmesh      (hA0_sp hA1_sp c01 --- hA_sp) */
int  spinfo();      /* spinfo      (hA_sp --- ) */
int  spsum();       /* spsum       (hA_sp hD hR hC lower --- hB_sp) */
int  spflip_sym();  /* spflip_sym  (hL_sp --- hLU_sp)               */
int  spinflate();   /* spinflate   (hA_sp Rind Cind R C --- hB_sp)  */
int  index_align(); /* index_align (hX hY --- hx_Map hy_Map)        */
int  index_merge(); /* index_merge (hX hY --- hy_Map hZ) */
int  spadd_ijv();   /* spadd_ijv   (hA hB --- hC)        */

void test_random_subset();
int  f_sp_error(stkitem *X);
int  random_subset(int  max_value,
                   int  N        ,
                   int *list);

int sparse_stk(int   nRows      , /* in  */ 
               int   nCols      , /* in  */
               int   nStr       , /* in  number of strings       */
               int   nNz_terms  , /* in  number of nonzero terms */
               int   cmplx      , /* in  0=real  1=complex       */
               int   indexed    , /* in  0=no    1=yes           */
               char *name       , /* in  */
               SparseMatrix *m);  /* out */
void sp_data_offsets(int  nRows       ,
                     int  nCols       ,
                     int  nStr        , /* in  number of strings       */
                     int  nNz         , /* in  number of nonzero terms */
                     int  cmplx       , /* in     0=real  1=complex    */
                     int  indexed     , /* in     0=no    1=yes        */
                     int *ptr_H       , /* out int offset to H[]       */
                     int *ptr_S_start , /* out int offset to S_start[] */
                     int *ptr_N_start , /* out int offset to S_start[] */
                     int *ptr_S       , /* out int offset to S[]       */
                     int *ptr_row_idx , /* out int offset to row_idx[] */
                     int *ptr_col_idx , /* out int offset to col_idx[] */
                     int *ptr_N       , /* out int offset to N[]       */
                     int *size);        /* out # bytes taken by matrix */

void sp_set_header(stkitem      *X,    /* in  pointer to a stack item  */
                   SparseMatrix *m,
                   int   ptr_H,        /* in  byte offset to H[]       */
                   int   ptr_S_start,  /* in  byte offset to S_start[] */
                   int   ptr_N_start,  /* in  byte offset to N_start[] */
                   int   ptr_S,        /* in  byte offset to S[]       */
                   int   ptr_row_idx,  /* in  int offset to row_idx[]  */
                   int   ptr_col_idx,  /* in  int offset to col_idx[]  */
                   int   ptr_N,        /* in  byte offset to N[]       */
                   int   size,         /* in  # bytes taken by matrix  */
                   int   nRols,        /* in */
                   int   nCols,        /* in */
                   int   cmplx,        /* in     0=real  1=complex */
                   int   nStr,         /* in */
                   int   nNz);         /* in */
void stringify_vector(int           nrows,  /* in     number of rows */
                      int           c,      /* in     column index */
                      double       *vector, /* in     the dense vector */
                      int           cmplx,  /* in     0=real  1=complex */
                      SparseMatrix *m,      /* in/out */
                      int          *iS,     /* in/out index to S */
                      int          *iN);    /* in/out index to N */
void f_sp_dump(const char *title, stkitem *X);
int  f_str_overlap(str_t A,              /* in                            */ 
                   str_t B,              /* in                            */
                   int   allow_adjacent, /* in adjoining strings overlap? */
                   int  *A_offset,       /* out Offset to A.start_row     */
                   int  *B_offset,       /* out Offset to B.start_row     */
                   int  *length  );      /* out Length of overlap.        */
int  f_col_overlap(int    n_str_A   ,    /* in  number of strings in {A}          */
                   str_t *A         ,    /* in  array of {A} strings              */
                   int    n_str_B   ,    /* in  number of strings in {B}          */ 
                   str_t *B         ,    /* in  array of {A} strings              */
                   int    allow_adjacent,/* in  [see f_str_overlap()]             */
                   int  *A_str_idx  ,    /* out index to the active string in {A} */
                   int  *B_str_idx  ,    /* out index to the active string in {B} */
                   int  *A_offset   ,    /* out offset to A[A_str_idx].start_row  */
                                         /*     to the beginning of the overlap   */
                   int  *B_offset   ,    /* out offset to B[B_str_idx].start_row  */
                                         /*     to the beginning of the overlap   */
                   int  *overlap_len);   /* out length of the overlap             */
void dump_sp_detail(SparseMatrix m);
int  comp_IJV_cr(const void *a, const void *b );
int  comp_IJV_rc(const void *a, const void *b );
int  comp_int_pair(const void *a, const void *b );
void get_sort_sequence( int  N , int *In, int *Out );
int  get_sort_sequence2(int  N , int *In, int *Out2);
void sp_copy_index_data(int ncols,            /* in */
                        int nstr,             /* in */
                        SparseMatrix m_in,    /* in */
                        SparseMatrix m_out);  /* out */
int  get_fp_scalar(double  *Re,    /* out */
                   double  *Im     /* out */
                  );
void strings_in_list(int  n_terms,        /* in  length of list[]            */
                     int *list,           /* in                              */
                     int *n_str,          /* out number of strings in list[] */
                     int *start_ind,      /* out index of 1st string terms   */
                     int *str_len,        /* out length of each string       */
                     int *max_length);    /* out longest string in list[]    */
void sp_copy_column(int col_index_in,     /* in  */
                    int col_index_out,    /* in  */
                    SparseMatrix m_in,    /* in  */
                    SparseMatrix m_out);  /* out */
int  int_compare(const void *a, const void *b);
void gprint_list(char *title, int N, int *list);
int  f_speye(int indexed);
int  f_spnull(int indexed);
int  f_index_align(int  X_size,   /* in  */
                   int *X     ,   /* in  */
                   int  Y_size,   /* in  */
                   int *Y     ,   /* in  */
                   int *x_Map ,   /* out */
                   int *y_Map ,   /* out */
                   int *err_cod); /* out */
int  f_index_merge(int  X_size,   /* in  */
                   int *X     ,   /* in  */
                   int  Y_size,   /* in  */
                   int *Y     ,   /* in  */
                   int *y_Map ,   /* out array, Y_size terms */
                   int *Z_size,   /* out # terms in Z ( <= X_size+Y_size) */
                   int *Z     ,   /* out array, (max X_size+Y_size terms) */
                   int *err_cod); /* out */
int  f_indices_match(stkitem *A,  /* in  */
                     stkitem *B); /* in  */
int  f_sp2ijv(SparseMatrix m      , /* in  */
              int          indexed, /* in  */
              IJVMatrix   *IJV      /* out */
             );
int  f_ijv2sp(char      *name       ,  /* in  */
              IJVMatrix  IJV        ,  /* in  */
              int        indexed    ,  /* in  1 make room for int. indices */
              int        transpose);   /* in  1 = treat rows as cols       */
int  f_ijv_reorder(IJVMatrix *IJV    , /* in/out */
                   int       *row_map, /* in  length should be == IJV.nRows */
                   int       *col_map);/* in  length should be == IJV.nCols */
int  f_ijv_add(IJVMatrix  A ,  /* in      [A]                           */
               IJVMatrix *B ,  /* in/out  [B] (might be reordered)      */
               IJVMatrix *C ); /* out [C] (malloc'ed here) = [A] + [B]  */
void f_ijv_dump(IJVMatrix  IJV);
int  malloc_IJVMatrix(IJVMatrix *IJV   ,   /* out */
                      int        nRows ,   /* in  */
                      int        nCols ,   /* in  */
                      int        n_Nonz,   /* in  */
                      int        indexed); /* in  */
void free_IJVMatrix(IJVMatrix IJV);  /* in  */
