/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* prs.c  April 2005

Copyright (c) 2005  D. R. Williamson

   Functions for parsing text:

      token_get(): extract tokens from input in the order given

      eval_expX(), X=0,1,...: evaluate token expressions as 
         recursive data structures, where eval_expX() calls 
         eval_expX+1()

   The arrangement of functions eval_expX determines the response to a 
   phrase of tokens, as infix symbols from token_get are placed in post-
   fix order, the order of stack machine operations.

   References:

      1. Schildt, H., "C: The Complete Reference," third edition,
         Osborne McGraw-Hill, 1995.

      2. Levine, J. R., et al, "lex & yacc," second edition, O'Reilly
         & Associates, Inc., 1995.

      3. Notes on this parser are in: doc/parse.doc, doc/design.doc

   The recursive-descent parser in this file is patterned after the 
   one in the C interpreter of Reference 1.

   Reference 2 shows the use of lexical analyzer lex, to turn "regular 
   expressions" into tokens, and parser yacc to assemble tokens into
   meaningful relationships (flex and bison are more recent programs 
   that are equivalent to lex and yacc, respectively).  

   For the modest needs of this program, function token_get() and 
   recursive functions eval_expX() are sufficient to do the work of 
   lex and yacc.
*/
#define _POSIX_SOURCE /* GNU C Lib: features.h will set __USE_POSIX */
#include <stdio.h>

#include <ctype.h>

#define __USE_SVID

#undef  __EXTENSIONS__
#define __EXTENSIONS__ /* for stdlib.h, Sun */
#include <stdlib.h>

#include <setjmp.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"

#include "ctrl.h"
#include "exe.h"
#include "lib.h"
#include "mat.h"
#include "math1.h"
#include "mem.h"
#include "prs.h"
#include "sys.h"
#include "tex.h"

/* inpo.h: */
extern int filefound(); /* filefound (qS --- 0 or qFile1 f) */
extern char *fileload(char *filename);
extern int gprintf(char *format, ...);
extern char *mprintf(char *format, ...); /* WARNING: returned character
                                            pointer must be freed by the
                                            calling function. */
extern int nc(); /* nl ( --- ) */
extern int number(char *str, double *x);
extern int stradv(char *s, char *s1);
extern int xray(); /* xray (x or hFile bytes --- hT) */

/*--------------------------------------------------------------------*/

/* Two stacks are used by the functions eval_expX() during recursive
   descent parsing of infix text, that is started by function eval_exp0:

      WS - the working stack, which is the program's main stack
      FS - the function stack, a stack made for the parser

   Generally, FS holds operators for math functions, relational opera-
   tors, and items that are on the LHS of an equate.  WS holds the 
   pairs of numbers or matrices involved in a math operation, items 
   compared in a relational operation, and the RHS of an equate.

   Parentheses, brackets and braces define levels on FS and WS.  Items 
   within the innermost parentheses and brackets are at the top levels 
   of the stacks (since they are seen last) until the end of recursve 
   parsing, signified at first by right parenthesis or bracket and fi-
   nally by semicolon.  Then FS is popped back onto WS and items wind
   up somewhat reversed as they fall into postfix order, the sequential
   order of operations on a stack machine.

   This traceback made during debugging shows the recursive descent in 
   action.  Backstep #16 shows eval_exp0() starting the calls to func-
   tions lower and lower in the heirarchy (think of a higher number X 
   in name eval_expX() as greater depth in the descent; function atom()
   is at the bottom of the descent):

      [dale@clacker] /opt/tops/tops/src > gdb tops core
      (gdb) where
      #0  0x080a29fe in token_peek (Token_type_peek=0x0) at prs.c:2591
      #1  0x080a261c in token_get () at prs.c:2526
      #2  0x080a00aa in eval_exp14 () at prs.c:1142
      #3  0x0809fed2 in eval_exp13 () at prs.c:1092
      #4  0x0809fd62 in eval_exp12 () at prs.c:1049
      #5  0x0809fbec in eval_exp11 () at prs.c:1011
      #6  0x0809fa78 in eval_exp10 () at prs.c:972
      #7  0x0809f949 in eval_exp9 () at prs.c:931
      #8  0x0809f7d8 in eval_exp8 () at prs.c:893
      #9  0x0809f653 in eval_exp7 () at prs.c:853
      #10 0x0809f4b4 in eval_exp6 () at prs.c:813
      #11 0x0809f30a in eval_exp5 () at prs.c:751
      #12 0x0809f238 in eval_exp4 () at prs.c:727
      #13 0x0809ee2e in eval_exp3 () at prs.c:587
      #14 0x0809ecca in eval_exp2 () at prs.c:521
      #15 0x0809ea8e in eval_exp1 () at prs.c:461
      #16 0x0809e9e4 in eval_exp0 () at prs.c:442
      #17 0x080a0a01 in parse () at prs.c:1486
      #18 0x08072faf in catexe (word=0x8ab6a3e "parse") at exe.c:171
      #19 0x08087109 in perform () at main.c:81
      #20 0x08072e14 in xmain (flag=1) at exe.c:2536
      #21 0x08073247 in execute () at exe.c:682
      #22 0x08071fc9 in run () at exe.c:1981
      #23 0x080705e3 in exeinline () at exe.c:761
      #24 0x08072faf in catexe (word=0x8ab6840 "console") at exe.c:171
      #25 0x08087109 in perform () at main.c:81
      #26 0x08087333 in source () at main.c:158
      #27 0x08087b53 in main (argc=1, argv=0xbffff814) at main.c:42
      #28 0x401d7507 in __libc_start_main (main=0x8087b28 <main>, 
          argc=1, 
          ubp_av=0xbffff814, init=0x806b24c <_init>, fini=0x825e520 
          <_fini>, 
          rtld_fini=0x4000dc14 <_dl_fini>, stack_end=0xbffff80c)
          at ../sysdeps/generic/libc-start.c:129
      (gdb) 

   When eval_exp0() finishes, FS is empty and the program's stack used 
   for WS contains postfix text collected into a string ready to be run
   by word main, saved to a file, or turned into an inline by words in-
   line or inlinex.

   The postfix text on the program's stack replaces the infix text that
   came in.  Items on the stack that were below the incoming infix text
   remain as they were.

   Word ptrace will activate a trace that shows operations during pars-
   ing, and ptraceoff will turn it off.  

   The program's file pointers from ctrl.h, used for postfix token
   matching (inpo.c), are used here.  While the pointers are set up to
   allow for NBUF multiple run levels during postfix execution, this 
   capability has not been required for parsing:

      char *pBUF[NBUF];     // array of start pointers for buffers 
      unsigned char *ip;    // ptr to byte in *(pBUF+onbuf)
      unsigned char *ipend; // ptr to end of text

   Defining the token buffer for parsing:

      Because run level function bufup() does not carry along the be-
      ginning pointer of the token array, pBUF (set by bufjump()), it 
      is null at each new bufup() run level.  So global pointer pTOK is
      defined here to establish the beginning pointer that goes with
      moving pointer ip and ending pointer ipend.

      Pointer pTOK is set in function parse_init().  If multiple run 
      levels are used, pTOK should be used at any run level, onbuf,
      instead of pointer *(pBUF+onbuf).

      Pointers pTOK, ip and ipend will correctly point to the token
      buffer through any run level movements of bufup() and bufdn()
      if they are ever used.

      With pTOK defined as the beginning pointer, the token buffer
      pointers can be described as follows:

         Pointer ip points to the current token byte, so *ip is
         the value of the current token byte at ip.

         Initially, ip = pTOK, the beginning of the buffer.

         The size of the buffer is (ipend - pTOK), and it contains
         all of the text being parsed. */

/* Pointer to the beginning of the token buffer: */
unsigned char *pTOK;  /* the beginning of the token buffer */

/* Current token (upper case T is used to define tokens in this file,
   to avoid conflict with global token declared in inpo.h): */
#define NTOK 1024
unsigned char Token[NTOK+1]; /* token string null terminated */
unsigned char Token_type;    /* token type */
unsigned char Token_int;     /* token internal representation */

unsigned char Token_was[NTOK+1]; /* old token string null terminated */
unsigned char Token_type_was;    /* old token type */

unsigned char Token_peek[NTOK+1]; /* used by token_peek() and peekFS()*/

/* Local enumerations. */

/* Token type: */
enum tok_type {FUNCTION,VARIABLE,NUMBER,STRING,OPERATOR,PUNCTUATION,
               KEYWORD,FINISHED,PNDING,UNKNOWN,PRN,BRK,BRC,NOTDEF=255};

/* Token internal representation: */
enum tok_int {IF=1,ELSE1,FOR,DO,WHILE1,SWITCH,DEFINE1,RETURN,END,
              UNSET=255};

/* These additional tokens are defined as characters above 127 (see 
   below, "Example: running Task 2"): */
/* Relational tokens (for < <= > >= == !=): */
enum rel_tok {LT=128,LE,GT,GE,EQ,NE,NGT,NLT};
/* Logical tokens (for && || !): */
enum log_tok {AND=1+NLT,OR,NOT,LAST_LOG_TOK};
/* Punctuation tokens (in addition to ( ) = [ ] ; ,): */
enum punct_tok {BRK_SEMI=1+LAST_LOG_TOK,BRK_COMMA};
/* Arithmetic tokens (in addition to +,-,*,^,/,%): */
enum arith_tok {NEG=1+BRK_COMMA,POS,INC,DEC,INC_EQU,DEC_EQU,MPY_EQU,
                DIV_EQU,DIV_EL,MUL_EL,POW_EL,DIAG_PRE,DIAG_POST};
enum extra_tok {CE=1+DIAG_POST};

/* Keyword lookup tables. */
typedef struct { 
   char command[20];
   char tok;
} commands;

/* Lookup table of parser keywords (primarily C-like functions): */
commands C_table[]= { 
   {"if",      IF},
   {"else",    ELSE1},
   {"for",     FOR},
   {"do",      DO},
   {"while",   WHILE1},
   {"switch",  SWITCH},
   {"function",DEFINE1},
   {"return"  ,RETURN},
   {"",        END} /* mark the end of C_table */
};

/* Parse driver tasks: */
enum parse_tasks {SHOW_TOKENS,EVAL_EXP,SHOW_ENUM};

/* Parser recursive functions, from top to bottom: */
void eval_exp0(void);  /* parse an expression                         */
void eval_exp1(void);  /* semicolon ;                                 */
void eval_exp2(void);  /* comma ,                                     */
void eval_exp3(void);  /* bracket pile ;                              */
void eval_exp4(void);  /* bracket park ,                              */
void eval_exp5(void);  /* logical operators && ||                     */
void eval_exp6(void);  /* relational operators < <= > >= == != !      */
void eval_exp7(void);  /* add or subtract + -                         */
void eval_exp8(void);  /* multiply or divide * / % .* ./ \* *\        */
void eval_exp9(void);  /* post-unary operator -- ++                   */
void eval_exp10(void); /* right parenthesis )                         */
void eval_exp11(void); /* right bracket ]                             */
void eval_exp12(void); /* raise to a power ^                          */
void eval_exp13(void); /* pre-unary operator + - ++ --                */
void eval_exp14(void); /* transpose operator '                        */
void eval_exp15(void); /* right brace }                               */
void eval_exp16(void); /* left parenthesis (                          */
void eval_exp17(void); /* left bracket [                              */
void eval_exp18(void); /* left brace {                                */
void eval_exp19(void); /* equate = += -= *= /=                        */
void atom(void);       /* put a new token on WS                       */

/*
Precedence.

Functions eval_expXX() with greater value XX are deeper in the descent 
parser, and have higher precedence than those with lower value XX.  

Thus eval_exp12() to raise a number to a power has higher precedence
than eval_exp8() that multiplies two numbers, and in the expression

   x = 5 + 4 * 3 ^ 2;

number 3 is squared before multiplication by 4 is performed; and the
addition of 5 occurs last since precedence of eval_exp7(), the add 
function, is lowest of all.  Here is the postfix phrase created by the
parser for the infix expression above:

   5 4 3 2 ^ * + "x" book

All postfix phrases assume operations with the program's last-in-first-
out (LIFO) stack (see note below).  The phrase above shows numbers push-
ed to the stack in the correct order, then the firing of ^ * + in se-
quence and finally the booking of x into the catalog.

Note: These are the steps taken by the postfix stack machine when oper-
ating on a postfix phrase like the one above:
   - fetch next token from source
   - if it is a number, push it to the stack
   - else if quotes surround the text, push the text to the stack
   - else find a word in the catalog that matches and run it
   - else report an error
   - repeat until source ends

A descent ends at function atom() which is called by eval_expXX() with
greatest XX.  Function atom() begins an ascent, probably one of many
ascents and descents.  The process ends when an ascent returns through 
eval_exp0(). */ 

/* Local headers. */
int args_in; /* when creating a definition; names are in input_args */
int args_out; /* when creating a definition */
unsigned char BRKTYP[NBUF]; /* see brk_type */
enum brk_type {MAKE=1,INDX,INDX_LIST};
void clearFS();
char _COLS[3]={'-','C','\0'};
int COMMAcount[NBUF]; /* comma counts at each LEVEL */
int d0[NBUF]={0}; /* stack depth when starting new runlevel */
int DEF_PENDING; /* the making of a definition (DEFINE) is underway */
int def_end();
void def_init(int entry);
#define depthFS _fdepth
patitem *EQUIV[NBUF]; /* pairs of equivalent strings; k+1 replaces k */
char EQU[NBUF]; /* 1 if RHS of equate is in progress on LEVEL */
int EQU_DEPTH[NBUF]; /* absolute WS depth for equate on LEVEL */
char *EQU_NAME[NBUF]; /* *char to equate name on LEVEL */
char EQwarn[NBUF]; /* parenthesis warning for equates at LEVEL */
void eval_init();
int exe_bracket_end();
int exe_def_init();
int exe_else();
int exe_if();
int exe_then();
int finish(); /* (hT --- qS) */
char FScount[NBUF]; /* array of FS counts at each LEVEL */
int initialFS;
char *input_args[NBUF]; /* pointers to input arg names */
unsigned char *IP_ERR; /* *ip at latest activity, for sntx_err */
int is_arg_in(char *s);
int is_constant(char *s);
int is_function(char *s);
int is_keyword();
int is_var(char *s);
int isblank1(char c);
int isdelim1(char c);
int KWcount; /* active KEYWORD count */
int lpush1();
char LEV3[4];
int LEVEL; /* level defined by left paren or left bracket */
void parse_init();
#define PATWID 80 /* max width of EQUIV strings */
unsigned char *peekFS();
void place_prep();
int popFS(); 
void pop_mathFS();
int popWS(); 
int pushFS(); 
int pushWS(char *quote);
struct {
   int (*exe)(void); /* execute on the closing right brace */
} RBexe[NBUF];
struct {
   int (*exe)(void); /* execute on the closing right paren */
} RPexe[NBUF];
char _ROWS[3]={'-','R','\0'};
struct {
   int (*exe)(void); /* execute on the next semicolon for KW */
} SEMIexe[NBUF];
char *show(unsigned char *Token, int Type);
void show_tokens(void);
void showWS();
void sntx_err(char *error);
void sntx_warn(char *warning);
int _token_get(void);
int token_get(void);
unsigned char *token_peek(int *token_type);
unsigned char TYP[NBUF]; /* symbol for LEVEL (tok_type BRC,BRK,PRN) */
int WScount[NBUF]; /* array of WS counts at each LEVEL */

/*--------------------------------------------------------------------*/

void atom()
/* This is the low end of the recursive descent.

   If the current token is a FUNCTION, VARIABLE or NUMBER, push its 
   name to the working stack (WS) and get another token and return.

   If the current token is a STRING, push it to the working stack and 
   get another token and return.

   If the current token is a KEYWORD, set up the functions to be run 
   during the return ascent, and get another token and return.

   Otherwise, the token is an OPERATOR or PUNCTUATION.  Keep the same 
   token and return.

   The return ascent will go until it reaches a function that processes
   the token.  Then it will fall back, processing more tokens read along
   the way and eventually coming here again. */
{
   char p[1+PATWID],*p1,*p2;
   double d;
   int indx,loc,op;

   if(PTRACE) {
      gprintf("                                      "); /* atom */
      gprintf("## (atom now)  Token: %s type: %d LEVEL: %d FSCOUNT: %d",
         show(Token,Token_type),Token_type,LEVEL,
         *(FScount+LEVEL));
      nc();
   }
/* This is an arbitrary limit.  It could probably be much higher. */
   if(abs(*(WScount+LEVEL))>4096)
      sntx_err(mprintf("halting due to working stack internal fault"));

   switch(Token_type) {

      default:
      case PUNCTUATION:
      case OPERATOR:
      break;

      case VARIABLE:
         pushWS((char *)Token);
         p1=strtok((char *)Token,".");
         p2=p1;
         p2=strtok('\0',".");

         if(p1 && p2 && strlen(p2)) {
         /* If Token is of the form A.B, it means "word B from the li-
            brary of word A."  But function is_var() only sees A.B as 
            a possible variable, and the program lands here.

            Flag Token to be of the form A.B by surrounding it with
            \t: */
            pushstr("\t");
            swap();
            cat(); /* \tA.B on tos */
            pushstr("\t");
            cat(); /* \tA.B\t on tos */

         /* Make string A.B unique by appending a unique index.  For 
            the index, use the growing rows of EQUIV table: */
            pushstr("*");
            indx=((*(EQUIV+onbuf))->use)-1; /* 0-based index */
            pushint(indx);
            intstr();
            cat();
            pushstr("*");
            cat();
            cat(); /* unique string on tos: \tA.B\t*ID* */

         /* Add token A.B to EQUIV table, entry k: */
            if(tos->col+1>PATWID) {
               sntx_err(mprintf("table width %d is exceeded by: %s",
                  PATWID,tos->tex));
            }
            memset(p,' ',PATWID);
            memcpy(p,tos->tex,tos->col);
            *(p+tos->col)='\0';
            patadd(p,*(EQUIV+onbuf),&loc);

            if(!(loc>indx)) /* EQUIV table must be growing */
               sntx_err(mprintf("pattern is not unique: %s",tos->tex));

         /* Add phrase '"A" "B" yank*' to EQUIV table, entry k+1.  This
            makes the machine extract word B from the library of word
            A, and is the substitution for A.B (entry k) that will be 
            made when finish() is later run. */
            pushstr(p1); /* p1 = lib = word name */
            quoted1();
            pushstr(" ");
            pushstr(p2); /* p2 = name to yank from p1's lib */
            quoted1();
            pushstr(" yank* "); /* command to extract p2 from p1 lib */
            cat();
            cat();
            cat(); /* "A" "B" yank* on tos */

         /* Adding '"A" "B" yank*' on tos to EQUIV table, entry k+1: */
            if(tos->col>PATWID) {
               sntx_err(mprintf("table width %d is exceeded by: %s",
                  PATWID,tos->tex));
            }
            memset(p,' ',PATWID);
            memcpy(p,tos->tex,tos->col);
            *(p+tos->col)='\0';
            patadd1(p,*(EQUIV+onbuf),NULL);

            drop(); /* drop tos phrase to yank; A.B is now on tos */
         }
         token_get();
      break;

      case NUMBER:
         if(!number((char *)Token,&d)) 
            sntx_err(mprintf("not a number"));
         pushWS((char *)Token);
         token_get();
      break;

      case FUNCTION:
         pushWS((char *)Token);
         token_get();
      break;

      case STRING:
         pushWS((char *)Token);
         token_get();
      break;

      case UNSET:
      case UNKNOWN:
         IP_ERR=ip;
         sntx_err(mprintf("type unknown for token %s",
            show(Token,Token_type)));
      break;

      case KEYWORD:
         op=Token_int;
         if(PTRACE) {
            gprintf("                                      "); /*atom*/
            gprintf("KEYWORD: %s op: %d",
               show(Token,Token_type),op);
            nc();
         }
         switch(op) {

            case IF:
               token_get();
               if(*Token!='(') {
                  sntx_err(mprintf("left parenthesis required after %s",
                     show(Token_was,Token_type_was)));
               }
            /* Function to close out an if expression: */
               RPexe[LEVEL].exe=(int (*)()) exe_if;
            break;

            case ELSE1: /* process an else statement */
               exe_else();

            /* There is no semicolon following else, so reading the 
               next token with token_get will produce a syntax error.  
               Use _token_get instead, which skips the syntax check: */
               _token_get(); 

               atom(); /* reenter for new token just gotten */
            break;

            case DEFINE1: /* initialize creation of a definition */
               def_init(0);
               token_get();
            break;

            case RETURN: /* force "return" to "ret__" in a definition */
               if(DEF_PENDING) {
                  memcpy(Token,"ret__",5); /* to ret__ if definition */
                  *(Token+5)='\0';
               }
               Token_type=FUNCTION; /* make return into a FUNCTION */
               atom(); /* go again and do return FUNCTION */
            break;

            case WHILE1: /* process a while loop */
     /*        exec_while(); */
            break;
            case FOR: /* process a for loop */
     /*        exec_for();   */
            break;
            case DO: /* process a do-while loop */
     /*        exec_do();   */
            break;
            case SWITCH: /* process a switch */
     /*        exec_switch();   */
            break;
         }
      break;
   }
   if(PTRACE) {
      gprintf("                                      "); /* atom */
      gprintf("<< (atom new)  Token: %s type: %d LEVEL: %d FSCOUNT: %d",
         show(Token,Token_type),Token_type,LEVEL,
         *(FScount+LEVEL));
      nc();
   }
}

void clearFS() 
/* Clear the function stack:  */
{
   while(depthFS>initialFS) {
      pullfs();
      if(PTRACE) {
         gprintf("pop fro FS: %s",tos->tex);
         gprintf(" tok name: %s",tos->tok);
         nc();
      }
      drop();
   }
   memset(FScount,0,NBUF); /* zero FS counts */
}

int def_end()
/* End creation of a parsed function word that was started with 
   def_init().  

   Push "RET__ end"  to WS and free the arrays of input argument 
   names. 

   Pile all the lines for the definition into a VOL and make the 
   pattern substitutions from the EQUIV table on this runlevel, 
   then free the EQUIV table.

   Finally, drop down to the runlevel being used before def_init() 
   was called to make this function word. */
{
   int k=0;

   if(PTRACE) gprintf("   Running def_end()\n");
   pushWS("RET__ end");

   DEF_PENDING=0;

   for(;k<args_in;k++) {
      if(*(input_args+k)) free(*(input_args+k));
      *(input_args+k)=NULL;
   }
   if(stkdepth()>*(d0+onbuf)) {
      pushint(stkdepth()-*(d0+onbuf));
      pilen();
      finish();
   }
/* Free the pattern table: */
   if(EQUIV+onbuf) {
      mallfree((void *) *(EQUIV+onbuf)); /* first p->pat */
      mallfree((void *) (EQUIV+onbuf));  /* then patitem p itself */
   }
   bufdn(); /* down from the runlevel set in def_init() */
   return 0;
}

void def_init(int entry)
/* Initialize WS and FS for the creation of a parsed function word.

   This function runs for keyword pattern "function."

   Word "function" is not an "official" catalog word, but is a pattern
   sensed by this front end parser (see lookup table above of parser 
   keywords, C_table).  So that "man function" does not come up empty, 
   though, there is a catalog word called "function" created in file
   word.p, but it is simply noop().

   Examples.  Here is running the parser for various function strings
   and displaying the resulting postfix text that comes primarily 
   from def_init() and def_end():

   Note: these examples may reflect earlier code.

      [dale@clacker] /opt/tops/tops/src > tops -p
               Tops 3.0.1
      Wed Jul 20 05:49:52 PDT 2005
      [tops@clacker] ready >

   Case 1:
      ready > "function (Phi,Omeg)=reig(M,K) { }" >> parse << .
      inline: reig
      [ defname 'name__' book 0 'ercnt__' book
      2 'arg_inp__' book
      2 'arg_ret__' book

      "0 pickfs" "M" inlinex
      "1 pickfs" "K" inlinex

      PARSER_FUNCTION 'PRE' yank
      "pushfs pushfs "
      pile 'PRE__' inlinex

      'Phi dup type tpurged "Phi" book'
      'Omeg dup type tpurged "Omeg" book'
      2 pilen
      "pullfs drop pullfs drop "
      PARSER_FUNCTION 'RET' yank
      'return2'
      4 pilen 'RET__' inlinex

      'Phi type tpurged "Phi" book'
      'Omeg type tpurged "Omeg" book'
      2 pilen
      "pullfs drop pullfs drop "
      PARSER_FUNCTION 'ret' yank
      'return2'
      4 pilen 'ret__' inlinex

      ] PRE__
         (... definition body goes here ...)
      RET__ end

      This shows the function being assembled into an inline and
      placed into the catalog:
      ready > >> function (Phi,Omeg)=reig(M,K) { } <<
       word M,0:reig into catalog
       word K,0:reig into catalog
       word PRE__,0:reig into catalog
       word RET__,0:reig into catalog
       word ret__,0:reig into catalog
       word reig into catalog

   Case 2:
      ready > "function reig(M,K) { }" >> parse << .           
      inline: reig
      [ defname 'name__' book 0 'ercnt__' book
      2 'arg_inp__' book
      0 'arg_ret__' book
      "0 pickfs" "M" inlinex
      "1 pickfs" "K" inlinex

      PARSER_FUNCTION 'PRE' yank
      "pushfs pushfs "
      pile 'PRE__' inlinex

      PARSER_FUNCTION 'RET' yank
      'return2'
      pile 'RET__' inlinex

      'return2'
      'ret__' inlinex

      ] PRE__
         (... definition body goes here ...)
      RET__ end

   Case 3:
      ready > "function (Phi,Omeg)=reig() { }" >> parse << .
      inline: reig
      [ defname 'name__' book 0 'ercnt__' book
      0 'arg_inp__' book
      2 'arg_ret__' book
      PARSER_FUNCTION 'PRE' yank
      'PRE__' inlinex

      'Phi dup type tpurged "Phi" book'
      'Omeg dup type tpurged "Omeg" book'
      2 pilen
      PARSER_FUNCTION 'RET' yank
      'return2'
      3 pilen 'RET__' inlinex

      'Phi type tpurged "Phi" book'
      'Omeg type tpurged "Omeg" book'
      2 pilen
      'return2'
      pile 'ret__' inlinex

      ] PRE__
         (... definition body goes here ...)
      RET__ end

   Case 4:
      ready > "function reig() { }" >> parse << .           
      inline: reig
      [ defname 'name__' book 0 'ercnt__' book
      0 'arg_inp__' book
      0 'arg_ret__' book
      PARSER_FUNCTION 'PRE' yank
      'PRE__' inlinex

      PARSER_FUNCTION 'RET' yank
      'return2'
      pile 'RET__' inlinex

      'return2'
      'ret__' inlinex

      ] PRE__
         (... definition body goes here ...)
      RET__ end

   After this function runs, the body of the definition is created by 
   simply continuing to run the parser.  When the closing } is reached,
   function def_end() adds "RET__ end" to WS, completing the definition.

   When the postfix interpreter later creates the definition, the text 
   created in def_init() that is inside brackets runs only at that time
   (this is a feature that has always been in the postfix interpreter).
   The inlines and variables created go into the library of the word 
   (see man [).

   For example, inlines PRE__ and RET__ placed by def_init() within 
   brackets are the preface and return inlines that will always be run 
   when the word starts and ends (as shown above, PRE__ and RET__ pre-
   cede and follow the body of the word).

   And if return() is used in the body of the word, ret__ is substituted
   (see RETURN in atom(), prs.c).  Inline ret__ is just like RET__ ex-
   cept arguments are not returned unless they are in the argument list
   of return().

   This shows creation of reig, along with four inline words which in-
   clude words PRE__, RET__ mentioned above:

      [dale@clacker] /opt/tops/tops/src > tops -i
               Tops 3.0.1
      Wed Jul 20 06:21:04 PDT 2005
      >> 
      >> function (Phi,Omeg)=reig(M,K) { }
       word M,0:reig into catalog
       word K,0:reig into catalog
       word PRE__,0:reig into catalog
       word RET__,0:reig into catalog
       word ret__,0:reig into catalog
       word reig into catalog

   Also in the library of reig are inlines M and K.  Text in def_init()
   causes input arguments to be pushed to the function stack (fs) by 
   the line in word PRE__ that says "pushfs pushfs " (see Case 1 above).

   And inline words M and K are also created to run word pickfs for the
   appropriate level, as these definitions (Case 1 above) show:
      "0 pickfs" "M" inlinex 
      "1 pickfs" "K" inlinex

   Thus no memory transfer is done.  Word M will fetch the top of fs 
   for matrix M, and word K will fetch next on fs for matrix K.  (The 
   function stack, fs, mentioned here and used at runtime, is the 
   same one used by the parser in this file, and called FS.)  

   Trying to yank one of the input arguments from a function will pro-
   duce an error; input arguments never really exist in the library of 
   a function. */
{  
   int i=0,k=0,lines,ret_pile;
   char *p;
   double d;

   IP_ERR=ip;
   if(LEVEL!=0) 
      sntx_err(mprintf("unbalanced parentheses, brackets or braces"));

   if(entry) goto bottom; /* reentry from exe_def_init() */
   if(PTRACE) gprintf(" Running def_init(0)\n");

/* Initialize just like left paren: */
   *(COMMAcount+LEVEL)=0; /* COMMAcount starts at zero */
   *(WScount+LEVEL)=0;    /* WScount starts over again */

   KWcount=0;

/* Jump to next run level and start a pattern table.  The table is freed
   and bufdn() is called by def_end() when everything is finished: */
   bufup();
   *(d0+onbuf)=stkdepth();
   *(EQUIV+onbuf)=(patitem *)patnew(32,4,PATWID);

   if(*token_peek(NULL)=='(') {
      token_get();
      atom();

   /* Read the paren list of output args: */
      while(!(*Token==')' || Token_type==FINISHED)) {
         token_get();
         if(*Token!=',' && 
         /* Output args can match VARIABLE names and even FUNCTION names
            since they won't really replace them; here the intent is to
            disallow things like numbers: */
            Token_type!=VARIABLE && 
            Token_type!=FUNCTION && 
            *Token!=')') {
               sntx_err(mprintf("%s is an invalid output argument name",
                  show(Token,Token_type)));
         }
         atom(); /* pushes VARIABLE to WS */
      }
   }
   args_out=*(WScount+LEVEL);
   for(k=0;k<args_out;k++) pushFS(); /* output args from WS to FS */

/* First item on WS is 'define:' or 'inline:': */
   if(PTRACE) pushWS("define: "); /* make define word if PTRACE */
   else pushWS("inline: ");

   if(*token_peek(NULL)=='=') {
      token_get(); /* next token after ) is = */
      atom(); /* check in, but PUNCTUATION will just return */

      if(*Token!='=') {
         sntx_err(mprintf("expect function equate symbol"));
      }
   }
   _token_get(); /* get name of word */

/* Note that function name initially looks like type VARIABLE, but
   becomes type FUNCTION once it has been put into the catalog.  On a 
   second pass, as when fixing an error, name will have become type 
   FUNCTION which this test allows for: */
   if(!(Token_type==VARIABLE || Token_type==FUNCTION))
      sntx_err(mprintf("expect function name"));

/* Function name followed by = probably means this is not the function
   name, but one of the output args: */
   if(*token_peek(NULL)=='=') {
      _token_get(); /* move to bad part */
      IP_ERR=ip;
      sntx_err(mprintf(\
         "expect enclosed (...) output arg list on left of ="));
   }
/* Function name must be followed by ( for input arg list: */
   if(*token_peek(NULL)!='(') {
      IP_ERR=ip;
      sntx_err(mprintf(\
         "expect enclosed (...) input arg list after function name"));
   }
/* Concatenate define: or inline: with name of word: */
   pushstr((char *)Token);
   cat();
 
/* Function to catch the end of the input args list on right paren (RP)
   and return here: */
   RPexe[LEVEL].exe=(int (*)()) exe_def_init;

   Token_type=KEYWORD; /* phony type so ( doesn't push VARIABLE to FS */
   DEF_PENDING=1; /* set flag that a DEFINE is in progress */
   return;

   bottom:
   if(PTRACE) {
      gprintf(" Running def_init(1); WS contents are:\n");
      showWS();
   }
/* The program is here upon reentry from exe_def_init(), after the list 
   of output args has been read.  The postfix text for the preface of
   the parsed function is made below.  This includes text from the
   PARSER_FUNCTION library of file boot.v.  Examples of preface con-
   struction by this code are shown above.

   After the preface words are stored on WS, the parser continues to
   run to build the body of the definition.  On the closing }, def_end()
   will run to complete the text for the parsed function. */
 
   args_out=*(FScount+LEVEL);

/* Input arguments were collected one level up (due to left paren that
   starts the argument list) and then right paren brought them down to 
   this LEVEL.  Because this LEVEL started out with one item, the string
   "inline: funcname," the total number of input arguments is one less 
   than WScount(LEVEL): */

   args_in=*(WScount+LEVEL)-1;

   if(args_in>NBUF) 
      sntx_err(
         mprintf("%d input arguments will overflow input_args array",\
         args_in));

   if(args_in>(DEPLOCSTK-_ldepth)) 
      sntx_err(
         mprintf("%d input arguments will overflow local stack",\
         args_in));

   if(args_in) { 
      pushint(args_in);
      revn(); /* reverse input arguments so first is tos */
   }
   for(k=0;k<args_in;k++) { /* null-terminated str for each arg_in */
      if((p=malloc(1+(tos-k)->col))==NULL) {
         stkerr(" def_init: ",MEMNOT);
         sntx_err(NULL);
      }
      *(p+(tos-k)->col)='\0';
      *(input_args+k)=p;
   }
   for(k=0;k<args_in;k++) {
      p=*(input_args+k);
      memcpy(p,tos->tex,tos->col); /* kth arg on stk to input_args(k) */
      popWS();                     /* input arg dropped from WS */
      if(number(p,&d) || isdelim1(*p)) 
         sntx_err(mprintf("%s is an invalid input argument name",p));
      for(i=0;i<args_out;i++) {
         pushint(i);
         pickfs(); /* output arg from FS */
         if(!strcmp((char *)tos->tex,p)) {
            drop();
            sntx_err(
               mprintf("input and output names \"%s\" cannot match",p));
         }
         drop();
      }
   }
/* Now all input args are in array input_args, and all output args are 
   on FS.  The large body of code that follows creates the words of the
   preface and pushes lines to WS, then piles all the lines into one 
   item (a VOL) on WS: */

   lines=1; /* first on WS is 'inline: name' or 'define: name' */

   pushstr("[ defname 'name__' book* 0 'ercnt__' book*");
   lines++;

   pushint(args_in);  intstr(); pushstr(" 'arg_inp__' book*"); cat();
   lines++;
   pushint(args_out); intstr(); pushstr(" 'arg_ret__' book*"); cat();
   lines++;

/* Inlines for args in, and inline PRE__: */

   if(args_in) {
      for(k=0;k<args_in;k++) {
         pushint(k); intstr(); 
         pushstr(" pickfs" ); cat(); quoted1(); spaced();
         pushstr(*(input_args+k)); quoted1(); cat();
         pushstr(" inlinex"); cat();
         lines++;
      }
      pushstr("PARSER_FUNCTION 'PRE' yank*");
      lines++;
      for(k=0;k<args_in;k++) { /* input args to fs */
         pushstr("pushfs ");
         if(k>0) cat();
      }
      quoted1();
      lines++;
      pushstr("pile* 'PRE__' inlinex");
      lines++;
   }
   else {
      pushstr("PARSER_FUNCTION 'PRE' yank*");
      lines++;
      pushstr("'PRE__' inlinex");
      lines++;
   }
/* Inline RET__: */

   ret_pile=0;
   if(args_out) {
      for(k=0;k<args_out;k++) {
         pushint(k);
         pickfs(); /* output arg from FS */
         pushstr(" dup* type* tpurged* ");
         over(); quoted1();
         pushstr(" book*"); 
         cat();
         cat();
         cat();
         quoted1();
         lines++;
      }
      pushint(args_out); 
      intstr(); 
      pushstr(" pilen*"); 
      cat();
      lines++;
      ret_pile++;
   }
   if(args_in) { /* Clean off fs */
      for(k=0;k<args_in;k++) { /* input args from fs */
         pushstr("pullfs drop* ");
         if(k>0) cat();
      }
      quoted1();
      lines++;
      ret_pile++;
   }
   pushstr("PARSER_FUNCTION 'RET' yank*");
   lines++;
   ret_pile++;

   pushstr("'return2*'"); /* RET__ will cause the word to return */
   lines++;
   ret_pile++;

   pushint(ret_pile);
   intstr(); 
   pushstr(" pilen* 'RET__' inlinex");
   cat();
   lines++;

/* Inline ret__ (used for return called inside the function): 

   Output args, or other items returned on the stack, are listed in 
   the output args of return, like: return(Phi, Omeg); 

   The return functions RET__ and ret__ use word return2 to cause an
   exit from the function that runs them. 

   Unlike RET__, ret__ does not clean up named output args because 
   they may never have been created. */ 

   ret_pile=0;
   if(args_in) { /* Clean off fs */
      for(k=0;k<args_in;k++) { /* input args from fs */
         pushstr("pullfs drop* ");
         if(k>0) cat();
      }
      quoted1();
      lines++;
      ret_pile++;
   }
   pushstr("PARSER_FUNCTION 'ret' yank*");
   lines++;
   ret_pile++;

   pushstr("'return2*'"); /* ret__ will cause the word to return */
   lines++;
   ret_pile++;

   pushint(ret_pile);
   intstr();
   pushstr(" pilen* 'ret__' inlinex");
   cat();
   lines++;

/* The very first executable word in the definition is PRE__: */
   pushstr("] PRE__");
   lines++;

/* All lines on WS are now piled into one VOL.  Thus WScount will 
   still be 1 as it was when pushWS was run above to put 'define:' 
   or 'inline:' on WS: */
   pushint(lines);
   pilen(); 

   while(*(FScount+LEVEL)) { /* output args off FS */
      popFS(); /* remove name from FS, push to WS */
      popWS(); /* and drop it from WS */
   }
/* Verify that the next token is { that follows the ) we are on: */
   if(*token_peek(NULL)!='{') {
      IP_ERR=ip;
      sntx_err(mprintf("syntax error; expecting ) followed by {")); 
   }
/* Next token is a left brace, and it will raise LEVEL by one.  Assign
   the function to finish this definition on the closing right brace
   that follows it, also on LEVEL+1 : */
   RBexe[LEVEL+1].exe=(int (*)()) def_end; /* RB (right brace) funct */
}

void eval_exp0()
/* Parse an expression

   Expressions end in semicolon or null.  For expressions ending in
   semicolon, the stack is cleared; otherwise, equated items (items 
   on the LHS of the equal sign) remain on the stack. */
{
   eval_init();

   token_get();

   if(!*Token) {
      ip=ipend;
      return;
   }
   if(*Token==';') {
      pushWS("xx"); /* semicolon clears the stack */
      return;
   } 
   while(*(TYP+LEVEL)==BRK || !(*Token==';' || Token_type==FINISHED) ) {

      if(PTRACE) {
         gprintf(">> (expression get)  Token: %s",
            show(Token,Token_type));
         nc();
      }
      eval_exp1();

      if(PTRACE) {
         gprintf("<< (expression got)  Token: %s",
            show(Token,Token_type));
         nc();
      }
   }
   if(LEVEL!=0) sntx_err(
      mprintf("unbalanced parentheses, brackets or braces"));
}

void eval_exp1()
/* Semicolon.  There are two uses for semicolon:

      1: ending an expression

      2: piling numbers or matrices while inside brackets

   This function is for ending an expression.  

   The second use is handled in eval_exp3().  Its context is deter-
   mined in _token_get() when operation inside brackets is noted by
   *(TYP+LEVEL)==BRK, and Token=BRK_SEMI is returned instead of ordi-
   nary semicolon. */
{
   int kmax;
   int (*exe)();

   eval_exp2();

   while(*Token==';' || Token_type==FINISHED) {

      if(PTRACE) {
         gprintf("  "); /* 1 */
         gprintf(">> (semicolon get)  Token: %s KWcount: %d LEVEL: %d",
            show(Token,Token_type),KWcount,LEVEL);
         nc();
      }
      if(LEVEL!=0 && *(TYP+LEVEL)!=BRC) 
         sntx_err(mprintf("unbalanced parentheses or brackets"));

   /* Clear the FS at LEVEL: */
      while(*(FScount+LEVEL)) popFS();

   /* Push xx to WS, making semicolon clear the stack: */
      if(*Token==';') {
         pushWS("xx"); 
         memset(COMMAcount,-1,NBUF); /* signed comma count set to -1 */
         memset(EQU,0,NBUF); /* zero the equate flag */
      }
   /* Execute the KEYWORD "semicolon" function: */
      if((exe=(SEMIexe+KWcount)->exe)) {
        exe();
        memset(COMMAcount,-1,NBUF); /* signed comma count set to -1 */
      }
      if(Token_type==FINISHED) break;

      if(*(TYP+LEVEL)!=BRC) {
         if(*Token==';') break;
         token_get();

         if(*Token==';' || Token_type==FINISHED) break;
         eval_init();
      }
      else
         token_get();

      eval_exp2();

      if(PTRACE) {
         gprintf("  "); /* 1 */
         gprintf("<< (semicolon got)  Token: %s KWcount: %d LEVEL: %d",
            show(Token,Token_type),KWcount,LEVEL);
         nc();
      }
   }
/* Run any remaining KEYWORD "semicolon" functions: */
   kmax=NBUF-1; 
   while(KWcount && kmax) {
      if((exe=(SEMIexe+kmax)->exe)) exe();
      kmax--;
   }
   if(PTRACE) {
      gprintf("  ");
      gprintf("<< (semicolon exit)  Token: %s KWcount: %d LEVEL: %d",
         show(Token,Token_type),KWcount,LEVEL);
      nc();
   }
   return;
}

void eval_exp2()
/* Comma.  There are two uses for comma:

      1: counting items in a paren list

      2: parking numbers or matrices while inside brackets

   This function is for counting items in a paren list.

   The second use is handled in eval_exp4().  Its context is deter-
   mined in _token_get() when operation inside brackets is noted by
   *(TYP+LEVEL)==BRK, and Token=BRK_COMMA is returned instead of ordi-
   nary comma. */
{
   int k;

   eval_exp3();

   while(*Token==',') {

      if(*(COMMAcount+LEVEL)<0 && *(TYP+LEVEL)!=PRN) {
         sntx_err(mprintf(\
         "comma-separated items need to be enclosed in parentheses"));
      }
   /* Higher comma levels must be uninitialized: */
      k=NBUF-1;
      while(k>LEVEL) { 
         *(COMMAcount+k)=-1;
         k--;
      }
      (*(COMMAcount+LEVEL))++;
      token_get();

      if(PTRACE) {
         gprintf("    "); /* 2 */
         gprintf(">> (comma get)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
      eval_exp3();

      if(PTRACE) {
         gprintf("    "); /* 2 */
         gprintf("<< (comma got)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
   }
}

void eval_exp3()
/* Semicolon bracket.
   Pile two stack items while inside brackets (semicolon). */
{
   unsigned char op[2]={0,0};

   eval_exp4();

   while(*Token==BRK_SEMI) {

      *op=*Token;

      token_get();

      if(PTRACE) {
         gprintf("      "); /* 3 */
         gprintf(">> [;] get  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      if(*(BRKTYP+LEVEL)!=MAKE) sntx_err(mprintf("invalid token"));

      pushWS("pile");
      pushFS();

      eval_exp4();

      if(PTRACE) {
         *op=*Token;
         gprintf("      "); /* 3 */
         gprintf("<< [;] got  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      if(*(FScount+LEVEL)) {
         if(!strcmp((char *)peekFS(),"pile")) popFS();
      }
   }
}

void eval_exp4()
/* Comma bracket.  
   Park two stack items while inside brackets (comma), or collect
   indices for an equate. */
{
   unsigned char op[2]={0,0};

   eval_exp5();

   while(*Token==BRK_COMMA) {

      *op=*Token;

      (*(COMMAcount+LEVEL))++;

      if(*(BRKTYP+LEVEL)==MAKE) {
         pushWS("park");
         pushFS();
      }
      else {
      /* Doing array row index for fetching or storing. */

         while(*(FScount+LEVEL)) popFS(); /* all FS at LEVEL to WS */

         if(*(COMMAcount+LEVEL)>1)
            sntx_err(mprintf("subscript error: too many commas"));

         if(*(WScount+LEVEL)==0) { /* nothing specified means all */
            pushWS("-ALL");
         }
         pushWS(_ROWS); /* doing rows on LEVEL */
          
         *(BRKTYP+LEVEL)=INDX; /* put back to plain INDX */

      /* Peek at next token, and if it is a * (meaning take "all" 
         columns), read it so it will be skipped: */
         if(*token_peek(NULL)=='*') {
            token_get();
            pushWS("-ALL");
            pushWS(_COLS); /* doing rows on LEVEL */
         }
      }
      token_get();

      if(PTRACE) {
         gprintf("        "); /* 4 */
         gprintf(">> [,] get  Token: %s op: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL,
            *(BRKTYP+LEVEL));
         nc();
      }
      eval_exp5();

      if(PTRACE) {
         *op=*Token;
         gprintf("        "); /* 4 */
         gprintf("<< [,] got  Token: %s op: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL,
            *(BRKTYP+LEVEL));
         nc();
      }
      if(*(FScount+LEVEL)) {
         if(!strcmp((char *)peekFS(),"park")) popFS();
      }
   }
}

void eval_exp5()
/* Process logical operators. 

   Note: Logical operator NOT (!) is processed in eval_exp6() with
   relational operators, to place it correctly in the recursive
   heirarchy with respect to AND and OR. */
{
   char *name;

   unsigned char idtoks[3]={AND,OR,0},op;  /* log_tok */

   eval_exp6();

   op=*Token;
   while(strchr((char *)idtoks,op) && op) {

      token_get();

      if(PTRACE) {
         gprintf("          "); /* 5 */
         gprintf(">> (logical ops get)  Token: %s Token op: %d",
            show(Token,Token_type),op);
         nc();
      }
      name=NULL;

      switch(op) {
         case AND: /* logical and, not bitwise */
            name="and_";
         break;

         case OR: /* logical or, not bitwise */
            name="or_";
         break;
      }
      if(name) {
      /* Push to FS the logical operator: */
         pushWS(name);
         pushFS();
      }
      eval_exp6();
      op=*Token;

      if(name) { 
         if(PTRACE) {
         gprintf("          "); /* 5 */
            gprintf("<< (logical ops got)  Token: %s Token op: %s",
               show(Token,Token_type),name);
            nc();
         }
         if(*(FScount+LEVEL)) {
            if(!strcmp((char *)peekFS(),name)) popFS();
         }
      }
   }
}

void eval_exp6()
/* Process relational operators. 

   Note: Also processed here in addition to relational operators:

       - logical operator NOT, to place it correctly in the recursive
         heirarchy with respect to AND and OR

       - colon (:) and CE (:=), to place it lower in precedence to 
         +,-,/ and * so that: 1:5*2 gives 1,2,3,...,10 and 2+3:9 
         gives 5,6,7,8,9 and infix "1:x0+dx-1" becomes postfix 
         "1 x0 dx + 1 - :" */
{
/* rel_tok+NOT+colon+CE */
   unsigned char idtoks[12]={LT,LE,GT,GE,EQ,NE,NGT,NLT,NOT,':',CE,0};
   unsigned char op;
   char *name,*peek;

   eval_exp7();

   op=*Token;
   while(strchr((char *)idtoks,op) && op) {

      token_get();

      if(PTRACE) {
         gprintf("            "); /* 6 */
         gprintf(">> (rel ops get)  Token: %s Token op: %d",
            show(Token,Token_type),op);
         nc();
      }
      name=NULL;

      switch(op) { 

         case LT:
            name="<";
         break;

         case LE:
            name="<=";
         break;

         case GT:
            name=">";
         break;

         case GE:
            name=">=";
         break;

         case EQ:
            name="=";
         break;

         case NE:
            name="<>";
         break;

         case NGT:
            name="> not";
         break;

         case NLT:
            name="< not";
         break;

         case NOT: /* logical not; ! means not(not equal to zero) */
            name="0<> not";
         break;

         case ':':
            name=":"; 
            if(*(BRKTYP+LEVEL)==INDX) 
            /* Plain INDX becomes INDX_LIST: */
               *(BRKTYP+LEVEL)=INDX_LIST;
         break;

         case CE:
            name="items"; 
            if(*(BRKTYP+LEVEL)==INDX) 
            /* Plain INDX becomes INDX_LIST: */
               *(BRKTYP+LEVEL)=INDX_LIST;
         break;
      }
      if(name) {
      /* Push to FS the relational operator: */
         pushWS(name);
         pushFS();
      }
      eval_exp7();
      op=*Token;

      if(name) {
         if(PTRACE) {
           gprintf("            "); /* 6 */
            gprintf("<< (rel ops got)  Token: %s Token op: %s",
               show(Token,Token_type),name);
            nc();
         }
         if(*(FScount+LEVEL)) {
            if(!strcmp(peek=(char *)peekFS(),"<") || 
               !strcmp(peek,"<=") || 
               !strcmp(peek,">") ||
               !strcmp(peek,">=") ||
               !strcmp(peek,"=") ||
               !strcmp(peek,"<>") ||
               !strcmp(peek,"> not") ||
               !strcmp(peek,"< not") ||
               !strcmp(peek,"0<> not") ||
               !strcmp(peek,":") ||
               !strcmp(peek,"items")
            ) popFS();
         }
      }
   }
}

void eval_exp7()
/* Add or subtract two stack items. */
{
   unsigned char idtoks[3]={'+','-',0},op[2]={0,0};
   char *peek;

   eval_exp8();

   while(strchr((char *)idtoks,*Token) && *Token) {

      *op=*Token;

      token_get();

      if(PTRACE) {
         gprintf("              "); /* 7 */
         gprintf(">> (+ - get)  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      pushWS((char *)op);
      pushFS();

      eval_exp8();

      if(PTRACE) {
         gprintf("              "); /* 7 */
         gprintf("<< (+ - %% got)  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      if(*(FScount+LEVEL)) {
         if(!strcmp((char *)(peek=(char *)peekFS()),"+") || 
            !strcmp((char *)peek,"-")) popFS();
      }
   }
}

void eval_exp8()
/* Multiply or divide two stack items. */
{
   char *name;
   unsigned char idtoks[8]={'*','/','%',MUL_EL,DIV_EL,DIAG_PRE,
      DIAG_POST,0};
   unsigned char op[2]={0,0};

   eval_exp9();

   while(strchr((char *)idtoks,*Token) && *Token) {

      *op=*Token;

      token_get();

      if(PTRACE) {
         gprintf("                "); /* 8 */
         gprintf(">> (* / %% ./ .* get)  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      name=NULL;

      switch(*op) {
         default:
            name=(char *)op;
         break;

         case DIV_EL:
            name="/by";
         break;

         case MUL_EL:
            name="*by";
         break;

         case DIAG_PRE:
            name="diagpre";
         break;

         case DIAG_POST:
            name="diagpost";
         break;

      }
      if(name) {
         pushWS(name);
         pushFS();
      }
      eval_exp9();

      if(name) {
         if(PTRACE) {
            *op=*Token;
            gprintf("                "); /* 8 */
            gprintf("<< (* / %% ./ .* got)  Token: %s op: %s LEVEL: %d",
               show(Token,Token_type),
               show(op,Token_type),LEVEL);
            nc();
         }
         pop_mathFS();
      }
   }
}

void eval_exp9()
/* Process a post-unary operator. */
{
   unsigned char idtoks[4]={'\'',DEC,INC,0}; 

   eval_exp10();

   while(strchr((char *)idtoks,*Token) && *Token) {

      switch(*Token) {
         case '\'':
            pushWS("transpose");
         break;

         case DEC:
            pushWS("-1 over bump"); /* x-- (use and then bump) */
         break;

         case INC:
            pushWS("1 over bump"); /* x++ (use and then bump) */
         break;
      }
      token_get();

      if(PTRACE) {
         gprintf("                  "); /* 9 */
         gprintf(">> (unary post get)  Token: %s",
            show(Token,Token_type));
         nc();
      }
      eval_exp10();

      if(PTRACE) {
         gprintf("                  "); /* 9 */
         gprintf("<< (unary post got)  Token: %s",
            show(Token,Token_type));
         nc();
      }
   }
}

void eval_exp10()
/* Right parenthesis. */
{
   int (*exe)();

   eval_exp11();

   while(*Token==')') {

      if(*(TYP+LEVEL)!=PRN) 
         sntx_err(mprintf("unbalanced parentheses or brackets"));

      while(*(FScount+LEVEL)) {
         popFS(); /* all FS at LEVEL to WS */

      /* Warning: this line for COMMAcount must be inside this loop: */
         *(COMMAcount+LEVEL)=-1; /* COMMAcount undefined if popFS */
      }
      *(EQwarn+LEVEL)=0;

      *(EQU+LEVEL-1)=*(EQU+LEVEL);           /* propagate down */
      *(EQU+LEVEL)=0;
      *(EQU_NAME+LEVEL-1)=*(EQU_NAME+LEVEL); /* propagate down */
      *(EQU_NAME+LEVEL)=NULL;
      *(EQU_DEPTH+LEVEL-1)=*(EQU_DEPTH+LEVEL); /* propagate down */
      *(EQU_DEPTH+LEVEL)=0;

      *(TYP+LEVEL)=NOTDEF;

      LEVEL--;
      if(LEVEL<0) sntx_err(mprintf("unbalanced parentheses"));

   /* Pull down WScount, but leave COMMAcount alone.  The original
      COMMAcount is needed to see if an equate is nested in BRK and
      thus would require parentheses.  When semicolon is processed,
      COMMAcount will be set to uninitialized -1 at all levels. */
      *(WScount+LEVEL)+=*(WScount+LEVEL+1);
      *(WScount+LEVEL+1)=0;

      if(PTRACE) {
         gprintf(" WS contents for right paren are:\n");
         showWS();
      }
   /* Execute the closing "right parenthesis" function on LEVEL for 
      expressions of keywords, such as if(...expression...): */
      if((exe=(RPexe+LEVEL)->exe)) {
         IP_ERR=ip;
         if(PTRACE) gprintf(" Running right paren function\n");
         if(*token_peek(NULL)==';') 
            sntx_err(mprintf("syntax error at ;"));

         exe();
         (RPexe+LEVEL)->exe=NULL;

         _token_get(); /* keyword() has no semicolon; skip the check */
      }
      else {
         if(!LEVEL) _token_get(); /* no semicolon check for LEVEL=0 */
         else token_get();
      }
      if(PTRACE) {
         gprintf("                    "); /* 10 */
         gprintf(">> (right paren get)  Token: %s LEVEL: %d WS: %d",
            show(Token,Token_type),LEVEL,*(WScount+LEVEL));
         nc();
      }
      eval_exp11();

      if(PTRACE) {
         gprintf("                    "); /* 10 */
         gprintf("<< (right paren got)  Token: %s LEVEL: %d",
            show(Token,Token_type),LEVEL);
         nc();
      }
      pop_mathFS();
   }
}

void eval_exp11()
/* Right bracket. */
{
   int args,commas=0,indx;

   eval_exp12();

   while(*Token==']') {

      if(*(TYP+LEVEL)!=BRK) 
         sntx_err(mprintf("unbalanced parentheses or brackets"));

      commas=*(COMMAcount+LEVEL);

      while(*(FScount+LEVEL)) {
         popFS(); /* all FS at LEVEL to WS */

      /* Warning: this line for COMMAcount must be inside this 
         loop: */
         *(COMMAcount+LEVEL)=-1; /* COMMAcount undefined if popFS */
      }
      if(*(BRKTYP+LEVEL)!=MAKE) { 
         if(commas>1)
            sntx_err(mprintf("subscript error: too many commas"));

      /* Doing array column index for fetching or storing. */
         if(*(WScount+LEVEL)>0) { /* something specified */

            if(strcmp((char *)tos->tex,_ROWS)) { 
               if(commas==0) /* row is leading index */
                  pushWS(_ROWS); /* doing rows on LEVEL */
            }
            if(!strcmp((char *)tos->tex,_ROWS)) { 
            /* If WS contains _ROWS, no _COLS specification has 
               been made.  Push "all" before pushing the _COLS 
               specifier: */
               pushWS("-ALL");
               pushWS(_COLS);
            }
            if(strcmp((char *)tos->tex,_COLS)) { 
               pushWS(_COLS); /* doing cols on LEVEL */
            } 
            if(*(FScount+LEVEL))  /* FS(LEVEL) should be empty */
               sntx_err(mprintf("internal FS error"));

         /* Load WS with words for R, and C to be used as input to 
            word place1().

            Currently, WS holds something like these examples:
               "A 3 -R 4 -C" as in A[3,4]
               "A -ALL -R -ALL -C" as in A[*,*]
               "A D 10 reach -R" as in A[reach(D,10)]

            A is on WS(LEVEL-1) and the other words are up one level
            on this WS(LEVEL): */

            if(PTRACE) {
               gprintf(\
                  "WS contents before right bracket processing:\n");
               showWS();
            }
            indx=0;
            args=*(WScount+LEVEL);

         /* Eliminate -R and -C, and substitute "purged" for -ALL,
            and push what is left to FS: */
            while(indx<args) {
               if(!strcmp((char *)tos->tex,_COLS)) {
                  popWS();
               }
               else {
                  if(!strcmp((char *)tos->tex,_ROWS)) {
                     popWS();
                  }
                  else {
                     if(!strcmp((char *)tos->tex,"-ALL")) {
                        popWS();
                        pushWS("purged"); /* means "all" in place1() */
                     }
                     pushFS();
                  }
               }
               indx++;
            }
         /* Prepare WS for running place1: (hA hR hC hB op --- hA1). */
            place_prep(); /* prepare A */

         /* Now move the words for R and C back from FS to WS: */
            while(*(FScount+LEVEL)) popFS();

         /* Use purged B matrix and op=0 to run default place1(): */
            pushWS("purged 0 place1"); /* complete place1 set up */
            
         /* COMMAcount is undefined */
            *(COMMAcount+LEVEL)=-1; 
         }
         *(EQU+LEVEL)=1; /* flag for equate function if it is run */
      }
      if(PTRACE) {
         gprintf("                      "); /* 11 */
         gprintf(\
            ">> (right bracket get)  Token: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),LEVEL,*(BRKTYP+LEVEL));
         nc();
      }
      *(EQU+LEVEL-1)=*(EQU+LEVEL);           /* propagate down */
      *(EQU+LEVEL)=0;
      *(EQU_NAME+LEVEL-1)=*(EQU_NAME+LEVEL); /* propagate down */
      *(EQU_NAME+LEVEL)=NULL;
      *(EQU_DEPTH+LEVEL-1)=*(EQU_DEPTH+LEVEL); /* propagate down */
      *(EQU_DEPTH+LEVEL)=0;

      *(EQwarn+LEVEL)=0;
      *(TYP+LEVEL)=NOTDEF;
      *(BRKTYP+LEVEL)=NOTDEF;

      LEVEL--;

   /* Pull WScount down to this lower LEVEL: */
      *(WScount+LEVEL)+=*(WScount+LEVEL+1);
      *(WScount+LEVEL+1)=0;

      if(LEVEL<0)
         sntx_err(mprintf("unbalanced brackets or parentheses"));

      if(PTRACE) {
         gprintf("WS contents after right bracket processing:\n");
         showWS();
      }
      token_get();

      eval_exp12();

      if(PTRACE) {
         gprintf("                      "); /* 11 */
         gprintf(\
            "<< (right bracket got)  Token: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),LEVEL,*(BRKTYP+LEVEL));
         nc();
      }
      pop_mathFS();
   }
}

void eval_exp12()
/* Raise a number to a power. */
{
   unsigned char idtoks[3]={'^',POW_EL,0};
   unsigned char op[2]={0,0};
   char *name,*peek;

   eval_exp13();

   while(strchr((char *)idtoks,*Token) && *Token) {

      *op=*Token;

      token_get();

      if(PTRACE) {
         gprintf("                        "); /* 12 */
         gprintf(">> (power get)  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      name=NULL;

      switch(*op) {
         default:
            name=(char *)op;
         break;

         case POW_EL:
            name="^by";
         break;

      }
      if(name) {
         pushWS(name);
         pushFS();
      }
      eval_exp13();

      if(PTRACE) {
         *op=*Token;
         gprintf("                        "); /* 12 */
         gprintf("<< (power got)  Token: %s op: %s LEVEL: %d",
            show(Token,Token_type),show(op,Token_type),LEVEL);
         nc();
      }
      if(*(FScount+LEVEL)) {
         if(!strcmp(peek=(char *)peekFS(),"^") ||
            !strcmp(peek,"^by")
         ) popFS();
      }
   }
}

void eval_exp13()
/* Process a pre-unary operator, the - in -A or the + in +5. */  
{
   char *name;
   unsigned char idtoks[4]={NEG,POS,'@',0},op; /* rel_toks */

   eval_exp14();

   op=*Token;
   while(strchr((char *)idtoks,op) && op) {

      token_get();

      if(PTRACE) {
         gprintf("                          "); /* 13 */
         gprintf(">> (unary pre get)  Token: %s Token op: %d LEVEL: %d",
            show(Token,Token_type),op,LEVEL);
         nc();
      }
      name=NULL;

      switch(op) { 

         case NEG:
            name="negate";
         break;

         case POS:
            name="noop"; /* the + sign has no operation */
         break;

         case '@':
            name="@";
         break;
      }
      if(name) {
      /* Push to FS the operator: */
         pushWS(name);
         pushFS();
      }
      eval_exp14();
      op=*Token;

      if(name) {
         if(PTRACE) {
            gprintf("                          "); /* 13 */
            gprintf(
               "<< (unary pre got)  Token: %s Token op: %s LEVEL %d",
               show(Token,Token_type),name,LEVEL);
            nc();
         }
         if(*(FScount+LEVEL)) {
            if(!strcmp((char *)peekFS(),name)) popFS();
         }
      }
   }
}

void eval_exp14()
/* Process the transpose operator. */
{
   char idtoks[2]={'\'',0}; 

   eval_exp15();

   while(strchr((char *)idtoks,*Token) && *Token) {

      switch(*Token) {
         case '\'':
            pushWS("transpose");
         break;
      }
      token_get();

      if(PTRACE) {
         gprintf("                            "); /* 14 */
         gprintf(">> (transpose get)  Token: %s",
            show(Token,Token_type));
         nc();
      }
      eval_exp15();

      if(PTRACE) {
         gprintf("                            "); /* 14 */
         gprintf("<< (transpose got)  Token: %s",
            show(Token,Token_type));
         nc();
      }
   }
}

void eval_exp15()
/* Right brace. */
{
   int (*exe)();

   eval_exp16();

   while(*Token=='}') {

      if(*(TYP+LEVEL)!=BRC) sntx_err(mprintf("unbalanced braces"));

      while(*(FScount+LEVEL)) {
         popFS(); /* all FS at LEVEL to WS */

      /* Warning: this line for COMMAcount must be inside this loop: */
         *(COMMAcount+LEVEL)=-1; /* COMMAcount undefined if popFS */
      }
   /* Execute the closing "right brace" function for LEVEL: */
      if((exe=(RBexe+LEVEL)->exe)) {
         if(PTRACE) gprintf(" Running right brace function\n");
         exe();
         (RBexe+LEVEL)->exe=NULL;
      }
      token_get();

      if(PTRACE) {
         gprintf("                              "); /* 15 */
         gprintf(\
            ">> (right brace get)  Token: %s LEVEL: %d",
            show(Token,Token_type),LEVEL);
         nc();
      }
      *(EQwarn+LEVEL)=0;
      *(TYP+LEVEL)=NOTDEF;
      *(BRKTYP+LEVEL)=NOTDEF;

      LEVEL--;
      if(LEVEL<0)
         sntx_err(
            mprintf("unbalanced brackets, parentheses or braces"));

      eval_exp16();

      if(PTRACE) {
         gprintf("                              "); /* 15 */
         gprintf("<< (right brace got)  Token: %s LEVEL: %d",
            show(Token,Token_type),LEVEL);
         nc();
      }
   }
}

void eval_exp16()
/* Left parenthesis. */
{
   eval_exp17();

   while(*Token=='(') {

      if(Token_type_was==FUNCTION || Token_type_was==VARIABLE) {
         pushFS(); /* name now on WS moved to FS */

      /* Lifting function or variable name to the next FS level. 

         For variables, this allows a post unary operator like ' to 
         be applied to a matrix before it is booked.  The phrase that 
         books now precedes the variable name on FS.  But after this 
         point the phrase that books will be last on FS and the name 
         will be first on FS+1: */
         (*(FScount+LEVEL+1))++; /* name is 1st on this level */
         (*(EQwarn+LEVEL+1))++;

         (*(FScount+LEVEL))--; /* one less item on former level */
         (*(EQwarn+LEVEL))--;
      }
      LEVEL++; /* to FS+1 */
      *(TYP+LEVEL)=PRN;
      *(EQU+LEVEL)=0;

      *(BRKTYP+LEVEL)=0; /* not a BRK type of level */
      *(COMMAcount+LEVEL)=0; /* start at zero */
      *(WScount+LEVEL)=0; /* WScounts start over again */

      token_get();

      if(PTRACE) {
         gprintf("                                "); /* 16 */
         gprintf(">> (left paren get)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
      eval_exp17();

      if(PTRACE) {
         gprintf("                                "); /* 16 */
         gprintf("<< (left paren got)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
   }
}

void eval_exp17()
/* Left bracket. */
{
   eval_exp18();

   while(*Token=='[') {

      LEVEL++;
      *(TYP+LEVEL)=BRK;

   /* Decide if case of [ ... ] is making a matrix (MAKE) or making
      indices for fetching from, or storing to, a matrix (INDX): */

      switch(Token_type_was) {

         default:
            *(BRKTYP+LEVEL)=MAKE;
         break;

         case VARIABLE:
         case FUNCTION:
            *(BRKTYP+LEVEL)=INDX;
         break;

         case PUNCTUATION:
            switch(*Token_was) {

               default:
                  *(BRKTYP+LEVEL)=MAKE;
               break;

               case '=':
                  *(BRKTYP+LEVEL)=MAKE;
               break;

               case ']':
               case ')':
                  *(BRKTYP+LEVEL)=INDX;
               break;
            }
         break;

         case OPERATOR:
            switch(*Token_was) {

               default:
                  *(BRKTYP+LEVEL)=MAKE;
               break;

               case '\'':
                  *(BRKTYP+LEVEL)=INDX;
               break;
         }
      }
      if(*(EQU_NAME+LEVEL)) {
         free(*(EQU_NAME+LEVEL));
         *(EQU_NAME+LEVEL)=NULL;
      }
      *(EQU_DEPTH+LEVEL)=0;

      *(COMMAcount+LEVEL)=0; /* start at zero */
      *(WScount+LEVEL)=0; /* WScounts start over again */
      *(EQU+LEVEL)=0;

      token_get();

      if(*(BRKTYP+LEVEL)==MAKE) {
         pushWS("hand"); /* turn NUM into 1-by-1 MAT, STR into VOL */
         pushFS();
      }
      else { /* *(BRKTYP+LEVEL)==INDX) */
      /* If this token is a * (meaning take "all" rows), get the next
         token: */
         if(*Token=='*') {
            pushWS("-ALL");
            token_get();
         }
      }
      if(PTRACE) {
         gprintf("                                  "); /* 17 */
         gprintf(\
            ">> (left bracket get)  Token: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),LEVEL,*(BRKTYP+LEVEL));
         nc();
      }
      eval_exp18();

      if(PTRACE) {
         gprintf("                                  "); /* 17 */
         gprintf(\
            "<< (left bracket got)  Token: %s LEVEL: %d BRKTYP: %d",
            show(Token,Token_type),LEVEL,*(BRKTYP+LEVEL));
         nc();
      }
   }
}

void eval_exp18()
/* Left brace. */
{
   eval_exp19();

   while(*Token=='{') {

      token_get();

      LEVEL++;
      *(TYP+LEVEL)=BRC;
      *(EQwarn+LEVEL)=0; /* no equate warnings on this level */

      *(COMMAcount+LEVEL)=0; /* start at zero */
      *(WScount+LEVEL)=0; /* WScounts start over again */

      if(DEF_PENDING && LEVEL>1 && KWcount==0) {
         pushWS("["); /* starting postfix bracket mode in DEFINE */

      /* Assigning the function on the closing } that will push right 
         bracket, ], to WS: */
         RBexe[LEVEL].exe=(int (*)()) exe_bracket_end;
      }
      if(PTRACE) {
         gprintf("                                    "); /* 18 */
         gprintf(\
            ">> (left brace get)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
      eval_exp19();

      if(PTRACE) {
         gprintf("                                    "); /* 18 */
         gprintf(\
            "<< (left brace got)  Token: %s LEVEL: %d COMMAS: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL));
         nc();
      }
   }
}

void eval_exp19()
/* Equate.

   Forms like these can have multiple LHS args and correspond to 
   PLACE=0 and apply only for the = symbol:

      A = random(10,10);
      (x, y, z) = (f(), g(), h());

   For more than one LHS argument, only operator = is allowed, but
   other equate operators can appear on the RHS as in:

      U = 10; (x, y, z) = (12, (U+=4), 16);


   Forms like these correspond to PLACE=1 and apply for symbols
   =, +=, -=, *= and /=:

      A += random(10,10);
      a[1:10] = [1:10];
      a[1:10] += [1:10];

   For PLACE=1 types, only one LHS argument is allowed; for example 
   this will not work: 

      (x, y) += 10;


Examples showing parentheses needed around items in RHS list.

>> (x,y)=(99,z=10);
line 1.12: parentheses required around entire nested equate
(x,y)=(99,z=10);
          ^^^
Parentheses around the entire equate z=10 are required, like this:
>> (x,y)=(99,(z=10)); x y z

 stack elements:
       0 number: z  10
       1 number: y  10
       2 number: x  99
 [3] ok!
>>

>> main(parse("(x,y,z)=(3,((u,v)=(4,5)));")) x y z u v

 stack elements:
       0 number: v  5
       1 number: u  4
       2 number: z  5
       3 number: y  4
       4 number: x  3
 [5] ok!
>>

This gives a warning and a fatal error:
>> main(parse(".m(x=[random(4,2),z=10*random(4,2)]);"))
line 1.5: warning: parentheses recommended around entire nested equate
.m(x=[random(4,2),z=10*random(4,2)]);
   ^^^
line 1.20: parentheses required around entire nested equate
.m(x=[random(4,2),z=10*random(4,2)]);
                  ^^^
After fixing:
>> main(parse(".m((x=[random(4,2),(z=10*random(4,2))]));"))
 Row 1:   0.3204   0.3238    7.313    9.322
 Row 2:   0.9697    0.807    3.298    7.337
 Row 3:   0.7303   0.5616    8.965    1.322
 Row 4:   0.4562   0.5675    6.335    1.613
>> */
{
   unsigned char idtoks[6]={'=',INC_EQU,DEC_EQU,MPY_EQU,DIV_EQU,0};
   unsigned char op[2]={0,0},*p1,*p2;
   int args,commas,k,PLACE=0;
   double d;
int len;

   atom();

   while(strchr((char *)idtoks,*Token) && *Token) {

      switch((unsigned char)*Token) {

         case '=': 
            *op='0';
         break;

         case INC_EQU: 
            *op='1';
         break;

         case DEC_EQU: 
            *op='2';
         break;

         case MPY_EQU: 
            *op='3';
         break;

         case DIV_EQU: 
            *op='4';
         break;

      }
      PLACE=(*(EQU+LEVEL)==1); /* WS(LEVEL) is set up to run place1() */

      if(!PLACE && *op>'0') { /* op>0 is for all op types except = */
      /* For cascading equates (of type op>0), this branch performs 
         steps similar to right bracket processing, building the WS 
         for word place1: place1 (hA hR hC hB op --- hA1).

         Set up the WS according to the place1 stack diagram above: */

      /* Prepare A currently on WS: */
         place_prep(); 

      /* Next, put purged R and C on WS: */
         pushWS("purged");
         pushWS("purged");

      /* Finally, define purged B and op=0 to run default place1() (up-
         coming equate processing will make this what it is supposed to
         be): */
         pushWS("purged 0 place1");

      /* COMMAcount is undefined */
         *(COMMAcount+LEVEL)=-1;

         *(EQU+LEVEL)=1; /* flag for equate function */
         PLACE=1;
      }
      token_get();

      if(*Token==';' || Token_type==FINISHED) {
         sntx_err(
            mprintf("equate has no right hand side"));
      }
      if(PLACE) { /* WS is set up to store terms into an array 

         PLACE is set to 1 when running something like:

            Kgg[2:7,3:5] = random(6,3)+10 

         to place terms into a LHS array.  The right bracket function
         has prepared the stack and set *(EQU+LEVEL) to 1.  A little
         more work has to be done in this branch. */

         if(PTRACE) {
            gprintf("equate PLACE operation 1.  WS contents are:\n");
            showWS();
         }
      /* The topmost stack item must read: "purged 0 place1."  But extra
         surrounding brackets, [...], may have put word "hand" on the 
         stack.  Word hand is unnecessary, so drop any references from 
         WS before checking for "purged 0 place1:" */
         while(!strcmp((char *)tos->tex,"hand")) popWS();

         if(strcmp((char *)tos->tex,"purged 0 place1")) {
            sntx_err(
               mprintf("invalid tokens for equate preceding ="));
         }
         popWS(); /* "purged 0 place1" not needed when equate */

      /* When do equate, RHS is pulled from temp stack:*/
         pushWS("pull"); 

      /* Push equate name to WS and free its allocated memory: */
         pushWS(*(EQU_NAME+LEVEL)); /* push equate name */
         free(*(EQU_NAME+LEVEL)); /* ok to free now */
         *(EQU_NAME+LEVEL)=NULL;

         if(PTRACE) {
            gprintf("equate PLACE operation 2.  WS contents are:\n");
            showWS();
         }
      }
      if(Token_type==KEYWORD) {
         sntx_err(
            mprintf("keyword %s is an invalid token for equate",
               show(Token,Token_type)));
      }
      if(*(COMMAcount+LEVEL)>0 || *(TYP+LEVEL)==BRK) {
      /* Cases requiring parentheses around nested equate:
            (x,y)=(1,z=10)
            x=[99,z=10];

         Example:
            >> x=[99,z=10];
            line 1.8: parentheses required around entire nested equate
            x=[99,z=10];
                  ^^^
      */
         sntx_err(mprintf(\
            "parentheses required around entire nested equate"));
      }
      if(*(TYP+LEVEL)==PRN && LEVEL>0 && *(EQwarn+LEVEL) &&
         *token_peek(NULL)!=')') {
         sntx_warn(mprintf(\
         "warning: parentheses recommended around entire nested equate"
         ));
      }
      if(*(FScount+LEVEL) && !strcmp((char *)peekFS(),"negate")) {
         sntx_warn(mprintf(\
         "warning: parentheses recommended around entire negated equate"
         ));
      }
      args=*(WScount+LEVEL);
      commas=MAX(0,*(COMMAcount+LEVEL));

   /* Preparing the equate left hand side (LHS). 

      Add WS and COMMA counts at levels above LEVEL to WS counts and
      COMMA counts at LEVEL level, and zero the counts above.  Note
      that right parenthesis processing brings WScount down to the
      lower level but leaves COMMAcount at the higher.  This loop
      brings everything to LEVEL. */

      k=LEVEL+1;
      while(k<NBUF) {
         if(*(WScount+k) || *(COMMAcount+k)>0) {

            commas+=MAX(0,*(COMMAcount+k));
            *(COMMAcount+k)=-1; /* uninitialize counts at level k */

            args+=*(WScount+k);
            *(WScount+k)=0; /* zero the counts at level k*/
         }
         k++;
      }
   /* Now all WS and COMMA counts are at level LEVEL: */
      *(WScount+LEVEL)=args; /* "stack depth" at this LEVEL level */
      *(COMMAcount+LEVEL)=commas; /* commas between listed items */

   /* The number of LHS args in the equate is equal to 1 plus the
      number of commas counted: */
      args=1+commas;

      if(PTRACE) {
         gprintf("                                      "); /* 19 */
         gprintf
          (">> (equate get)  Token: %s LEVEL: %d COMMAS: %d args: %d",
            show(Token,Token_type),LEVEL,commas,args);
         gprintf(" TYP: %d",*(TYP+LEVEL));
         nc();
      }
      if(*(WScount+LEVEL)<args) {
         sntx_err(
            mprintf("insufficient LHS names for %d arguments",args));
      }
      for(k=0;k<args;k++) {
      /* Push copy of output arg name from WS to FS: */
         pushint(k);
         pick();
         (*(WScount+LEVEL))++; /* bump WS count due to pick() */

      /* Name cannot be a number: */
         if(number(tos->tex,&d))
            sntx_err(\
            mprintf("number %s is invalid token on left side of =",
            tos->tex));

      /* Cannot be a function unless inside a definition: */
         if(!PLACE && (!DEF_PENDING && is_function(tos->tex))) {
           sntx_err(mprintf(\
              "function %s is invalid token on left side of =",tos->tex
           ));
         }
      /* Cannot be an input argument name to a definition: */
         if(is_arg_in(tos->tex)) {
            sntx_err(\
            mprintf("equate name cannot match input argument name %s",
            tos->tex));
         }
         pushFS(); /* push to FS, pop WS (and decrement count) */
      }
   /* Reverse order on the stack of the args for catalog assignment: */
      if(!PLACE) {
         pushint(args); /* how many to reverse */
         revn();
      }
   /* Empty WS by pushing reversed-order catalog assignment commands
      to FS: */
      for(k=0;k<args;k++) {
         if(*tos->tex=='\t') {
         /* When tos text is of form \tA.B\t, want to bank B into lib 
            of A.  Function atom() flagged this form by prepending and 
            appending \t to A.B: */

            len=stradv(tos->tex+1,"\t")-1; /* len of A.B */
            if(len>NTOK) sntx_err(mprintf("string A.B is too long"));

            memcpy((char *)Token_peek,tos->tex+1,len);
            *(Token_peek+len)='\0';

            p1=(unsigned char *)strtok((char *)Token_peek,".");
            p2=p1;
            p2=(unsigned char *)strtok('\0',".");

         /* Make phrase to bank to lib of a word: */
            drop();
            pushstr((char *)p1); /* p1 = lib = word name */
            quoted1();
            spaced();
            pushstr((char *)p2); /* p2 = name to bank into word's lib */
            quoted1();
            spaced();

            pushstr("bank* "); /* command to assign to lib of word */
            cat();
            cat();
         }
         else {
         /* Make phrase to book to lib: */
            quoted1(); /* quoted name to book lib */
            pushstr(" book* "); /* command to assign to lib */
            cat();
         }
         if(PLACE) {
         /* Run "op place1" before booking to lib */
            pushstr((char *)op);
            pushstr(" place1 ");
            cat();
            swap();
            cat();
            if(PTRACE) {
               gprintf("equate PLACE operation 3.  WS contents are:\n");
               showWS();
            }
         }
         pushFS(); /* push WS to FS, pop WS */
      }
      if(PLACE) {
         while(*(WScount+LEVEL) && _depth!=*(EQU_DEPTH+LEVEL)) pushFS();
         pushWS("push"); /* push RHS to temp stk for later pull */
         pushFS();
      }
   /* All LHS items and assignment commands have been pushed to FS,
      and WScount=0. */
      *(COMMAcount+LEVEL)=-1; /* commas now uninitialized */
      *(EQU+LEVEL)=0; /* zero the flag set by right bracket function */
      PLACE=0;
      *op=0;

      atom();
 
      if(PTRACE) {
         gprintf("                                      "); /* 19 */
         gprintf
            ("<< (equate got)  Token: %s LEVEL: %d COMMAS: %d args: %d",
            show(Token,Token_type),LEVEL,*(COMMAcount+LEVEL),
               args);
         gprintf(" TYP: %d",*(TYP+LEVEL));
         nc();
      }
   }
}

void eval_init() 
/* Initialize before parsing an expression. */
{
   int k=0;

   LEVEL=0;

   clrfs(); /* clears the function stack */
   initialFS=depthFS; /* function stack */

   memset(FScount,0,NBUF); /* zero function stack count */
   memset(WScount,0,NBUF*sizeof(int)); /* zero working stack count */
   memset(EQU,0,NBUF); /* zero the equate flag */
   memset(EQwarn,0,NBUF); /* zero equate warning */

   for(k=0;k<NBUF;k++) *(COMMAcount+k)=-1; /* signed comma count */

   for(k=0;k<NBUF;k++) *(EQU_NAME+k)=NULL;
   memset(EQU_DEPTH,0,NBUF*sizeof(int)); /* zero abs stk depth */
   for(k=0;k<NBUF;k++) (RPexe+k)->exe=NULL;
   for(k=0;k<NBUF;k++) (SEMIexe+k)->exe=NULL;

   *Token='\0';
   Token_type=UNSET;

   *Token_was='\0';
   Token_type_was=UNSET;
}

int exe_bracket_end()
/* This function is run at the closing right brace to provide ] that
   ends bracket mode in a postfix definition. */
{
   if(PTRACE) gprintf(" Running exe_bracket_end()\n");
   return(pushWS("]")); /* ending postfix bracket mode in DEFINE */
}

int exe_def_init()
/* This function is run at the closing right parenthesis of the
   input argument list of a definition. */
{
   if(PTRACE) gprintf(" Running exe_def_init()\n");
   def_init(1); /* final entry to this function */
   return 0;
}

int exe_else()
/* This function is run from atom() at KEYWORD else.  

   Push ELSE to WS and assign the function to run at the semicolon that
   ends the upcoming expression that follows else.  Count KWcount stays
   the same. */
{
   if(PTRACE) {
      gprintf("  ");
      gprintf("## (exe_else)  Token: %s KWcount: %d LEVEL: %d",
         show(Token,Token_type),KWcount,LEVEL);
      nc();
   }
   pushWS("ELSE");
   SEMIexe[KWcount].exe=(int (*)()) exe_then;

   return 0;
}

int exe_if()
/* This function is run at the closing right parenthesis of an if()
   expression.  

   Bump KEYWORD count, KWcount, assign the function to run at the 
   semicolon that ends the upcoming expression that follows if(), 
   and push IF to WS. */
{
   KWcount++;
   SEMIexe[KWcount].exe=(int (*)()) exe_then;

   if(PTRACE) {
      gprintf("  ");
      gprintf("## (exe_if)  Token: %s KWcount: %d LEVEL: %d",
         show(Token,Token_type),KWcount,LEVEL);
      nc();
   }
   pushWS("IF");
   return 0;
}

int exe_then()
/* This function is run at token semicolon during if() branch pro-
   cessing.  It may push THEN to WS to end an IF ... ELSE ... THEN 
   structure, and decrement KEYWORD count, KWcount. */
{
   unsigned char *peek;
   int peek_type;

   if(!KWcount) return 0;

/* Test for case of: else if(... */
   if(!strcmp((char *)Token_was,"else") && 
      !strcmp((char *)Token,    "if"  )) {
      if(PTRACE) {
         gprintf("  ");
         gprintf("## (exe_then)  Token: %s KWcount: %d LEVEL: %d",
            show(Token,Token_type),KWcount,LEVEL);
         nc();
      }
      return 0;
   }
   peek=token_peek(&peek_type); /* peek at next token */

   if(PTRACE) {
      gprintf("  ");
      gprintf("## (exe_then)  Token_peek: %s KWcount: %d LEVEL: %d",
         show(peek,peek_type),KWcount,LEVEL);
      nc();
   }
/* If next token is not else, push THEN to WS and decrement KWcount: */
   if(strcmp((char *)peek,"else")) {
      pushWS("THEN");

      (SEMIexe+KWcount)->exe=NULL;
      KWcount--;
   }
   return 0;
}

int finish() /* (hT --- qS) */
/* Finish the parsed text VOL T that is now on the stack.  

   Replace strings in T that match patterns in table EQUIV at rows 1,3,
   5,..., with substitution strings at rows 2,4,6,..., respectively.

   Odd numbered strings in table EQUIV must be unique.

   Turn T into STR S with NL (new line, 0xA) at the end of each 
   line; NL is important for verbatim postfix text being returned,
   so backslash (\, comment to end of a line) works correctly. */
{
   patitem *TABLE;
   int k=1,rows;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" finish: ",STRORVOLNOT);
      return 0;
   }
   TABLE=*(EQUIV+onbuf); /* originally set up by patnew() in parse() */

   if((rows=TABLE->use)) { /* patitem TABLE starts at k=1, not k=0 */
      for(;k<rows;k++) {

         pushstr((char *)patget(k,TABLE)); /* orig string */

         k++;
         pushstr((char *)patget(k,TABLE)); /* new to replace orig */

         strp(); 
      }
   }
   noblanklines();
   return(textput());
}

int is_arg_in(char *s)
/* Return 1 if s matches an input argument name. */
{
   register int k;

   if(!DEF_PENDING) return 0;

   for(k=0;k<args_in;k++)
      if(!strcmp(*(input_args+k),s)) return 1;
    
   return 0;
}

int is_constant(char *s)
/* Determine if identifier s is a constant.  Return 1 if s is a 
   constant, and 0 otherwise. */
{
/* Constants, such as pi, already exist and are type CONS in the 
   CODE__ region.  Return 1 if this is true. */

/* Native, constant, define, or inline are all in cat lib region CODE__,
   but type constant (like pi) is a numerical constant: */
   return(caton(tagged(s,"CODE__")) && oncat->typ==CONS);
}

int is_function(char *s)
/* Determine if identifier s is a function.  Return 1 if s is a 
   function, and 0 otherwise. */
{
/* Functions already exist and are in the CODE__ region.  Return 1
   if this is true. */

/* Native, constant, define, or inline are all in cat lib region CODE__,
   but type constant (like pi) is not a function: */
   return(caton(tagged(s,"CODE__")) && oncat->typ!=CONS);
}

int is_keyword()
/* See if Token is in C_table: */
{
   register int i;

   for(i=0;*C_table[i].command;i++) {
      if(!strcmp(C_table[i].command,(char *)Token))
         return C_table[i].tok; /* returning internal id, tok_int */
   }
   return 0; /* unknown command */
}

int is_var(char *s)
/* Determine if identifier s is a variable or could become a variable.

   Identifier s is a variable, or can become a variable, if it is not
   in the CODE__ region and it does not look like a number.  Return 1
   if this is true. */
{
   double d;
/* Native, constant, define, or inline (all are in CODE__): */
   return(!caton(tagged(s,"CODE__")) && !number(s,&d));
}

int isblank1(char c)
/* Return true if c is a space or a tab. */
{
   if(c==' ' || c=='\t') return 1;
   return 0;
}

int isdelim1(char c)
/* Return true if c is a delimiter. */
{
   if(strchr(" !;,+-<>'/%*^=()[]{}:",c) || c==9 || c=='\n' || c=='\0')
      return 1;
   return 0;
}

int lpush1() /* pushfs (x y --- x) */
/* Push quote from working stack, WS, to the function stack, FS,
   and pop WS. */
/* Note: This function is made from a copy of pushfs(), stk.c. */

/* Moves top of stack item to top of function stack.  The item's stack
   count, cnt, is unchanged.  */
{
   if(tosfun==(stkfunction+DEPFUNSTK)) {
      sntx_err(mprintf("FS full at token %s",
         show(Token,Token_type)));
      return 0;
   }
   if(tos==stack) {
      sntx_err(mprintf("cannot pop empty WS at token %s",
         show(Token,Token_type)));
      return 0;
   }
   tosfun++; /* adding to function stack */
   memcpy(tosfun,tos,sizeof(stkitem));
   tos--; /* removing from stack, as drop() does */

   return 1;
}

int parse() /* parse (qInfix --- qPostfix) */
/* Convert infix expressions into postfix text for running.

   This word is reentrant and preserves the INFIX flag that is present
   on entry. */
{
   char *s[NBUF];
   int err[NBUF],f=0,initFS[NBUF],k;
   int infix_save[NBUF],len[NBUF];

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" parse: ",STRORVOLNOT);
      return 0;
   }
   if((*(s+onbuf)=(char *)memgetnNL(
      tos->tex,tos->row,tos->col,(len+onbuf)))==NULL) 
      return 0;

   if(PTRACE) {
      gprintf("parse input: "); dup1s(); dot(); nc();
   }
   if(bufjump(*(s+onbuf),*(len+onbuf))) { /* to next run level */
      drop(); /* ok to free tos.tex because words are in s */
      *(infix_save+onbuf)=INFIX;
      *(err+onbuf)=stkerrcount;
      infix(); /* to infix mode */

      *(d0+onbuf)=stkdepth();
      setjmp(*(jmpenv+onbuf)); /* <<-- longjmp lands here */

      if(!*(jmpready+onbuf)) { /* falls through first time */
         *(jmpready+onbuf)=ENDSOU;

         parse_init();

         *(EQUIV+onbuf)=(patitem *)patnew(32,4,PATWID);

         *(initFS+onbuf)=initialFS;
         while(ip+1<ipend && Token_type!=FINISHED) eval_exp0();
      }
      if(*(infix_save+onbuf)) infix(); else postfix();

      if(depthFS-*(initFS+onbuf)) {
         if((depthFS-*(initFS+onbuf)>0)) {
            gprintf(" parse: function stack is not empty; depth = %d",\
               depthFS-*(initFS+onbuf));
            nc();
         /* Pop the topmost FS item and show it: */
            LEVEL=0;
            while(*(FScount+LEVEL)==0) LEVEL++;
            popFS();
            gprintf(" parse: topmost FS item is %s at level %d\n",
               tos->tex,LEVEL);
            popWS();
            nc();
         }
         else {
            gprintf(" parse: cannot pop function stack");
            nc();

         /* Sat Jan 28 07:43:43 PST 2012.  If PTRACE is on, abort the
            program here: */
            if(!PTRACE) {
               gprintf(" parse: turn on ptrace to abort here");
               nc();
            }
            else {
               gprintf(" parse: aborting program");
               nc();
               pushint(1);
               abort1();
            }
         }
         clearFS();
         stkerr("","");
      }
      if(stkdepth()>*(d0+onbuf)) {
         pushint(stkdepth()-*(d0+onbuf));
         pilen();
         finish();
      }
      else pushstr(""); /* return empty string at least */

      if(EQUIV+onbuf) {
         mallfree((void *) *(EQUIV+onbuf)); /* first p->pat */
         mallfree((void *) (EQUIV+onbuf));  /* then patitem p itself */
      }
      if(PTRACE) {
         dup1s();
         asciify();
         strchop();
         gprintf("parse output: %s",show((unsigned char *)tos->tex,-1));
         nc();
         drop();
      }
      f=(*(err+onbuf)==stkerrcount);
      bufreturn(); /* s is freed in bufreturn() */
   }
   for(k=0;k<NBUF;k++) {
      if(*(EQU_NAME+k)) free(*(EQU_NAME+k));
      *(EQU_NAME+k)=NULL;
   }
   for(k=0;k<args_in;k++) {
      if(*(input_args+k)) free(*(input_args+k));
      *(input_args+k)=NULL;
   }
   return f;
}

int parse1() /* parse1 (qInfix --- qPostfix) */
/* This function receives Infix keyboard input, runs the parser, and
   delivers Postfix output.  It is called by word console in key.v. */
{
   int ret=0;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" parse1: ",STRORVOLNOT);
      return 0;
   }
   chop();

/* If the leftmost chars are <<, switch to postfix and return to run
   the postfix text that follows: */
   if(*tos->tex=='<' && *(tos->tex+1)=='<') {
      cop();
      *(tos->tex)=' ';
      *(tos->tex+1)=' ';
      return(postfix());
   }
   if(*tos->tex=='>' && *(tos->tex+1)=='>') {
      cop();
      *(tos->tex)=' ';
      *(tos->tex+1)=' ';
      infix();
      chop();
   }
   if(!tos->col) return 1;
   if(!INFIX) return 1;

   ret=parse(); /* run the parser */

   if(INFIX) /* running parse() may have changed INFIX flag */
      return(ret && infix());
   else 
      return(ret && postfix());
}

int parse_driver() /* parse_driver (qFile nTask --- hT) */
/* Run parsing tasks using File for input.

   Task=0: Token display.
      Read tokens from File and display their types.

   Task=1: Parsing.
      Parse phrases in file named on the stack and return VOL T 
      suitable for postfix running.

   Task=2: Show enumerations (run "anyfilename" 2 parse_driver): 

Example: running Task 2:
>> parse_driver("prs.o",2);
 LT: 128
 LE: 129
 GT: 130
 GE: 131
 EQ: 132
 NE: 133
 NGT: 134
 NLT: 135
 AND: 136
 OR: 137
 NOT: 138
 LAST_LOG_TOK: 139
 BRK_SEMI: 140
 BRK_COMMA: 141
 NEG: 142
 POS: 143
 INC: 144
 DEC: 145
 INC_EQU: 146
 DEC_EQU: 147
 MPY_EQU: 148
 DIV_EQU: 149
 MUL_EL: 151
 DIV_EL: 150
 POW_EL: 152
 DIAG_PRE: 153
 DIAG_POST: 154
 CE: 155
*/
{
   char *filename,*s;
   int d0,err,f,infix_save,TASK=-1;

   if(!popint(&TASK)) return 0;

   if(tos->typ!=STR && tos->typ!=VOL) {
      stkerr(" parse_driver: ",STRORVOLNOT);
      return 0;
   }
   strchop();
   filename=memgetn(tos->tex,tos->col); /* first line for file name */

   if(PTRACE) {
      gprintf(" Parsing file %s",filename);
      nc();
   }
   filefound(); /* add path */
   popint(&f); /* popping found flag from stack */
   if(!f) {
      gprintf(" parse_driver: file %s not found",filename);
      nc();
      stkerr("","");
      mallfree((void *)&filename);
      return 0;
   }
   if((s=fileload(tos->tex))==NULL) { /* tos->tex has path/filename */

      mallfree((void *)&filename);
      return 0;
   }
   drop(); /* qFile off stack */

   if(bufjump(s,strlen(s))) { /* to next run level */
      err=stkerrcount;

      f=setjmp(*(jmpenv+onbuf)); /* <<-- longjmp lands here */

      if(!*(jmpready+onbuf)) { /* falls through first time */
         *(jmpready+onbuf)=ENDSOU;

         parse_init();

         switch(TASK) {

            case SHOW_TOKENS:
               infix_save=INFIX;
               infix();

               show_tokens();

               if(infix_save!=INFIX) postfix();
            break;

            case EVAL_EXP:
               d0=stkdepth();
               infix_save=INFIX;
               infix();

               eval_init();

               while(ip+1<ipend && Token_type!=FINISHED) {
                  eval_exp0();
               }
               if(infix_save!=INFIX) postfix();

               pushint(stkdepth()-d0);
               pilen();
            break;

            case SHOW_ENUM:
               gprintf(" LT: %d\n",LT);
               gprintf(" LE: %d\n",LE);
               gprintf(" GT: %d\n",GT);
               gprintf(" GE: %d\n",GE);
               gprintf(" EQ: %d\n",EQ);
               gprintf(" NE: %d\n",NE);
               gprintf(" NGT: %d\n",NGT);
               gprintf(" NLT: %d\n",NLT);

               gprintf(" AND: %d\n",AND);
               gprintf(" OR: %d\n",OR);
               gprintf(" NOT: %d\n",NOT);
               gprintf(" LAST_LOG_TOK: %d\n",LAST_LOG_TOK);

               gprintf(" BRK_SEMI: %d\n",BRK_SEMI);
               gprintf(" BRK_COMMA: %d\n",BRK_COMMA);

               gprintf(" NEG: %d\n",NEG);
               gprintf(" POS: %d\n",POS);

               gprintf(" INC: %d\n",INC);
               gprintf(" DEC: %d\n",DEC);

               gprintf(" INC_EQU: %d\n",INC_EQU);
               gprintf(" DEC_EQU: %d\n",DEC_EQU);
               gprintf(" MPY_EQU: %d\n",MPY_EQU);
               gprintf(" DIV_EQU: %d\n",DIV_EQU);

               gprintf(" MUL_EL: %d\n",MUL_EL);
               gprintf(" DIV_EL: %d\n",DIV_EL);
               gprintf(" POW_EL: %d\n",POW_EL);

               gprintf(" DIAG_PRE: %d\n",DIAG_PRE);
               gprintf(" DIAG_POST: %d\n",DIAG_POST);

               gprintf(" CE: %d\n",CE);
            break;
         }
      }
      else {
         if(f==ABORT) {
            mallfree((void *)&filename);
            bufreturn(); /* s is freed in bufreturn() */
            longjmp(abortbuf,ABORT);
         }
      }
      if(depthFS-initialFS) {
         gprintf(" parse_driver: function stack is not empty");
         nc();
         sntx_err(mprintf("missing ;"));
      }
      if(PTRACE) {
         gprintf(" parse_driver: end of file %s",filename);
         nc();
      }
      f=(err==stkerrcount);
      mallfree((void *)&filename);
      bufreturn(); /* s is freed in bufreturn() */

      return(f);
   }
   return 0;
}

void parse_init()
/* Initialize before parsing. */
{
   *Token=' ';
   Token_type=UNSET;
   Token_int=UNSET;

   *Token_was=' ';
   Token_type_was=FINISHED;
   Token_int=UNSET;

   pTOK=(unsigned char *)*(pBUF+onbuf); /* start of Token buffer */
   initialFS=depthFS; /* function stack */
   KWcount=0;

   DEF_PENDING=0;
   *input_args=NULL;
}

unsigned char *peekFS()
/* Peek at tos item from function stack. */
{
   int len,ret=0;

   if(!*(FScount+LEVEL)) sntx_err(\
      mprintf("cannot peek at empty FS at token %s",
         show(Token,Token_type)));

   ret=(
      pushint(0) &&
      pickfs()
   );
   if(!ret) sntx_err(\
      mprintf("function stack error at token %s",
         show(Token,Token_type)));

   memcpy((char *)Token_peek,tos->tex,(len=tos->row*tos->col));
   drop();
   *(Token_peek+len)='\0';
   return(Token_peek);
}

void place_prep()
/* Prepare WS for running place1(). */
{
   char *p1,*p2;
   int len;

   if(!strcmp((char *)tos->tex,"purged 0 place1")) {

   /* When "purged 0 place1" is on top of the stack, have a case like 
      T[*,2][3] and the second right bracket is being processed.  

      In this case, "... purged 0 place1" will already have put T[*,2] 
      on the stack and there is nothing to prepare.  Just return.

      Here is an example with three bracketed expressions for sub-
      matrices.  

         >> T = [ 1,2,3,4 ;10,20,30,40]'; T .m
          Row 1:        1       10
          Row 2:        2       20
          Row 3:        3       30
          Row 4:        4       40

      Reading right to left in the following expression for V: the 2nd 
      term in the 2:3 pair of column 2 of T is 30:
         >> V = T[*,2][2:3][2]; V .m
          Row 1:       30 

      Each bracked expression operates on the matrix that results from
      operations on its left. */

      return ; /* return when already have matrix on the left */
   }
/* Build WS for word place1(): place1 (hA hR hC hB op --- hA1)

   Reading B and op has yet to be done when upcoming =, +=,
   -= *= or /= is read.  R and C have just been moved to FS.  
   So, A is currently sitting on tos.  Get it ready: */

/* Make a copy of name A that the equate function uses if called: */
   if(*(EQU_NAME+LEVEL)) free(*(EQU_NAME+LEVEL));
   *(EQU_NAME+LEVEL)=memgetn(tos->tex,tos->col);
   *(EQU_DEPTH+LEVEL)=_depth-1; /* abs WS depth under EQU_NAME */

   if(*tos->tex=='\t') { /* have the form \tA.B\t
      Function atom() flagged this form by prepending and appending \t 
      to A.B.  Make postfix phrase "A" "B" localref fetchif that will 
      fetch MAT to stack for place1: */

      len=stradv(tos->tex+1,"\t")-1; /* len of A.B */
      if(len>NTOK) sntx_err(mprintf("string A.B is too long"));

      memcpy((char *)Token_peek,tos->tex+1,len);
      *(Token_peek+len)='\0';

      p1=strtok((char *)Token_peek,".");
      p2=p1;
      p2=strtok('\0',".");

   /* Phrase to make catalog name of MAT B in lib of A: */
      drop();
      pushstr(p1); /* p1 = lib = word name = "A" */
      quoted1();
      spaced();
      pushstr(p2); /* p2 = name of word in A's lib = "B" */
      quoted1();
      spaced();
      pushstr("localref "); /* command to make catlog name */
      cat();
      cat();
   }
   else {
      quoted1();
      spaced();
   }
/* Make phrase: "A" fetchif */
   pushstr("fetchif"); 
   cat();

   if(PTRACE) {
      gprintf("place_prep(); WS contents are:\n");
      showWS();
   }
}

int popFS()
/* Pop tos item from function stack. */
{
   int ret=0;

   if(!*(FScount+LEVEL)) sntx_err(\
      mprintf("cannot pop empty FS at token %s",
         show(Token,Token_type)));

   ret=pullfs();

   (*(FScount+LEVEL))--;
   (*(WScount+LEVEL))++;

   (*(EQwarn+LEVEL))--;

   if(PTRACE) {
      gprintf(\
       "pop fro FS, push to WS: %s LEVEL: %d FScount: %d WScount: %d",
       tos->tex,LEVEL,*(FScount+LEVEL),*(WScount+LEVEL));
      nc();
   }
   return(ret);
}

void pop_mathFS()
/* Pop multiply and divide tokens from FS(LEVEL).  This is done on the 
   ascent in three places to keep the correct precedence with plus and 
   minus: 
      eval_exp8(void);  // multiply or divide * / % 
      eval_exp15(void); // right parenthesis )  
      eval_exp16(void); // right bracket ] */
{
   int more=1;
   char *peek;

   while(*(FScount+LEVEL) && more) {
      if(!strcmp((peek=(char *)peekFS()),"*") ||
         !strcmp(peek,"/") ||
         !strcmp(peek,"%") ||
         !strcmp(peek,"*by") ||
         !strcmp(peek,"/by") ||
         !strcmp(peek,"diagpre") ||
         !strcmp(peek,"diagpost") ||
         !strcmp(peek,"negate") ||
         !strcmp(peek,"@")
        ) popFS();
      else more=0;
   }
}

int popWS() /* popWS (... X --- ... ) */
/* Pop tos item from working stack. */
{
   if(!*(WScount+LEVEL)) sntx_err(\
      mprintf("cannot pop empty WS at token %s",
         show(Token,Token_type)));

   (*(WScount+LEVEL))--;
   if(PTRACE) {
      gprintf("pop fro WS: %s LEVEL: %d WScount: %d KWcount: %d",
       tos->tex,LEVEL,*(WScount+LEVEL),KWcount);
      nc();
   }
   return(drop());
}

int ptrace() /* ptrace ( --- ) */
/* Trace intermediate parsing operations. */
{
   PTRACE=1;
   return 1;
}

int ptraceoff() /* ptraceoff ( --- ) */
{
   PTRACE=0;
   return 1;
}

int pushFS() /* pushFS (... X --- ... ) */
/* Copy WS item to FS and pop WS.  Uses the same LEVEL for WS and FS. */
{
   if(!*(WScount+LEVEL)) {
      sntx_err(mprintf("cannot pop empty WS at token %s LEVEL %d",
         show(Token,Token_type),LEVEL));
      return 0;
   }
   (*(FScount+LEVEL))++;
   if(PTRACE) {
      gprintf("push to FS: %s LEVEL: %d FScount: %d",
         tos->tex,LEVEL,*(FScount+LEVEL));
      nc();
   }
   (*(EQwarn+LEVEL))++;

/* This part is identical to popWS() except it doesn't drop(), since
   lpush1() will move the WS item to the FS: */
   if(!*(WScount+LEVEL)) sntx_err(\
      mprintf("cannot pop empty WS at token %s",
         show(Token,Token_type)));

   (*(WScount+LEVEL))--;
   if(PTRACE) {
      gprintf("pop fro WS: %s LEVEL: %d WScount: %d KWcount: %d",
         tos->tex,LEVEL,*(WScount+LEVEL),KWcount);
      nc();
   }
   return(lpush1());
}

int pushWS(char *quote)
/* Push quote to the working stack, WS.
   Note: This function is made from a copy of pushstr(), stk.c. */
{  char *str;
   int   len;
   #define NOTAG 0

   len = strlen(quote);
   if ((str = (char *) memgetn(quote, len)) != NULL) {
      (*(WScount+LEVEL))++;
      if(PTRACE) {
         gprintf("push to WS: %s LEVEL: %d WScount: %d KWcount: %d",
           quote,LEVEL,*(WScount+LEVEL),KWcount);
         nc();
      }
      return(push(STR, NULL, NOTAG, 0, NULL, str, 1, len, NULL));
   } return 0;

   #undef NOTAG
}

char *show(unsigned char *Token, int Type)
/* Return char string for Token. */
{
   if(!Token) return("error: null Token pointer");

   switch(*Token) {

      default:
         if(Type==STRING) { 
            if(*Token=='{') return("{\"");
            else return("\"");
         }
         else return((char *)Token);

      case 0: return("null");

      case LT: return("<");
      case LE: return("<=");
      case GT: return(">");
      case GE: return(">=");
      case EQ: return("==");
      case NE: return("!=");
      case NGT: return("!>");
      case NLT: return("!<");

      case AND: return("&&");
      case OR: return("||");
      case NOT: return("!");

      case BRK_SEMI: return(";");
      case BRK_COMMA: return(",");

      case NEG: return("-");
      case POS: return("+");
      case INC: return("++");
      case DEC: return("--");

      case INC_EQU: return("+=");
      case DEC_EQU: return("-=");
      case MPY_EQU: return("*=");
      case DIV_EQU: return("/=");

      case MUL_EL: return(".*");
      case DIV_EL: return("./");
      case POW_EL: return(".^");

      case DIAG_PRE: return("\\*");
      case DIAG_POST: return("*\\");

      case CE: return(":=");
   }
}

void show_tokens()
/* Show tokens and types in the current run level buffer for these
   enumerations: tok_type, rel_tok, log_tok, arith_tok, punct_tok,
   extra_tok. */
{
   int d,d0;

   d0=stkdepth();
   pushstr(" token type id offset");
   hand();

   do {
      Token_type=_token_get();

      d=stkdepth();
      if( /* range 32 - 127: */
         (unsigned char)*Token<LT &&
         (unsigned char)*Token>' ') pushstr((char *)Token);

      else {
         switch((unsigned char)*Token) {
            default:
               pushstr("unknown");
            break;
            case 0:
               pushstr("null");
            break;
            case EQ:
               pushstr("==");
            break;
            case GE:
               pushstr(">=");
            break;
            case GT:
               pushstr(">");
            break;
            case LE:
               pushstr("<=");
            break;
            case LT:
               pushstr("<");
            break;
            case NE:
               pushstr("!=");
            break;
            case NGT: 
               pushstr("!>");
            break;
            case NLT: 
               pushstr("!<");
            break;
            case AND:
               pushstr("&&");
            break;
            case OR:
               pushstr("||");
            break;
            case NOT:
               pushstr("!");
            break;
            case BRK_SEMI:
               pushstr(";");
            break;
            case BRK_COMMA:
               pushstr(",");
            break;
            case NEG:
               pushstr("-");
            break;
            case POS:
               pushstr("+");
            break;
            case INC:
               pushstr("++");
            break;
            case DEC:
               pushstr("--");
            break;
            case INC_EQU:
               pushstr("+=");
            break;
            case DEC_EQU:
               pushstr("-=");
            break;
            case MPY_EQU:
               pushstr("*=");
            break;
            case DIV_EQU:
               pushstr("/=");
            break;
            case MUL_EL:
               pushstr(".*");
            break;
            case DIV_EL:
               pushstr("./");
            break;
            case POW_EL:
               pushstr(".^");
            break;
            case DIAG_PRE:
               pushstr("\\*");
            break;
            case DIAG_POST:
               pushstr("*\\");
            break;
            case CE:
               pushstr(":=");
            break;
         }
      }
      spaced(); hand();

      switch(Token_type) {
         default:
            pushstr("unknown");
         break;
         case FUNCTION:
            pushstr("function");
         break;
         case VARIABLE:
            pushstr("variable");
         break;
         case NUMBER:
            pushstr("number");
         break;
         case OPERATOR:
            pushstr("operator");
         break;
         case PUNCTUATION:
            pushstr("punctuation");
         break;
         case KEYWORD:
            pushstr("keyword");
         break;
         case PNDING:
            pushstr("pending");
         break;
         case STRING:
            pushstr("string");
         break;
      }
      spaced(); hand();

      pushint(Token_type);
      intstr();
      spaced(); hand();

      pushuint((unsigned int)(ip-pTOK)
         -(unsigned int)strlen((char *)Token));
      intstr();
      spaced(); hand();

      pushint(stkdepth()-d);
      parkn();

   }  while(ip<ipend && Token_type!=FINISHED);

   pushint(stkdepth()-d0);
   pilen();
   neat();
   dot();
   nc();

   bufunwind(ENDSOU,ENDSOU,1); /* longjmp back to parse_driver() */
}

void showWS() 
/* Show WS at LEVEL.  Place this function at various places while 
   debugging to see what is currently on top level WS. */
{
   int args,i=0;

   if(!(*(WScount+LEVEL))) {
      gprintf(" WS at LEVEL %d is empty\n",LEVEL);
      return;
   }
   args=*(WScount+LEVEL);
   gprintf(" WS items on LEVEL %d: ",LEVEL);

   for(i=args-1;i>-1;i--) {
      pushint(i);
      pick();
      asciify();
      chop();
      gprintf("%s ",tos->tex);
      drop();
   }
   nc();
}

void sntx_err(char *error)
/* Show a line with error.  

   WARNING: If char *error is not NULL, or is not a mallocked pointer 
   as from mprintf(), this function will seg fault.

   Due to recursion, pointer ip may be well past the offset of the
   error.  IP_ERR is set at important points in hopes that it will
   be on the same line as the error.  Where an error straddles one
   line to another, the second line may be shown. */
{
   unsigned char *p;
   int chars,k,linecount=0,offset=0;

/* Find line number of error */
   p=pTOK;
   while(p<(IP_ERR-strlen((char *)Token)-1) && p<ipend) {
      if(*p=='\n') {
         linecount++; 
         offset=0;  
      }
      p++;
      offset++;
   }
/* Report 1based linecount and offset, like gcc: */
   gprintf("line %d.%d: ",linecount+1,offset+1);
   if(error) {
      gprintf(error); 
      mallfree((void *)&error); /* free allocation from mprintf() */
   }
   nc();
   pushstr((char *)pTOK);
   textget0();

   pushint(tos->row);
   ndx();
   pushint(linecount+XBASE);
   min1(); /* min index of linecount and rows */

   quote();
   notrailing();
   chars=tos->col;
   dot();
   nc();

   k=0;
   offset=MIN(offset,chars);
   while(k<offset-1) { k++; gprintf(" "); }
   gprintf("^^^");
   nc();

   clearFS(); /* clear the function stack */

/* Bump the error counts like stkerr() does, but don't call reperr(): */
   stkerrcount++;
   stkerrabs++;

   p=ipend;
   Token_type=FINISHED;
   _depthMIN=0;

   bufunwind(ENDSOU,ENDSOU,1); /* longjmp back to parse() */
}

void sntx_warn(char *warning)
/* Show a line with warning.  

   Example of warning about parentheses in ! expression.  The resulting
   postfix phrases shown in the cases below are completely different:

      without parentheses: z not 0 = 

      with parentheses: z 0 = not 

   Unless z=true (-1) or z=false (0), these phrases produce different 
   results, so parentheses can be essential.

      >> "if(x==0 && y==0 || !z==0)"
      >> parse asciify .
      line 1.20: warning: recommend parentheses around expression for !
      if(x==0 && y==0 || !z==0)
                        ^^^
   These forms with parentheses eliminate the ambiguity of the case 
   above that gives the warning:

      >> "if(x==0 && y==0 || !(z==0))"
      >> "if(x==0 && y==0 || !(z)==0)"
      >> "if(x==0 && y==0 || (!z)==0)"

   WARNING: If char *warning is not NULL, or is not a mallocked pointer
   as from mprintf(), this function will seg fault. */
{
   unsigned char *p;
   int chars,k,linecount=0,offset=0;

/* Find line number of warning */
   p=pTOK;
   while(p<(IP_ERR-strlen((char *)Token)-1) && p<ipend) {
      if(*p=='\n') {
         linecount++; 
         offset=0;  
      }
      p++;
      offset++;
   }
/* Report 1based linecount and offset, like gcc: */
   gprintf("line %d.%d: ",linecount+1,offset+1);
   if(warning) {
      gprintf(warning); 
      mallfree((void *)&warning); /* free allocation from mprintf() */
   }
   nc();
   pushstr((char *)pTOK);
   textget0();

   pushint(tos->row);
   ndx();
   pushint(linecount+XBASE);
   min1(); /* min index of linecount and rows */

   quote();
   notrailing();
   chars=tos->col;
   dot();
   nc();

   k=0;
   offset=MIN(offset,chars);
   while(k<offset-1) { k++; gprintf(" "); }
   gprintf("^^^");
   nc();
 
   return;
}

int _token_get(void)
/* Get next token from pTOK.  Returns token type, and also sets global 
   variable Token_type to token type. 

   Patterned after Reference 1, function get_token(). */ 
{
   register unsigned char *temp;
   unsigned char *ip0;
   unsigned int len;
   static int reenter=0;
   static unsigned char sym1,sym2,sym3;

   if(*Token) {
      strncpy((char *)Token_was,(char *)Token,NTOK);
      Token_type_was=Token_type;
   }
   temp=Token;
   *temp='\0';

   Token_type=UNSET;
   Token_int=UNSET;

/* Skip over blanks. */
   while(isblank1(*ip) && *ip) ++ip;

/* Skip over a carriage return. */
   if(*ip=='\r') {
      ++ip;
   /* Skip over blanks */
      while(isblank1(*ip) && *ip) ++ip;
   }
/* Skip over a new line. */
   if(*ip=='\n') {
      ++ip;
   /* Skip over blanks */
      while(isblank1(*ip) && *ip) ++ip;
   }
/* Look for a null terminator. */
   if(*ip=='\0') {
      *Token='\0';
      return(Token_type=FINISHED);
   }
/* Advance to newline char on # unless # has a nondelimiter on its 
   left, indicating that # is part of a name (this mimics postfix
   blank() in inpo.c): */
   if(*ip=='#' && (ip==pTOK || isdelim1(*(ip-1)))) {
      ip++;
      while(*ip && *ip!='\n') ip++;
   }
/* Advance to newline char on double slash, //: */
   if(*ip=='/' && *(ip+1)=='/') {
      ip++;
      ip++;
      while(*ip && *ip!='\n') ip++;
   }
/* Skip over a multiple-line comment (two forms are shown below).
   Multiple-line comments can be nested. */
   if(*ip=='{' || *ip=='/') {
      sym1=*ip;
      sym2=*(ip+1);

      if((sym1=='{' && sym2=='#' )  || /* 1. {# ... #} */
         (sym1=='/' && sym2=='*' )) {  /* 2. C style */

         if(sym1=='{') sym3='}';
         else sym3=sym1;

         ip+=2;
         do { /* find end of comment */
            IP_ERR=ip;
            while(ip<ipend &&
               (*ip!=sym2 || (*ip==sym1 && *(ip+1)==sym2))) {
               if(*ip==sym1 && *(ip+1)==sym2) {
                  reenter++;
                  _token_get(); /* reenter this function */
                  reenter--;
               }
               ip++;
            }
            ip++;
         } while(ip<ipend && *ip!=sym3);
         ip++;
         ip=MIN(ipend,ip);

         if(!*ip || ip>=ipend)
            sntx_err(mprintf("end of comment not found"));
      }
      if(reenter) return 0;
   }
/* This does C style comment (where nesting is not allowed), left here 
   for reference: 
   if(*ip=='/') {
      if(*(ip+1)=='*') { 
         ip+=2;
         do { // find end of comment //
            while(ip<ipend && *ip!='*') ip++;
            ip++;
         } while(ip<ipend && *ip!='/');
         ip++;
         ip=MIN(ipend,ip);
      }
   }
*/
/* Look for a quoted string. */
   if(*ip=='"') {
      IP_ERR=ip;
      *temp=*ip;
      ip++;
      temp++;

      while(*ip!='"' && *ip!='\n' && ip<ipend) *temp++ = *ip++;
      if(*ip!='"') {
         sntx_err(mprintf("require ending quote on same line"));
      }
      *temp=*ip;
      temp++;
      *temp='\0';
      ip++;
      return(Token_type=STRING);
   }
/* Look for text surrounded by {" ... "} 
   Put text including words {" and "} that will be run to gather the
   text and place a VOL on the stack. */
   if(*ip=='{') {
      if(*(ip+1)=='"') {
         IP_ERR=ip; /* starting ip for brace-quote string */
         ip+=2;
         do { /* find ending "} */
            while(ip<ipend && *ip!='"') ip++;
            ip++;
         } while(ip<ipend && *ip!='}');
         ip++;

         if(!*ip || ip>=ipend)
            sntx_err(mprintf("end of brace-quoted text not found"));

         len=ip-IP_ERR;
         if(!strstk(len,"_bq")) /* STRING of text on WS */
            sntx_err(mprintf("no memory for brace-quoted text"));

         (*(WScount+LEVEL))++;
         memcpy(tos->tex,IP_ERR,len);
      }
   }
/* Pass along verbatim text with no changes. */
   if(!INFIX || *ip=='<') {

      if(!INFIX || *(ip+1)=='<') {

         postfix(); /* symbol << means start verbatim; infix is off */

      /* Do not pass symbol << to a definition: */
         if(DEF_PENDING && (*ip=='<' && *(ip+1)=='<')) ip+=2;

         ip0=ip;
         do { /* find end of verbatim signified by symbol >>  */
            while(ip<ipend && *ip!='>') ip++;
            ip++;
         } while(ip<ipend && *ip!='>');

         ip++;
         ip=MIN(ipend,ip);
         len=(unsigned int)(ip-ip0);

         if(*(ip0+len-1)=='>' && *(ip0+len-2)=='>') {
            infix(); /* symbol >> means end verbatim; infix is on */
         /* Do not pass symbol >> to a definition: */
            if(DEF_PENDING) len-=2;
         }
      /* Push non-null terminated string to WS (can't use pushWS()): */
         pushq2((char *)ip0,len); /* verbatim text */
      /* These lines are like the ones in pushWS(): */
         (*(WScount+LEVEL))++;
         if(PTRACE) {
            gprintf("push to WS: %s LEVEL: %d WScount: %d KWcount: %d",
              tos->tex,LEVEL,*(WScount+LEVEL),KWcount);
            nc();
         }
      }
   }
   ip=MIN(ipend,ip);

/* Look for a null terminator. */
   if(*ip=='\0') {
      *Token='\0';
      return(Token_type=FINISHED);
   }
/* Act on >>, the switch-to-infix symbol. */
   if(*ip=='>' && *(ip+1)=='>') {
      infix(); /* >> means end verbatim; infix is on */
      if(!DEF_PENDING) pushstr(">>"); /* final verbatim text */
      ip+=2;
   }
   ip=MIN(ipend,ip);

/* Look for a null terminator. */
   if(*ip=='\0') {
      *Token='\0';
      return(Token_type=FINISHED);
   }
/* If PTRACE is on, look for special character to halt. */
   if(PTRACE) {
   /* HALT on ` for interactive debugging: */
      if(*ip=='`') sntx_err(mprintf("HALTING")); 
   }
/* Look for a relational operator. */
   if(strchr("!<>=",*ip)) { 
      switch(*ip) {
         case '=':
            if(*(ip+1)=='=') {
               ip++; 
               ip++; 
               *temp=EQ;
               temp++; 
               *temp=EQ; 
               temp++;
               *temp='\0';
            }
         break;
         case '!':
            if(*(ip+1)=='=') {
               ip++; 
               ip++; 
               *temp=NE;
               temp++; 
               *temp=NE; 
               temp++;
               *temp='\0';
            }
            else {
               if(*(ip+1)=='>') {
                  ip++;
                  ip++;
                  *temp=NGT;
                  temp++;
                  *temp=NGT;
                  temp++;
                  *temp='\0';
               }
               else {
                  if(*(ip+1)=='<') {
                     ip++;
                     ip++;
                     *temp=NLT;
                     temp++;
                     *temp=NLT;
                     temp++;
                     *temp='\0';
                  }
               }
            }
         break;
         case '<':
            if(*(ip+1)=='=') {
               ip++; 
               ip++; 
               *temp=LE;
               temp++; 
               *temp=LE;
            }
            else {
               ip++;
               *temp=LT;
            }
            temp++;
            *temp='\0';
         break;
         case '>':
            if(*(ip+1)=='=') {
               ip++;
               ip++;
               *temp=GE;
               temp++;
               *temp=GE;
            }
            else {
               ip++;
               *temp=GT;
            }
            temp++;
            *temp='\0';
         break;
      }
      if(*Token) return(Token_type=OPERATOR);
   }
/* Look for a logical operator. */
   if(strchr("&|!",*ip)) {
      switch(*ip) {
         case '&':
            if(*(ip+1)=='&') {
               ip++;
               ip++;
               *temp=AND;
               temp++;
               *temp=AND;
               temp++;
               *temp='\0';
            }
         break;
         case '|':
            if(*(ip+1)=='|') {
               ip++;
               ip++;
               *temp=OR;
               temp++;
               *temp=OR;
               temp++;
               *temp='\0';
            }
         break;
         case '!':
            ip++;
            *temp=NOT;
         break;
      }
      if(*Token) return(Token_type=OPERATOR);
   }
/* Look for an arithmetic operator (note dot for ./ and .*
   and \ for \*). */
   if(strchr(".+-*^/%'\\",*ip)) {
      switch(*ip) {
         default:
            *temp=*ip;
            ip++; /* advance to next position */
         break;

         case '*':
            if(*(ip+1)=='\\') {
               ip++;
               ip++;
               *temp=DIAG_POST;
               temp++;
               *temp=DIAG_POST;
            }
            else {
               if(*(ip+1)=='=') {
                  ip++;
                  ip++;
                  *temp=MPY_EQU;
                  temp++;
                  *temp=MPY_EQU;
               }
               else {
                  *temp=*ip;
                  ip++; /* advance to next position */
               }
            }
         break;

         case '+':
            if(*(ip+1)=='=') {
               ip++;
               ip++;
               *temp=INC_EQU;
               temp++;
               *temp=INC_EQU;
            }
            else {
               if(*(ip+1)=='+') {
                  ip++;
                  ip++;
                  *temp=INC;
                  temp++;
                  *temp=INC;
               }
               else {
                  if(*Token_was!='\'' && ( /* + following ' cannot
                                              be POS; must be plus */
                     Token_type_was==UNSET || Token_type_was==OPERATOR 
                     || (Token_type_was==PUNCTUATION && *Token_was!=')' 
                     && *Token_was!=']'))
                     ) {
                     *temp=POS;
                     ip++;
                  }
                  else {
                     *temp=*ip;
                     ip++;
                  }
               }
            }
         break;

         case '-':
            if(*(ip+1)=='=') {
               ip++;
               ip++;
               *temp=DEC_EQU;
               temp++;
               *temp=DEC_EQU;
            }
            else {
               if(*(ip+1)=='-') {
                  ip++;
                  ip++;
                  *temp=DEC;
                  temp++;
                  *temp=DEC;
               }
               else {
                  if(*Token_was!='\'' && ( /* - following ' cannot 
                                              be NEG; must be minus */
                     Token_type_was==UNSET || Token_type_was==OPERATOR 
                     || (Token_type_was==PUNCTUATION && *Token_was!=')' 
                     && *Token_was!=']'))
                     ) {
                     *temp=NEG;
                     ip++;
                  }
                  else {
                     *temp=*ip;
                     ip++;
                  }
               }
            }
         break;

         case '/':
            if(*(ip+1)=='=') {
               ip++;
               ip++;
               *temp=DIV_EQU;
               temp++;
               *temp=DIV_EQU;
            }
            else {
               *temp=*ip;
               ip++; /* advance to next position */
            }
         break;

         case '.':
            if(*(ip+1)=='/') {
               ip++;
               ip++;
               *temp=DIV_EL;
               temp++;
               *temp=DIV_EL;
            }
            else {
               if(*(ip+1)=='*') {
                  ip++;
                  ip++;
                  *temp=MUL_EL;
                  temp++;
                  *temp=MUL_EL;
               }
               else {
                  if(*(ip+1)=='^') {
                     ip++;
                     ip++;
                     *temp=POW_EL;
                     temp++;
                     *temp=POW_EL;
                  }
                  else goto next;
               }
            }
         break;

         case '\\':
            if(*(ip+1)=='*') {
               ip++;
               ip++;
               *temp=DIAG_PRE;
               temp++;
               *temp=DIAG_PRE;
            }
            else goto next;
         break;
      }
      temp++;
      *temp='\0';
      return(Token_type=OPERATOR);
   }
   next:
/* Look for punctuation. */
   if(strchr("(,;)={}[]:@",*ip)) { 
      if((*(TYP+LEVEL)==BRK || *(BRKTYP+LEVEL)==INDX || 
         *(BRKTYP+LEVEL)==INDX_LIST) && (*ip==';' || *ip==',')) {
         if(*ip==';') {
            *temp=BRK_SEMI;
            temp++;
            *temp='\0';
            ip++;
         }
         else {
            *temp=BRK_COMMA;
            temp++;
            *temp='\0';
            ip++;
         }
      }
      else {
         if(*ip==':' && *(ip+1)=='=') {
            ip++;
            ip++;
            *temp=CE;
            temp++;
            *temp=CE;
         }
         else {
            *temp=*ip;
            temp++;
            *temp='\0';
            ip++;
         }
      }
      return(Token_type=PUNCTUATION);
   }
/* Get command or catalog item.  Allow names starting with dot or
   underscore or @: */
   if(isalpha(*ip) || (*ip=='.' && !strchr("0123456789",*(ip+1))) ||
      *ip=='_' || *ip=='@') {
      *temp++ = *ip++;
      while(!isdelim1(*ip) && 
      /* watch out for ./ and .* and .^: */
         !(*ip=='.' && 
          (*(ip+1)=='/' || *(ip+1)=='*' || *(ip+1)=='^')) &&
      /* watch out for \*: */
         !(*ip=='\\' && *(ip+1)=='*')) *temp++ = *ip++;
      Token_type=PNDING;
   }
   *temp='\0'; 
/* See if a string is a keyword. */
   if(Token_type==PNDING) {
      
      if((Token_int=is_keyword((char *)Token))) 
         return(Token_type=KEYWORD);
      else {
         Token_int=UNSET;
         if(is_var((char *)Token)) return(Token_type=VARIABLE);
      
         else if(is_function((char *)Token)) 
            return(Token_type=FUNCTION);
      
         else if(is_constant((char *)Token)) 
            return(Token_type=FUNCTION);
   
         else {
            Token_type=UNKNOWN;
            IP_ERR=ip;
            sntx_err(mprintf("unknown type"));
         }
      }
   }
/* Look for an unsigned floating point number. 

   These are the production rules used here for parsing an unsigned 
   floating point number:
 
      unumber --> [0123456789] [.] [0123456789]* (rule 1)
      or
      unumber --> [.] [0123456789]+ (rule 2)

      unumber --> unumber [D|d|E|e] [0123456789]+ (rule 3)
      or
      unumber --> unumber [D|d|E|e] [+|-] [0123456789]+ (rule 4)

   where * means "followed by zero or more of the preceding," and + 
   means "followed by one or more of the preceding."  
*/
   if(isdigit(*ip)) {
      *temp++=*ip++;
      while(strchr(".0123456789",*ip)) *temp++=*ip++; /* rule 1 */
      Token_type=PNDING;
   }
   else {
      if(*ip=='.' && isdigit(*(ip+1))) {
         *temp++=*ip++;
         while(strchr("0123456789",*ip)) *temp++=*ip++; /* rule 2 */
         Token_type=PNDING;
      }
   }
   if(Token_type==PNDING) {
      if(!strchr("DdEe",*ip)) { /* not rule 3 or rule 4 */
         *temp='\0';
         return(Token_type=NUMBER); /* success on rule 1 or rule 2 */
      }
      *temp++=*ip++;
      if(strchr("+-",*ip)) *temp++=*ip++; /* + or -, rule 4 */

      if(!strchr("0123456789",*ip)) { /* rule 3 or 4 needs exponent */
         *temp='\0';
         return(Token_type=UNSET); /* failed rule 3 or rule 4 */
      }
      *temp++=*ip++;
      while(strchr("0123456789",*ip)) *temp++=*ip++;
      *temp='\0';
      return(Token_type=NUMBER); /* success on rule 3 or rule 4 */
   }
   if(Token_type==(unsigned char)UNSET) {
      if(*ip>' ') {
         *temp=*ip;
         temp++;
         *temp='\0';
         ip++; /* advance to next position */
      }
      else _token_get(); /* skip some more blanks or \n or \r */
   }
   return Token_type;
}

int token_get(void)
/* Get next token from pTOK.  Return token type, and also set global 
   variable Token_type to token type. 

   Before returning, check previous token against current to see if 
   it is allowed according to this table:

      Token was         Token current allowed
        NUMBER      ] ) , : := ; OPERATOR FUNCTION NUMBER STRING
        STRING      ] ) , ; OPERATOR FUNCTION 
        KEYWORD     ) , OPERATOR ; (
        }           { } [ VARIABLE FUNCTION KEYWORD
        )           [ ] ) , ; ' = OPERATOR VARIABLE FUNCTION { : :=
        ]           [ ] ) , ; ' = OPERATOR VARIABLE FUNCTION : :=
        FUNCTION    [ ] ) , : := OPERATOR ; ' ( VARIABLE FUNCTION NUMBER
                       @ STRING KEYWORD (allows: >> if(...) )
        VARIABLE    [ ] ) , : := OPERATOR ; ' = ( VARIABLE FUNCTION

      Token was      Token current allowed
        *        BRK_COMMA , ]
        :        VARIABLE FUNCTION NUMBER [ ] ( )
        :=       VARIABLE FUNCTION NUMBER [ ] ( )
        '        OPERATOR ] ) , ; = '
        ++       OPERATOR ) , ; = '
        --       OPERATOR ) , ; = '
        ;        ( [ { } NUMBER FUNCTION VARIABLE KEYWORD ;
        ,        ( ) [ * @ NUMBER FUNCTION VARIABLE KEYWORD STRING 
                    OPERATOR NEG POS
        =        ( NUMBER FUNCTION VARIABLE KEYWORD + - @ STRING 
                    NEG POS
        (        ( , NUMBER FUNCTION VARIABLE KEYWORD ) + - @ [ OPERATOR
        OPERATOR [ ( ) , @ NUMBER FUNCTION VARIABLE KEYWORD OPERATOR 
                     STRING

   where tokens ) ] } , ; = ' ( [ { are type PUNCTUATION and OPERATOR
   includes arithemetic, relational, and logical types, most or all of
   which are in the following list:
      -  + * / % -- ++ ^ += -= > >= < <= == != !> !< && || !  '  */
{        
   int err=0,type;

   type=_token_get();

   if(type==FINISHED) return type;

   IP_ERR=ip;

/* Testing the upper portion of the table above: */

   switch(Token_type_was) { 
      default:
      break;

      case VARIABLE:
         if(Token_type==FUNCTION) goto bottom;
         if(Token_type==VARIABLE) goto bottom;
         if(*Token=='=') goto bottom;
         if(*Token=='(') goto bottom;

         if(*Token==NOT) sntx_err(mprintf("parse error before !"));

         if(Token_type==OPERATOR || (strchr("')[],;:@",*Token) ||
            *Token==CE)) goto bottom;

         if(*Token==BRK_SEMI || *Token==BRK_COMMA) goto bottom;

         err=1;
      break;

      case FUNCTION:
         if(*Token=='=') {
            if(DEF_PENDING) { /* ok to switch use inside a function */
               Token_type_was=VARIABLE;
               break;
            }
            sntx_err(mprintf(\
               "%s is a word in the catalog and cannot be equated",\
               show(Token_was,Token_type_was))); 
         }
         if(Token_type==FUNCTION) goto bottom;
         if(Token_type==VARIABLE) goto bottom;
         if(Token_type==NUMBER) goto bottom;
         if(Token_type==STRING) goto bottom;
         if(Token_type==KEYWORD) goto bottom; /* allows: >> if( ... ) */
         if(*Token=='(') goto bottom;
         if(Token_type==OPERATOR || (strchr("')[],;:@",*Token) ||
            *Token==CE)) goto bottom;
         if(*Token==BRK_SEMI || *Token==BRK_COMMA) goto bottom;
         err=1;
      break;

      case KEYWORD:
         if(*Token=='(') goto bottom;
         if(Token_type==OPERATOR || strchr("),;",*Token)) goto bottom;
         err=1;
      break;

      case PUNCTUATION:
         if(*Token_was!=')' && *Token_was!=']' && *Token_was!='}') 
            break;

         if(*Token_was==')' && (strchr("{",*Token))) break;

         if((*Token_was==')' || *Token_was==']') && 
            (
             Token_type==VARIABLE || Token_type==FUNCTION ||
             strchr(":",*Token) || *Token==CE
            ))break;

         if(*Token_was=='}') {
            if(Token_type==VARIABLE || Token_type==FUNCTION ||
               Token_type==KEYWORD  || *Token=='}' || *Token=='[' || 
               *Token=='{' || *Token=='(') break;
            err=1;
            break;
         }
         if(Token_type==OPERATOR || strchr("[]),;='",*Token)) break;
         if(*Token==BRK_SEMI || *Token==BRK_COMMA) break;
         err=1;
      break;

      case NUMBER:
         if(Token_type==OPERATOR || Token_type==FUNCTION ||
            Token_type==NUMBER || Token_type==STRING ||
            strchr("]),;:",*Token) || *Token==CE) goto bottom;
         if(*Token==BRK_SEMI || *Token==BRK_COMMA) goto bottom;
         err=1;
      break;

      case STRING:
         if(Token_type==OPERATOR || Token_type==FUNCTION
            || strchr("]),;",*Token)) goto bottom;
         if(*Token==BRK_SEMI || *Token==BRK_COMMA) goto bottom;
         err=1;
      break;
   }
   if(err) goto bottom;
    
/* Testing the lower portion of the table above: */

   switch(Token_type_was) { 
      default:
      break;

      case PUNCTUATION:
/*
gprintf(" PUNCTUATION. *Token_was: %c *Token: %c *Token: %d Token_type: %d\n",*Token_was,*Token,*Token,Token_type);
*/
         if((*Token_was==BRK_SEMI || *Token_was==BRK_COMMA) &&
            *Token==']') {
            err=1;
            break;
         }
         if(!strchr(":;,=(",*Token_was) || *Token_was==CE) goto bottom;

         if(Token_type==VARIABLE || Token_type==NUMBER || 
            Token_type==FUNCTION || Token_type==KEYWORD) goto bottom;

         if(*Token_was==';' && (strchr("([{}",*Token))) goto bottom;

         if(*Token_was==',') {
            if(strchr("[-+*)(@",*Token) || Token_type==STRING || 
               Token_type==OPERATOR || *Token==NEG || *Token==POS) 
               goto bottom;
         }
         if(*Token_was=='(') {
            if(strchr("-+),([@",*Token) || Token_type==STRING || 
               Token_type==OPERATOR || *Token==NEG || *Token==POS) 
               goto bottom;
         }
         if(*Token_was=='=') {
            if(strchr("-+([@",*Token) || Token_type==STRING || 
               *Token==NEG || *Token==POS) goto bottom;
         }
         if((*Token_was==':' || *Token_was==CE) && 
            (strchr("[]()@",*Token))) goto bottom;

         err=1;
      break;

      case OPERATOR:
/*
gprintf(" OPERATOR. *Token_was: %c *Token: %c *Token: %d Token_type: %d\n",*Token_was,*Token,*Token,Token_type);
*/
         if(Token_type==STRING) goto bottom;

         if(*Token_was=='\'') {
            if(Token_type==OPERATOR) goto bottom;
            if(strchr("]),;=",*Token)) goto bottom;
            if(*Token==BRK_SEMI || *Token==BRK_COMMA) goto bottom;
         }
         if(Token_type==OPERATOR) goto bottom;

      /* For A[*,*], allow comma, BRK_COMMA or ] to follow *: */
         if(*Token_was=='*') {
            if(*Token==BRK_COMMA || strchr("]",*Token)) goto bottom;
            else { /* due to special use of * as "all" operator: */
               if(strchr(")",*Token)) {
                 sntx_err(
                 mprintf(\
                 "as 'all' operator, * cannot have surrounding tokens",
                 show(Token,Token_type)));
                 goto bottom;
               }
            }
         }
         if(strchr("[(),@",*Token)) goto bottom;      

         if(*Token_was==INC || *Token_was==DEC) {
            if(strchr("),;='",*Token)) goto bottom;
         }
         if(*Token_was==NOT && Token_type==VARIABLE &&
            (*token_peek(NULL)<(unsigned char)LAST_LOG_TOK &&
             *token_peek(NULL)>=(unsigned char)LT)) {
            sntx_warn(mprintf(\
            "warning: recommend parentheses around expression for !"));
            goto bottom1;
         }
         if(Token_type==VARIABLE || Token_type==NUMBER || 
            Token_type==FUNCTION || Token_type==KEYWORD) goto bottom;

         err=1;
      break;
   }
   bottom:
   if(err) {
/*
gprintf(" ERR. *Token_was: %c *Token: %c *Token: %d Token_type: %d\n",*Token_was,*Token,*Token,Token_type);
*/
      sntx_err(
         mprintf(\
            "a separator such as ]),;=([ is needed between %s and %s",
            show(Token_was,Token_type_was),
            show(Token,Token_type)));
   }
/*
   Before returning, check current token against previous to see if 
   it is allowed according to this table:

      Token current       Token_was allowed

       '                ] ) FUNCTION VARIABLE
*/
   switch(Token_type) {
      default:
      break;

      case OPERATOR:
         if(!strchr("'",*Token)) goto bottom1;

         if(Token_type_was==VARIABLE || Token_type_was==FUNCTION ||
            *Token_was==')' || *Token_was==']' || 
            *Token_was=='\'') goto bottom1;
         sntx_err(mprintf("misplaced transpose operator"));
      break;
   }
   bottom1:
   return type;
}

unsigned char *token_peek(int *Token_type_peek) 
/* Peek at next token. */
{
   unsigned char Token_save[NTOK+1]={0};
   int Token_type_save,Token_int_save;
   unsigned char Token_was_save[NTOK+1]={0};
   int Token_type_was_save;
   int type_peek;
   unsigned char *ipsave=NULL;
   int WScount_save;

   strncpy((char *)Token_save,(char *)Token,NTOK);
   Token_type_save=Token_type;
   Token_int_save=Token_int;

   strncpy((char *)Token_was_save,(char *)Token_was,NTOK);
   Token_type_was_save=Token_type_was;

   ipsave=ip;
   WScount_save=*(WScount+LEVEL);

   type_peek=_token_get();

/* If incoming pointer is not NULL, store type_peek: */
   if(Token_type_peek) *Token_type_peek=type_peek;

   strncpy((char *)Token_peek,(char *)Token,NTOK);
   strncpy((char *)Token,(char *)Token_save,NTOK);
   strncpy((char *)Token_was,(char *)Token_was_save,NTOK);

   Token_type=Token_type_save;
   Token_int=Token_int_save;
   Token_type_was=Token_type_was_save;

   ip=ipsave;

/* In _token_get(), << pushes items to WS; restore saved depth: */
   while(*(WScount+LEVEL)>WScount_save) popWS();

   return(Token_peek);
}

/* end prs.c functions */
