/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author and copyright holder of ordering.c:  Al Danial <al.danial@gmail.com>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* ordering.c

   Albert Danial  May 2001

   Routines for determining the ascending sort sequence of a numeric
   array, and for reordering columns of a matrix in-place according
   to a known sort sequence.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ordering.h"
#include "main.h"
#include "stk.h"

#include "exe.h"
#include "ctrl.h"
#include "inpo.h"
#include "mem.h"

#ifdef BLAS
#ifdef FORT_UDSC
#define DCOPY dcopy_
#else
#define DCOPY dcopy
#endif
#endif
       
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int ascending() /* ascending (hA --- f ) */
/*
 * Puts 0 on the stack if terms in the array A are not in ascending
 * order, -1 if they are.
 */
{
    int     i, N;
    double *A;
    if (tos->row > tos->col) {
        if (tos->col != 1) {
            stkerr(" ascending: ",ARRAYNOT);
            return 0;
        }
        N = tos->row;
    } else {
        if (tos->row != 1) {
            stkerr(" ascending: ",ARRAYNOT);
            return 0;
        }
        N = tos->col;
    }
    A   = tos->mat;
    for (i = 0; i < N-1; i++) {
        if (A[i] > A[i+1]) {
            pushint(0);
            return 1;
        }
    }
    pushint(-1);
    return 1;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int reorder() /* reorder (hS hA --- hS hA' ) */
/*
 * Reorders columns of the matrix A (or rows, if A has only one
 * column) *in-place* according to the swap sequence defined by 
 * swap index pair array S.  Use sort_seq to obtain S 
 * for a numeric array.
 */
{   int     k, nPair2, LDA, col_i, col_j;
#ifdef BLAS
    int     int_1       =   1;
#else
    int m;
#endif

    double *A, *Seq, *temp;

    if (tos->typ != MAT) {
       stkerr(" reorder: ",MATNOT);
       return 0;
    }
    if ((tos-1)->typ != MAT) {
       stkerr(" reorder: ",MATNOT);
       return 0;
    }
    if (tos->col == 1) {
        LDA = 1;
    } else {
        LDA = tos->row;
    }
    A   = tos->mat;

    Seq = (tos-1)->mat;
    nPair2 = ((tos-1)->row);

    if (!nPair2) {
        /* no terms in S; array is already in order */
        return 1; 
    }
    if ((temp = malloc(1+LDA*sizeof(double))) == NULL) {
        stkerr(" reorder: ",MEMNOT);
        return 0;
    }
    for (k = 0; k < nPair2; k += 2) {
        /* exchange columns i and j */
        col_i = (int) Seq[k];
        col_j = (int) Seq[k+1];
#ifdef BLAS
        DCOPY(&LDA, &A[col_i*LDA], &int_1,  temp,         &int_1);
        DCOPY(&LDA, &A[col_j*LDA], &int_1, &A[col_i*LDA], &int_1);
        DCOPY(&LDA,  temp,         &int_1, &A[col_j*LDA], &int_1);
#else
        for (m = 0; m < LDA; m++) {
            temp[m]          = A[col_i*LDA + m];
            A[col_i*LDA + m] = A[col_j*LDA + m];
            A[col_j*LDA + m] = temp[m];
        }
#endif
    }
    free(temp);

    return 1;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void update_ordering(int N, int new_i, int old_i, int nPerm, Sort_D_Record *Perm) {
    int i;

    if (new_i == old_i) return;
    for (i = old_i; i < N; i++) {
        if (Perm[i].position == old_i) {
            Perm[i].position =  new_i;
            break;
        }
    }
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int sort_seq() /* sort_seq (hA --- hS ) */
/*
 * Returns an array hS of 2P integers.  If terms of
 * the input array were swapped according to the P pairs of indices
 * in S, the input array A would be in ascending numerical order.
 * The swapping must be done in order so that, at step 1,  
 * A[S[0]] is exchanged with A[S[1]], through step P, 
 * where A[S[2*(P-1)]] is exchanged with A[S[2*P-1]].  The number of
 * pairs, P, is less than or equal to the number of terms in the array A.
 */

{   int     i, N, nPairs;
    double *A, *Seq;
    Sort_D_Record *D_data;
    /*Sort_A_Record *A_data;*/  /* text input not yet supported */

    if(tos->typ != MAT) {
       stkerr(" sort_seq: ",MATNOT);
       return 0;
    }
    if (tos->row > tos->col) {
        if (tos->col != 1) {
            stkerr(" sort_seq: ",ARRAYNOT);
            return 0;
        }
        N = tos->row;
    } else {
        if (tos->row != 1) {
            stkerr(" sort_seq: ",ARRAYNOT);
            return 0;
        }
        N = tos->col;
    }
    A = tos->mat;

   /* make an array to contain the sort sequence */
   if(!matstk(2*N,1,"_Seq")) return 0;
   Seq = tos->mat;

    /* copy user's numerical data into the sort working array */
    if ((D_data = calloc(N, sizeof(Sort_D_Record))) == NULL) {
        stkerr(" sort_seq: ",MEMNOT);
        return (int) NULL;
    }
    for (i = 0; i < N; i++) {
        D_data[i].position =   i;
        D_data[i].value    = A[i];
    }

    qsort(D_data, N, sizeof(D_data[0]), &comp_D_Record);

    /* traverse the sort sequence to figure out which pairs to swap */
    nPairs = 0;
    for (i = 0; i < N; i++) {
        if (i != D_data[i].position) {
            Seq[2*nPairs]   = i;
            Seq[2*nPairs+1] = D_data[i].position;
            ++nPairs;
            update_ordering(N, D_data[i].position, i, N, D_data);
        }
    }
    free(D_data);

    tos->row = 2*nPairs; /* valid rows in S; others are uninitialized */
    return(lop());

    return 1;
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int comp_D_Record(const void *a, const void *b ) {

    if (((Sort_D_Record *)a)->value > ((Sort_D_Record *)b)->value) {
        return  1;
    } else {
        return -1;
    }
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

/* this function isn't called because sort_seq does not yet
 * handle text arrays */
int comp_A_Record(const void *a, const void *b ) {

    return strcmp(((Sort_A_Record *)a)->string, ((Sort_A_Record *)b)->string);
}

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

int new_name() /* new_name ( A qName --- B ) */
/* 
   Renames stack item A to the given string.  ("rename" is a libc function)
   */
{
    if (tos->typ != STR) {
        stkerr(" rename: ",STRNOT);
        return 0;
    }
    strcpy((tos-1)->tok, tos->tex);
    drop();
    return 1;
}
/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */
