/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2014  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

/* mmath.c  October 2000

Copyright (c) 2000-2014  D. R. Williamson

*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

#include "main.h"
#include "stk.h"

#include "exe.h"
#include "inpo.h"
#include "math1.h"
#include "mat.h"
#include "mem.h"
#include "mmath.h"
#include "sparse.h"
#include "tag.h"

#ifdef LAPACK
   #include "lapack.h"
#endif

#ifdef ESSL
   #include "essl.h"
#endif

/*--------------------------------------------------------------------*/

/* Some of this code was generated from Fortran using gnu f2c. */

/* In some functions, this is used for indexing in double arrays: */
#define locf2c(col,rows) (col-1)*rows 

/*--------------------------------------------------------------------*/

int ceig() /* ceig (hC f --- hAr hAi) */
/* Interface to complex eigenvalue and eigenvector functions. */
{
   double *C,*C1;
   int bal,ret,rows;
   register double *Rr,*Ri,*Ri1,w;
   register int i;

   char *clear="2 roll drop";
/*
   char *filtering="swap 1e-10 filter, swap 1e-10 filter";
*/

/* Pointers for eigenvectors (function modald_()): */
   double *B,*A,*EAR,*EAI,*AR,*AI,*PIVOTS,*WORK;
   int imp,iout,*kWORK,nopt,rows1;
   float tol;

   if(!popint(&bal)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" ceig: ",MATNOT);
      return 0;
   }
   if(!cop()) return 0;

   rows=tos->row;
   if(tos->col!=rows) {
      stkerr(" ceig: ","matrix is not square");
      return 0;
   }
   C=tos->mat;
   if(!matstk(rows,1,"_Rr")) return 0;
   Rr=tos->mat;
   if(!matstk(rows,1,"_Ri")) return 0;
   Ri=tos->mat;

   if((C1=(double *)memget(rows,rows))==NULL) return 0;
   memcpy(C1,C,sizeof(double)*rows*rows);

/* run balanc here */

   if(!eig_(C1,Rr,Ri,rows)) {
      mallfree((void *)&C1);
      drop2();
      return 0;
   } 
   mallfree((void *)&C1);

   if(rows>1) { /* sorting eigenvalues: */
      over();
      over();
      park(); /* Rr Ri into 2-column matrix: [Rr Ri] */
      pushint(xFALSE); /* sorting in descending order */
      pushint(XBASE); /* sorting first column, Rr */
      sorton(); 

      Ri1=(tos->mat)+rows; /* pointer to sorted Ri */
      for(i=1;i<rows;i++) { /* setting complex conjugates */
         Ri1++;
         if((w=fabs(*Ri1))!=0) {
            if(fabs(*(Ri1-1))==w) {
               *Ri1=w;
               *(Ri1-1)=-w;
            }
         }
      }
      memcpy(Rr,tos->mat,sizeof(double)*rows);
      memcpy(Ri,(tos->mat)+rows,sizeof(double)*rows);
      drop();
   }
/* Pointers for eigenvector calculations: */

   if((A=(double *)memget(1,1))==NULL) return 0; /* nopt=1, A not used*/
   if((B=(double *)memget0(rows,rows))==NULL) return 0;
   for(i=0;i<rows;i++) *(B+i+locvec(i,rows))=1; /* B=identity mat */

   if((AR=(double *)memget(rows+1,rows))==NULL) return 0;
   if((AI=(double *)memget(rows+1,rows))==NULL) return 0;

   if((PIVOTS=(double *)memget(rows+1,1))==NULL) return 0;
   if((WORK=(double *)memget(rows+1,3))==NULL) return 0;
   if((kWORK=(int *)malloc((rows+1)*sizeof(int)))==NULL) return 0;

   if(!matstk(rows+1,rows,"_Ar")) return 0;
   EAR=tos->mat;
   if(!matstk(rows+1,rows,"_Ai")) return 0;
   EAI=tos->mat;

/* Eigenvector calculations: 
     imp=2 for amp/phase form [not supported], otherwise real/imag form
     nopt=1 for state space form [used here], otherwise quadratic 
*/
   imp=0;
   nopt=1;
   rows1=rows+1;
   tol=1e-6;
   iout=0;

   ret=modald_(C,&rows,&rows,Rr,Ri,B,A,&rows,&rows,&rows,
              EAR,EAI,&rows1,AR,AI,&nopt,&imp,&tol,
              PIVOTS,kWORK,WORK,&iout);
   mallfree((void *)&A);
   mallfree((void *)&B);
   mallfree((void *)&AR);
   mallfree((void *)&AI);
   mallfree((void *)&PIVOTS);
   mallfree((void *)&WORK);
   mallfree((void *)&kWORK);

/* Stack contents at this point: hC hRr hRi hAi hAr */

   if(!ret) { /* if eigvec error, returning eigvals hRr hRi on stack */
      stkerr(" ceig: ","eigenvector error");
      drop2();
      pushq2(clear,strlen(clear));
      xmain(0);
      return 0;
   }
/* DON'T FILTER HERE
   else {
      pushq2(filtering,strlen(filtering));
      xmain(0);
   }
*/
   return( /* returning Ar Ai on stack; eigvals are the 1st row */
      pushq2(clear,strlen(clear)) && xmain(0) &&
      pushq2(clear,strlen(clear)) && xmain(0) &&
      pushq2(clear,strlen(clear)) && xmain(0) 
   );
}

/*
   Thu Feb 14 15:48:37 PST 2013. 

   Inverse fft function cfti() is a work in progress, and is probably
   not well thought out.  

   The idea of inverting an fft from cft() (_lamp()) will only work if
   the number of time points equals the number of frequency points.

   But ffts from cft() are usually sought where the number of frequency
   points is much smaller.

   References to cfti in mmath.h and word.p have been removed and the
   code below is commented out.

int cfti() // cfti (hFr hFi hf ht --- hA) //
// Wed Feb  6 18:43:54 PST 2013

   Invert complex continuous Fourier transform Fr+i*Fi at frequencies
   f, to obtain the original time history, real A(t).

   //
{
   double *A,*f,*f0,*Fi,*Fi0,*Fr,*Fr0,*t0;
   int cols,nf,nt;
   register int i,j,k=0;
   const double pi2=2*3.14159265358979323846;
   register double *A1,*A2,b11,b12,b21,b22,ct,d,dFr,dFi,dt,Omeg,\
      st,t,tr,ti;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT || \
      (tos-3)->typ!=MAT) {
      stkerr(" cfti: ",MATNOT);
      return 0;
   }
   t0=tos->mat;
   f0=(tos-1)->mat;

   Fi0=(tos-2)->mat;
   Fr0=(tos-3)->mat;

   if((tos-3)->col!=(tos-2)->col) {
      stkerr(" cfti: ",COLSNOT);
      return 0;
   }
   cols=(tos-2)->col;

   if((tos-3)->row!=(tos-2)->row) {
      stkerr(" cfti: ",ROWSNOT);
      return 0;
   }
   nf=(tos-1)->row;
   if(((tos-2)->row)!=nf) {
      gprintf(" cfti: frequency rows does not match (Fr, Fi) rows");
      nc();
      stkerr("","");
      return 0;
   }       
   nt=tos->row;
   if(!matstk(nt,cols,"_A")) return 0;
   A=tos->mat;
   memset(A,0,nt*cols*sizeof(double)); 

// Showing loops:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.0
      Thu Feb  7 13:01:11 PST 2013
      [tops@kaffia] ready > ww
      cft.  cols 1 nf 2 nt 4
      k 0 cols 1
         k 0 i 0
            i 0 j 1 
            i 0 j 2 
            i 0 j 3 
         k 0 i 1
            i 1 j 1 
            i 1 j 2 
            i 1 j 3 
      cfti.  cols 1 nf 2 nt 4
      k 0 cols 1
         k 0 i 1
            i 1 j 3
            i 1 j 2
            i 1 j 1
         k 0 i 0
            i 0 j 3
            i 0 j 2
            i 0 j 1

      [tops@kaffia] ready > // 

   gprintf("cfti.  cols %d nf %d nt %d\n",cols,nf,nt);
   for(;k<cols;k++) { // for each column of (Fr+i*Fi): //
      gprintf("k %d cols %d\n",k,cols);

      for(i=nf-1;i>-1;i--) { // for each frequency, f(i): //
         gprintf("   k %d i %d\n",k,i);

         for(j=nt-1;j>0;j--) { // for each t(i): //
            gprintf("      i %d j %d\n",i,j,nf);

         }
      }
   }
   return 1; //

   for(;k<cols;k++) { // for each column of (Fr+i*Fi): //
      Fr=Fr0;
      Fi=Fi0;

      for(i=nf-1;i>-1;i--) { // for each frequency, f(i): //
         A1=A;
         A2=A+1;
         t=*(t0+i-1);
         dt=*(t0+i)-t;
         f=f0;

         for(j=nt-1;j>0;j--) { // for each t(i): //
            Omeg=*(f+j-1)*pi2;
            st=sin(Omeg*t);
            ct=cos(Omeg*t);

            dFr=*(Fr+j)-*(Fr+j-1);
            dFi=*(Fi+j)-*(Fi+j-1);
gprintf(" i %d j %d Omeg %f t %f Fr1 %0.4e Fr2 %0.4e\n",i,j,Omeg,t,*(Fr+j),*(Fr+j-1));
            tr=ct*dFr-st*dFi;
            ti=st*dFr+ct*dFi;

            st=sin(Omeg*dt);
            ct=cos(Omeg*dt);

            d=dt*Omeg*Omeg;
            b11=(1-ct)/d;
            b12=st/Omeg-(1-ct)/d;
            b21=st/d-1/Omeg;
            b22=ct/Omeg-st/d;

            d=b11*b22-b21*b12; // determinant of 2x2 //
            *A1+=(tr*b11-ti*b12)/d; 
            *A2+=(-tr*b21+ti*b11)/d;
         }
         A++;

      }
      Fr0+=nf;
      Fi0+=nf;
   }
   return(lop() && lop() && lop() && lop());
}
*/

int chmats_(double *rr, double *ri, double *ar, double *ai, int *mx, 
   double *a, double *b, double *c__, int *mda, int *mdb, int *mdc, 
   int *n, int *nopt)
/* chmats.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int ar_dim1, ar_offset, ai_dim1, ai_offset, a_dim1, a_offset, b_dim1, b_offset, c_dim1, c_offset, i__1, i__2;
    double d__1, d__2, d__3;

    /* Local variables */
    static double amod, vmax;
    static int j, k;
    static double rrmri;
    static int n2;

/*         ASSEMBLES THE CHARACTERISTIC MATRIX FOR A GIVEN EIGENVALUE */
/*         S = (RR,RI).  THE CHARACTERISTIC MATIX HAS THE FORM */
/*                     [AR,AI] = [(A*S + B)*S + C] */
/*         UNLESS NOPT = 1, IN WHICH CASE THE MATRIX HAS THE FORM */
/*                         [AR,AI] = [B*S - C] */

/*         AR AND AI ARE ROW SCALED BY THE MAXIMUM MODULUS OF THE ROW. */

/*         DECLARE DOUBLE PRECISION FUNCTIONS: */

    /* Parameter adjustments */
    ai_dim1 = *mx;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;
    ar_dim1 = *mx;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    a_dim1 = *mda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    b_dim1 = *mdb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    c_dim1 = *mdc;
    c_offset = 1 + c_dim1 * 1;
    c__ -= c_offset;

    /* Function Body */
    rrmri = *rr * *rr - *ri * *ri;
    n2 = *n;
    i__1 = n2;
    for (j = 1; j <= i__1; ++j) {
 vmax = 0.;

 i__2 = n2;
 for (k = 1; k <= i__2; ++k) {
     if (*nopt == 1) {
  goto L626;
     }
     ar[j + k * ar_dim1] = a[j + k * a_dim1] * rrmri + b[j + k * 
      b_dim1] * *rr + c__[j + k * c_dim1];
     ai[j + k * ai_dim1] = *ri * (b[j + k * b_dim1] + *rr * 2. * a[j + 
      k * a_dim1]);
/* Computing 2nd power */
     d__1 = a[j + k * a_dim1] * rrmri;
/* Computing 2nd power */
     d__2 = b[j + k * b_dim1] * *rr;
/* Computing 2nd power */
     d__3 = *ri * (b[j + k * b_dim1] + *rr * 2. * a[j + k * a_dim1]);
     amod = d__1 * d__1 + d__2 * d__2 + c__[j + k * c_dim1] * c__[k + 
      j * c_dim1] + d__3 * d__3;
     goto L627;
L626:
     ar[j + k * ar_dim1] = b[j + k * b_dim1] * *rr - c__[j + k * 
      c_dim1];
     ai[j + k * ai_dim1] = b[j + k * b_dim1] * *ri;
/* Computing 2nd power */
     d__1 = b[j + k * b_dim1] * *rr;
/* Computing 2nd power */
     d__2 = b[j + k * b_dim1] * *ri;
     amod = d__1 * d__1 + c__[j + k * c_dim1] * c__[j + k * c_dim1] + 
      d__2 * d__2;
L627:
     amod = sqrt(amod);
     vmax = MAX(amod,vmax);
/* L629: */
 }

 if (vmax == 0.) {
     goto L630;
 }
 i__2 = n2;
 for (k = 1; k <= i__2; ++k) {
     ar[j + k * ar_dim1] /= vmax;
     ai[j + k * ai_dim1] /= vmax;
/* L628: */
 }
L630:
 ;
    }
    return 0;
} /* chmats_ */

int conj1() /* conj (hA --- hB) */
/* The complex conjugate of matrix A is returned in B. */
{
   double *A,*B;
   int k=0,spars=0;

   if(tos->typ==NUM) {
      cop();
      if(is_complex(tos)) tos->imag=-tos->imag;
      else set_complex(tos);
      return 1;
   }
   if(is_sparse(tos)) {
      dense();
      spars=1;
   }
   if(tos->typ!=MAT) {
      stkerr(" conj: ",MATNOT);
      return 0;
   }
   if(!is_complex(tos)) {
      return(
         cop() &&
         dup1s() &&
         dims() &&
         null() &&
         dblcmplx()
      );
   }
   A=tos->mat;

   if(!matstk(tos->row,tos->col,"_B")) return 0;
   B=tos->mat;
   set_complex(tos);

   for(;k<(tos->row*tos->col)/2;k++) {
      *B=*A;
      A++;
      B++;
      *B=-(*A);
      A++;
      B++;
   }
   if(spars) sparse();

   return(lop());
}

int cross_correlation() /* cross-correlation (hX hY d --- hR) */
/* Cross-correlation, also called covariance, of X and Y, with delay
   d applied to Y.

   X and Y must have the same dimensions.  This is not checked.

   Each row in X and Y is a data record, and each column is a step
   in time.  Columns of Y are delayed for d steps and the cross-
   correlation is computed (Ref: Bendat and Piersol, p. 120):
      Rxy(k) = [X'(k) * Y(k-d)] */
{
   double *R,*R1,*X,*Y;
   int cols,d,i=1,j=0,rows;

   if((tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" cross-correlation: ",MATNOT2);
      return 0;
   }
   if(!popint(&d)) return 0;

   rows=tos->row;
   cols=tos->col;

   Y=tos->mat; /* starting at first column */
   X=(tos-1)->mat + locvec(d,tos->row); /* starting at column d */

   if(!matstk(cols,1,"_R")) return 0;
   R=tos->mat;

/* The first d rows of R are null: */
   memset(R,0,MIN(d,cols)*sizeof(double)); 
   if(d>=cols) return(lop() && lop());

   R1=R+d;

#ifdef LAPACK
   for (; j < cols-d; j++) {
      R1[j] = DDOT(&rows, &X[j*rows], &i, &Y[j*rows], &i);
   }
#else
   #ifdef ESSL
      dndot(cols-d,rows,R1,i,i,X,i,rows,Y,i,rows);
   #else
      for(;j<cols-d;j++) {
         *R1=0;
         for(i=0;i<rows;i++) {
            *R1+=*X*(*Y);
            X++;
            Y++;
         }
         R1++;
      }
   #endif
#endif
   return(lop() && lop());
}

int diagpre() /* diagpre (hV hA --- hB) */
/* Pre-multiply A by matrix diagonals stored in columns of matrix 
   V (produces row scaling of A). */
{
   double *A,*A0,*B,*B0,*V;
   register double *A1,*B1,d;
   register int i,j,k=0,rA;
   int cA,cV;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" diagpre: ",MATNOT2);
      return 0;
   }
   if(is_complex(tos) || is_complex(tos-1)) return(diagpre_complex());

   if((rA=(tos-1)->row)!=tos->row) {
      stkerr(" diagpre: ","rows of V must match rows of A");
      return 0;
   }
   V=(tos-1)->mat;
   cV=(tos-1)->col;

   A0=tos->mat;
   cA=tos->col;

   if(!matstk(rA,cA*cV,"_B")) return 0;
   B0=tos->mat;

   for(;k<cV;k++) {
      A=A0;
      B=B0+locvec(cA*k,rA);
      for(i=0;i<rA;i++) {
         d=*V;
         A1=A;
         B1=B;
         for(j=0;j<cA;j++) {
            *B1=(*A1)*d;
            A1+=rA;
            B1+=rA;
         }
         A++;
         B++;
         V++;
      }
   }
   return(
      lop() && lop()
   );
}

int diagpre_complex() /* (hV hA --- hB) */
/* Called by diagpre when one or both matrices are complex.  

   This function separates things into real and imaginary and calls 
   diagpre() with real matrices like this high level word in mmath.v:

   inline: diagpre_complex (hVr hVi hAr hAi --- hBr hBi)
\     Pre-multiplication of complex A by a complex diagonal matrix
\     stored in vector V:
\        Br = Vr*Ar - Vi*Ai
\        Bi = Vi*Ar + Vr*Ai
\     where * denotes operation by word diagpre.

      "Ai" book "Ar" book "Vi" book "Vr" book
      Vr Ar diagpre, Vi Ai diagpre less (hBr) "_Br" naming
      Vi Ar diagpre, Vr Ai diagpre plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end
*/
{
   cmplxmatch();
   swap();
   cmplxdbl(); /* Vr Vi */
   rot();
   cmplxdbl(); /* Ar Ai */

/* stack: Vr Vi Ar Ai
          3  2  1  0  */
   pushint(2); pick(); pushint(2); pick(); /* Vi Ar */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vi*Ar
                         4  3  2  1  0     */
   pushint(4); pick(); pushint(2); pick(); /* Vr Ai */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vi*Ar Vr*Ai
                         5  4  3  2  1     0  */
   plusm(); lpush(); /* Bi = Vi*Ar + Vr*Ai to temp stack */

/* stack: Vr Vi Ar Ai
          3  2  1  0  */
   pushint(3); pick(); pushint(2); pick(); /* Vr Ar */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vr*Ar
                         4  3  2  1  0     */
   pushint(3); pick(); pushint(2); pick(); /* Vi Ai */ 
   diagpre(); /* stack: Vr Vi Ar Ai Vr*Ar Vi*Ai
                         5  4  3  2  1     0  */
   minusm(); /* Br = Vr*Ar - Vi*Ai */

   lpull();  /* Bi = Vi*Ar - Vr*Ai */
   dblcmplx(); /* hB on tos */

/* Lop Ai Ar Vi Vr: */
   return(lop() && lop() && lop() && lop());
}

int diagpost() /* diagpost (hA hV --- hB) */
/* Post-multiply A by matrix diagonals stored in columns of matrix 
   V (produces column scaling of A). */
{
   register double *A,*A0,*B,*B0,d,*V;
   register int i,j,k=0;
   int cA,cV,rA;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" diagpost: ",MATNOT2);
      return 0;
   }
   if(is_complex(tos) || is_complex(tos-1)) return(diagpost_complex());

   if((cA=tos->row)!=(tos-1)->col) {
      stkerr(" diagpost: ","rows of V must match columns of A");
      return 0;
   }
   A0=(tos-1)->mat;
   rA=(tos-1)->row;

   V=tos->mat;
   cV=tos->col;

   if(!matstk(rA,cA*cV,"_B")) return 0;
   B0=tos->mat;

   for(;k<cV;k++) {
      A=A0;
      B=B0+locvec(cA*k,rA);
      for(j=0;j<cA;j++) {
         d=*V;
         for(i=0;i<rA;i++) {
            *B=(*A)*d;
            A++;
            B++;
         }
         V++;
      }
   }
   return(
      lop() && lop()
   );
}

int diagpost_complex() /* (hA hV --- hB) */
/* Called by diagpost when one or both matrices are complex.  

   This function separates things into real and imaginary and calls 
   diagpost() with real matrices like this high level word in mmath.v:

   inline: diagpost_complex (hAr hAi hVr hVi --- hBr hBi)
\     Post-multiplication of complex A by a complex diagonal matrix
\     stored in vector V:
\        Br = Ar*Vr - Ai*Vi
\        Bi = Ai*Vr + Ar*Vi
\     where * denotes operation by word diagpost.

      "Vi" book "Vr" book "Ai" book "Ar" book
      Ar Vr diagpost, Ai Vi diagpost less (hBr) "_Br" naming
      Ar Vi diagpost, Ai Vr diagpost plus (hBi) "_Bi" naming
      freed is Ar, freed is Ai, freed is Vr, freed is Vi
   end
*/
{
   cmplxmatch();
   swap();
   cmplxdbl(); /* Ar Ai */
   rot();
   cmplxdbl(); /* Vr Vi */

/* stack: Ar Ai Vr Vi
          3  2  1  0  */
   pushint(2); pick(); pushint(2); pick(); /* Ai Vr */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ai*Vr
                         4  3  2  1  0     */
   pushint(4); pick(); pushint(2); pick(); /* Ar Vi */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ai*Vr Ar*Vi
                         5  4  3  2  1     0  */
   plusm(); lpush(); /* Bi = Ai*Vr + Ar*Vi to temp stack */

/* stack: Ar Ai Vr Vi
          3  2  1  0  */
   pushint(3); pick(); pushint(2); pick(); /* Ar Vr */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ar*Vr
                         4  3  2  1  0     */
   pushint(3); pick(); pushint(2); pick(); /* Ai Vi */ 
   diagpost(); /* stack: Ar Ai Vr Vi Ar*Vr Ai*Vi
                         5  4  3  2  1     0  */
   minusm(); /* Br = Ar*Vr - Ai*Vi */

   lpull();  /* Bi = Ai*Vr - Ar*Vi */
   dblcmplx(); /* hB on tos */

/* Lop Vi Vr Ai Ar: */
   return(lop() && lop() && lop() && lop());
}

void dot1(double *Ain, double *Bin, int len, double *dot)
/* Dot product of vectors A and B. */
{
   register double *A,*B,d=0;
   register int k=0;
   A=Ain;
   B=Bin;
   for(;k<len;k++) d+=*(A+k)*(*(B+k));
   *dot=d;
}

double dsign(double z, double p)
/* z receives the sign of p */
{
   z=fabs(z);
   if(p<0) return(-z);
   else return(z);
}

int eig_(double *a, double *rr, double *ri, int n)
{
   int i=0;

   elmhes_(a,n); /* converting a to Hessenberg upper form */

   if(hqr_(a,n,rr,ri)) { /* finding eigenvalues in Hessenberg matrix */

      for (;i<n;i++) { /* making tiny terms equal to zero: */
         if(fabs(rr[i])<1e-8) rr[i]=0.;
         if(fabs(ri[i])<1e-8) ri[i]=0.;
      }
      return 1;
   }
   return 0;
}

int elmhes_(double *a1, int n)
/* From Press, W. H., et al, "Numerical Recipes in FORTRAN," 
   Cambridge Univeristy Press, second edition, 1992. 

   Reduction to upper Hessenberg form by the elimination method.

   The real, nonsymmetric n-by-n matrix a, stored in n-by-n 
   array, is replaced by an upper Hessenberg matrix with identi-
   cal eigenvalues.  Recommended, but not required, is that this
   routine be preceded by balanc.f.  On output, the Hessenberg 
   matrix is in elements a(i,j) with i <= j+1.  Elements with
   i > j+1 are to be thought of as zero, but are returned with
   random values. */
{
/* Most of this code was generated from Fortran to C using gnu f2c. */

    /* System generated locals */
    double d__1;
    int a_dim1, i__1, i__2, i__3;

    /* Local variables */
    double *a;
    static int i__, j, m;
    static double x, y;

/*  Adjusting incoming pointer a1 for Fortran 1-based indexing 
    used below in the function body: */
    a=a1;
    --a;
    a_dim1 = n;

    /* Function Body */

    i__1 = n - 1;
/* m is called r+1 in Reference (p. 478) */
for (m = 2; m <= i__1; ++m) { 
   x = 0.;
   i__ = m;
   i__2 = n;

   for (j = m; j <= i__2; ++j) {
/* find the pivot */
      if((d__1 = a[j + locf2c((m - 1),a_dim1)], fabs(d__1)) > fabs(x)) {
         x = a[j + locf2c((m - 1),a_dim1)];
         i__ = j;
      }
   }
 if (i__ != m) {
/* interchange rows i__ and m: */
    i__2 = n;
    for (j = m - 1; j <= i__2; ++j) { 
       y = a[i__ + locf2c(j,a_dim1)];
       a[i__ + locf2c(j,a_dim1)] = a[m + locf2c(j,a_dim1)];
       a[m + locf2c(j,a_dim1)] = y;
    }
/* interchange columns i__ and m: */
    i__2 = n;
    for (j = 1; j <= i__2; ++j) {
       y = a[j + locf2c(i__,a_dim1)];
       a[j + locf2c(i__,a_dim1)] = a[j + locf2c(m,a_dim1)];
       a[j + locf2c(m,a_dim1)] = y;
    }
 }
 if (x != 0.) {
/* carry out the elimination */
    i__2 = n;
    for (i__ = m + 1; i__ <= i__2; ++i__) {
       y = a[i__ + locf2c((m - 1),a_dim1)];

       if (y != 0.) {
          y /= x;
          a[i__ + locf2c((m - 1),a_dim1)] = y;

          i__3 = n;
          for (j = m; j <= i__3; ++j) { 
             a[i__ + locf2c(j,a_dim1)] -= y * a[m + locf2c(j,a_dim1)];
          }

          i__3 = n;
          for (j = 1; j <= i__3; ++j) {
             a[j + locf2c(m,a_dim1)] += y * a[j + locf2c(i__,a_dim1)];
          }
       }
    }
 }

}    
   return 1; 
} /* elmhes_ */

int hqr_(double *a1, int n, double *wr1, double *wi1)
/* From Press, W. H., et al, "Numerical Recipes in FORTRAN,"
   Cambridge Univeristy Press, second edition, 1992.

   Find all eigenvalues of an n-by-n upper Hessenberg matrix a1
   that is stored in an n-by-n array.  On input, a1 can be
   exactly as output from elmhes_(); on output it is destroyed.
   The real and imaginary parts of the eigenvalues are returned
   in wr1 and wi1. */
{
/* Most of this code was generated from Fortran to C using gnu f2c. */

/* Initialized data */

/* levels of iteration */
    static int max1= 10;
    static int max2= 20;
    static int max3= 30;

    /* System generated locals */
    int a_dim1, i__1, i__2, i__3, i__4;
    double d__1, d__2, d__3, d__4;

    /* Local variables */
    static int i__, j, k, l, m;
    static double p, q, r__, s, t, u, v, w, x, y, z__, anorm;
    static int nn, its;
    double *a,*wr,*wi;

/*  Adjusting incoming pointer a1 for Fortran 1-based indexing 
    used below in the function body: */
    a=a1;
    --a;
    wr=wr1;
    --wr;
    wi=wi1;
    --wi;

    a_dim1 = n;

    /* Function Body */

/* compute matrix norm for possible use in locating single small 
    subdiagonal element: */
    anorm = (d__1 = a[1], fabs(d__1));
    i__1 = n;
    for (i__ = 2; i__ <= i__1; ++i__) {
 i__2 = n;
 for (j = i__ - 1; j <= i__2; ++j) {
     anorm += (d__1 = a[i__ + locf2c(j,a_dim1)], fabs(d__1));
 }
    }
    nn = n;
    t = 0.; /* gets changed only by an exceptional shift */
L1:
    if (nn >= 1) { /* begin search for next eigenvalue */
 its = 0;

L2: /* begin iteration: look for single small subdiagonal element: */
 for (l = nn; l >= 2; --l) { 
     s = (d__1 = a[l - 1 + locf2c((l - 1),a_dim1)], fabs(d__1)) + (d__2 = a[l 
      + locf2c(l,a_dim1)], fabs(d__2));
     if (s == 0.) {
  s = anorm;
     }
     if ((d__1 = a[l + locf2c((l - 1),a_dim1)], fabs(d__1)) + s == s) {
  goto L3;
     }
 }
 l = 1;
L3:
 x = a[nn + locf2c(nn,a_dim1)];
 if (l == nn) { /* one root found */
     wr[nn] = x + t;
     wi[nn] = 0.;
     --nn;
 } else {
     y = a[nn - 1 + locf2c((nn - 1),a_dim1)];
     w = a[nn + locf2c((nn - 1),a_dim1)] * a[nn - 1 + locf2c(nn,a_dim1)];
     if (l == nn - 1) { /* two roots found ... */
  p = (y - x) * .5; 
/* Computing 2nd power */
  d__1 = p;
  q = d__1 * d__1 + w;

  z__ = sqrt(fabs(q));
  x += t;
  if (q >= 0.) { /* ... a real pair */
      z__ = p + dsign(z__, p); /* abs z times sign of p */
      wr[nn] = x + z__;
      wr[nn - 1] = wr[nn];
      if (z__ != 0.) {
   wr[nn] = x - w / z__;
      }
      wi[nn] = 0.;
      wi[nn - 1] = 0.;
  } else { /* ... a complex pair */
      wr[nn] = x + p;
      wr[nn - 1] = wr[nn];
      wi[nn] = z__;
      wi[nn - 1] = -z__;
  }
  nn += -2;
     } else { /* no roots found; continue iteration */
  if (its >= max3) {
      stkerr(" hqr eigensolver: ","too many iterations");
      return 0;
  }
/* form exceptional shift */
  if (its == max1 || its == max2) {
      t += x;
      i__1 = nn;
      for (i__ = 1; i__ <= i__1; ++i__) {
   a[i__ + locf2c(i__,a_dim1)] -= x;
      }
      s = (d__1 = a[nn + locf2c((nn - 1),a_dim1)], fabs(d__1)) + (d__2 
       = a[nn - 1 + locf2c((nn - 2),a_dim1)], fabs(d__2));
      x = s * .75f;
      y = x;
/* Computing 2nd power */
      d__1 = s;
      w = d__1 * d__1 * -.4375f;
  }
  ++its;
  i__1 = l;
  for (m = nn - 2; m >= i__1; --m) {
/* form shift and then look for two consecutive small subdiagonal 
   elements */
      z__ = a[m + locf2c(m,a_dim1)];
      r__ = x - z__;
      s = y - z__;
      p = a[m + locf2c((m + 1),a_dim1)]; /* eq. 11.6.23 */
      if (a[m + 1 + locf2c(m,a_dim1)] != 0.) {
   p += (r__ * s - w) / a[m + 1 + locf2c(m,a_dim1)];
      }
      q = a[m + 1 + locf2c((m + 1),a_dim1)] - z__ - r__ - s;
      r__ = a[m + 2 + locf2c((m + 1),a_dim1)];
      s = fabs(p) + fabs(q) + fabs(r__);
/* scale to prevent overflow or underflow */
      p /= s;
      q /= s;
      r__ /= s;
      if (m == l) { /* eq. 11.6.26 */
   goto L4;
      }
      u = (d__1 = a[m + locf2c((m - 1),a_dim1)], fabs(d__1)) * (fabs(q) 
       + fabs(r__));
      v = fabs(p) * ((d__1 = a[m - 1 + locf2c((m - 1),a_dim1)], fabs(
       d__1)) + fabs(z__) + (d__2 = a[m + 1 + locf2c((m + 1),a_dim1)], fabs(d__2)));
      if (u + v == v) {
   goto L4;
      }
  }
L4:
  i__1 = nn;
  for (i__ = m + 2; i__ <= i__1; ++i__) {
      a[i__ + locf2c((i__ - 2),a_dim1)] = 0.;
      if (i__ != m + 2) {
   a[i__ + locf2c((i__ - 3),a_dim1)] = 0.;
      }
  }
  i__1 = nn - 1;
  for (k = m; k <= i__1; ++k) {
/* double QR step on rows l to nn and columns m to nn begin setup 
   of Householder vector: */
      if (k != m) {
   p = a[k + locf2c((k - 1),a_dim1)];
   q = a[k + 1 + locf2c((k - 1),a_dim1)];
   r__ = 0.;
   if (k != nn - 1) {
       r__ = a[k + 2 + locf2c((k - 1),a_dim1)];
   }
   x = fabs(p) + fabs(q) + fabs(r__);
   if (x != 0.) {
       p /= x;
/* scale to prevent overflow or overflow */
       q /= x;
       r__ /= x;
   }
      }
/* Computing 2nd power */
      d__2 = p;
/* Computing 2nd power */
      d__3 = q;
/* Computing 2nd power */
      d__4 = r__;
      d__1 = sqrt(d__2 * d__2 + d__3 * d__3 + d__4 * d__4);
      s = dsign(d__1, p);
      if (s != 0.) {
   if (k == m) {
       if (l != m) {
    a[k + locf2c((k - 1),a_dim1)] = -a[k + locf2c((k - 1),a_dim1)];
       }
   } else {
       a[k + locf2c((k - 1),a_dim1)] = -s * x;
   }
   p += s; /* eq. 11.6.24 */
   x = p / s;
   y = q / s;
   z__ = r__ / s;
   q /= p;
   r__ /= p;
   i__2 = nn;
   for (j = k; j <= i__2; ++j) { /* row modification */
       p = a[k + locf2c(j,a_dim1)] + q * a[k + 1 + locf2c(j,a_dim1)];
       if (k != nn - 1) {
    p += r__ * a[k + 2 + locf2c(j,a_dim1)];
    a[k + 2 + locf2c(j,a_dim1)] -= p * z__;
       }
       a[k + 1 + locf2c(j,a_dim1)] -= p * y;
       a[k + locf2c(j,a_dim1)] -= p * x;
   }
/* Computing MIN */
   i__3 = nn, i__4 = k + 3;
   i__2 = MIN(i__3,i__4);
   for (i__ = 1; i__ <= i__2; ++i__) { /* column modification */
       p = x * a[i__ + locf2c(k,a_dim1)] + y * a[i__ + locf2c((k + 1),a_dim1)];
       if (k != nn - 1) {
    p += z__ * a[i__ + locf2c((k + 2),a_dim1)];
    a[i__ + locf2c((k + 2),a_dim1)] -= p * r__;
       }
       a[i__ + locf2c((k + 1),a_dim1)] -= p * q;
       a[i__ + locf2c(k,a_dim1)] -= p;
   }
      }
  }
  goto L2; /* ... for next iteration of current eigenvalue */
     }
 }
 goto L1; /* ... for next eigenvalue */
    }
    return 1;
} /* hqr_ */

int inverd_(int *ndim, double *a, int *n, double *b, int *m, \
   double *determ, int *ising, int *index)
/* inverd.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int a_dim1, a_offset, b_dim1, b_offset, index_dim1, index_offset, 
     i__1, i__2, i__3, i__4;
    static int equiv_0[1], equiv_1[1];
    static double equiv_2[1];

    /* Local variables */
#define amax (equiv_2)
#define swap (equiv_2)
#define irow (equiv_0)
#define jrow (equiv_0)
    static double d__;
    static int i__, j, k, l;
#define t (equiv_2)
    static int l1;
    static double pivot;
#define icolum (equiv_1)
#define jcolum (equiv_1)

/* NDIM IS THE ACTUAL SIZE OF A IN CALLING PROGRAM, 
   E.G., A(NDIM,NDIM). 
   A IS SQUARE MATRIX TO BE INVERTED. 
   N IS SIZE OF UPPER LEFT PORTION BEING INVERTED. 
   B IS COLUMN OF CONSTANTS (OPTIONAL INPUT).  SUPPLY SPACE 
   B(NDIM,1) MINIMUM.
   M IS THE NUMBER OF COLUMNS IN B. 
   DETERM RETURNS THE VALUE OF DETERMINANT IF NON-SINGULAR 
   ISING RETURNS 2 IF MATRIX A(N,N) IS SINGULAR
                 1 IF MATRIX A(N,N) IS NON-SINGULAR.
   INDEX IS WORKING STORAGE (N,3).
   INVERSE RETURNS IN A.
   SOLUTION VECTORS RETURN IN B.
*/

/*     DOUBLE PRECISION A,B,AMAX,T,SWAP,DETERM,PIVOT */

/*         INITIALIZE */
    /* Parameter adjustments */
    b_dim1 = *ndim;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    a_dim1 = *ndim;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    index_dim1 = *n;
    index_offset = 1 + index_dim1 * 1;
    index -= index_offset;

    /* Function Body */
    *determ = 1.;
    i__1 = *n;
    for (j = 1; j <= i__1; ++j) {
       /* L10: */
       index[j + index_dim1 * 3] = 0;
    }
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {

      /*         SEARCH FOR PIVOT */
       *amax = 0.;
       i__2 = *n;
       for (j = 1; j <= i__2; ++j) {
          if (index[j + index_dim1 * 3] == 1) {
             goto L40;
          }
          i__3 = *n;
          for (k = 1; k <= i__3; ++k) {
             if ((i__4 = index[k + index_dim1 * 3] - 1) < 0) {
                goto L20;
             } 
             else if (i__4 == 0) {
                goto L30;
             } 
             else {
               goto L190;
             }
             L20:
             d__ = a[j + k * a_dim1];
             if (fabs(d__) <= *amax) {
                goto L30;
             }
             *irow = j;
             *icolum = k;
             *amax = fabs(d__);
             L30: ; 
          }
          L40: ;
       }
       ++index[*icolum + index_dim1 * 3];
       index[i__ + index_dim1] = *irow;
       index[i__ + (index_dim1 << 1)] = *icolum;

/*         INTERCHANGE ROWS TO PUT PIVOT ELEMENT ON DIAGONAL */
       if (*irow == *icolum) {
          goto L70;
       }
       *determ = -(*determ);
       i__2 = *n;
       for (l = 1; l <= i__2; ++l) {
          *swap = a[*irow + l * a_dim1];
          a[*irow + l * a_dim1] = a[*icolum + l * a_dim1];
          /* L50: */
          a[*icolum + l * a_dim1] = *swap;
       }
       if (*m <= 0) {
          goto L70;
       }
       i__2 = *m;
       for (l = 1; l <= i__2; ++l) {
          *swap = b[*irow + l * b_dim1];
          b[*irow + l * b_dim1] = b[*icolum + l * b_dim1];
          /* L60: */
          b[*icolum + l * b_dim1] = *swap;
       }

/*         DIVIDE THE PIVOT ROW BY PIVOT ELEMENT */
L70:
       pivot = a[*icolum + *icolum * a_dim1];
       *determ *= pivot;
       if (pivot == 0.) {
          goto L190;
       }
       a[*icolum + *icolum * a_dim1] = 1.;
       i__2 = *n;
       for (l = 1; l <= i__2; ++l) {
          /* L80: */
          a[*icolum + l * a_dim1] /= pivot;
       }
       if (*m <= 0) {
          goto L100;
       }
       i__2 = *m;
       for (l = 1; l <= i__2; ++l) {
          /* L90: */
          b[*icolum + l * b_dim1] /= pivot;
       }

   /*         REDUCE NON-PIVOT ROWS */
       L100:
       i__2 = *n;
       for (l1 = 1; l1 <= i__2; ++l1) {
          if (l1 == *icolum) {
             goto L130;
          }
          *t = a[l1 + *icolum * a_dim1];
          a[l1 + *icolum * a_dim1] = 0.;
          i__3 = *n;
          for (l = 1; l <= i__3; ++l) {
             /* L110: */
             a[l1 + l * a_dim1] -= a[*icolum + l * a_dim1] * *t;
          }
          if (*m <= 0) {
             goto L130;
          }
          i__3 = *m;
          for (l = 1; l <= i__3; ++l) {
             /* L120: */
             b[l1 + l * b_dim1] -= b[*icolum + l * b_dim1] * *t;
          }
          L130: ;
       }
    }

/*         INTERCHANGE COLUMNS */
    i__2 = *n;
    for (i__ = 1; i__ <= i__2; ++i__) {
       l = *n + 1 - i__;
       if (index[l + index_dim1] == index[l + (index_dim1 << 1)]) {
          goto L150;
       }
       *jrow = index[l + index_dim1];
       *jcolum = index[l + (index_dim1 << 1)];
       i__1 = *n;
       for (k = 1; k <= i__1; ++k) {
          *swap = a[k + *jrow * a_dim1];
          a[k + *jrow * a_dim1] = a[k + *jcolum * a_dim1];
          a[k + *jcolum * a_dim1] = *swap;
          /* L140: */
       }
       L150: ;
    }
    i__2 = *n;
    for (k = 1; k <= i__2; ++k) {
       if (index[k + index_dim1 * 3] == 1) goto L160;
       *ising = 2;
       goto L190;
      L160: ;
    }
    *ising = 1; return 1;
    L190: *ising = 2; return 0;
} /* inverd_ */

#undef jcolum
#undef icolum
#undef t
#undef jrow
#undef irow
#undef swap
#undef amax

void _lag(double *V, int rows, int n, double *A)
/* Wed Jan 15 04:30:37 PST 2014

   A(k) equals vector V with all rows shifted down by n rows; n<rows.
   The first n+1 values of A(k) are set to V(0).

   This matches code in the central loop of lagV(), file math.c. */
{
      register int i;
      int len;

      len=(rows-n)*sizeof(double);
      memcpy(A+n,V,len);
      for(i=0;i<n;i++) *(A+i)=*V;
}

int _lamp() /* _lamp (hA ht hf --- hFr hFi), cft (hA ht hf --- hFr hFi)
   At frequencies f, compute the complex Fourier transform of time
   histories in A(t).  Each column in A is a time history, and each
   column in returned Fr and Fi is for the corresponding column of A.

   Adapted from original Fortran function lamp created by John Frye,
   using integration of a piece-wise linear ramp function between 
   time points A(t1) and A(t2). */
{
   double *A,*f,*f0,*Fi,*Fi0,*Fr,*Fr0,*t;
   int cols,nf,nt,rows;
   register int i,j,k=0;
   const double pi2=2*3.14159265358979323846;
   register double A1,A2,ct1,dt,Omeg,slp,st1,t1,tr,ti;

   hand();
   if(tos->typ!=MAT || (tos-1)->typ!=MAT || (tos-2)->typ!=MAT) {
      stkerr(" _lamp: ",MATNOT);
      return 0;
   }
   A=(tos-2)->mat;
   rows=(tos-2)->row;
   cols=(tos-2)->col;

   t=(tos-1)->mat;
   f0=(tos  )->mat;

   nt=MIN((tos-1)->row,(tos-2)->row);
   nf=tos->row;

   if(!matstk(nf,cols,"_Fr")) return 0;
   Fr0=tos->mat;

   if(!matstk(nf,cols,"_Fi")) return 0;
   Fi0=tos->mat;

/* Showing loops:
   gprintf("cft.  cols %d nf %d nt %d\n",cols,nf,nt);
   for(;k<cols;k++) { // for each time history, A(t): //
      gprintf("k %d cols %d\n",k,cols);

      for(i=0;i<nf;i++) { // for each frequency, f(i): //
         gprintf("   k %d i %d\n",k,i);

         for(j=1;j<nt;j++) { // integrating over all t(i): //
            gprintf("      i %d j %d \n",i,j);
         }
      }
   }
   return 1; */

   for(;k<cols;k++) { /* for each time history, A(t): */
      f=f0;
      Fr=Fr0;
      Fi=Fi0;

      for(i=0;i<nf;i++) { /* for each frequency, f(i): */
         *Fr=0;
         *Fi=0;
         Omeg=(*f)*(pi2);

         for(j=1;j<nt;j++) { /* integrating over all t(i): */
            t1=*(t+j-1);
            dt=*(t+j)-t1;

            A1=*(A+j-1);
            A2=*(A+j);

            slp=(A2-A1)/dt;
            st1=sin(Omeg*dt);
            ct1=cos(Omeg*dt);

            tr=A2*st1/Omeg-slp/(Omeg*Omeg)*(1-ct1);
            ti=-(A1-A2*ct1)/Omeg-slp/(Omeg*Omeg)*st1;

            st1=sin(Omeg*t1);
            ct1=cos(Omeg*t1);

            *Fr+=(tr*ct1+ti*st1);
            *Fi+=(ti*ct1-tr*st1);
         }
         Fr++;
         Fi++;
         f++;
      }
      A+=rows;
      Fr0+=nf;
      Fi0+=nf;
   }
   return(
      lpush() && lpush() &&
      drop() && drop2() &&
      lpull() && lpull()
   );
}

int lsqfit() /* lsqfit (hA dt --- hBd hBdd) */
/* Sat Apr  6 09:47:58 PDT 2013.  Least-squares quadratic fit.

   For step size dt, fit least-squared error quadratic curves to data
   in vector A using a moving window of 3 steps.

   B(k) is the value on a fitted curve made from 3 values of A in the
   set that includes A(k), given by [A(k), A(k-1), A(k-2)].

   When B(k)=A(k), Bd(k) and Bdd(k) are the first and second deriva-
   tives of A(k) under the quadratic fit; they are returned by this 
   function.  

   References are given in function matB().

   Notes: 

      Results from this function for n>3 eventually converge to the 
      same result for built-in n=3.

      Functions s3coef() and s3eval() produce results that agree with
      results from this function (see Testing 2).

   Testing:

      1. Extracting the acceleration of gravity from free-fall data:

            >> t = uniform(0.05, 21);         
            g = -386.0; // acceleration of gravity, in/sec/sec

         // The distance of free-fall under acceleration g is:
            X = (1/2)*g*t^2; 
            (Xd, Xdd) = lsqfit(X, 0.05);   
            nl; nl;.m(parkn(X, Xd, Xdd, 3));
   
         Pasting the previous lines at the ready prompt gives:

                            X       Xd      Xdd
              Row 1:       -0        0        0
              Row 2:  -0.4825   -14.47     -193
              Row 3:    -1.93    -38.6     -386
              Row 4:   -4.343    -57.9     -386
              Row 5:    -7.72    -77.2     -386
              Row 6:   -12.06    -96.5     -386
              Row 7:   -17.37   -115.8     -386
              Row 8:   -23.64   -135.1     -386
              Row 9:   -30.88   -154.4     -386
             Row 10:   -39.08   -173.7     -386
             Row 11:   -48.25     -193     -386
             Row 12:   -58.38   -212.3     -386
             Row 13:   -69.48   -231.6     -386
             Row 14:   -81.54   -250.9     -386
             Row 15:   -94.57   -270.2     -386
             Row 16:   -108.6   -289.5     -386
             Row 17:   -123.5   -308.8     -386
             Row 18:   -139.4   -328.1     -386
             Row 19:   -156.3   -347.4     -386
             Row 20:   -174.2   -366.7     -386
             Row 21:     -193     -386     -386
            >>

         After initial start up (rows 1 and 2), these results are cor-
         rect by inspection.

      2. Functions s3coef() and s3eval() produce the same result and
         have no problem with rows 1 and 2:

            >> t = uniform(0.05, 21);         
            g = -386.0;                    
            X = (1/2)*g*t^2; 
            (Y, Yd, Ydd) = s3eval(s3coef(X, t), t);
            nl; nl;.m(parkn(Y, Yd, Ydd, 3));

                            Y        Yd      Ydd
              Row 1:        0 4.136e-15     -386
              Row 2:  -0.4825     -19.3     -386
              Row 3:    -1.93     -38.6     -386
              Row 4:   -4.343     -57.9     -386
              Row 5:    -7.72     -77.2     -386
              Row 6:   -12.06     -96.5     -386
              Row 7:   -17.37    -115.8     -386
              Row 8:   -23.64    -135.1     -386
              Row 9:   -30.88    -154.4     -386
             Row 10:   -39.08    -173.7     -386
             Row 11:   -48.25      -193     -386
             Row 12:   -58.38    -212.3     -386
             Row 13:   -69.48    -231.6     -386
             Row 14:   -81.54    -250.9     -386
             Row 15:   -94.57    -270.2     -386
             Row 16:   -108.6    -289.5     -386
             Row 17:   -123.5    -308.8     -386
             Row 18:   -139.4    -328.1     -386
             Row 19:   -156.3    -347.4     -386
             Row 20:   -174.2    -366.7     -386
             Row 21:     -193      -386     -386
            >>

         Function s3coef() uses spline3() to compute the cubic spline
         coefficients, and spline3() takes extra steps for derivatives
         at steps 1 and 2.  These functions also are not restricted to
         constant step size.
         
      3. Setting BTEST from 0 to 1 and recompling allows a test to be
      run that verifies D=matD() is the inverse of B=matB().  This test
      is run once, and then B=matB() is never needed again.

         [dale@kaffia] /opt/tops/tops/src > tops
                  Tops 3.2.0
         Sat Apr  6 11:18:43 PDT 2013

         [tops@kaffia] ready > 180 (sec) lsqfit
         lsqfit: Test of B times D

          Matrix B:
          Row 1:        3       -540   1.62e+05
          Row 2:     -540   1.62e+05 -5.249e+07
          Row 3: 1.62e+05 -5.249e+07  1.785e+10

          Matrix D:
          Row 1:         1  0.008333 1.543e-05
          Row 2:  0.008333 0.0002006 5.144e-07
          Row 3: 1.543e-05 5.144e-07 1.429e-09

          Matrix D*B:
          Row 1:          1 -3.78e-14 1.256e-11
          Row 2:  1.279e-17         1 1.342e-12
          Row 3: -1.222e-20 3.591e-18         1

         [tops@kaffia] ready >

      Off-diagonal terms in D*B are sufficiently small to conclude that
      D*B is an identity matrix, and that D is the inverse of B.  Set 
      BTEST to 0 because B=matB() is no longer required. */
{
#define BTEST 0

   double *A,*B,*Bd,*Bdd,Bn,*C,*D,*DT,dt,*S,tsum,t2sum,tmp;
   int i,k=0,rows;
   static int n=3; 

   if(tos->typ!=NUM) {
      stkerr(" lsqfit: ",NUMNOT);
      return 0;
   }
   popd(&dt);

   if(!matstk(3,3,"_D")) return 0;
   D=tos->mat;
   matD(D,n,dt);

#if BTEST==1
   if(!matstk(3,3,"_B")) return 0;
   B=tos->mat;
   matB(B,n,dt);

   gprintf("lsqfit: Test of B times D"); nc(); nc();
   gprintf(" Matrix B:"); nc(); pushstr("dup .m nl"); xmain(0); nc();
   gprintf(" Matrix D:"); nc(); pushstr("over .m nl"); xmain(0); nc();
   gprintf(" Matrix D*B:"); nc(); pushstr("* .m nl"); xmain(0);

   return 1;
#endif

   swap(); /* A to tos */
   rows=tos->row; /* rows of A */

   if(!matstk(rows,1,"_Bd")) return 0;
   Bd=tos->mat;
   
   if(!matstk(rows,1,"_Bdd")) return 0;
   Bdd=tos->mat;

   rot(); /* A to tos */
   if(tos->typ!=MAT) {
      stkerr(" lsqfit: ",MATNOT);
      return 0;
   }
   A=tos->mat;

   if(!matstk(n+rows,1,"_B")) return 0;
   B=tos->mat; /* n more rows than A */
   for(;k<n;k++) {
      *B=*A; /* pad n past values of B with A(0) */
      B++;
   }
   memcpy(B,A,rows*sizeof(double)); /* the rest of B equals A */
   lop(); /* drop A off the stack and use B */

   if(!matstk(n,1,"_DT")) return 0;
   DT=tos->mat;
   S=DT;
   for(i=0;i<n;i++) *(S+i)=-i*dt;

   tsum=t2sum=0;
   for(i=1;i<n;i++) {
      Bn=*(DT+i);
      tsum+=Bn;
      t2sum+=Bn*Bn;
   }
   if(!matstk(3,1,"_S")) return 0;
   S=tos->mat;

   if(!matstk(3,1,"_C")) return 0;
   C=tos->mat;

   for(k=0;k<rows;k++) {
      Bn=*B;
      memset(S,0,3*sizeof(double)); 
      memset(C,0,3*sizeof(double)); 

   /* S(1) through S(3) are sigma0 through sigma2 in equation (4)
      of Reference 1: */
      for(i=n-1;i>0;i--) {
         tmp=*(B-i)-Bn;
         *S+=tmp;
         *(S+1)+=tmp*(*(DT+i));
         *(S+2)+=tmp*(*(DT+i))*(*(DT+i));
      }
      mpyad1(C,D,S,3,3,1,C); /* C=D*S; Reference (1), equation (7) */

      *Bd=*(C+1);
      *Bdd=*(C+2)*2;

      B++;
      Bd++;
      Bdd++;
   }
   return(drop2() && drop2() && rot() && drop());
}

#if BTEST==1
void matB(double *B, int n, double t)
/* Sat Apr  6 09:17:58 PDT 2013.  Matrix B for quadratic fit of least 
   squares error.  Require n>2, and valid only for constant step t.

   B=matB() is only used to verify that D=matD() is the inverse of B.

   References (personal notes): 

   1. Dale Williamson notes, "Numerical Integration by a Method of 
      Least-Squared Error," March 19, 1986.

   2. Robert O'Donnell notes, "Analysis of Williamson's Numerical 
      Integration Method," February 26, 1987.

   3. Robert O'Donnell notes, revisions to Reference 2, undated.

   This function returns matrix B from Reference 1, as it was rewritten
   for constant time step in Reference 3. */
{   
   double N,tt;

   tt=t*t;
   N=n;

   *B=N;
   *(B+2)=*(B+4)=*(B+6)=N*(n-1)*(2*n-1)*tt/6;
   *(B+8)=N*(2*n-1)*(n-1)*(3*n*n-3*n-1)*tt*tt/30;

   *(B+1)=*(B+3)=-N*(n-1)*t/2;
   *(B+5)=*(B+7)=-N*N*(n-1)*(n-1)*t*tt/4;
}
#endif
#undef BTEST

void matD(double *D, int n, double t)
/* Sat Apr  6 09:17:58 PDT 2013.  Matrix D for quadratic fit of least 
   squares error.  Require n>2, and valid only for constant step t.

   D=matD() is the inverse of B=matB().

   This function returns matrix D from Reference 3 of the references
   given in function matB(). */
{   
   double N,tt;

   tt=t*t;
   N=3.0/((n+2)*(n+1)*n);

   *D=N*(3*n*n-3*n+2);
   *(D+4)=N*4*(2*n-1)*(8*n-11)/((n-1)*(n-2)*tt);
   *(D+8)=N*60/((n-1)*(n-2)*tt*tt);

   *(D+1)=*(D+3)=N*6*(2*n-1)/t;
   *(D+2)=*(D+6)=N*10/tt;
   *(D+5)=*(D+7)=N*60/((n-2)*t*tt);
}

int lsqprj() /* lsqprj (hA n m dt --- hB hBd hBdd) */
/* Sun Apr  7 05:24:09 PDT 2013.  Least-squares quadratic projection.

   For step size dt, fit least-squared error quadratic curves to data
   in vector A using a moving window of n steps, then project the fit-
   ted curve ahead in time by m steps.

   A quadratic fitted curve is made from n values of A in the set that
   includes A(k), given by [A(k), A(k-1), ..., A(k-n+1)].  The solution
   for least-squared error provides c0, c1 and c2 for fitted quadratic
   curve B.

   These are the expressions for quadratic B and its derivatives; when
   m>0, the curves represent A projected ahead by m steps:

      B = c0 + c1*m*dt + c2*(m*dt)^2
      Bd = c1 + 2*c2*m*dt
      Bdd = 2*c2

   References are given in function matB(). 

   Testing:

      1. Extracting the acceleration of gravity from free-fall data,
         zero projection (m=0):

            >>
            g = -386.0;
            dt = 0.005;
            t = uniform(dt, 21);
            X = (1/2)*g*t^2;
            n = 3;
 
            m = 0;
            (Y, Yd, Ydd) = lsqprj(X, n, m, dt);
            W = parkn(X, Y, g*t, Yd, Ydd, 5);
            .m(W); nl; nl;
 
                          X         Y       Xd       Yd      Ydd
           Row 1:        -0         0       -0        0        0
           Row 2: -0.004825 -0.004825    -1.93   -1.448     -193
           Row 3:   -0.0193   -0.0193    -3.86    -3.86     -386
           Row 4:  -0.04342  -0.04343    -5.79    -5.79     -386
           Row 5:   -0.0772   -0.0772    -7.72    -7.72     -386
           Row 6:   -0.1206   -0.1206    -9.65    -9.65     -386
           Row 7:   -0.1737   -0.1737   -11.58   -11.58     -386
           Row 8:   -0.2364   -0.2364   -13.51   -13.51     -386
           Row 9:   -0.3088   -0.3088   -15.44   -15.44     -386
          Row 10:   -0.3908   -0.3908   -17.37   -17.37     -386
          Row 11:   -0.4825   -0.4825    -19.3    -19.3     -386
          Row 12:   -0.5838   -0.5838   -21.23   -21.23     -386
          Row 13:   -0.6948   -0.6948   -23.16   -23.16     -386
          Row 14:   -0.8154   -0.8154   -25.09   -25.09     -386
          Row 15:   -0.9457   -0.9457   -27.02   -27.02     -386
          Row 16:    -1.086    -1.086   -28.95   -28.95     -386
          Row 17:    -1.235    -1.235   -30.88   -30.88     -386
          Row 18:    -1.394    -1.394   -32.81   -32.81     -386
          Row 19:    -1.563    -1.563   -34.74   -34.74     -386
          Row 20:    -1.742    -1.742   -36.67   -36.67     -386
          Row 21:     -1.93     -1.93    -38.6    -38.6     -386
 
      2. Extracting the acceleration of gravity from free-fall data
         and projecting Y two steps ahead (m=2); lag() is used to move
         Y results to higher numbered rows to compare with X:

            m = 2;
            (Y, Yd, Ydd) = lsqprj(X, n, m, dt);
            Y = lag(Y, m); Yd = lag(Yd, m); Ydd = lag(Ydd, m);
            W = parkn(X, Y, g*t, Yd, Ydd, 5);
            .m(W); nl;

                          X        Y       Xd       Yd      Ydd
           Row 1:        -0        0       -0        0        0
           Row 2: -0.004825  -0.0386    -1.93   -3.378     -193
           Row 3:   -0.0193        0    -3.86        0        0
           Row 4:  -0.04342  -0.0386    -5.79   -3.378     -193
           Row 5:   -0.0772  -0.0965    -7.72    -7.72     -386
           Row 6:   -0.1206  -0.1399    -9.65    -9.65     -386
           Row 7:   -0.1737   -0.193   -11.58   -11.58     -386
           Row 8:   -0.2364  -0.2557   -13.51   -13.51     -386
           Row 9:   -0.3088  -0.3281   -15.44   -15.44     -386
          Row 10:   -0.3908  -0.4101   -17.37   -17.37     -386
          Row 11:   -0.4825  -0.5018    -19.3    -19.3     -386
          Row 12:   -0.5838  -0.6031   -21.23   -21.23     -386
          Row 13:   -0.6948  -0.7141   -23.16   -23.16     -386
          Row 14:   -0.8154  -0.8347   -25.09   -25.09     -386
          Row 15:   -0.9457   -0.965   -27.02   -27.02     -386
          Row 16:    -1.086   -1.105   -28.95   -28.95     -386
          Row 17:    -1.235   -1.254   -30.88   -30.88     -386
          Row 18:    -1.394   -1.414   -32.81   -32.81     -386
          Row 19:    -1.563   -1.583   -34.74   -34.74     -386
          Row 20:    -1.742   -1.761   -36.67   -36.67     -386
          Row 21:     -1.93   -1.949    -38.6    -38.6     -386 */
{
   double *A,*B,*Bd,*Bdd,Bn,*C,*D,*DT,dt,*S,tsum,t2sum,tmp;
   int i,k=0,m,n,rows;

   if(tos->typ!=NUM || (tos-1)->typ!=NUM || (tos-2)->typ!=NUM) {
      gprintf(" lsqprj: expect three numbers on stack");
      nc();
      stkerr("","");
      return 0;
   }
   popd(&dt);
   popint(&m);
   popint(&n);

   if(!matstk(3,3,"_D")) return 0;
   D=tos->mat;
   matD(D,n,dt);

   swap(); /* orig A to tos */
   rows=tos->row; /* rows of orig A */

   if(!matstk(rows,1,"_B")) return 0;
   B=tos->mat;
   
   if(!matstk(rows,1,"_Bd")) return 0;
   Bd=tos->mat;
   
   if(!matstk(rows,1,"_Bdd")) return 0;
   Bdd=tos->mat;

   pushstr("3 roll"); xmain(0); /* orig A to tos */
   if(tos->typ!=MAT) {
      stkerr(" lsqprj: ",MATNOT);
      return 0;
   }
   C=tos->mat; /* orig A */

   if(!matstk(n+rows,1,"_A")) return 0;
   A=tos->mat; /* n more rows than orig A */
   for(;k<n;k++) {
      *A=*C; /* pad n past values of A with A(0)=C(0) */
      A++;
   }
   memcpy(A,C,rows*sizeof(double)); /* the rest of A equals orig A */
   lop(); /* drop orig A off the stack and use new A */

   if(!matstk(n,1,"_DT")) return 0;
   DT=tos->mat;
   S=DT;
   for(i=0;i<n;i++) *(S+i)=-i*dt;

   tsum=t2sum=0;
   for(i=1;i<n;i++) {
      Bn=*(DT+i);
      tsum+=Bn;
      t2sum+=Bn*Bn;
   }
   if(!matstk(3,1,"_S")) return 0;
   S=tos->mat;

   if(!matstk(3,1,"_C")) return 0;
   C=tos->mat;

   for(k=0;k<rows;k++) {
      Bn=*A;
      memset(S,0,3*sizeof(double)); 
      memset(C,0,3*sizeof(double)); 

   /* S(1) through S(3) are sigma0 through sigma2 in equation (4)
      of Reference 1: */
      for(i=n-1;i>0;i--) {
         tmp=*(A-i)-Bn;
         *S+=tmp;
         *(S+1)+=tmp*(*(DT+i));
         *(S+2)+=tmp*(*(DT+i))*(*(DT+i));
      }
      mpyad1(C,D,S,3,3,1,C); /* C=D*S; Reference (1), equation (7) */

      *Bdd=*(C+2)*2;
      *Bd=*(C+1)+*Bdd*dt*m;
      *B=*C+Bn+*Bd*dt*m;

      B++;
      Bd++;
      Bdd++;
      A++;
   }
   return(drop2() && drop2() && lpush() && rot() && drop() && lpull());
}

int mmax1() /* mmax1 (hV ht ntau --- hV1) */
/* Moving maximum.  V1(k) holds the maximum value of vector V in the 
   interval t(k) to t(k)-tau. */
{
   double max,*t,*t0,tau,*ti,tk,*V,*V1,*Vi;
   int f,i,k=0,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmax1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmax1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   t=tos->mat;
   t0=t;

   V=(tos-1)->mat;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   tk=*t+tau;
   max=*V;
   Vi=V;
   while(*t<tk && k<rows) {
      max=MAX(max,*Vi);
      *V1=max;
      Vi++;
      V1++;
      t++;
      k++;
   }
   for(;k<rows;k++) {
      tk=*t-tau;
      f=bsearchd(tk,t0,rows,&i); /* f=1 if tk is found in t0 */
      i+=1-f; /* use next index if f=0 (tk not found) */

      Vi=V+i;
      ti=t0+i;
      max=*Vi;

      while(i<=k) {
         max=MAX(max,*Vi);
         Vi++;
         i++;
      }
      *V1=max;
      V1++;
      t++;
   }
   return(lop() && lop());
}

/*

This version of mmax1 is not suitable real time analysis because of the
backward running loop, which can hit different past values when future 
points are added, thus "revising" previous results that may have been 
used.

It has been replaced by the uncommented version of mmax1.

int mmax1() // mmax1 (hV ht ntau --- hV1) //
// Moving maximum.  V1(k) holds the maximum value of vector V in the
   interval t(k) to t(k)-tau. //
{
   double max,*t,t0,tau,*ti,tk,*V,*V1,*Vi;
   int i,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmax1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmax1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   ti=(tos->mat)+rows-1;
   t0=*(tos->mat);

   Vi=((tos-1)->mat)+rows-1;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=(tos->mat)+rows-1;

   for(i=rows-1;i>-1;i--) {
      V=Vi;
      t=ti;
      max=*V;
      tk=MAX(*t-tau,t0);
      while(*t>tk) {
         t--;
         V--;
         max=MAX(max,*V);
      }
      *V1=max;
      V1--;
      Vi--;
      ti--;
   }
   return(lop() && lop());
}
*/

void _mmax(double *A, int rows, int n, double *B)
/* Tue Jan 14 11:35:43 PST 2014
   This code was moved here from mmax() in wapp.c.

   Moving max; B(i) is max of A(i-n+1) to A(i).

   Fri Jun 13 05:05:43 PDT 2014.  Replace two loops with one.  
   Testing:

      [dale@kaffia] /opt/tops/tops/src > tops
               Tops 3.2.1
      Fri Jun 13 05:14:32 PDT 2014
      [tops@kaffia] ready > syspath "../src/wapp.c" + \
         "#def Testing mmaxf" "#end Testing mmaxf" msource1

       using _mmax
       mmax and mmaxf results match
       flag from mmaxf is valid
       rows from mmaxf are valid
       using _mmin
       mmin and mminf results match
       flag from mminf is valid
       rows from mminf are valid

       press Enter to continue 

      [tops@kaffia] ready > */
{
   register int i,k1,k2;
   double X;
   int kX;

 /*gprintf(" using _mmax\n");*/

   k2=0;
   X=*B=*A;
   kX=k2;
   k2++;
   while(k2<rows) {
      if(*(A+k2)>X) { 
         X=*(B+k2)=*(A+k2); /* max is X at kX=k2 */
         kX=k2;
      }
      else {
         k1=k2-n+1;
         if(kX<k1) { /* max is too old; find new max X at kX<=k2 */
            X=*(A+k1);
            kX=k1;

            i=k1+1;
            while(i<=k2) {
               if(*(A+i)>X) { 
                  X=*(A+i); /* max is X at kX<=k2 */
                  kX=i;
               }
               i++;
            }
         }
         *(B+k2)=X;
      }
    /*gprintf(" _mmax. k2 %d kX %d k1 %d A %d X %d B %d\n",\
      k2,kX,k1,(int)*(A+k2),(int)X,(int)*(B+k2));*/

      k2++;
   }
}

int mmin1() /* mmin1 (hV ht ntau --- hV1) */
/* Moving minimum.  V1(k) holds the minimum value of vector V in the 
   interval t(k) to t(k)-tau. */
{
   double min,*t,*t0,tau,*ti,tk,*V,*V1,*Vi;
   int f,i,k=0,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmin1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmin1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   t=tos->mat;
   t0=t;

   V=(tos-1)->mat;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=tos->mat;

   tk=*t+tau;
   min=*V;
   Vi=V;
   while(*t<tk && k<rows) {
      min=MIN(min,*Vi);
      *V1=min;
      Vi++;
      V1++;
      t++;
      k++;
   }
   for(;k<rows;k++) {
      tk=*t-tau;
      f=bsearchd(tk,t0,rows,&i); /* f=1 if tk is found in t0 */
      i+=1-f; /* use next index if f=0 (tk not found) */

      Vi=V+i;
      ti=t0+i;
      min=*Vi;

      while(i<=k) {
         min=MIN(min,*Vi);
         Vi++;
         i++;
      }
      *V1=min;
      V1++;
      t++;
   }
   return(lop() && lop());
}

/*

This version of mmin1 is not suitable real time analysis because of the
backward running loop, which can hit different past values when future 
points are added, thus "revising" previous results that may have been 
used.

It has been replaced by the uncommented version of mmin1.

int mmin1() // mmin1 (hV ht ntau --- hV1) //
// Moving minimum.  V1(k) holds the minimum value of vector V in the 
   interval t(k) to t(k)-tau. //
{
   double min,*t,t0,tau,*ti,tk,*V,*V1,*Vi;
   int i,rows;

   if(!popd(&tau)) return 0;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" mmin1: ",MATNOT2);
      return 0;
   }
   if(tos->row != (tos-1)->row) {
      stkerr(" mmin1: ",ROWSNOT);
      return 0;
   }
   rows=tos->row;
   ti=(tos->mat)+rows-1;
   t0=*(tos->mat);

   Vi=((tos-1)->mat)+rows-1;

   if(!matstk(rows,1,"_V1")) return 0;
   V1=(tos->mat)+rows-1;

   for(i=rows-1;i>-1;i--) {
      V=Vi;
      t=ti;
      min=*V;
      tk=MAX(*t-tau,t0);
      while(*t>tk) {
         t--;
         V--;
         min=MIN(min,*V);
      }
      *V1=min;
      V1--;
      Vi--;
      ti--;
   }
   return(lop() && lop());
}
*/

void _mmin(double *A, int rows, int n, double *B)
/* Tue Jan 14 12:03:05 PST 2014
   This code was moved here from mmin() in wapp.c.

   Moving min; B(i) is min of A(i-n+1) to A(i).

   Fri Jun 13 05:05:43 PDT 2014.  Replace two loops with one.
   Testing: see _mmax() in this file. */
{
   register int i,k1,k2;
   double X;
   int kX;

 /*gprintf(" using _mmin\n");*/

   k2=0;
   X=*B=*A;
   kX=k2;
   k2++;
   while(k2<rows) {
      if(*(A+k2)<X) {
         X=*(B+k2)=*(A+k2); /* min is X at kX=k2 */
         kX=k2;
      }
      else {
         k1=k2-n+1;
         if(kX<k1) { /* min is too old; find new min X at kX<=k2 */
            X=*(A+k1);
            kX=k1;

            i=k1+1;
            while(i<=k2) {
               if(*(A+i)<X) {
                  X=*(A+i); /* min is X at kX<=k2 */
                  kX=i;
               }
               i++;
            }
         }
         *(B+k2)=X;
      }
    /*gprintf(" _mmin. k2 %d kX %d k1 %d A %d X %d B %d\n",\
      k2,kX,k1,(int)*(A+k2),(int)X,(int)*(B+k2));*/

      k2++;
   }
}

int modald_(double *c__, int *n, int *m, double *rr, double *ri, 
   double *b, double *a, int *mdc, int *mdb, int *mda, 
   double *ear, double *eai, int *mx, double *ar, double *ai, 
   int *nopt, int *imp, float *tol, double *pivots, int *kwork, 
   double *work, int *w)
/* modald.f -- translated by f2c (version 19991025). */

/* Sizes: kwork needs 1 column; work needs 3 columns */
{
    /* System generated locals */
    int c_dim1, c_offset, b_dim1, b_offset, a_dim1, a_offset, ear_dim1, 
     ear_dim2, ear_offset, eai_dim1, eai_dim2, eai_offset, ar_dim1, 
     ar_offset, ai_dim1, ai_offset, work_dim1, work_offset, i__1, i__2;
    double d__1;

    /* Local variables */
    static int ncol, irel, nvct;
    static float zero, tiny;
    static int multiple;
    static int i__, j, ncalc, ipass, n2, jc, ji, ki, ir;
    static float ridiff;
    static float rrdiff;
    static int jcm, irm;

/*         GIVEN M EIGENVALUES S = (RR,RI), THIS S/R PRESIDES OVER */
/*         THE CALCULATION OF THE REAL AND COMPLEX PARTS (EAR,EAI) */
/*         OF THE MODAL MATRIX OF THE N-DIMENSIONAL QUADRATIC SYSTEM */

/*                   [(A*S + B)*S + C]*(EAR,EAI) = 0 */

/*         IF NOPT = 1, A IS CONSIDERED NULL (AND NOT USED) AND C */
/*         IS GIVEN A NEGATIVE SIGN, SO THAT THE DYNAMIC SYSTEM IS */
/*         ASSUMED TO BE IN STATE SPACE FORM: */

/*                        [B*S - C]*(EAR,EAI) = 0 */

/*         IF IMP = 2, AMP/PHASE FORM OF MODE SHAPES IS CALCULATED */

    /* Parameter adjustments */
    --rr;
    --ri;
    c_dim1 = *mdc;
    c_offset = 1 + c_dim1 * 1;
    c__ -= c_offset;
    b_dim1 = *mdb;
    b_offset = 1 + b_dim1 * 1;
    b -= b_offset;
    a_dim1 = *mda;
    a_offset = 1 + a_dim1 * 1;
    a -= a_offset;
    work_dim1 = *mx;
    work_offset = 1 + work_dim1 * 1;
    work -= work_offset;
    --kwork;
    --pivots;
    ai_dim1 = *mx;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;
    ar_dim1 = *mx;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    eai_dim1 = *mx;
    eai_dim2 = *m;
    eai_offset = 1 + eai_dim1 * (1 + eai_dim2 * 1);
    eai -= eai_offset;
    ear_dim1 = *mx;
    ear_dim2 = *m;
    ear_offset = 1 + ear_dim1 * (1 + ear_dim2 * 1);
    ear -= ear_offset;

    /* Function Body */
    tiny = (float)1e-6;
    zero = (float)1e-6;
    if (*tol > (float)0.) {
 zero = *tol;
    }

/*         INITIALIZE */
    ipass = 1;
    n2 = *n;
/*     WRITE(W,51) */
/*  51 FORMAT(1H1) */
    ncol = 1;
    ncalc = 0;

/*         TOP OF MAIN LOOP */
    i__ = 1;
    while(i__ <= *m) {
 irel = 0;
 nvct = 1;

/*     Check for multiple roots.  Assumes roots are sorted so multiple */
/*     ones are next to each other. */
 j = i__ + 1;
 multiple = 1;
 while(j <= *m && multiple) {
     rrdiff = (d__1 = rr[i__] - rr[j], fabs(d__1));
     ridiff = (d__1 = ri[i__] - ri[j], fabs(d__1));
     if (rrdiff < tiny && ridiff < tiny) {
  ++nvct;
/* bump vector count */
     } else {
  multiple = 0;
     }
     ++j;
 }
/*         TEST TO SEE IF THIS ROOT IS A COMPLEX CONJUGATE OF THE LAST */
 if (ri[i__] == 0.) {
     goto L625;
 }
 if (i__ == 1) {
     goto L625;
 }
 ridiff = (d__1 = ri[i__ - 1] + ri[i__], fabs(d__1));
 if (ridiff < tiny) {
     i__1 = nvct;
     for (ji = 1; ji <= i__1; ++ji) {
  i__2 = n2;
  for (ki = 1; ki <= i__2; ++ki) {
      ai[ki + ji * ai_dim1] = -ai[ki + ji * ai_dim1];
  }
     }
     goto L633;
 }
/*         SET UP THE COMPLEX MATRIX REQUIRED TO COMPUTE VECTOR(S) */
L625:
 chmats_(&rr[i__], &ri[i__], &ar[ar_offset], &ai[ai_offset], mx, &a[
  a_offset], &b[b_offset], &c__[c_offset], mda, mdb, mdc, n, 
  nopt);

/*         SET REAL FLAG, IREL, IF SYSTEM IS REAL (I.E., RR(I) = 0): */
 if (rr[i__] == 0.) {
     irel = 1;
 }
/*         CALCULATE THE VECTOR(S) */
 vectrd_(&ar[ar_offset], &ai[ai_offset], &n2, mx, mx, &irel, &nvct, &
  zero, &pivots[1], &kwork[1], &work[work_dim1 + 1], &work[(
  work_dim1 << 1) + 1], w);

/*         PLACE THE EIGENVALUES AND VECTORS INTO THE READOUT MATRIX */
L633:
 i__1 = nvct;
 for (jc = 1; jc <= i__1; ++jc) {
     jcm = jc + ncol - 1;
     ear[(jcm + ipass * ear_dim2) * ear_dim1 + 1] = rr[i__];
     eai[(jcm + ipass * eai_dim2) * eai_dim1 + 1] = ri[i__];
     i__2 = n2;
     for (ir = 1; ir <= i__2; ++ir) {
  irm = ir + 1;
  ear[irm + (jcm + ipass * ear_dim2) * ear_dim1] = ar[ir + jc * 
   ar_dim1];
  eai[irm + (jcm + ipass * eai_dim2) * eai_dim1] = ai[ir + jc * 
   ai_dim1];
/* L637: */
     }
/* L638: */
 }
 ncol += nvct;
 ++ncalc;

/*         WRITE THE PIVOTS */
 if (*w > 0) {
/*
Message goes here
*/
 }

 if (ncol > *m) {
     goto L680;
 }
/* L675: */
 i__ += nvct;
    }
/*         BOTTOM OF MAIN LOOP */

L680:
    --ncol;
    return 1;
} /* modald_ */

int s3coef() /* s3coef (hX ht --- hC) */
/* Sat Apr  6 22:47:59 PDT 2013.  See "Testing 2" in lsqfit() (this 
   file) for a free-fall gravity test case using s3coef() and s3eval().

   Create coefficients for cubic spline fit to data in X at points t.

   Each column in X defines a curve at points t, so N-by-M matrix X
   and corresponding N-by-1 vector t define M curves at N points.

   The columnwise structure of returned coefficient matrix C is as
   follows:

      0  1  2  3  4   5  6  7  8   ...
      t; x1 b1 c1 d1; x2 b2 c2 d2; ... xM bM cM dM

   where xk is column k of X, and bk ck dk are the spline coefficients
   for column k of X, and M is the number of columns of X.

   Note: If rows of incoming X and t do not match, it is assumed that
   X contains data stored by columns and X is transposed to align rows
   of X to match rows of t. */
{
   register double *X,*C,*t;
   register int bytes,cols,j=0,k=1,rows;
   int ccols;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" s3coef: ",MATNOT2);
      return 0;
   }
   if((rows=tos->row)!=(tos-1)->row) {
      if((tos-1)->col==rows) {
         swap(); bend(); swap(); /* assuming X cols match t rows */
      }
      else {
         stkerr(" s3coef: ",ROWSNOT);
         return 0;
      }
   }
   bytes=rows*sizeof(double);
   cols=(tos-1)->col;
   X=(tos-1)->mat;
   t=tos->mat;

   if(!matstk(rows,(ccols=1+4*cols),"_s3coef")) return 0;
   C=tos->mat;
   memcpy(C,t,bytes);

   for(;j<cols;j++) {
      memcpy(C+locvec(k,rows),X+locvec(j,rows),bytes);
      k++;
      spline3(rows,t,X+locvec(j,rows),C+locvec(k,rows));
      k+=3;
   }
   return(lop() && lop());
}

int nlag() /* nlag (hA f --- hN) */
/* Sat Jan 18 09:20:33 PST 2014

   Given discrete time series A(t).  

   For incoming number f true, return N where N(k) is the row offset
   (number of steps) backward from k, into the past, to where value 
   A(k-N(k)) is greater than or equal to A(k).

   For incoming number f false, return N where N(k) is the row offset
   (number of steps) backward from k, into the past, to where value
   A(k-N(k)) is less than or equal to A(k).

   Initial N is indeterminate and N(0)=INF.  

   Also, if A(k) is equal to the minimum or maximum of A(i), (i=0,k), 
   then N(k) is indeterminate in the former case if f=false (no chance
   of finding smaller) and in the latter case if f=true (no chance of
   finding greater); in these cases value N(k)=INF.

   Testing Sat Jan 18 11:27:13 PST 2014: drop the following line at the
   ready prompt:
      syspath "../src/mmath.c" + "test_nlag___" msource

    \ Testing steps in a sine wave.

      test_nlag___
      "sine" missing IF "mmath.v" source THEN

    \ A = 1000 + 1000*sin(10*2pi*t):
      1 10 2pi * 0 0.001 200 (steps) 1+ (points) sine (hA ht)
      "t" book 1000 * 0.5 + integer 1000 + "A" book
      1 A rows items "r" book

    \ Using nlag:
      A no nlag "n1" book
      A yes nlag "n2" book

    \ Verify A(k)<=A(k-n1):
      A r n1 - xbase max reach "A1" book A A1 <= "f1" book 

    \ Verify A(k)>=A(k-n2):
      A r n2 - xbase max reach "A2" book A A2 >= "f2" book

    \ Display results:
      "    Case of f=false      Case of f=true"
      "k A n1 A1 f1" 0 A rows items int$ 
      A n1 A1 f1 4 parkn mtext park pile neat
      "A n2 A2 f2"
      A n2 A2 f2 4 parkn mtext pile neat 5 indent park pile iview

    \ Show verification:
      f1 totals abs @ f1 rows = \ all rows true
      IF " OK no nlag" ELSE " no nlag ERROR" THEN . nl
      f2 totals abs @ f2 rows = \ all rows true
      IF " OK yes nlag" ELSE " yes nlag ERROR" THEN . nl
      halt

   Results are correct by inspection. */
{
   register double *A;
   double *A0,Am,Ak,*N;

   register int i;
   int f,k=1,rows;

   if(!popint(&f)) return 0;

   if(tos->typ!=MAT) {
      stkerr(" nlag: ",MATNOT);
      return 0;
   }
   A0=tos->mat;
   Am=*A0;
   A0++;

   rows=tos->row;
   if(!matstk(rows,1,"_nlag")) return 0;
   N=tos->mat;
   memset(N,0,rows*sizeof(double));
   *N=INF;
   N++;

   if(f) { /* find N for A(N)>A(t) */
      for(;k<rows;k++) {
         Ak=*A0;
         Am=MAX(Ak,Am);
         A=A0-1;
         if(Am>Ak && Ak>*A) { /* dA(t)>0 */
          /*gprintf("  k %d Am %d Ak %d\n",k,(int)Am,(int)Ak);*/
            i=k-1;
            while(i>-1 && *A<Ak) {
             /*gprintf("    i %d A %d Ak %d\n",i,(int)*A,(int)Ak);*/
               A--;
               i--;
            }
            *N=k-i;
         }
         else {
            if(Ak==Am) *N=INF;
            else if(Ak>=*A) *N=1;
         }
         A0++;
         N++;
      }
   }
   else { /* find N for A(N)<A(t) */
      for(;k<rows;k++) {
         Ak=*A0;
         Am=MIN(Ak,Am);
         A=A0-1;
         if(Am<Ak && Ak<*A) { /* dA(t)<0 */
          /*gprintf("  k %d Am %d Ak %d\n",k,(int)Am,(int)Ak);*/
            i=k-1;
            while(i>-1 && *A>Ak) {
             /*gprintf("    i %d A %d Ak %d\n",i,(int)*A,(int)Ak);*/
               A--;
               i--;
            }
            *N=k-i;
         }
         else {
            if(Ak==Am) *N=INF;
            else if(Ak<=*A) *N=1;
         }
         A0++;
         N++;
      }
   }
   return(lop());
}

int s3eval() /* s3eval (hC ht --- hX hX' hX'') */
/* Evaluate cubic spline at points in t using coefficients C from
   s3coef().  Points in t need not match the original ones used to
   create coefficients C.

   The columnwise structure of coefficient matrix C is as follows:

      0   1   2   3   4    5   6   7   8    ...
      tf; Xf1 bf1 cf1 df1; Xf2 bf2 cf2 df2; ... XfM bfM cfM dfM

   where the first column, tf, contains points where Xf was fitted by
   s3coef(), Xfk is column k of fitted Xf, bfk cfk dfk are the spline
   coefficients of Xfk, and M is the number of columns (functions) in
   fitted Xf.

   Returned matrices X, X' and X'' will contain M columns for the M
   functions in C, and N rows for the N rows of incoming t. */
{
   double *bf,*cf,*df,dt,*t,*tf,*Xf;
   double *X,*Xd,*Xdd,*x,*xd,*xdd;
   int i=0,j,N,Nf,Nf4,M,r;
   char *roll4="4 roll";

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" s3eval: ",MATNOT2);
      return 0;
   }
   N=tos->row;
   M=(((tos-1)->col)-1)/4;
   Nf=(tos-1)->row;
   Nf4=Nf*4;

   t=tos->mat;

   tf=(tos-1)->mat; /* first column of C */

   if(!matstk(N,M,"_X")) return 0;
   X=tos->mat;
   if(!matstk(N,M,"_Xd")) return 0;
   Xd=tos->mat;
   if(!matstk(N,M,"_Xdd")) return 0;
   Xdd=tos->mat;

   for(;i<N;i++) {
      bsearchd(*(t+i),tf,Nf,&r); /* nearest-below ti in tf */
      dt=*(t+i)-*(tf+r);
      x=X;
      xd=Xd;
      xdd=Xdd;

      Xf=tf+Nf;
      bf=Xf+Nf;
      cf=bf+Nf;
      df=cf+Nf;

      /* Evaluation at ti:
          x = Xf(r) + dt*(bf(r) + dt*(cf(r) + dt*df(r)))
         xd = bf(r) + dt*(2*cf(r) + 3*dt*df(r))
        xdd = 2*cf(r) + 6*dt*df(r) */

      for(j=0;j<M;j++) {
         *x=*(Xf+r) + dt*(*(bf+r) + dt*(*(cf+r) + dt*(*(df+r))));
         *xd=*(bf+r) + dt*(*(cf+r)*2 + 3*dt*(*(df+r)));
         *xdd=2*(*(cf+r)) + 6*dt*(*(df+r));

         Xf+=Nf4;
         bf+=Nf4;
         cf+=Nf4;
         df+=Nf4;

         x+=N;
         xd+=N;
         xdd+=N;
      }
      X++;
      Xd++;
      Xdd++;
   }
   pushq2(roll4,strlen(roll4));
   main1();
   pushq2(roll4,strlen(roll4));
   main1();
   return(drop2());
}

int skyline() /* skyline (hA --- hA1) */
/* A1 is flat at all peaks of A columns.

   Peaks in columns of A are where:
      the value is greater than or equal to the previous
      the value is greater than or equal to the next
      and both of these are true.

   This word replaces high level skyline now in the Appendix of math.v
   and works the same except for differences due to using greater-than-
   or-equal tests rather than greater-than tests. */
{
   double *A,*R;
   int cols,i,j=0,rows;

   if(tos->typ!=MAT) {
      stkerr(" skyline: ",MATNOT);
      return 0;
   }
   cols=tos->col;
   rows=tos->row;
   lpush(); /* put A on local stack */

   for(;j<cols;j++) {
      lpeek();
      pushint(XBASE+j);
      catch(); /* column Aj on stack */
      A=tos->mat;
      A++;

      if(!matstk(rows,1,"_R")) return 0;
      R=tos->mat;
      *R=0;
      R++;

      for(i=1;i<rows-1;i++) {
         if(*A>=*(A-1) && *A>=*(A+1)) *R=1;
         else *R=0;
         A++;
         R++;
      }
      *R=0;
      looking(); /* (hAj hR --- hA1j) */
   }
   pushint(cols);
   return(parkn() && lpull() && drop());
}

int solve() /* solve (hb hA --- hx) */
/* Solve A*x=b for x. */
{
#if defined(LAPACK) || defined(ESSL)

 #ifdef LAPACK

   if(is_complex(tos)) return(zgesv1());
   else return(dgesv1());

 #endif

 #ifdef ESSL

   return(solveSSL()); /* b and A can be complex */

 #endif

#else

/* Solve A*x=b for x using gaussian elimination. */

   double *A,*b=NULL,determ;
   int cols,*index,ising,rows;

   if(tos->typ!=MAT || (tos-1)->typ!=MAT) {
      stkerr(" solve: ",MATSNOT);
      return 0;
   }
   rows=tos->row;
   if(tos->col!=rows) {
      stkerr(" solve: ","matrix A is not square");
      return 0;
   }
   if((tos-1)->col>0 && (tos-1)->row!=rows) {
      stkerr(" solve: ","matrix b not compatible with A");
      return 0;
   }
   cols=(tos-1)->col;

   A=tos->mat;
   if(!matstk(rows,rows,"_A")) return 0;
   memcpy(tos->mat,A,sizeof(double)*rows*rows);
   A=tos->mat;

   if(cols) {
      b=(tos-2)->mat;
      if(!matstk(rows,cols,"_x")) return 0;
      memcpy(tos->mat,b,sizeof(double)*rows*cols);
      b=tos->mat;
   }
   if((index=(int *)malloc(1+rows*3*sizeof(int)))==NULL) {
      stkerr(" solve: ",MEMNOT);
      return 0;
   }

   if(!inverd_(&rows,A,&rows,b,&cols,&determ,&ising,index)) {
      gprintf(" solve: singular matrix"); nc();
      mallfree((void *)&index);
      if(cols) drop2();
      else drop();
      return 0;
   }
   mallfree((void *)&index);
   if(cols) return(lop() && lop() && lop());
   else return(lop() && lop());

#endif
}


void spline3(int n, double *x, double *y, double *coef)
/* Computing the coefficients for fitting a cubic spline to n data
   point pairs (x,y); points x are in ascending order.

   Reference: Forsythe, G. E. and Michael A. Malcolm, "Computer
   Methods for Mathematical Computations," Prentice-Hall, Inc.,
   1977.

   The spline representation s(x) is

      s(x) = y(i) + b(i)*(x - x(i)) + c(i)*(x - x(i))**2
             + d(i)*(x - x(i))**3

   for x(i) .ge. x .le. x(i+1). */
{
   register int i=0,nm1;
   register double *b,*c,*d,t;

   b=coef;
   c=b+n;
   d=c+n;

   if(n<2) {
      *b=0;
      *c=0;
      *d=0;
      return;
   }
   if(n<3) {
      *b=(*(y+1) - *y)/(*(x+1)- *x);
      *c=0;
      *d=0;
      *(b+1)=*b;
      *(c+1)=0;
      *(d+1)=0;
      return;
   }
   nm1=n-1;

/* Setting up tridiagonal system
      b = diagonal, d = offdiagonal, c = right hand side */
   *d=*(x+1)-*x;
   *(c+1)=(*(y+1)-*y)/(*d);

   for(i=1;i<nm1;i++) {
      *(d+i)=*(x+i+1)-*(x+i);
      *(b+i)=2*(*(d+i-1)+*(d+i));
      *(c+i+1)=(*(y+i+1)-*(y+i))/(*(d+i));
      *(c+i)=*(c+i+1)-*(c+i);
   }
/* End conditions; third derivatives at x(1) and x(n)
   obtained from divided differences (see reference): */
   *b=-*d;
   *(b+nm1)=-*(d+nm1-1);
   *c=0;
   *(c+nm1)=0;
   if(n>3) {
      *c=*(c+2)/(*(x+3)-*(x+1)) - *(c+1)/(*(x+2)-*x);
      *(c+nm1)=*(c+nm1-1)/(*(x+nm1) - *(x+nm1-2)) \
               - *(c+nm1-2)/(*(x+nm1-1)-*(x+nm1-3));
      *c=*c*((*d)*(*d))/(*(x+3)-*x);
      *(c+nm1)=-*(c+nm1)*(*(d+nm1-1)*(*(d+nm1-1))) \
               /(*(x+nm1)-*(x+nm1-3));
   }
/* Forward elimination: */
   for(i=1;i<n;i++) {
      t=*(d+i-1)/(*(b+i-1));
      *(b+i)=*(b+i) - t*(*(d+i-1));
      *(c+i)=*(c+i) - t*(*(c+i-1));
   }
/* Back substitution: */
   *(c+nm1)=*(c+nm1)/(*(b+nm1));
   for(i=nm1-1;i>=0;i--) {
      *(c+i)=(*(c+i) - *(d+i)*(*(c+i+1)))/(*(b+i));
   }
/* c(i) is now sigma(i) of the reference;
      computing polynomial coefficients: */
   *(b+nm1)=(*(y+nm1) - *(y+nm1-1))/(*(d+nm1-1)) \
            + *(d+nm1-1)*(*(c+nm1-1) + *(c+nm1)*2);
   for(i=0;i<nm1;i++) {
      *(b+i)=(*(y+i+1) - *(y+i))/(*(d+i)) \
             - *(d+i)*(*(c+i+1) + *(c+i)*2);
      *(d+i)=(*(c+i+1) - *(c+i))/(*(d+i));
      *(c+i)=*(c+i)*3;
   }
   *(c+nm1)=*(c+nm1)*3;
   *(d+nm1)=*(d+nm1-1);

   return;
}

int vectrd_(double *ar, double *ai, int *n, int *nrdim, int *nidim,
  int *irel, int *nvct, float *zro, double *pivots, int *ncol,
  double *wr, double *wi, int *iout)
/* vectrd.f -- translated by f2c (version 19991025). */
{
    /* System generated locals */
    int ar_dim1, ar_offset, ai_dim1, ai_offset, i__1, i__2, i__3;
    double d__1;

    /* Local variables */
    static float amax;
    static double temp, tsti, tstr;
    static int i__, j, k, l, m, icame;
    static float aiwas, arwas;
    static int ipass;
    static double pvtsz, fc, fd;
    static int nc, is;
    static float aitest;
    static int nczero;
    static float artest;
    static int km1, nvctnu;
    static int nmk, iro;

/*         THIS SUBROUTINE FINDS NVCT EIGENVECTORS FOR THE MATRIX */
/*         AR + I*AI.  THE VECTORS ARE RETURNED IN THE LEFT COLUMNS */
/*         OF AR AND AI.  ZERO PIVOT IS ENCOUNTERED EARLIER THAN */
/*         DEFINED BY NVCT. */
/*         IF THE PARTICULAR PROBLEM BEING SOLVED INVOLVES A DISTINCT */
/*         EIGENVALUE, THE INDEX NVCT WILL BE UNITY.  ZRO INDICATES */
/*         THE PRACTICAL SIZE OF ZERO.  DIFFERENCES TAKEN WHICH ARE */
/*         LESS THAN ZRO TIMES ONE OF THE SUBTRACTED VARIABLES ARE SET */
/*         TO ZERO. */
/*         THE ARRAY, PIVOTS, PRESENTS THE SQUARES OF THE MODULI OF THE */
/*         PIVOTS.  MATRICES AR AND AI ARE NXN, DIMENSIONED TO NRDIM */
/*         AND NIDIM ROWS, RESPECTIVELY, IN THE MAIN PROGRAM.  IF AI IS */
/*         NULL, IREL = 1 TO INDICATE THAT THE SYSTEM IS REAL. */


/*         DECLARE DOUBLE PRECISION FUNCTIONS: */

    /* Parameter adjustments */
    --wi;
    --wr;
    --ncol;
    --pivots;
    ar_dim1 = *nrdim;
    ar_offset = 1 + ar_dim1 * 1;
    ar -= ar_offset;
    ai_dim1 = *nidim;
    ai_offset = 1 + ai_dim1 * 1;
    ai -= ai_offset;

    /* Function Body */
    nvctnu = *nvct;
    i__1 = *n;
    for (i__ = 1; i__ <= i__1; ++i__) {
/* L100: */
 pivots[i__] = 0.;
    }
    i__1 = *n;
    for (ipass = 1; ipass <= i__1; ++ipass) {
 k = *n - ipass + 1;

/*         SCAN THE KXK SUBMATRIX FOR THE PIVOT */
 pvtsz = 0.;
 ncol[ipass] = 0;
 l = 0;
 i__2 = k;
 for (i__ = 1; i__ <= i__2; ++i__) {
     i__3 = k;
     for (j = 1; j <= i__3; ++j) {
  if (*irel == 1) {
      temp = ar[i__ + j * ar_dim1] * ar[i__ + j * ar_dim1];
  }
  if (*irel != 1) {
      temp = ar[i__ + j * ar_dim1] * ar[i__ + j * ar_dim1] + ai[
       i__ + j * ai_dim1] * ai[i__ + j * ai_dim1];
  }
  if (temp < pvtsz) {
      goto L149;
  }
  ncol[ipass] = j;
  pvtsz = temp;
  l = i__;
L149:
  ;
     }
/* L150: */
 }

/*         SAVE THE PIVOT */
 pivots[ipass] = pvtsz;
/* L155: */

/*         CHECK IF UPPER LEFT K BY K MATRIX IS NULL (I.E., IF THE */
/*         PIVOT HAS VANISHED). */
 amax = sqrt(pvtsz);
 if (amax <= (float)1e-6) {
     goto L1001;
 }

 if (k > *nvct) {
     goto L160;
 }

/*         IF HERE, THERE ARE LESS THAN NVCT VECTORS FOR THE REPEATED */
/*         EIGENVALUE.  WE WILL CONTINUE THE PROCESS UNTIL THE PIVOT */
/*         DOES VANISH, OR K = 1. */
/* moved here, below zero pivot test 7/7/ */

/* 
Message goes here 
 dmxout_(&ar[ar_offset], n, n, nrdim, iout, "AR  ", n, &k, &nvctnu, (
  ftnlen)4);
 dmxout_(&ai[ai_offset], n, n, nidim, iout, "AI  ", n, &k, &nvctnu, (
  ftnlen)4);
*/

 if (k == 1) {
     goto L1001;
 }
 --nvctnu;

L160:

/*         EXCHANGE ROWS AND COLUMNS TO MOVE THE L,NCOL(IPASS) ELEMENT */
/*         TO THE PIVOTAL POSITION - POSITION K,K.  EXCHANGE ROWS FIRST. */
 i__2 = k;
 for (j = 1; j <= i__2; ++j) {
     temp = ar[k + j * ar_dim1];
     ar[k + j * ar_dim1] = ar[l + j * ar_dim1];
     ar[l + j * ar_dim1] = temp;
     if (*irel == 1) {
  goto L170;
     }
     temp = ai[k + j * ai_dim1];
     ai[k + j * ai_dim1] = ai[l + j * ai_dim1];
     ai[l + j * ai_dim1] = temp;
L170:
     ;
 }

/*         EXCHANGE COLUMNS. */
 m = ncol[ipass];
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     temp = ar[i__ + k * ar_dim1];
     ar[i__ + k * ar_dim1] = ar[i__ + m * ar_dim1];
     ar[i__ + m * ar_dim1] = temp;
     if (*irel == 1) {
  goto L180;
     }
     temp = ai[i__ + k * ai_dim1];
     ai[i__ + k * ai_dim1] = ai[i__ + m * ai_dim1];
     ai[i__ + m * ai_dim1] = temp;
L180:
     ;
 }

/*         DIVIDE THE PIVOTAL ROW BY THE PIVOT.  INSERT THE ROW INTO */
/*         THE WORKING MATRIX. */
 fc = ar[k + k * ar_dim1] / pvtsz;
 fd = ai[k + k * ai_dim1] / pvtsz;
 km1 = k - 1;
 i__2 = km1;
 for (j = 1; j <= i__2; ++j) {
     if (*irel == 1) {
  goto L185;
     }
     temp = ar[k + j * ar_dim1];
     ar[k + j * ar_dim1] = ar[k + j * ar_dim1] * fc + ai[k + j * 
      ai_dim1] * fd;
     ai[k + j * ai_dim1] = ai[k + j * ai_dim1] * fc - temp * fd;
     wr[j] = ar[k + j * ar_dim1];
     wi[j] = ai[k + j * ai_dim1];
     goto L190;
L185:
     ar[k + j * ar_dim1] *= fc;
     wr[j] = ar[k + j * ar_dim1];
L190:
     ;
 }
 ar[k + k * ar_dim1] = 1.;
 ai[k + k * ai_dim1] = 0.;

/*         REDUCE THE KTH COLUMN TO ZERO */
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     if (i__ == k) {
  goto L220;
     }
     if (ar[i__ + k * ar_dim1] != 0.) {
  goto L197;
     } else {
  goto L195;
     }
L195:
     if (*irel == 1) {
  goto L220;
     }
     if (ai[i__ + k * ai_dim1] != 0.) {
  goto L197;
     } else {
  goto L220;
     }
L197:
     i__3 = km1;
     for (j = 1; j <= i__3; ++j) {
  tstr = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  if (*irel == 1) {
      goto L207;
  }
  tsti = (d__1 = ai[i__ + j * ai_dim1], fabs(d__1));
  ar[i__ + j * ar_dim1] = ar[i__ + j * ar_dim1] - ar[i__ + k * 
   ar_dim1] * ar[k + j * ar_dim1] + ai[i__ + k * ai_dim1]
    * ai[k + j * ai_dim1];
  ai[i__ + j * ai_dim1] = ai[i__ + j * ai_dim1] - ai[i__ + k * 
   ai_dim1] * ar[k + j * ar_dim1] - ar[i__ + k * ar_dim1]
    * ai[k + j * ai_dim1];
  temp = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  artest = *zro * tstr;
  arwas = ar[i__ + j * ar_dim1];
  if (temp < artest) {
      ar[i__ + j * ar_dim1] = 0.;
  }
  temp = (d__1 = ai[i__ + j * ai_dim1], fabs(d__1));
  aitest = *zro * tsti;
  aiwas = ai[i__ + j * ai_dim1];
  if (temp < aitest) {
      ai[i__ + j * ai_dim1] = 0.;
  }
  goto L209;
L207:
  ar[i__ + j * ar_dim1] -= ar[i__ + k * ar_dim1] * ar[k + j * 
   ar_dim1];
  temp = (d__1 = ar[i__ + j * ar_dim1], fabs(d__1));
  artest = *zro * tstr;
  arwas = ar[i__ + j * ar_dim1];
  if (temp < artest) {
      ar[i__ + j * ar_dim1] = 0.;
  }
L209:

  if (ar[i__ + j * ar_dim1] == 0. && arwas != 0. && *iout > 0) {

/* Message here */
  }
  if (*irel == 1) {
      goto L210;
  }

  if (ai[i__ + j * ai_dim1] == 0. && aiwas != 0. && *iout > 0) {
/* Message here */
  }

L210:
  ;
     }
     ar[i__ + k * ar_dim1] = 0.;
     ai[i__ + k * ai_dim1] = 0.;
L220:
     ;
 }

/* L1000: */
    }

L1001:

/*         K VECTORS CAN BE FORMED.  THEY WILL OCCUPY THE LEFT COLUMNS */
/*         OF MARTICES AR AND AI. */
    i__1 = k;
    for (j = 1; j <= i__1; ++j) {
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     if (i__ == j) {
  ar[i__ + j * ar_dim1] = 1.;
     }
     if (i__ <= k) {
  ai[i__ + j * ai_dim1] = 0.;
     }
     if (i__ <= k && i__ != j) {
  ar[i__ + j * ar_dim1] = 0.;
     }
     if (i__ <= k) {
  goto L325;
     }
     ar[i__ + j * ar_dim1] = -ar[i__ + j * ar_dim1];
     ai[i__ + j * ai_dim1] = -ai[i__ + j * ai_dim1];
L325:
     ;
 }
/* L350: */
    }
    if (nvctnu == *nvct) {
 goto L355;
    }

/*         ZERO OUT THE OTHER COLUMNS, MAKING THEM TRIVIAL VECTORS. */
    nczero = *nvct - nvctnu;
    i__1 = nczero;
    for (j = 1; j <= i__1; ++j) {
 nc = nvctnu + j;
 i__2 = *n;
 for (i__ = 1; i__ <= i__2; ++i__) {
     ar[i__ + nc * ar_dim1] = 0.;
/* L352: */
     ai[i__ + nc * ai_dim1] = 0.;
 }
    }
L355:

/*         ARRANGE THE VECTORS IN PROPER ORDER (ACCOUNT FOR COLUMNAL */
/*         PIVOTING). */
    i__2 = k;
    for (j = 1; j <= i__2; ++j) {
 nmk = *n - k;
 i__1 = nmk;
 for (i__ = 1; i__ <= i__1; ++i__) {
     is = nmk - i__ + 1;
     icame = ncol[is];
     iro = k + i__;
     temp = ar[iro + j * ar_dim1];
     ar[iro + j * ar_dim1] = ar[icame + j * ar_dim1];
     ar[icame + j * ar_dim1] = temp;
     if (*irel == 1) {
  goto L360;
     }
     temp = ai[iro + j * ai_dim1];
     ai[iro + j * ai_dim1] = ai[icame + j * ai_dim1];
     ai[icame + j * ai_dim1] = temp;
L360:
     ;
 }
/* L370: */
    }
    return 0;
} /* vectrd_ */
