/* {{{1 GNU General Public License

Program Tops - a stack-based computing environment
Copyright (C) 1999-2005  Dale R. Williamson

Author: Dale R. Williamson <dale.williamson@prodigy.net>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software Foundation,
Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
1}}} */

#ifdef MLAB
/* matlab.c  April 2001

   Copyright (c) 2001   D. R. Williamson

   Reference:
      "Matlab Application Program Interface Guide, Version 5,"
      The Math Works, January 1997.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "main.h"
#include "stk.h"

#include "matlab.h"

int _engClose() /* _engClose (ep --- f) */
/* Closing the Matlab engine.  Returns f true (-1) if error. */
{
   union {
      double d;
      Engine *ep;
   } eng;

   if(tos->typ!=NUM) {
      stkerr(" _engClose: ",NUMNOT);
      return 0;
   }
   popd(&eng.d);

   if(eng.d) {
      if(engClose(eng.ep)) {
         stkerr(" _engClose: ","can't close Matlab engine");
         pushint(-1);
         return 0;
      }
      else return(pushint(0));
   }
   else {
      stkerr(" _engClose: ","Matlab engine is not on");
      pushint(-1);
      return 0;
   }
}

int _engEval() /* _engEval (qS ep --- f) */
/* Evaluating string S in Matlab.  Returns f true (-1) if error. */
{
   union {
      double d;
      Engine *ep;
   } eng;

   if(tos->typ!=NUM) {
      stkerr(" _engEval: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" _engEval: ",STRNOT);
      return 0;
   }
   if(!tos->real) {
      stkerr(" _engEval: ","Matlab engine is not on");
      return 0;
   }
   popd(&eng.d);

   if(TRACE) {
      gprintf(" Executing string in Matlab: %s",tos->tex);
      nc();
   }
   if(engEvalString(eng.ep,tos->tex)) {
      stkerr(" engEval: ","error executing string in Matlab");
      drop();
      pushint(-1);
      return 0;
   }
   return(drop() && pushint(0));
}

int _engGet() /* _engGet (qA ep --- -1 | hA 0) */
/* Getting matrix A from Matlab.  Returns f true (-1) if error. */
{
   union {
      double d;
      Engine *ep;
   } eng;
   int bad,cols,len,rows;
   double *Ai,*Ar;
   char *p;

   if(tos->typ!=NUM) {
      stkerr(" _engGet: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=STR) {
      stkerr(" _engGet: ",STRNOT);
      return 0;
   }
   if(!tos->real) {
      stkerr(" _engGet: ","Matlab engine is not on");
      return 0;
   }
   popd(&eng.d);

   if((p=malloc(3+(len=tos->col)))==NULL) {
      stkerr(" _engGet: ",MEMNOT);
      return 0;
   }
   *(p)='_';
   memcpy(p+1,tos->tex,len);
   *(p+1+len)='\0';
   drop(); /* qA off the stack */

   bad=engGetFull(eng.ep,p+1,&rows,&cols,&Ar,&Ai);

   if(bad) {
      stkerr(" _engGet: ","error getting matrix from Matlab");
      free(p);
      pushint(-1);
      return 0;
   }
   if(!Ar && !Ai) {
      stkerr(" _engGet: ","matrix not found in Matlab");
      free(p);
      pushint(-1);
      return 0;
   }
   if(Ar) {
      if(Ai) {
         *(p+1+len)='r';
         *(p+2+len)='\0';
      }
      if(!matstk(rows,cols,p)) {
         mxFree(Ar);
         mxFree(Ai);
         free(p);
         pushint(-1);
         return 0;
      }
      memcpy(tos->mat,Ar,rows*cols*sizeof(double));
      mxFree(Ar);
   }
   if(Ai) {
      *(p+1+len)='i';
      *(p+2+len)='\0';
      if(!matstk(rows,cols,p)) {
         mxFree(Ar);
         mxFree(Ai);
         free(p);
         pushint(-1);
         return 0;
      }
      memcpy(tos->mat,Ai,rows*cols*sizeof(double));
      mxFree(Ai);
   }
   free(p);
   return(pushint(0));
}

int _engOpen() /* _engOpen (qS --- ep) */
/* Starting Matlab engine; returned engine pointer ep is zero if error.

   Strings to matlab function engOpen:

      Start the matlab engine locally by executing the string "matlab"
      which is the default if string \0 is received.

      To start matlab on a remote host, use the name of the host.

      For general case, use any string and it will be executed 
      literally to start matlab.  Example (quote in a single line):

         "rsh bach 'export DISPLAY=141.148.32.48:0.0; 
            /usr/local/matlab/bin/matlab'"
*/
{
   union {
      double d;
      Engine *ep;
   } eng;

   if(tos->typ!=STR) {
      stkerr(" _engOpen: ",STRNOT);
      return 0;
   }
   if(!(eng.ep=engOpen(tos->tex))) {
      stkerr(" _engOpen: ","can't start Matlab engine");
      drop();
      pushint(0);
      return 0;
   }
   return(drop() && pushd(eng.d));
}

int _engPut() /* _engPut (hA ep --- f) */
/* Sending matrix A to Matlab.  Returns f true (-1) if error. */
{
   union {
      double d;
      Engine *ep;
   } eng;
   int bad;
   mxArray *A;

   if(tos->typ!=NUM) {
      stkerr(" _engPut: ",NUMNOT);
      return 0;
   }
   if((tos-1)->typ!=MAT) {
      stkerr(" _engPut: ",MATNOT);
      return 0;
   }
   if(!tos->real) {
      stkerr(" _engPut: ","Matlab engine is not on");
      return 0;
   }
   popd(&eng.d);

   A=mxCreateDoubleMatrix(tos->row,tos->col,mxREAL);

   memcpy(mxGetPr(A),tos->mat,((tos->row*tos->col)*sizeof(double)));

   pushq2("named notag",11); xmain(0); /* firing named also drops hA */
   mxSetName(A,tos->tex);
   drop(); /* name string off stack */

   bad=engPutArray(eng.ep,A);
   if(bad) {
      stkerr(" _engPut: ","error sending matrix to Matlab");
      pushint(-1);
      return 0;
   }
   else {
      mxDestroyArray(A);
      return(pushint(0));
   }
}
#endif
